#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <poll.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <sys/asound_common.h>
#if defined(__QNX__)
#include <gulliver.h>
#elif defined(__LINUX__)
#include <endian.h>
#define ENDIAN_LE16(x) htole16(x)
#define ENDIAN_LE32(x) htole32(x)
#endif

#define FIXED_SIZE      3072

#define UNIMPLEMENTED   printf("%s not implemented\n", __FUNCTION__);

typedef struct io_audio_info {
    snd_pcm_ioplug_t ext;
    snd_pcm_t *pcm;
    int fd;
    int channel_flip;
    int bps;
    int bytes_written;
    unsigned int voices;
    const char *filename;
    snd_pcm_stream_t stream;
    snd_pcm_channel_info_t info;
} io_audio_info_t;

struct
{
    char        riff_id[4];
    uint32_t    wave_len;
    struct
    {
        char        fmt_id[8];
        uint32_t    fmt_len;
        struct
        {
            uint16_t    format_tag;
            uint16_t    voices;
            uint32_t    rate;
            uint32_t    char_per_sec;
            uint16_t    block_align;
            uint16_t    bits_per_sample;
        }
        fmt;
        struct
        {
            char        data_id[4];
            uint32_t    data_len;
        }
        data;
    }
    wave;
}
riff_hdr =
{
    {'R', 'I', 'F', 'F' },
    sizeof (riff_hdr.wave),
    {
        {'W', 'A', 'V', 'E', 'f', 'm', 't', ' '    },
        sizeof (riff_hdr.wave.fmt),
        {
            1, 0, 0, 0, 0, 0
        },
        {
            {'d', 'a', 't', 'a' },
            0,
        }
    }
};

static void file_free(io_audio_info_t *info)
{
    free(info);
}

static int file_start(snd_pcm_ioplug_t *io)
{
    return 0;
}

static snd_pcm_sframes_t file_pointer(snd_pcm_ioplug_t *io)
{
    return 0;
}

snd_pcm_sframes_t file_transfer_out(snd_pcm_ioplug_t *io,
                    const snd_pcm_channel_area_t *areas,
                    snd_pcm_uframes_t offset,
                    snd_pcm_uframes_t size)
{
    int ret;
    int target;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    target = size * info->voices * info->bps / _BITS_BYTE;
#if defined FIXED_SIZE
    if( target % FIXED_SIZE == 0 ) {
        ret = write(info->fd, areas->addr + (areas->first + areas->step * offset) / 8, target);
    } else {
        ret = 0;
    }
#else
    ret = write(info->fd, areas->addr + (areas->first + areas->step * offset) / 8, target);
#endif
    info->bytes_written += ret;
    return ret == target ? size : ret * size / target;
}

snd_pcm_sframes_t file_transfer_in(snd_pcm_ioplug_t *io,
                    const snd_pcm_channel_area_t *areas,
                    snd_pcm_uframes_t offset,
                    snd_pcm_uframes_t size)
{
    printf("Capture not yet supported\n");
    return 0;
}

static int file_hw_params(snd_pcm_ioplug_t *io, snd_pcm_hw_params_t *params)
{
    int ret;
    unsigned int rate_num, rate_den;
    snd_pcm_access_t access;
    snd_pcm_format_t format;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    if( (ret = snd_pcm_hw_params_get_access(params, &access)) ) {
        return ret;
    }

    if( (ret = snd_pcm_hw_params_get_rate_numden(params, &rate_num, &rate_den)) ) {
        return ret;
    }

    // Rate must be an integer at this point
    if( rate_den != 1 ) {
        return -EINVAL;
    }

    if( (ret = snd_pcm_hw_params_get_channels(params, &info->voices)) ) {
        return ret;
    }
    if( (ret = snd_pcm_hw_params_get_format(params, &format)) ) {
        return ret;
    }

    // Open the file
    if( info->fd ) {
        close( info->fd );
    }
    info->fd = open(info->filename, O_RDWR | O_CREAT | O_TRUNC, 0700);
    if(info->fd == -1 ) {
        info->fd = 0;
        return -errno;
    }

    info->bps = snd_pcm_format_width(format);

    riff_hdr.wave.fmt.voices = ENDIAN_LE16 (info->voices);
    riff_hdr.wave.fmt.rate = ENDIAN_LE32 (rate_num);
    riff_hdr.wave.fmt.char_per_sec =
        ENDIAN_LE32 (rate_num * info->voices * info->bps / 8);
    riff_hdr.wave.fmt.block_align = ENDIAN_LE16 (info->voices * info->bps / 8);
    riff_hdr.wave.fmt.bits_per_sample = ENDIAN_LE16 (info->bps);
    riff_hdr.wave.data.data_len = ENDIAN_LE32 (0);
    riff_hdr.wave_len = ENDIAN_LE32 (0 + sizeof (riff_hdr) - 8);
    write(info->fd, &riff_hdr, sizeof (riff_hdr));

    return 0;
}

static int file_sw_params(snd_pcm_ioplug_t *io, snd_pcm_sw_params_t *params)
{
    return 0;
}

static snd_pcm_chmap_query_t **file_query_chmaps(snd_pcm_ioplug_t *io)
{
    snd_pcm_chmap_query_t **map = malloc(sizeof(snd_pcm_chmap_query_t *)*3);
    if( map == NULL ) return NULL;

    map[0] = malloc(sizeof(snd_pcm_chmap_query_t) + sizeof(int) * 2);
    map[1] = malloc(sizeof(snd_pcm_chmap_query_t) + sizeof(int) * 2);
    map[2] = NULL;
    if( map[0] == NULL || map[1] == NULL ) {
        free(map[0]);
        free(map[1]);
        free(map);
        return NULL;
    }

    map[0]->type = SND_CHMAP_TYPE_FIXED;
    map[0]->map.channels = 2;
    map[0]->map.pos[0] = SND_CHMAP_FL;
    map[0]->map.pos[1] = SND_CHMAP_FR;
    map[1]->type = SND_CHMAP_TYPE_FIXED;
    map[1]->map.channels = 2;
    map[1]->map.pos[0] = SND_CHMAP_FR;
    map[1]->map.pos[1] = SND_CHMAP_FL;

    return map;
}

static snd_pcm_chmap_t *file_get_chmap(snd_pcm_ioplug_t *io)
{
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    snd_pcm_chmap_t *map = malloc(sizeof(snd_pcm_chmap_t) + sizeof(int) * 2);
    if( map == NULL ) return NULL;
    map->channels = 2;
    if( info->channel_flip ) {
        map->pos[0] = SND_CHMAP_FR;
        map->pos[1] = SND_CHMAP_FL;
    } else {
        map->pos[0] = SND_CHMAP_FL;
        map->pos[1] = SND_CHMAP_FR;
    }
    return map;
}

static int file_set_chmap(snd_pcm_ioplug_t *io, const snd_pcm_chmap_t *map)
{
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    if( map->channels == 2 && map->pos[0] == SND_CHMAP_FL && map->pos[1] == SND_CHMAP_FR) {
        info->channel_flip = 0;
    } else if( map->channels == 2 && map->pos[0] == SND_CHMAP_FL && map->pos[1] == SND_CHMAP_FR) {
        info->channel_flip = 1;
    } else {
        return -EINVAL;
    }

    return EOK;
}

static int file_close(snd_pcm_ioplug_t *io)
{
    int32_t size;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    // Write RIFF size
    lseek(info->fd, 4, SEEK_SET);
    size = ENDIAN_LE32(info->bytes_written + sizeof(riff_hdr) - 8);
    write(info->fd, &size, 4);

    // Write data size
    lseek(info->fd, sizeof(riff_hdr) - 4, SEEK_SET);
    size = ENDIAN_LE32(info->bytes_written);
    write(info->fd, &size, 4);

    close(info->fd);

    snd_pcm_ioplug_delete( io );

    file_free(info);

    return EOK;
}

snd_pcm_ioplug_callback_t file_playback_callback = {
    .start = file_start,
    .pointer = file_pointer,
    .transfer = file_transfer_out,
    .close = file_close,
    .hw_params = file_hw_params,
    .sw_params = file_sw_params,
    .query_chmaps = file_query_chmaps,
    .get_chmap = file_get_chmap,
    .set_chmap = file_set_chmap
};

SND_PCM_PLUGIN_DEFINE_FUNC(file)
{
    struct io_audio_info *info;
    int err;
    unsigned int val;
    long channels = 2;
    long rate = 48000;
    snd_config_t *result;

    info = calloc(1, sizeof(*info));
    if (info == NULL)
        return -ENOMEM;

    if( stream != SND_PCM_STREAM_PLAYBACK ) {
        free( info);
        return -EINVAL;
    }
    info->ext.version = SND_PCM_IOPLUG_VERSION;
    info->ext.name = "file";
    info->ext.callback = &file_playback_callback;
    info->bytes_written = 0;
    info->stream = stream;
    info->ext.private_data = info;
    info->filename = "file.wav";

    if( snd_config_search(conf, "filename", &result) == EOK ) {
        snd_config_get_string(result, &info->filename);
    }
    if( snd_config_search(conf, "rate", &result) == EOK ) {
        snd_config_get_integer(result, &rate);
    }
    if( snd_config_search(conf, "channels", &result) == EOK ) {
        snd_config_get_integer(result, &channels);
    }

    err = snd_pcm_ioplug_create(&info->ext, name, stream, mode);
    if (err < 0) {
        file_free(info);
        return err;
    }

    // Set capabilities
    val = SND_PCM_ACCESS_RW_INTERLEAVED;
    snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_ACCESS, 1, &val);
    val = SND_PCM_FORMAT_S16_LE;
    snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_FORMAT, 1, &val);
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_CHANNELS, channels, channels);
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_RATE, rate, rate);

#if defined(FIXED_SIZE)
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_PERIOD_BYTES,
            FIXED_SIZE,
            FIXED_SIZE);
#endif

    *pcmp = info->ext.pcm;
    return 0;
}

SND_PCM_PLUGIN_SYMBOL(file);

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/plugins/file.c $ $Rev: 781520 $")
#endif
