#ifdef __QNX__
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <stdint.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdbool.h>
#include <devctl.h>
#include <sys/asound_common.h>
#include <pcm_internal.h>

#define AUDIOMAN_ROUTER_PPS_PATH        "/pps/services/audio/audio_router_control"
#define PPS_WAIT                        "?nopersist,wait"
#define AUDIOMAN_READ_TIMEOUT 12 //secs, BT times out in 10 seconds...
#define AUDIOMAN_ROUTER_GET_HANDLE      "get_handle"
#define AUDIOMAN_ROUTER_FREE_HANDLE     "free_handle"
#define AUDIOMAN_ROUTER_HANDLE          "audioman_handle"
#define AUDIOMAN_ROUTER_BIND_HANDLE     "pcm_bind"
#define AUDIOMAN_ROUTER_UNBIND_HANDLE   "pcm_unbind"

typedef struct io_audio_info {
    snd_pcm_ioplug_t ext;
    snd_pcm_t *pcm;
    bool nonblock;
    int fd;
    int chmap_size;
    snd_pcm_stream_t stream;
    snd_pcm_channel_info_t info;
    snd_pcm_channel_params_t params;
    int audioman_handle;
    int external_audioman_handle;
    struct _snd_pcm_hw_params *hwparams;
    struct _snd_pcm_sw_params *swparams;
} io_audio_info_t;

int snd_pcm_format_mapping[] = {
    SND_PCM_FMT_S8,
    SND_PCM_FMT_U8,
    SND_PCM_FMT_S16_LE,
    SND_PCM_FMT_S16_BE,
    SND_PCM_FMT_U16_LE,
    SND_PCM_FMT_U16_BE,
    SND_PCM_FMT_S24_LE,
    SND_PCM_FMT_S24_BE,
    SND_PCM_FMT_U24_LE,
    SND_PCM_FMT_U24_BE,
    SND_PCM_FMT_S32_LE,
    SND_PCM_FMT_S32_BE,
    SND_PCM_FMT_U32_LE,
    SND_PCM_FMT_U32_BE,
    SND_PCM_FMT_FLOAT_LE,
    SND_PCM_FMT_FLOAT_BE,
    SND_PCM_FMT_FLOAT64_LE,
    SND_PCM_FMT_FLOAT64_BE,
    SND_PCM_FMT_IEC958_SUBFRAME_LE,
    SND_PCM_FMT_IEC958_SUBFRAME_BE,
    SND_PCM_FMT_MU_LAW,
    SND_PCM_FMT_A_LAW,
    SND_PCM_FMT_IMA_ADPCM,
    SND_PCM_FMT_MPEG,
    SND_PCM_FMT_GSM
};

int snd_pcm_format_reverse_mapping[] = {
    SND_PCM_FORMAT_U8,
    SND_PCM_FORMAT_S8,
    SND_PCM_FORMAT_U16_LE,
    SND_PCM_FORMAT_U16_BE,
    SND_PCM_FORMAT_S16_LE,
    SND_PCM_FORMAT_S16_BE,
    SND_PCM_FORMAT_U24_LE,
    SND_PCM_FORMAT_U24_BE,
    SND_PCM_FORMAT_S24_LE,
    SND_PCM_FORMAT_S24_BE,
    SND_PCM_FORMAT_U32_LE,
    SND_PCM_FORMAT_U32_BE,
    SND_PCM_FORMAT_S32_LE,
    SND_PCM_FORMAT_S32_BE,
    SND_PCM_FORMAT_A_LAW,
    SND_PCM_FORMAT_MU_LAW,
    SND_PCM_FORMAT_IEC958_SUBFRAME_LE,
    SND_PCM_FORMAT_IEC958_SUBFRAME_BE,
    SND_PCM_FORMAT_SPECIAL,
    SND_PCM_FORMAT_FLOAT_LE,
    SND_PCM_FORMAT_FLOAT_BE,
    SND_PCM_FORMAT_FLOAT64_LE,
    SND_PCM_FORMAT_FLOAT64_BE,
    SND_PCM_FORMAT_IMA_ADPCM,
    SND_PCM_FORMAT_GSM,
    SND_PCM_FORMAT_MPEG,
    SND_PCM_FORMAT_SPECIAL
};

static int snd_pcm_notify_audioman_generic( char * msg, char * res, unsigned int res_size )
{
    char tmp[500]={0};
    int fd;
    int ret = EOK;
    int nread;
    fd_set readfds;
    struct timeval timeout;
    timeout.tv_sec = AUDIOMAN_READ_TIMEOUT;
    timeout.tv_usec = 0;

    // Check if /pps exists first
    if(access("/pps", X_OK) != 0) {
        // if pps service is not available, we cannot post to 
        // audioman
        return EOK;
    }

    if ((fd = open(AUDIOMAN_ROUTER_PPS_PATH PPS_WAIT, O_RDWR)) < 0) {
        // audioman service is absent? we cannot post the 
        // notification
        return EOK;
    }

    write(fd, msg, strlen(msg) + 1);
    // wait for the read to finish or timeout
    FD_ZERO( &readfds );
    FD_SET( fd, &readfds );
    if ( select( fd+1, &readfds, NULL, NULL, &timeout ) <= 0 ) {
        close(fd);
        return EINVAL;
    }
    nread = read(fd, tmp, sizeof(tmp) - 1);
    if ( nread > 0 ) {
        // make sure the string operations are not going to crash
        tmp[sizeof(tmp)-1] = '\0';
        if ( res ) {
            strncpy( res, tmp, res_size );
            res[res_size-1] = '\0';
        }
    }
    close(fd);

    return ret;
}

static int snd_pcm_notify_audioman_pcm_binding( io_audio_info_t * info, uint32_t audioman_handle, int32_t bind_handles )
{
    char tmp[512]={0};
    int ret = EOK;
    char * ptr;

    snprintf(tmp, sizeof(tmp),
            "msg::%s\nid::PCM-%d-%d\ndat:json:{\"audioman_handle\":%d, \"pcm_handle\":%"PRIdPTR"}",
            bind_handles ? AUDIOMAN_ROUTER_BIND_HANDLE :
            AUDIOMAN_ROUTER_UNBIND_HANDLE,
            gettid(),
            __LINE__,
            audioman_handle,
            (intptr_t)info->pcm);

    if ( !(ret = snd_pcm_notify_audioman_generic( tmp, tmp, sizeof(tmp) ) ) ) {
        // check whether the command succeeded
        if ( (ptr = strstr(tmp, "dat")) != NULL ) {
            ret = EOK;
        } else {
            ret = EINVAL;
        }
    }

    return ret;
}

static int snd_pcm_notify_audioman_remove( io_audio_info_t * info )
{
    char tmp[512]={0};
    int ret = EOK;

    if ( info->audioman_handle ) {
        snprintf(tmp, sizeof(tmp),
                "msg::%s\nid::PCM-%d-%d\ndat:json:{\"audioman_handle\":%u}",
                AUDIOMAN_ROUTER_FREE_HANDLE,
                gettid(),
                __LINE__,
                info->audioman_handle);

        if ( !(ret = snd_pcm_notify_audioman_generic( tmp, tmp, sizeof(tmp) ) ) ) {
            // default to invalid handle, just in case any error happens
            info->audioman_handle = 0;
        }
    }

    return ret;
}

static void io_audio_free(io_audio_info_t *info)
{
    int ret;

    if ( !info->external_audioman_handle ) {
        ret = snd_pcm_notify_audioman_remove( info );
    } else {
        ret = snd_pcm_notify_audioman_pcm_binding( info, info->audioman_handle, 0 );
        info->audioman_handle = 0;
        if( ret == EOK ) {
            ret = snd_pcm_notify_audioman_remove( info );
        } else {
            snd_pcm_notify_audioman_remove( info );
        }
    }

    close( info->fd );
    snd_pcm_hw_params_free( info->hwparams );
    snd_pcm_sw_params_free( info->swparams );
    free(info);
}

static int is_interleaved_access_type(snd_pcm_access_t access)
{
    switch( access ) {
        case SND_PCM_ACCESS_MMAP_INTERLEAVED:
        case SND_PCM_ACCESS_RW_INTERLEAVED:
            return 1;
        default:
            return 0;
    }
}


static int apply_configuration(snd_pcm_hw_params_t *hwparams, io_audio_info_t *info, snd_pcm_t *pcm)
{
    snd_pcm_format_t format;
    snd_pcm_uframes_t buffersize;
    snd_pcm_channel_setup_t setup;
    snd_mixer_eid_t eid;
    snd_mixer_gid_t gid;
    int ret;
    iov_t   iov[3];
    unsigned int rate_num, rate_den, channels;
    snd_pcm_access_t access;
    snd_pcm_uframes_t frames;

    if( (ret = snd_pcm_hw_params_get_access(hwparams, &access)) ) {
        return ret;
    }

    if( (ret = snd_pcm_hw_params_get_rate_numden(hwparams, &rate_num, &rate_den)) ) {
        return ret;
    }

    // Rate must be an integer at this point
    if( rate_den != 1 ) {
        return -EINVAL;
    }

    if( (ret = snd_pcm_hw_params_get_channels(hwparams, &channels)) ) {
        return ret;
    }

    memset(&info->params, 0, sizeof(info->params));
    info->params.channel = info->stream;
    info->params.mode = SND_PCM_MODE_BLOCK;
    info->params.format.interleave = is_interleaved_access_type(access);
    info->params.format.rate = rate_num;
    info->params.format.voices = channels;

    if( snd_pcm_hw_params_get_format(hwparams, &format) ) {
        return -EINVAL;
    }
    if( snd_pcm_hw_params_get_period_size(hwparams, &frames, NULL) ) {
        return -EINVAL;
    }
    if( format >= 0 && format < sizeof(snd_pcm_format_mapping)/sizeof(snd_pcm_format_mapping[0]) ) {
        info->params.format.format = snd_pcm_format_mapping[format];
    } else if( format == SND_PCM_FORMAT_SPECIAL ) {
        info->params.format.format = SND_PCM_FMT_SPECIAL;
    }  else {
        return -EINVAL;
    }

    // TODO - support alternate start modes
    info->params.start_mode = PCM_START_FULL;
    info->params.buf.block.frag_size = frames * snd_pcm_format_width( info->params.format.format ) / 8 * channels;
    info->params.stop_mode = SND_PCM_STOP_STOP;
    info->params.buf.block.frags_min = 1;
    snd_pcm_hw_params_get_buffer_size_min( hwparams, &buffersize );
    info->params.buf.block.frags_max = buffersize / info->params.buf.block.frag_size;

    if (ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_PARAM_FIT, &info->params) < 0 ) {
        return -errno;
    }
    if (ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_PREPARE) < 0) {
        return -errno;
    }
    memset(&setup, 0, sizeof(snd_pcm_channel_setup_t));
    setup.channel = (info->stream == SND_PCM_STREAM_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE;
    SETIOV (&iov[0], &setup, sizeof (snd_pcm_channel_setup_t));
    SETIOV (&iov[1], &eid, sizeof (snd_mixer_eid_t));
    SETIOV (&iov[2], &gid, sizeof (snd_mixer_gid_t));
    if ((ret = devctlv (info->fd, SND_PCM_IOCTL_CHANNEL_SETUP, 3, 3, iov, iov, NULL)) != EOK)
        return -ret;
    snd_pcm_hw_params_set_period_size( pcm, hwparams, setup.buf.block.frag_size * 8 / snd_pcm_format_width( info->params.format.format ) / channels, 0 );
    snd_pcm_hw_params_set_periods( pcm, hwparams, setup.buf.block.frags + 1, 0 );

    return 0;
}

static int hw_start(snd_pcm_ioplug_t *io)
{
    int ret;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    ret = ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_GO);

    return ret;
}

static int hw_stop(snd_pcm_ioplug_t *io)
{
    int ret;
    int on = 1;

    io_audio_info_t *info = (io_audio_info_t *) io->private_data;
    ret = ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_PAUSE, &on);

    return ret;
}

static snd_pcm_sframes_t hw_pointer(snd_pcm_ioplug_t *io)
{
    snd_pcm_channel_status_t status;

    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    if (ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_STATUS, &status) < 0) {
        return -errno;
    }

    return status.frag;
}

static void set_nonblock(int fd, bool nonblock)
{
    int flags;
    if ((flags = fcntl(fd, F_GETFL)) < 0) {
        return;
    }
    if (nonblock) {
        flags |= O_NONBLOCK;
    } else {
        flags &= ~O_NONBLOCK;
    }
    fcntl(fd, F_SETFL, flags);
}

snd_pcm_sframes_t hw_transfer_out(snd_pcm_ioplug_t *io,
                    const snd_pcm_channel_area_t *areas,
                    snd_pcm_uframes_t offset,
                    snd_pcm_uframes_t size)
{
    int ret, target;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    if( io->nonblock != info->nonblock ) {
        io->nonblock = info->nonblock;
        set_nonblock( info->fd, io->nonblock );
    }

    target = size * info->params.format.voices * snd_pcm_format_width(info->params.format.format) / _BITS_BYTE;
    ret = write(info->fd, areas->addr + (areas->first + areas->step * offset) / 8, target);
    if( ret < 0 ) ret = 0;

    return ret == target ? size : ret * size / target;
}

snd_pcm_sframes_t hw_transfer_in(snd_pcm_ioplug_t *io,
                    const snd_pcm_channel_area_t *areas,
                    snd_pcm_uframes_t offset,
                    snd_pcm_uframes_t size)
{
    int ret, target;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    if( io->nonblock != info->nonblock ) {
        io->nonblock = info->nonblock;
        set_nonblock( info->fd, io->nonblock );
    }

    target = size * info->params.format.voices * snd_pcm_format_width(info->params.format.format) / _BITS_BYTE;
    ret = read(info->fd, areas->addr + (areas->first + areas->step * offset) / 8, target);
    if( ret < 0 ) ret = 0;

    return ret == target ? size : ret * size / target;
}

static int hw_prepare(snd_pcm_ioplug_t *io)
{
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    if (ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_PREPARE) < 0) {
        return -errno;
    }
    return EOK;
}

static int hw_drain(snd_pcm_ioplug_t *io)
{
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;
    if (ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_DRAIN) < 0)
        return -errno;
    return EOK;
}

static int hw_resume(snd_pcm_ioplug_t *io)
{
    int off = 0;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;
    if (ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_PAUSE, &off) < 0)
        return -errno;
    return EOK;
}

static int hw_hw_params(snd_pcm_ioplug_t *io, snd_pcm_hw_params_t *params)
{
    int ret;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    ret = apply_configuration(params, info, info->pcm);
    snd_pcm_hw_params_copy(info->hwparams, params);
    return ret;
}

static int hw_sw_params(snd_pcm_ioplug_t *io, snd_pcm_sw_params_t *params)
{
    int ret;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    snd_pcm_sw_params_copy(info->swparams, params);
    ret = apply_configuration(info->hwparams, info, info->pcm);
    return ret;
}

static snd_pcm_chmap_query_t **hw_query_chmaps(snd_pcm_ioplug_t *io)
{
    int ret;
    int mapcount, i, j;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;
    snd_pcm_chmap_query_t **map = NULL;

    // Make sure we can at least see the size of buffer needed
    if( info->chmap_size < 8 ) info->chmap_size = 8;

    unsigned int *data = malloc(info->chmap_size);
    if( data == NULL ) return NULL;

    while(1) {
        data[0] = (info->stream == SND_PCM_STREAM_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE;
        ret = devctl(info->fd, SND_PCM_IOCTL_QUERY_CHANNEL_MAP, data, info->chmap_size, NULL);
        switch( ret ) {
            case EOK:
                switch( data[0] ) {
                    case EOK:
                        // Parse the chmap info. The format is an int for
                        // number of channels followed by an int for type
                        // followed by channel layout of that many channels
                        // terminated by a layout of size 0
                        for(mapcount = 0, i = 1; data[ i ] != 0; mapcount++,i += data[i]+2);
                        map = calloc(mapcount+1, sizeof(snd_pcm_chmap_query_t *));
                        if( !map ) {
                            free( data );
                            return NULL;
                        }
                        for( i = 0, j = 1; i < mapcount; j += data[j]+2, i++ ) {
                            map[i] = malloc(sizeof(snd_pcm_chmap_query_t) + data[j] * sizeof(int));
                            if( map[ i ] == NULL ) {
                                for( ; i >=0; i -- ) {
                                    free( map[ i ] );
                                }
                                free( map );
                                free( data );
                                return NULL;
                            }
                            map[i]->type = data[j+1];
                            map[i]->map.channels = data[j];
                            memcpy(map[i]->map.pos, data+j+2, data[j]*sizeof(int));
                        }
                        free(data);
                        return map;
                    case ENOMEM:
                        info->chmap_size = ((int *)data)[1];
                        {
                            unsigned int *newdata;

                            newdata = realloc( data, info->chmap_size*sizeof(int) );
                            if( newdata == NULL ) {
                                info->chmap_size = 0;
                                free( data );
                                return NULL;
                            }
                            data = newdata;
                        }
                        break;
                    default:
                        free(data);
                        return NULL;
                }
                break;
            case ENOMEM:
                {
                    info->chmap_size ++;
                    unsigned int *newdata;
                    newdata = realloc( data, info->chmap_size*sizeof(int) );
                    if( newdata == NULL ) {
                        info->chmap_size = 0;
                        free( data );
                        return NULL;
                    }
                    data = newdata;
                }
                break;
            default:
                free(data);
                return NULL;
        }
    }
}

static snd_pcm_chmap_t *hw_get_chmap(snd_pcm_ioplug_t *io)
{
    int ret;
    // Size for max number of supported channels + error code + channel count
    char msgbuf[32*sizeof(int)+2];
    int *intMsgBuf = (int *)msgbuf;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    intMsgBuf[0] = (info->stream == SND_PCM_STREAM_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE;
    ret = devctl(info->fd, SND_PCM_IOCTL_GET_CHANNEL_MAP, msgbuf, sizeof(msgbuf), NULL);
    if( ret != EOK || msgbuf[0] != EOK ) {
        return NULL;
    }
    snd_pcm_chmap_t *map = malloc(sizeof(snd_pcm_chmap_t) + sizeof(int) * msgbuf[1]);
    if( map == NULL ) return NULL;
    map->channels = msgbuf[1];
    memcpy(map->pos, msgbuf+2, sizeof(int) * map->channels);
    return map;
}

static int hw_set_chmap(snd_pcm_ioplug_t *io, const snd_pcm_chmap_t *map)
{   int ret;
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;
    int mode = (info->stream == SND_PCM_STREAM_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE;
    unsigned int *channelMsg = malloc(sizeof(snd_pcm_chmap_t)+sizeof(unsigned int)*(map->channels+2));
    if( channelMsg == NULL ){
        return -ENOMEM;
    }
    channelMsg[0] = mode;
    channelMsg[1] = 0; //angle
    memcpy(channelMsg+2, map, sizeof(snd_pcm_chmap_t)+sizeof(unsigned int)*map->channels);
    ret = devctl(info->fd, SND_PCM_IOCTL_SET_CHANNEL_MAP, channelMsg, sizeof(snd_pcm_chmap_t)+sizeof(unsigned int)*(map->channels+2), NULL);
    free( channelMsg );
    return -ret;
}

static int hw_close(snd_pcm_ioplug_t *io)
{
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;
    close(info->fd);

    snd_pcm_ioplug_delete( io );

    io_audio_free(info);

    return 0;
}

int hw_poll_descriptors_count(snd_pcm_ioplug_t *io)
{
    return 1;
}

int hw_poll_descriptors(snd_pcm_ioplug_t *io, struct pollfd *pfds, unsigned int space)
{
    io_audio_info_t *info = (io_audio_info_t *) io->private_data;

    if( space < 1 ) return -ENOMEM;
    pfds->fd = info->fd;

    return EOK;
}

snd_pcm_ioplug_callback_t io_playback_callback = {
    .start = hw_start,
    .stop = hw_stop,
    .pointer = hw_pointer,
    .transfer = hw_transfer_out,
    .prepare = hw_prepare,
    .drain = hw_drain,
    .close = hw_close,
    .poll_descriptors_count = hw_poll_descriptors_count,
    .poll_descriptors = hw_poll_descriptors,
    .query_chmaps = hw_query_chmaps,
    .get_chmap = hw_get_chmap,
    .set_chmap = hw_set_chmap,
    .hw_params = hw_hw_params,
    .sw_params = hw_sw_params,
    .resume = hw_resume,
};

snd_pcm_ioplug_callback_t io_capture_callback = {
    .start = hw_start,
    .stop = hw_stop,
    .pointer = hw_pointer,
    .transfer = hw_transfer_in,
    .prepare = hw_prepare,
    .drain = hw_drain,
    .close = hw_close,
    .poll_descriptors_count = hw_poll_descriptors_count,
    .poll_descriptors = hw_poll_descriptors,
    .query_chmaps = hw_query_chmaps,
    .get_chmap = hw_get_chmap,
    .set_chmap = hw_set_chmap,
    .hw_params = hw_hw_params,
    .sw_params = hw_sw_params,
    .resume = hw_resume,
};

SND_PCM_PLUGIN_DEFINE_FUNC(hw)
{
    int i;
    int formats;
    unsigned int *formatlist;
    struct io_audio_info *info;
    int min_bps = INT_MAX, max_bps = 0; // min and max bytes per sample
    int err;
    const char *device = "pcmPreferred";
    char buf[FILENAME_MAX];
    snd_config_t *result;
    unsigned interleaved_list [] = {
        SND_PCM_ACCESS_RW_INTERLEAVED,
        SND_PCM_ACCESS_RW_NONINTERLEAVED,
    };

    info = calloc(1, sizeof(*info));
    if (info == NULL) {
        return -ENOMEM;
    }

    // See if the caller specified a device
    if( snd_config_search(conf, "device", &result) == EOK ) {
        snd_config_get_string(result, &device);
    }

    info->ext.version = SND_PCM_IOPLUG_VERSION;
    info->ext.name = "plughw";
    info->ext.callback = stream == SND_PCM_STREAM_PLAYBACK ? &io_playback_callback : &io_capture_callback;
    info->stream = stream;
    info->ext.private_data = info;
    if( (err = snd_pcm_hw_params_malloc( &info->hwparams )) != EOK ) {
        return err;
    }
    if( (err = snd_pcm_sw_params_malloc( &info->swparams )) != EOK ) {
        snd_pcm_hw_params_free( info->hwparams );
        return err;
    }

    // Open the file
    snprintf(buf, sizeof(buf), "/dev/snd/%s%c", device, stream == SND_PCM_STREAM_PLAYBACK ? 'p' : 'c');
    info->fd = open(buf, O_RDWR | O_CLOEXEC);
    if(info->fd == -1 ) {
        snd_pcm_hw_params_free( info->hwparams );
        snd_pcm_sw_params_free( info->swparams );
        free(*pcmp);
        return -ENOENT;
    }

    err = snd_pcm_ioplug_create(&info->ext, name, stream, mode);
    if (err < 0) {
        io_audio_free(info);
        return err;
    }

    // Get capabilities
    if (ioctl (info->fd, SND_PCM_IOCTL_CHANNEL_INFO, &info->info) < 0 ) {
        io_audio_free(info);
        return -errno;
    }

    // Set access methods. We don't support mmap yet
    if( info->info.flags & SND_PCM_CHNINFO_INTERLEAVE ) {
        if( info->info.flags & SND_PCM_CHNINFO_NONINTERLEAVE ) {
            snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_ACCESS, 2, &interleaved_list[0]);
        } else {
            snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_ACCESS, 1, &interleaved_list[0]);
        }
    } else {
        snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_ACCESS, 1, &interleaved_list[1]);
    }

    // Set supported formats
    // Count the formats
    formats = 0;
    for( i = 0; i < sizeof(info->info.formats) * _BITS_BYTE; i ++ )  {
        if( info->info.formats & (1 << i) ) {
            formats++;
        }
    }
    formatlist = malloc(sizeof(unsigned int) * formats);
    if( formatlist == NULL ) {
        io_audio_free(info);
        return -ENOMEM;
    }
    formats = 0;
    for( i = 0; i < sizeof(info->info.formats) * _BITS_BYTE; i ++ )  {
        if( info->info.formats & (1 << i) ) {
            formatlist[ formats ] = snd_pcm_format_reverse_mapping[ i ];
            min_bps = MIN( min_bps, snd_pcm_format_width( formatlist[ formats ] ) );
            max_bps = MAX( max_bps, snd_pcm_format_width( formatlist[ formats ] ) );
            formats++;
        }
    }
    snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_FORMAT, formats, formatlist);
    free( formatlist );

    // Set voice count
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_CHANNELS, info->info.min_voices, info->info.max_voices);

    // TODO - if info->info.rates is set, we need to restrict the rate set
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_RATE, info->info.min_rate, info->info.max_rate);

    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_PERIOD_BYTES,
            info->info.min_fragment_size,
            info->info.max_fragment_size);

    // This comes from services/audio/mgr/main.c. Might want to be able to read it from io-audio
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_BUFFER_BYTES,
            0,
            256 * 1024);

    *pcmp = info->ext.pcm;
    return EOK;
}

SND_PCM_PLUGIN_SYMBOL(hw);

#endif

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/plugins/hw.c $ $Rev: 781169 $")
#endif
