#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#ifdef __QNX__
#include <asound_byteswap.h>
#else
#include <pcm_internal.h>
#include <byteswap.h>
#endif
#include <asound_endian.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>

typedef struct plug_info {
    snd_pcm_extplug_t ext;
    int *voicematrix;
    int vmw;
    int vmh;
} plug_info_t;

static void plug_free(plug_info_t *info)
{
    free(info->voicematrix);
    free(info);
}

// Normalize everything to signed 32-bit. Not super efficient, but can be optimized later. Assumes at least a 32-bit int
static int readval(void *data, snd_pcm_format_t format, int offset, int maxoffset)
{
    if( offset >= maxoffset ) {
        return 0;
    }

    switch( format ) {
        case SND_PCM_FORMAT_U8:
            return (((uint8_t *)data)[offset] - (1<<7)) << 24;
        case SND_PCM_FORMAT_S8:
            return ((int8_t *)data)[offset] << 24;
        case SND_PCM_FORMAT_U16_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            return (((uint16_t *)data)[offset] - (1<<15)) << 16;
#else
            return (bswap_16((uint16_t *)data)[offset] - (1<<15)) << 16;
#endif
        case SND_PCM_FORMAT_U16_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            return (((uint16_t *)data)[offset] - (1<<15)) << 16;
#else
            return bswap_16(((uint16_t *)data)[offset] - (1<<15)) << 16;
#endif
        case SND_PCM_FORMAT_S16_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            return ((int16_t *)data)[offset] << 16;
#else
            return bswap_s16((int16_t *)data[offset]) << 16;
#endif
        case SND_PCM_FORMAT_S16_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            return ((int16_t *)data)[offset] << 16;
#else
            return bswap_s16(((int16_t *)data)[offset]) << 16;
#endif
        case SND_PCM_FORMAT_U32_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            return ((uint32_t *)data)[offset] - (1<<31);
#else
            return bswap_32(((uint32_t *)data)[offset] - (1<<31));
#endif
        case SND_PCM_FORMAT_U32_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            return ((uint32_t *)data)[offset] - (1<<31);
#else
            return bswap_32(((uint32_t *)data)[offset] - (1<<31));
#endif
        case SND_PCM_FORMAT_S32_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            return ((int32_t *)data)[offset];
#else
            return bswap_s32((int32_t *)data[offset]);
#endif
        case SND_PCM_FORMAT_S32_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            return ((int32_t *)data)[offset];
#else
            return bswap_s32(((int32_t *)data)[offset]);
#endif
        default:
            // This should never happen
            return 0;
    }
}

// Normalize everything to signed 32-bit. Not super efficient, but can be optimized later
static void writeval(void *data, snd_pcm_format_t format, int offset, int maxoffset, int value)
{
    if( offset >= maxoffset ) {
        return;
    }

    switch( format ) {
        case SND_PCM_FORMAT_U8:
            ((uint8_t *)data)[offset] = (value >> 24) + (1<<7);
            break;
        case SND_PCM_FORMAT_S8:
            ((int8_t *)data)[offset] = (value >> 24);
            break;
        case SND_PCM_FORMAT_U16_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            ((uint16_t *)data)[offset] = (value >> 16) + (1<<15);
#else
            ((uint16_t *)data)[offset] = (bswap_16(value) >> 16) + (1<<15);
#endif
            break;
        case SND_PCM_FORMAT_U16_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            ((uint16_t *)data)[offset] = value >> 16 + (1<<15);
#else
            ((uint16_t *)data)[offset] = (bswap_16(value) >> 16) + (1<<15);
#endif
            break;
        case SND_PCM_FORMAT_S16_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            ((int16_t *)data)[offset] = (value >> 16);
#else
            ((int16_t *)data)[offset] = (bswap_s16(value) >> 16);
#endif
            break;
        case SND_PCM_FORMAT_S16_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            ((int16_t *)data)[offset] = (value >> 16);
#else
            ((int16_t *)data)[offset] = (bswap_s16(value) >> 16);
#endif
            break;
        case SND_PCM_FORMAT_U32_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            ((uint32_t *)data)[offset] = value + (1<<31);
#else
            ((uint32_t *)data)[offset] = bswap_32(value) + (1<<31);
#endif
            break;
        case SND_PCM_FORMAT_U32_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            ((uint32_t *)data)[offset] = value + (1<<31);
#else
            ((uint32_t *)data)[offset] = bswap_32(value) + (1<<31);
#endif
            break;
        case SND_PCM_FORMAT_S32_LE:
#if __BYTE_ORDER==__LITTLE_ENDIAN
            ((int32_t *)data)[offset] = value;
#else
            ((int32_t *)data)[offset] = bswap_s32(value);
#endif
            break;
        case SND_PCM_FORMAT_S32_BE:
#if __BYTE_ORDER==__BIG_ENDIAN
            ((int32_t *)data)[offset] = value;
#else
            ((int32_t *)data)[offset] = bswap_s32(value);
#endif
            break;
        default:
            // This should never happen
            return;
    }
}

snd_pcm_sframes_t plug_transfer(snd_pcm_extplug_t *ext,
                  const snd_pcm_channel_area_t *dst_areas,
                  snd_pcm_uframes_t dst_offset,
                  const snd_pcm_channel_area_t *src_areas,
                  snd_pcm_uframes_t src_offset,
                  snd_pcm_uframes_t size)
{
    int i, j, k, v;
    void *dp, *sp;
    snd_pcm_format_t srcf, dstf;
    int src_chan, dst_chan;
    int buffer[ext->slave_channels];
    int counts[ext->slave_channels];
    plug_info_t *info = (plug_info_t *) ext->private_data;

    if( ext->stream == SND_PCM_STREAM_PLAYBACK ) {
        srcf = ext->format;
        dstf = ext->slave_format;
        src_chan = ext->channels;
        dst_chan = ext->slave_channels;
    } else {
        srcf = ext->slave_format;
        dstf = ext->format;
        src_chan = ext->slave_channels;
        dst_chan = ext->channels;
    }
    dp = dst_areas->addr;
    sp = src_areas->addr;

    memset( dp, 0, size * dst_areas->step * dst_chan / 8 );
    for( i = 0; i < size; i ++ ) {
        memset( buffer, 0, sizeof(buffer) );
        memset( counts, 0, sizeof(counts) );
        for( j = 0; j < src_chan; j ++ ) {
            v = readval(sp, srcf, j, info->vmh);
            for( k = 0; k < dst_chan; k ++ ) {
                if( info->voicematrix[j*info->vmw + k] ) {
                    buffer[k] += v;
                    counts[k] ++;
                }
            }
        }
        for( j = 0; j < src_chan; j ++ ) {
            writeval( dp, dstf, j, info->vmw, counts[j] > 1 ? buffer[j]/counts[j] : buffer[j] );
        }
        dp = ((char *)dp) + dst_areas->step * dst_chan / 8;
        sp = ((char *)sp) + src_areas->step * src_chan / 8;
    }

    return size;
}

static int plug_hw_params(snd_pcm_extplug_t *ext, snd_pcm_hw_params_t *params)
{
    return EOK;
}

snd_pcm_chmap_query_t **plug_query_chmaps(snd_pcm_extplug_t *ext)
{
    // Supported chmaps will be all chmaps with any number of channels that the hardware supports
    return NULL;
}

snd_pcm_chmap_t *plug_get_chmap(snd_pcm_extplug_t *ext)
{
    return NULL;
}

int plug_set_chmap(snd_pcm_extplug_t *ext, const snd_pcm_chmap_t *map)
{
    return EOK;
}


static int plug_close(snd_pcm_extplug_t *ext)
{
    plug_info_t *info = (plug_info_t *) ext->private_data;

    snd_pcm_extplug_delete( ext );

    plug_free(info);

    return EOK;
}


snd_pcm_extplug_callback_t plug_callback = {
    .hw_params = plug_hw_params,
    .transfer = plug_transfer,
    .query_chmaps = plug_query_chmaps,
    .get_chmap = plug_get_chmap,
    .set_chmap = plug_set_chmap,
    .close = plug_close,
};

SND_PCM_PLUGIN_DEFINE_FUNC(plug)
{
    struct plug_info *info;
    snd_config_t *slave_conf;
    int err;
    unsigned int formats[] = {SND_PCM_FORMAT_U8, SND_PCM_FORMAT_S8, SND_PCM_FORMAT_U16_LE, SND_PCM_FORMAT_U16_BE, SND_PCM_FORMAT_S16_LE, SND_PCM_FORMAT_S16_BE, SND_PCM_FORMAT_U16_LE, SND_PCM_FORMAT_U16_BE, SND_PCM_FORMAT_S16_LE, SND_PCM_FORMAT_S16_BE };

    info = calloc(1, sizeof(*info));
    if (info == NULL)
        return -ENOMEM;

    info->ext.version = SND_PCM_EXTPLUG_VERSION;
    info->ext.name = "plug";
    info->ext.callback = &plug_callback;
    info->ext.private_data = info;

    // voicematrix defaults to 2x2 identity
    info->voicematrix = calloc( 2 * 2, sizeof(int) );
    if( info->voicematrix == NULL ) {
        return -ENOMEM;
    }
    memset( info->voicematrix, 0, 2 * 2 * sizeof(int) );
    info->voicematrix[ 0 ] = 1;
    info->voicematrix[ 3 ] = 1;
    info->vmw = info->vmh = 2;

    err = snd_config_search(conf, "slave", &slave_conf);
    if (err < 0) {
        plug_free( info );
        return err;
    }

    err = snd_pcm_extplug_create(&info->ext, name, root, slave_conf, stream, mode);
    if (err < 0) {
        plug_free(info);
        return err;
    }

    // Set capabilities
    snd_pcm_extplug_set_param_list(&info->ext, SND_PCM_EXTPLUG_HW_FORMAT, sizeof(formats)/sizeof(formats[0]), formats);
    snd_pcm_extplug_set_param_minmax(&info->ext, SND_PCM_EXTPLUG_HW_CHANNELS, 1, 32);
    snd_pcm_extplug_set_slave_param_minmax(&info->ext, SND_PCM_EXTPLUG_HW_CHANNELS, 1, 32);
    snd_pcm_extplug_set_slave_param_list(&info->ext, SND_PCM_EXTPLUG_HW_FORMAT, sizeof(formats)/sizeof(formats[0]), formats);

    *pcmp = info->ext.pcm;
    return EOK;
}

SND_PCM_PLUGIN_SYMBOL(plug);

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/plugins/plug.c $ $Rev: 756950 $")
#endif
