#include <errno.h>
#include <stdint.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>
#include <alsa/pcm_rate.h>
#include <stdlib.h>
#include <rate_poly.h>

#ifndef MIN
#define MIN(x,y) ((x)<(y)?(x):(y))
#endif
#ifndef MAX
#define MAX(x,y) ((x)>(y)?(x):(y))
#endif

typedef struct {
    ResampleState_t rs;
    int channels;
    int fragsize;
} rate_t;

static int rate_init(void *obj, snd_pcm_rate_info_t *info)
{
    int ret;
    rate_t *rate = (rate_t *)obj;

    rate->fragsize = MAX(info->in.period_size, info->out.period_size) * sizeof(int16_t) * info->channels;
    rate->channels = info->channels;

    ret = rate_poly_create( &rate->rs, info->in.rate, info->channels, info->out.rate, rate->fragsize, 0, info->in.period_size < info->out.period_size );
    if( ret != EOK ) {
        return ret;
    }

    rate_poly_reset( &rate->rs, rate->fragsize );

    return EOK;
}

static void rate_reset(void *obj)
{
    rate_t *rate = (rate_t *)obj;

    rate_poly_reset( &rate->rs, rate->fragsize );
}

static void rate_convert_s16(void *obj, int16_t *dst, unsigned int dst_frames,
            const int16_t *src, unsigned int src_frames)
{
    rate_t *rate = (rate_t *)obj;
    int fragsize = MAX(src_frames, dst_frames) * sizeof(int16_t) * rate->channels;

    if( fragsize != rate->fragsize ) {
        rate->fragsize = fragsize;
        rate_poly_reset( &rate->rs, rate->fragsize );
    }
    rate_poly_process( &rate->rs, src, src_frames * sizeof(int16_t) * rate->rs.channels, dst, dst_frames * sizeof(int16_t) * rate->rs.channels);
}

static void rate_free(void *obj)
{
    rate_t *rate = (rate_t *)obj;

    rate_poly_free( &rate->rs );
}

static int rate_get_supported_rates(void *obj, unsigned int *rate_min, unsigned int *rate_max)
{
    *rate_min = 8000;
    *rate_max = 48000;

    return 0;
}

static snd_pcm_uframes_t rate_input_frames(void *obj, snd_pcm_uframes_t frames)
{
    rate_t *rate = (rate_t *)obj;

    return frames * rate->rs.inRate / rate->rs.outRate;
}

static snd_pcm_uframes_t rate_output_frames(void *obj, snd_pcm_uframes_t frames)
{
    rate_t *rate = (rate_t *)obj;

    return frames * rate->rs.outRate / rate->rs.inRate;
}

static snd_pcm_rate_ops_t rate_ops =
{
    .init = rate_init,
    .reset = rate_reset,
    .free = rate_free,
    .version = SND_PCM_RATE_PLUGIN_VERSION,
    .get_supported_rates = rate_get_supported_rates,
    .convert_s16 = rate_convert_s16,
    .input_frames = rate_input_frames,
    .output_frames = rate_output_frames,
};


int SND_PCM_RATE_PLUGIN_ENTRY(samplerate)(unsigned int version, void **objp, snd_pcm_rate_ops_t *ops)
{
    *objp = calloc(1, sizeof(rate_t));
    *ops = rate_ops;
    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/plugins/rate_polyphase.c $ $Rev: 756950 $")
#endif
