#include <errno.h>
#include <math.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>

#define FREQ        440
#define UNIMPLEMENTED   printf("%s not implemented\n", __FUNCTION__);

typedef struct generator_info {
    snd_pcm_ioplug_t ext;
    long long samples;
    snd_pcm_uframes_t fragsize;
    unsigned int frags;
    unsigned int frag;
} generator_info_t;

static void generator_free(generator_info_t *info)
{
    free(info);
}

static int generator_start(snd_pcm_ioplug_t *io)
{
    return EOK;
}

static int generator_stop(snd_pcm_ioplug_t *io)
{
    return EOK;
}

static snd_pcm_sframes_t generator_pointer(snd_pcm_ioplug_t *io)
{
    generator_info_t *info = (generator_info_t *) io->private_data;

    return info->frag;
}

snd_pcm_sframes_t generator_transfer(snd_pcm_ioplug_t *io,
                    const snd_pcm_channel_area_t *areas,
                    snd_pcm_uframes_t offset,
                    snd_pcm_uframes_t size)
{
    int i;
    signed short *saddr;
    generator_info_t *info = (generator_info_t *) io->private_data;

    saddr = (signed short *)areas->addr;
    for( i = 0; i < size; i ++ ) {
        saddr[ i ] = sin((info->samples + i)*FREQ*2*M_PI/48000)*20000;
    }
    info->samples += size;
    // Our buffer is always full, so next frag is always one before the end
    info->frag = (info->samples / info->fragsize + info->frags - 1) % info->frags;
    return size;
}

static int generator_hw_params(snd_pcm_ioplug_t *io, snd_pcm_hw_params_t *params)
{
    int ret;

    generator_info_t *info = (generator_info_t *) io->private_data;

    ret = snd_pcm_hw_params_get_periods( params, &info->frags, NULL );
    if( ret != EOK ) return ret;
    info->frag = info->frags - 1;
    
    ret = snd_pcm_hw_params_get_period_size( params, &info->fragsize, NULL );
    if( ret != EOK ) return ret;

    return EOK;
}

static int generator_sw_params(snd_pcm_ioplug_t *io, snd_pcm_sw_params_t *params)
{
    return EOK;
}

static int generator_close(snd_pcm_ioplug_t *io)
{
    return EOK;
}

snd_pcm_ioplug_callback_t generator_callback = {
    .start = generator_start,
    .stop = generator_stop,
    .pointer = generator_pointer,
    .transfer = generator_transfer,
    .close = generator_close,
    .hw_params = generator_hw_params,
    .sw_params = generator_sw_params,
};

SND_PCM_PLUGIN_DEFINE_FUNC(sine_generator)
{
    struct generator_info *info;
    int err;
    unsigned int val;
    long rate = 48000;
    snd_config_t *result;

    info = calloc(1, sizeof(*info));
    if (info == NULL)
        return -ENOMEM;

    if( stream != SND_PCM_STREAM_CAPTURE ) {
        free( info);
        return -EINVAL;
    }
    info->ext.version = SND_PCM_IOPLUG_VERSION;
    info->ext.name = "file";
    info->ext.callback = &generator_callback;
    info->ext.private_data = info;

    if( snd_config_search(conf, "rate", &result) == EOK ) {
        snd_config_get_integer(result, &rate);
    }

    err = snd_pcm_ioplug_create(&info->ext, name, stream, mode);
    if (err < 0) {
        generator_free(info);
        return err;
    }

    // Set capabilities
    val = SND_PCM_ACCESS_RW_INTERLEAVED;
    snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_ACCESS, 1, &val);
    val = SND_PCM_FORMAT_S16_LE;
    snd_pcm_ioplug_set_param_list(&info->ext, SND_PCM_IOPLUG_HW_FORMAT, 1, &val);
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_CHANNELS, 1, 1);
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_RATE, rate, rate);
    snd_pcm_ioplug_set_param_minmax(&info->ext, SND_PCM_IOPLUG_HW_PERIOD_BYTES,
            3072,
            3072);

    *pcmp = info->ext.pcm;
    return 0;
}

SND_PCM_PLUGIN_SYMBOL(sine_generator);

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/plugins/sine_generator.c $ $Rev: 781520 $")
#endif
