#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <stdbool.h>
#include <alsa/pcm_extplug.h>
#include <alsa/pcm_external.h>
#include <alsa/pcm_rate.h>
#include "pcm_internal.h"

#define IS_POWER_OF_TWO(x) (!((x) & ((x) - 1)))

struct snd_pcm_rateplug
{
    snd_pcm_access_mask_t accessMask;
    snd_pcm_format_mask_t formatMask;
    unsigned int channelMask;
    unsigned int rate_min;
    unsigned int rate_max;
    int rate_count;
    int *rates;
    unsigned int period_bytes_min;
    unsigned int period_bytes_max;
    unsigned int buffer_bytes_min;
    unsigned int buffer_bytes_max;
    unsigned int periods_min;
    unsigned int periods_max;
    snd_pcm_t *pcm;
    snd_pcm_t *slave;
    void *objp;
    snd_pcm_channel_area_t dst;
    snd_pcm_rate_info_t info;
    snd_pcm_rate_ops_t ops;
    snd_pcm_hw_params_t slaveParams;
    bool useS16;
    int bitwidth;
};

static int rate_hw_params(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    int err;
    int dir = 0;
    unsigned int min_rate = 0, max_rate = 0;
    snd_pcm_access_t access;
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    if( snd_pcm_hw_params_get_format( params, &info->info.in.format ) == EOK
     && snd_pcm_hw_params_get_access( params, &access ) == EOK
     && (info->bitwidth = snd_pcm_format_width( info->info.in.format )) == 16
     && snd_pcm_format_signed( info->info.in.format )
     && access == SND_PCM_ACCESS_RW_INTERLEAVED ) {
        info->useS16 = true;
    } else {
        info->useS16 = false;
    }
    info->info.out.format = info->info.in.format;

    if( (err = snd_pcm_hw_params_get_channels( params, &info->info.channels )) != EOK ) {
        return err;
    }
    if( (err = snd_pcm_hw_params_get_rate( params, &info->info.in.rate, &dir )) != EOK ) {
        return err;
    }
    // For now, choose the nearest rate above
    if( (err = snd_pcm_hw_params_get_rate_min( &info->slaveParams, &min_rate, &dir )) != EOK ) {
        return err;
    }
    if( (err = snd_pcm_hw_params_get_rate_max( &info->slaveParams, &max_rate, &dir )) != EOK ) {
        return err;
    }
    if( info->info.in.rate && info->info.in.rate < min_rate ) {
        info->info.out.rate = min_rate;
    } else if( info->info.in.rate < max_rate ) {
        info->info.out.rate = info->info.in.rate;
    } else {
        info->info.out.rate = max_rate;
    }

    // Choose the same number of channels
    if( (err = snd_pcm_hw_params_set_channels( info->slave, &info->slaveParams, info->info.channels )) != EOK ) {
        return err;
    }

    if( (err = snd_pcm_hw_params_get_period_size( params, &info->info.in.period_size, &dir )) != EOK ) {
        return err;
    }
    info->info.out.period_size = info->info.in.period_size * info->info.out.rate / info->info.in.rate;
    if( (err = snd_pcm_hw_params_get_buffer_size( params, &info->info.in.buffer_size )) != EOK ) {
        return err;
    }
    info->info.out.buffer_size = info->info.in.buffer_size * info->info.out.rate / info->info.in.rate;

    if( (err = info->ops.init(info->objp, &info->info)) != EOK ) {
        return err;
    }

    return snd_pcm_hw_params( info->slave, &info->slaveParams );
}

static int rate_sw_params(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_sw_params(info->slave, params);
}

static int rate_recover(snd_pcm_t *pcm, int err, int silent)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_recover(info->slave, err, silent);
}

static int rate_drain(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_drain(info->slave);
}

static int rate_prepare(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_prepare(info->slave);
}

static int rate_resume(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_resume(info->slave);
}

static int rate_hw_params_any(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    int i;

    snd_pcm_rateplug_t *data = pcm->private_data;

    params->access_modes = data->accessMask.data;

    // Set formats
    params->formats = data->formatMask.data;
    if( IS_POWER_OF_TWO( params->formats ) ) {
        for( i = 0; i < sizeof(params->formats) * _BITS_BYTE; i ++ ) {
            if( params->formats & (1 << i) ) {
                params->format = i;
                break;
            }
        }
    }

    // Set channels
    params->channels = data->channelMask;
    if( IS_POWER_OF_TWO( params->channels ) ) {
        for( i = 0; i < sizeof(params->channels) * _BITS_BYTE; i ++ ) {
            if( params->channels & (1 << i) ) {
                params->voices = i;
                break;
            }
        }
    }

    // Set rate
    params->rates = data->rates;
    params->rate_count = data->rate_count;
    params->rate_min = data->rate_min;
    params->rate_max = data->rate_max;

    // Set period bytes
    params->period_bytes_min = data->period_bytes_min;
    params->period_bytes_max = data->period_bytes_max;

    // Set buffer bytes
    params->buffer_bytes_min = data->buffer_bytes_min;
    params->buffer_bytes_max = data->buffer_bytes_max;

    // Set periods
    params->periods_min = data->periods_min;
    params->periods_max = data->periods_max;

    return EOK;
}

static int rate_sw_params_any(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    return EOK;
}

static snd_pcm_sframes_t rate_writei(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size)
{
    int ret;
    int output_size;

    snd_pcm_rateplug_t *rate = (snd_pcm_rateplug_t *) pcm->private_data;

    output_size = rate->ops.output_frames( rate->objp, size );

    if( (ret = snd_pcm_area_size( &rate->dst, output_size, rate->info.channels, rate->info.in.format )) ) {
        return ret;
    }

    if( rate->useS16 ) {
        rate->ops.convert_s16(rate->objp, rate->dst.addr, output_size,
                buffer, size);
    } else {
        snd_pcm_channel_area_t areas;
        areas.addr = (void *)buffer;
        areas.first = 0;
        areas.step = rate->bitwidth;

        rate->ops.convert(rate->objp,
            &rate->dst, 0, size,
            &areas, 0, size);
    }
    return snd_pcm_writei(rate->slave, rate->dst.addr, output_size) * size / output_size;
}

static int rate_set_rate_resample(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val)
{
    snd_pcm_rateplug_t *rate = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_hw_params_set_rate_resample( rate->slave, params, val );
}

static int rate_poll_descriptors_count(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *rate = pcm->private_data;

    return snd_pcm_poll_descriptors_count( rate->slave );
}

static int rate_poll_descriptors(snd_pcm_t *pcm, struct pollfd *pfds, unsigned int space)
{
    snd_pcm_rateplug_t *rate = pcm->private_data;

    return snd_pcm_poll_descriptors( rate->slave, pfds, space );
}

static int rate_close(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *rate = pcm->private_data;

    snd_pcm_close(rate->slave);
    free(rate);

    return EOK;
}

snd_pcm_plugin_callbacks_t rate_callbacks = {
    .close = rate_close,
    .writei = rate_writei,
    .recover = rate_recover,
    .drain = rate_drain,
    .prepare = rate_prepare,
    .resume = rate_resume,
    .hw_params = rate_hw_params,
    .sw_params = rate_sw_params,
    .hw_params_any = rate_hw_params_any,
    .sw_params_any = rate_sw_params_any,
    .set_rate_resample = rate_set_rate_resample,
    .poll_descriptors_count = rate_poll_descriptors_count,
    .poll_descriptors = rate_poll_descriptors,
};


int snd_pcm_filterplug_create(snd_pcm_rateplug_t *rate, const char *name,
              snd_pcm_stream_t stream, int mode)
{
    int ret = snd_pcm_allocate_pcm(&rate->pcm, &rate_callbacks);
    if( !ret ) {
        rate->rate_max = UINT_MAX;
        rate->period_bytes_max = UINT_MAX;
        rate->buffer_bytes_max = UINT_MAX;
        rate->periods_max = UINT_MAX;
        rate->pcm->private_data = rate;
        rate->pcm->callback_param.rateplug = rate;
    }
    return ret;
}

int _snd_pcm_rate_open(snd_pcm_t **pcmp, const char *name, snd_config_t *root, snd_config_t *conf,
				  snd_pcm_stream_t stream, int mode)
{
    int err;
    struct snd_pcm_rateplug *info;
    snd_config_iterator_t i, next;
    const char *plugin_name = "samplerate";
    const char *slavename = NULL;
    int (*plugin)(unsigned int, void **, snd_pcm_rate_ops_t *);

    snd_config_for_each(i, next, conf) {
        snd_config_t *n = snd_config_iterator_entry(i);
        const char *id;
        if( snd_config_get_id(n, &id) < 0) {
            continue;
        }

        if( strcmp(id, "slavename") == 0 ) {
            err = snd_config_get_string(n, &slavename);
            if( err < 0 ) {
                return err;
            }
        }
    }

    // Rate plugin has to have a slave
    if( slavename == NULL ) {
        return -EINVAL;
    }

    info = calloc(1, sizeof(*info));
    if (info == NULL) {
        return -ENOMEM;
    }

    err = snd_pcm_filterplug_create(info, name, stream, mode);
    if( err < 0 ) {
        free(info);
        return err;
    }

    plugin = snd_pcm_get_rate_plugin(plugin_name);
    if( plugin == NULL ) {
        return -EINVAL;
    }

    // TODO Use proper version
    err = plugin(0, &info->objp, &info->ops);

    if( err < 0 ) {
        free(info);
        return err;
    }

    err = snd_pcm_open(&info->slave, slavename, stream, mode);
    if( err < 0 ) {
        free(info);
        return err;
    }

    // Start with the same capabilities as the slave
    snd_pcm_hw_params_any( info->slave, &info->slaveParams );

    // Copy access methods
    info->accessMask.data = info->slaveParams.access_modes;

    // Copy formats
    info->formatMask.data = info->slaveParams.formats;

    // Copy voices
    info->channelMask = info->slaveParams.channels;

    if( (err = info->ops.get_supported_rates(info->objp, &info->rate_min, &info->rate_max)) != EOK ) {
        free(info);
        return err;
    }

    *pcmp = info->pcm;
    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/rate.c $ $Rev: 781520 $")
#endif
