#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <stdbool.h>
#include <stdint.h>
#include <alsa/asoundlib.h>
#include <alsa/pcm_external.h>
#include <alsa/pcm_rate.h>
#include "pcm_internal.h"

#define IS_POWER_OF_TWO(x) (!((x) & ((x) - 1)))

struct snd_pcm_rateplug
{
    unsigned int rate_min;
    unsigned int rate_max;
    int rate_count;
    int *rates;
    snd_pcm_t *pcm;
    snd_pcm_t *slave;
    void *objp;
    snd_pcm_channel_area_t dst;
    snd_pcm_rate_info_t info;
    snd_pcm_rate_ops_t ops;
    snd_pcm_hw_params_t slaveParams;
    bool useS16;
    bool bypass;
    int bitwidth;
};

static int rate_hw_params(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    int err;
    int dir = 0;
    unsigned int min_rate = 0, max_rate = 0;
    snd_pcm_access_t access;
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;
    snd_pcm_rate_side_info_t *src, *dst;

    if( pcm->stream == SND_PCM_STREAM_PLAYBACK ) {
        src = &info->info.in;
        dst = &info->info.out;
    } else {
        src = &info->info.out;
        dst = &info->info.in;
    }

    if( snd_pcm_hw_params_get_format( params, &src->format ) == EOK
     && snd_pcm_hw_params_get_access( params, &access ) == EOK
     && (info->bitwidth = snd_pcm_format_width( src->format )) == 16
     && snd_pcm_format_signed( src->format )
     && access == SND_PCM_ACCESS_RW_INTERLEAVED ) {
        info->useS16 = true;
    } else {
        info->useS16 = false;
    }

    dst->format = src->format;

    if( (err = snd_pcm_hw_params_get_access( params, &access )) ) {
        return err;
    }

    if( (err = snd_pcm_hw_params_set_access( info->slave, &info->slaveParams, access)) ) {
        return err;
    }

    if( (err = snd_pcm_hw_params_get_channels( params, &info->info.channels )) != EOK ) {
        return err;
    }
    if( (err = snd_pcm_hw_params_get_rate( params, &src->rate, &dir )) != EOK ) {
        return err;
    }
    // For now, choose the nearest rate above
    if( (err = snd_pcm_hw_params_get_rate_min( &info->slaveParams, &min_rate, &dir )) != EOK ) {
        return err;
    }
    if( (err = snd_pcm_hw_params_get_rate_max( &info->slaveParams, &max_rate, &dir )) != EOK ) {
        return err;
    }
    if( src->rate && src->rate < min_rate ) {
        dst->rate = min_rate;
    } else if( src->rate <= max_rate ) {
        info->bypass = 1;
        dst->rate = src->rate;
    } else {
        dst->rate = max_rate;
    }

    // Choose the same number of channels
    if( (err = snd_pcm_hw_params_set_channels( info->slave, &info->slaveParams, info->info.channels )) != EOK ) {
        return err;
    }

    if( (err = snd_pcm_hw_params_set_rate( info->slave, &info->slaveParams, dst->rate, 0)) != EOK ) {
        return err;
    }

    if( (err = snd_pcm_hw_params_get_period_size_max( params, &src->period_size, &dir )) != EOK ) {
        return err;
    }
    // Validate that the period size didn't leave the valid range due to rounding - find the nearest
    // rate in the slave params to the parent params
    dst->period_size = transform_buffer_size( &info->slaveParams, params, src->period_size);
    if( (err = snd_pcm_hw_params_set_period_size_near( info->slave, &info->slaveParams, &dst->period_size, NULL)) != EOK ) {
        return err;
    }
    if( (err = snd_pcm_hw_params_get_buffer_size_max( params, &src->buffer_size )) != EOK ) {
        return err;
    }
    dst->buffer_size = src->buffer_size * dst->rate / src->rate;

    if( (err = snd_pcm_hw_params_set_buffer_size( info->slave, &info->slaveParams, dst->buffer_size )) ) {
        return err;
    }

    if( !info->bypass ) {
        if( (err = info->ops.init(info->objp, &info->info)) != EOK ) {
            return err;
        }
    }

    return snd_pcm_hw_params( info->slave, &info->slaveParams );
}

static int rate_sw_params(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_sw_params(info->slave, params);
}

static int rate_recover(snd_pcm_t *pcm, int err, int silent)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_recover(info->slave, err, silent);
}

static int rate_drain(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_drain( info->slave );
}

static int rate_drop(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_drop( info->slave );
}

static int rate_reset(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_reset( info->slave );
}

static int rate_status(snd_pcm_t *pcm, snd_pcm_status_t *status)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    // TODO - A rate plugin may cause delay due to integer rounding
    return snd_pcm_status( info->slave, status );
}

static snd_pcm_chmap_query_t **rate_query_chmaps(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_query_chmaps( info->slave );
}

static snd_pcm_chmap_t *rate_get_chmap(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_get_chmap( info->slave );
}

static int rate_set_chmap(snd_pcm_t *pcm, const snd_pcm_chmap_t *map)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_set_chmap( info->slave, map );
}

static int rate_start(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_start(info->slave);
}

static snd_pcm_sframes_t rate_avail(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_avail(info->slave);
}

static snd_pcm_sframes_t rate_avail_update(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_avail_update(info->slave);
}

static int rate_prepare(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;
    info->ops.reset(info->objp);

    return snd_pcm_prepare(info->slave);
}

static int rate_nonblock(snd_pcm_t *pcm, int nonblock)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_nonblock(info->slave, nonblock);
}

static int rate_resume(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *info = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_resume(info->slave);
}

static int rate_hw_params_any(snd_pcm_t *pcm, snd_pcm_hw_params_t *params)
{
    snd_pcm_rateplug_t *data = pcm->private_data;

    params->parent = &data->slaveParams;

    // Set rate
    params->rates = data->rates;
    params->rate_count = data->rate_count;
    params->rate_min = data->rate_min;
    params->rate_max = data->rate_max;
    params->params_set = (1 << PARAM_RATE_SET);

    return EOK;
}

static int rate_sw_params_any(snd_pcm_t *pcm, snd_pcm_sw_params_t *params)
{
    return EOK;
}

static snd_pcm_sframes_t rate_readi(snd_pcm_t *pcm, void *buffer, snd_pcm_uframes_t size)
{
    int ret;
    int input_size;

    snd_pcm_rateplug_t *rate = (snd_pcm_rateplug_t *) pcm->private_data;

    if( !rate->bypass ) {
        input_size = rate->ops.input_frames( rate->objp, size );

        if( (ret = snd_pcm_area_size( &rate->dst, input_size, rate->info.channels, rate->info.in.format )) ) {
            return ret;
        }

        if( (ret = snd_pcm_readi(rate->slave, rate->dst.addr, input_size)) != input_size ) {
            return ret;
        }

        if( rate->useS16 ) {
            rate->ops.convert_s16(rate->objp, buffer, size, rate->dst.addr, input_size);
        } else {
            snd_pcm_channel_area_t areas;
            areas.addr = (void *)buffer;
            areas.first = 0;
            areas.step = rate->bitwidth;

            rate->ops.convert(rate->objp,
                &areas, 0, size,
                &rate->dst, 0, size);
        }

        return size;
    } else {
        return snd_pcm_readi(rate->slave, buffer, size);
    }
}

static snd_pcm_sframes_t rate_writei(snd_pcm_t *pcm, const void *buffer, snd_pcm_uframes_t size)
{
    int ret;
    int output_size;

    snd_pcm_rateplug_t *rate = (snd_pcm_rateplug_t *) pcm->private_data;

    if( !rate->bypass ) {
        output_size = rate->ops.output_frames( rate->objp, size );

        if( (ret = snd_pcm_area_size( &rate->dst, output_size, rate->info.channels, rate->info.in.format )) ) {
            return ret;
        }

        if( rate->useS16 ) {
            rate->ops.convert_s16(rate->objp, rate->dst.addr, output_size,
                    buffer, size);
        } else {
            snd_pcm_channel_area_t areas;
            areas.addr = (void *)buffer;
            areas.first = 0;
            areas.step = rate->bitwidth;

            rate->ops.convert(rate->objp,
                &rate->dst, 0, size,
                &areas, 0, size);
        }
        return snd_pcm_writei(rate->slave, rate->dst.addr, output_size) * size / output_size;
    } else {
        return snd_pcm_writei(rate->slave, buffer, size);
    }
}

static int rate_set_rate_resample(snd_pcm_t *pcm, snd_pcm_hw_params_t *params, unsigned int val)
{
    snd_pcm_rateplug_t *rate = (snd_pcm_rateplug_t *) pcm->private_data;

    return snd_pcm_hw_params_set_rate_resample( rate->slave, params, val );
}

static int rate_poll_descriptors_count(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *rate = pcm->private_data;

    return snd_pcm_poll_descriptors_count( rate->slave );
}

static int rate_poll_descriptors(snd_pcm_t *pcm, struct pollfd *pfds, unsigned int space)
{
    snd_pcm_rateplug_t *rate = pcm->private_data;

    return snd_pcm_poll_descriptors( rate->slave, pfds, space );
}

static int rate_close(snd_pcm_t *pcm)
{
    snd_pcm_rateplug_t *rate = pcm->private_data;

    snd_pcm_close( rate->slave );
    rate->ops.free( rate->objp );
    free( rate->objp );
    free( rate->dst.addr );
    free( rate );

    return EOK;
}

snd_pcm_plugin_callbacks_t rate_callbacks = {
    .close = rate_close,
    .readi = rate_readi,
    .writei = rate_writei,
    .recover = rate_recover,
    .drain = rate_drain,
    .drop = rate_drop,
    .reset = rate_reset,
    .status = rate_status,
    .query_chmaps = rate_query_chmaps,
    .get_chmap = rate_get_chmap,
    .set_chmap = rate_set_chmap,
    .start = rate_start,
    .avail = rate_avail,
    .avail_update = rate_avail_update,
    .prepare = rate_prepare,
    .nonblock = rate_nonblock,
    .resume = rate_resume,
    .hw_params = rate_hw_params,
    .sw_params = rate_sw_params,
    .hw_params_any = rate_hw_params_any,
    .sw_params_any = rate_sw_params_any,
    .set_rate_resample = rate_set_rate_resample,
    .poll_descriptors_count = rate_poll_descriptors_count,
    .poll_descriptors = rate_poll_descriptors,
};


int snd_pcm_filterplug_create(snd_pcm_rateplug_t *rate, const char *name,
              snd_pcm_stream_t stream, int mode)
{
    int ret = snd_pcm_allocate_pcm(&rate->pcm, &rate_callbacks);
    if( !ret ) {
        rate->rate_max = UINT_MAX;
        rate->pcm->private_data = rate;
        rate->pcm->callback_param.rateplug = rate;
    }
    return ret;
}

int _snd_pcm_rate_open(snd_pcm_t **pcmp, const char *name, snd_config_t *root, snd_config_t *conf,
				  snd_pcm_stream_t stream, int mode)
{
    int err;
    struct snd_pcm_rateplug *info;
    const char *plugin_name = "samplerate";
    int (*plugin)(unsigned int, void **, snd_pcm_rate_ops_t *);

    if( (err = snd_config_search(conf, "slave", &conf)) < 0 ) {
        return err;
    }

    info = calloc(1, sizeof(*info));
    if (info == NULL) {
        return -ENOMEM;
    }

    err = snd_pcm_filterplug_create(info, name, stream, mode);
    if( err < 0 ) {
        free(info);
        return err;
    }

    plugin = snd_pcm_get_rate_plugin(plugin_name);
    if( plugin == NULL ) {
        return -EINVAL;
    }

    // TODO Use proper version
    err = plugin(0, &info->objp, &info->ops);

    if( err < 0 ) {
        free(info);
        return err;
    }

    err = snd_pcm_open_config(&info->slave, root, conf, stream, mode);
    if( err < 0 ) {
        free(info);
        return err;
    }

    // Start with the same capabilities as the slave
    snd_pcm_hw_params_any( info->slave, &info->slaveParams );

    if( (err = info->ops.get_supported_rates(info->objp, &info->rate_min, &info->rate_max)) != EOK ) {
        free(info);
        return err;
    }

    *pcmp = info->pcm;
    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/alsa/rate_alsa.c $ $Rev: 781520 $")
#endif
