/*
 * $QNXLicenseC:
 * Copyright 2015, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable license fees to QNX
 * Software Systems before you may reproduce, modify or distribute this software,
 * or any work that includes all or part of this software.   Free development
 * licenses are available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review this entire
 * file for other proprietary rights or license notices, as well as the QNX
 * Development Suite License Guide at http://licensing.qnx.com/license-guide/
 * for other information.
 * $
 */

#include <rate_poly.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

//uncomment the line below to enable profiling of the poly rate conversion
//#define POLY_ENABLE_PROFILING

//the line below should be always defined; comment out only for test coverage purposes
#define POLY_ENABLE_STEREO_OPTIMIZATION

// if the compiler flag has included NEON we enable NEON INTRINSICS
#if defined(__ARM_NEON__)
#define POLY_NEON_INTRINSICS
#endif

#if defined(VARIANT_v7) && defined(POLY_NEON_INTRINSICS)
#include <arm_neon.h>
#endif

#if defined(POLY_ENABLE_PROFILING) && defined(VARIANT_v7)
#include <sys/neutrino.h>
#include <sys/syspage.h>
#endif
#include <sys/slog.h>
#include <sys/slogcodes.h>


#define MAGIC_POLY 0xA5A5DEEC

// Number of taps in polyphase filter bank
#define FIR_TAP_16         16
#define FIR_TAP_24         24
#define FIR_TAP_32         32

//uncommment the line below to check delay line array out of bounds accesses
//#define DLYLINE_CHECK_OVERRUN

#ifdef DLYLINE_CHECK_OVERRUN
static int16_t* _dlyLinePtr;
static int _dlySize;
static int _channels;
#define DLYLINE_SET_INFO(dlyLinePtr, dlySize, channels) \
   _dlyLinePtr = (int16_t*)dlyLinePtr; _dlySize = dlySize; _channels = channels
#define DLYLINE_CHECK_IDX(dlyIdx) \
   if( dlyIdx < 0 || dlyIdx >= _dlySize ) slogf (_SLOGC_AUDIO, _SLOG_ERROR, "dlyIdx overrun: dlyIdx=%d, dlySize=%d", dlyIdx, _dlySize)
#define DLYLINE_CHECK_PTR(dlyPtr)\
   if( ( ( (_channels%2) == 0 ) && ( (int16_t*)dlyPtr < _dlyLinePtr || (int16_t*)dlyPtr >= _dlyLinePtr + 2*_dlySize ) ) || \
       ( ( (_channels%2) == 1 ) && ( (int16_t*)dlyPtr < _dlyLinePtr || (int16_t*)dlyPtr >= _dlyLinePtr + _dlySize ) ) ) \
       slogf (_SLOGC_AUDIO, _SLOG_ERROR, "dlyPtr overrun: dlyPtr=%p, dlyLinePtr=%p, dlySize=%d, channels=%d", dlyPtr, _dlyLinePtr, _dlySize, _channels)
#else
#define DLYLINE_SET_INFO(dlyLinePtr, dlySize, channels)
#define DLYLINE_CHECK_IDX(dlyIdx)
#define DLYLINE_CHECK_PTR(dlyPtr)
#endif

//-------------- Polyphase filter coefficients in 1.15 format-------------------
// FIR filters with kaiser window (alpha = 8.0)

/* Matlab code:

function [h, N]=makeFIR(ntaps, L, M, cutoffAdj, kaiserAlpha)
    if (L==1)
        N = ntaps*M;
    else
        N = ntaps*L;
    end
    h = fir1(N-1,cutoffAdj*min(1/M,1/L),kaiser(N,kaiserAlpha));
    h = h*L;
    outputFIR(h, N);
end

function outputFIR(h, N)
    int16Arr = int16(h*32768);
    str = '';
    for xx=1:N/2
        if ((xx~=0) && mod((xx-1),12) == 0)
            fmtStr = '\n    0x%s,';
        else
            fmtStr = ' 0x%s,';
        end
        str = strcat(str, sprintf(fmtStr, dec2hex(typecast(int16Arr(xx), 'uint16'),4)));
    end
    str = lower(str);
    disp(str);
end

*/

// ----- 6 : 1 ---------------
// makeFIR(24, 6, 1, .9, 8);
static const int16_t fir_6_1[ 6 * FIR_TAP_24 / 2] = {
    0x0002, 0x0003, 0x0004, 0x0005, 0x0003, 0xffff, 0xfff9, 0xfff2, 0xffec, 0xffea, 0xffed, 0xfff8,
    0x0009, 0x001f, 0x0034, 0x0041, 0x0040, 0x002d, 0x0008, 0xffd5, 0xffa0, 0xff74, 0xff63, 0xff76,
    0xffb2, 0x0012, 0x0084, 0x00f0, 0x0136, 0x013e, 0x00f7, 0x0062, 0xff95, 0xfeb8, 0xfe01, 0xfda5,
    0xfdcd, 0xfe88, 0xffc2, 0x0146, 0x02c1, 0x03d3, 0x0426, 0x0384, 0x01eb, 0xff93, 0xfcef, 0xfa96,
    0xf927, 0xf922, 0xfac6, 0xfdf7, 0x0234, 0x06a7, 0x0a48, 0x0c12, 0x0b42, 0x078b, 0x013e, 0xf954,
    0xf156, 0xeb20, 0xe895, 0xeb43, 0xf408, 0x02dc, 0x16b7, 0x2da3, 0x4502, 0x59ee, 0x69ac, 0x721e,
};


// makeFIR(24, 1, 6, .92, 8);
static const int16_t fir_1_6[ 6 * FIR_TAP_24 / 2] = {
    0x0000, 0x0000, 0x0001, 0x0001, 0x0001, 0x0001, 0x0000, 0xffff, 0xfffe, 0xfffc, 0xfffb, 0xfffc,
    0xfffd, 0x0001, 0x0005, 0x0009, 0x000c, 0x000d, 0x000a, 0x0003, 0xfff9, 0xffef, 0xffe6, 0xffe2,
    0xffe6, 0xfff1, 0x0003, 0x0019, 0x002d, 0x0039, 0x003a, 0x002b, 0x000e, 0xffe7, 0xffbf, 0xffa0,
    0xff94, 0xffa1, 0xffca, 0x0007, 0x004f, 0x008f, 0x00b5, 0x00b5, 0x0086, 0x002e, 0xffba, 0xff44,
    0xfeea, 0xfec7, 0xfeec, 0xff5e, 0x000d, 0x00d7, 0x018e, 0x0202, 0x020a, 0x018d, 0x0093, 0xff40,
    0xfdd3, 0xfca2, 0xfc06, 0xfc4b, 0xfda2, 0x0010, 0x036c, 0x0761, 0x0b76, 0x0f25, 0x11ef, 0x136f,
};

// ----- 4 : 1 ---------------
// makeFIR(24, 4, 1, .9, 8);
static const int16_t fir_4_1[ 4 * FIR_TAP_24 / 2] = {
    0x0002, 0x0004, 0x0004, 0x0000, 0xfff8, 0xffee, 0xffeb, 0xfff5, 0x000e, 0x002e, 0x0040, 0x0032,
    0xfffc, 0xffae, 0xff70, 0xff70, 0xffc8, 0x0066, 0x0103, 0x0140, 0x00d6, 0xffcc, 0xfe88, 0xfdb6,
    0xfdf2, 0xff6b, 0x01a7, 0x0398, 0x0410, 0x0263, 0xfeed, 0xfb20, 0xf905, 0xfa39, 0xfef5, 0x0591,
    0x0aec, 0x0bb7, 0x062d, 0xfb63, 0xef8d, 0xe8cb, 0xecdf, 0xfea0, 0x1c3f, 0x3f47, 0x5e72, 0x70c5,
};

// makeFIR(24, 1, 4, .92, 8);
static const int16_t fir_1_4[ 4 * FIR_TAP_24 / 2] = {
    0x0000, 0x0001, 0x0002, 0x0002, 0x0000, 0xfffd, 0xfffa, 0xfff9, 0xfffd, 0x0006, 0x000f, 0x0013,
    0x000c, 0xfffa, 0xffe3, 0xffd5, 0xffdc, 0xfffd, 0x002d, 0x0052, 0x0052, 0x0021, 0xffcb, 0xff7a,
    0xff60, 0xff9c, 0x0025, 0x00bf, 0x0115, 0x00e1, 0x001c, 0xff12, 0xfe4a, 0xfe4b, 0xff48, 0x00f5,
    0x028c, 0x031c, 0x0206, 0xff67, 0xfc3d, 0xfa29, 0xfaca, 0xff0a, 0x068f, 0x0faf, 0x17ea, 0x1ccb,
};


// ----- 3 : 1 ---------------
// makeFIR(24, 3, 1, .9, 8);
static const int16_t fir_3_1[ 3 * FIR_TAP_24 / 2] = {
    0x0002, 0x0004, 0x0001, 0xfff7, 0xffec, 0xfff3, 0x0013, 0x0038, 0x0036, 0xfff0, 0xff8c, 0xff6c,
    0xffdf, 0x00b8, 0x013c, 0x00b1, 0xff29, 0xfdce, 0xfe1f, 0x007e, 0x0353, 0x03eb, 0x00cf, 0xfbb6,
    0xf8fb, 0xfc37, 0x0473, 0x0b6e, 0x09bc, 0xfd68, 0xede8, 0xe937, 0xfabb, 0x21f0, 0x4ff3, 0x6ee7,
};

// makeFIR(24, 1, 3, .92, 8);
static const int16_t fir_1_3[ 3 * FIR_TAP_24 / 2] = {
    0x0000, 0x0001, 0x0002, 0x0000, 0xfffa, 0xfff7, 0xfffe, 0x000e, 0x0019, 0x000d, 0xffe9, 0xffc9,
    0xffd6, 0x001b, 0x0066, 0x0066, 0xfff6, 0xff5e, 0xff31, 0xffce, 0x00df, 0x0172, 0x00bc, 0xfefd,
    0xfda4, 0xfe3b, 0x00e1, 0x03a4, 0x03b6, 0xffe2, 0xfa5d, 0xf813, 0xfd70, 0x0ab4, 0x1ac5, 0x25b9,
};


// ----- 2 : 1 ---------------
// makeFIR(24, 2, 1, .9, 8);
static const int16_t fir_2_1[ 2 * FIR_TAP_24 / 2] = {
    0x0002, 0x0002, 0xfff5, 0xfff0, 0x001c, 0x003a, 0xffd9, 0xff6d, 0x0011, 0x0128, 0x005e, 0xfe12,
    0xfe92, 0x02b1, 0x0374, 0xfcfa, 0xf935, 0x0230, 0x0c00, 0x013d, 0xeb2c, 0xf40b, 0x2d9e, 0x69aa,
};

// makeFIR(24, 1, 2, .92, 8);
static const int16_t fir_1_2[ 2 * FIR_TAP_24 / 2] = {
    0x0001, 0x0003, 0xfffe, 0xfff4, 0x0002, 0x0022, 0x0008, 0xffb8, 0xffd5, 0x0080, 0x007b, 0xff43,
    0xfeeb, 0x00e7, 0x0215, 0xff31, 0xfc5e, 0x0026, 0x05fe, 0x01b8, 0xf5ec, 0xf8e9, 0x1620, 0x35cc,
};

// ----- 2 : 3 ---------------
// makeFIR(32, 2, 3, .95, 8);
static const int16_t fir_2_3[ 2 * FIR_TAP_32 / 2] = {
    0x0000, 0xfffd, 0xfffa, 0xffff, 0x000f, 0x001a, 0x0009, 0xffda, 0xffb5, 0xffd9, 0x004b, 0x00ab,
    0x0072, 0xff8a, 0xfeae, 0xfeef, 0x0092, 0x0256, 0x023c, 0xff8a, 0xfc2e, 0xfbaf, 0xffd7, 0x05fd,
    0x0819, 0x01f2, 0xf66b, 0xefb1, 0xf894, 0x1361, 0x35b3, 0x4db1,
};



// ----- 48 : 11 ---------------
// makeFIR(24, 48, 11, .9, 8);
static const int16_t fir_48_11[ 48 * FIR_TAP_24 / 2] = {
    0x0001, 0x0001, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0004, 0x0004, 0x0004, 0x0004,
    0x0003, 0x0003, 0x0002, 0x0002, 0x0001, 0x0001, 0x0000, 0xffff, 0xffff, 0xfffe, 0xfffd, 0xfffd,
    0xfffc, 0xfffb, 0xfffa, 0xfff9, 0xfff8, 0xfff7, 0xfff6, 0xfff5, 0xfff4, 0xfff4, 0xfff3, 0xfff2,
    0xfff1, 0xfff0, 0xffef, 0xffee, 0xffed, 0xffed, 0xffec, 0xffeb, 0xffeb, 0xffea, 0xffea, 0xffe9,
    0xffe9, 0xffe9, 0xffe8, 0xffe8, 0xffe8, 0xffe9, 0xffe9, 0xffe9, 0xffea, 0xffea, 0xffeb, 0xffec,
    0xffed, 0xffee, 0xffef, 0xfff0, 0xfff2, 0xfff3, 0xfff5, 0xfff7, 0xfff8, 0xfffa, 0xfffd, 0xffff,
    0x0001, 0x0004, 0x0006, 0x0009, 0x000b, 0x000e, 0x0011, 0x0014, 0x0017, 0x0019, 0x001c, 0x001f,
    0x0022, 0x0025, 0x0028, 0x002a, 0x002d, 0x0030, 0x0032, 0x0035, 0x0037, 0x0039, 0x003b, 0x003d,
    0x003f, 0x0040, 0x0042, 0x0043, 0x0044, 0x0044, 0x0045, 0x0045, 0x0045, 0x0045, 0x0044, 0x0043,
    0x0042, 0x0040, 0x003e, 0x003c, 0x003a, 0x0037, 0x0034, 0x0030, 0x002d, 0x0029, 0x0024, 0x0020,
    0x001b, 0x0016, 0x0011, 0x000b, 0x0005, 0xffff, 0xfff9, 0xfff2, 0xffec, 0xffe5, 0xffde, 0xffd7,
    0xffd0, 0xffc9, 0xffc2, 0xffbb, 0xffb4, 0xffad, 0xffa6, 0xffa0, 0xff99, 0xff93, 0xff8d, 0xff87,
    0xff81, 0xff7c, 0xff77, 0xff72, 0xff6e, 0xff6a, 0xff67, 0xff64, 0xff62, 0xff60, 0xff5f, 0xff5e,
    0xff5e, 0xff5f, 0xff60, 0xff62, 0xff64, 0xff67, 0xff6b, 0xff70, 0xff75, 0xff7b, 0xff81, 0xff88,
    0xff90, 0xff98, 0xffa1, 0xffab, 0xffb5, 0xffc0, 0xffcc, 0xffd7, 0xffe4, 0xfff1, 0xfffe, 0x000b,
    0x0019, 0x0027, 0x0036, 0x0045, 0x0053, 0x0062, 0x0071, 0x0080, 0x008f, 0x009d, 0x00ac, 0x00ba,
    0x00c8, 0x00d5, 0x00e2, 0x00ef, 0x00fb, 0x0106, 0x0111, 0x011b, 0x0124, 0x012c, 0x0134, 0x013a,
    0x0140, 0x0144, 0x0147, 0x0149, 0x014a, 0x014a, 0x0149, 0x0146, 0x0142, 0x013d, 0x0137, 0x012f,
    0x0126, 0x011b, 0x010f, 0x0102, 0x00f4, 0x00e4, 0x00d4, 0x00c2, 0x00af, 0x009a, 0x0085, 0x006f,
    0x0058, 0x0040, 0x0027, 0x000d, 0xfff3, 0xffd8, 0xffbd, 0xffa1, 0xff85, 0xff69, 0xff4c, 0xff30,
    0xff14, 0xfef8, 0xfedc, 0xfec1, 0xfea6, 0xfe8c, 0xfe73, 0xfe5a, 0xfe43, 0xfe2c, 0xfe17, 0xfe03,
    0xfdf1, 0xfde0, 0xfdd1, 0xfdc3, 0xfdb7, 0xfdad, 0xfda5, 0xfd9f, 0xfd9b, 0xfd9a, 0xfd9a, 0xfd9d,
    0xfda2, 0xfda9, 0xfdb3, 0xfdbf, 0xfdcd, 0xfdde, 0xfdf1, 0xfe07, 0xfe1f, 0xfe39, 0xfe55, 0xfe73,
    0xfe94, 0xfeb6, 0xfedb, 0xff01, 0xff29, 0xff53, 0xff7e, 0xffab, 0xffd9, 0x0008, 0x0038, 0x0069,
    0x009a, 0x00cc, 0x00fe, 0x0131, 0x0163, 0x0195, 0x01c7, 0x01f8, 0x0228, 0x0257, 0x0285, 0x02b2,
    0x02dd, 0x0307, 0x032e, 0x0353, 0x0376, 0x0397, 0x03b4, 0x03cf, 0x03e7, 0x03fc, 0x040e, 0x041c,
    0x0427, 0x042e, 0x0431, 0x0430, 0x042c, 0x0424, 0x0417, 0x0407, 0x03f2, 0x03da, 0x03bd, 0x039d,
    0x0378, 0x0350, 0x0323, 0x02f3, 0x02bf, 0x0288, 0x024d, 0x020e, 0x01cd, 0x0189, 0x0142, 0x00f8,
    0x00ac, 0x005e, 0x000e, 0xffbc, 0xff69, 0xff14, 0xfebf, 0xfe6a, 0xfe14, 0xfdbe, 0xfd69, 0xfd14,
    0xfcc1, 0xfc6e, 0xfc1e, 0xfbcf, 0xfb83, 0xfb39, 0xfaf2, 0xfaaf, 0xfa6f, 0xfa32, 0xf9fb, 0xf9c7,
    0xf998, 0xf96f, 0xf94a, 0xf92b, 0xf911, 0xf8fe, 0xf8f0, 0xf8e9, 0xf8e8, 0xf8ee, 0xf8fa, 0xf90d,
    0xf927, 0xf948, 0xf96f, 0xf99d, 0xf9d2, 0xfa0e, 0xfa50, 0xfa99, 0xfae9, 0xfb3e, 0xfb9a, 0xfbfc,
    0xfc63, 0xfcd0, 0xfd42, 0xfdb8, 0xfe33, 0xfeb2, 0xff35, 0xffbc, 0x0045, 0x00d1, 0x015f, 0x01ee,
    0x027f, 0x0310, 0x03a2, 0x0433, 0x04c3, 0x0552, 0x05de, 0x0669, 0x06f0, 0x0774, 0x07f3, 0x086e,
    0x08e4, 0x0954, 0x09be, 0x0a21, 0x0a7d, 0x0ad1, 0x0b1d, 0x0b60, 0x0b9a, 0x0bcb, 0x0bf2, 0x0c0f,
    0x0c22, 0x0c2a, 0x0c27, 0x0c18, 0x0bfe, 0x0bd9, 0x0ba8, 0x0b6b, 0x0b22, 0x0ace, 0x0a6e, 0x0a01,
    0x098a, 0x0907, 0x0878, 0x07df, 0x073b, 0x068c, 0x05d4, 0x0512, 0x0447, 0x0373, 0x0297, 0x01b3,
    0x00c8, 0xffd7, 0xfee1, 0xfde5, 0xfce6, 0xfbe3, 0xfadd, 0xf9d6, 0xf8ce, 0xf7c6, 0xf6bf, 0xf5b9,
    0xf4b7, 0xf3b8, 0xf2be, 0xf1c9, 0xf0db, 0xeff5, 0xef18, 0xee44, 0xed7a, 0xecbc, 0xec0b, 0xeb67,
    0xead2, 0xea4b, 0xe9d5, 0xe970, 0xe91d, 0xe8dc, 0xe8af, 0xe895, 0xe891, 0xe8a2, 0xe8c9, 0xe907,
    0xe95b, 0xe9c7, 0xea4b, 0xeae8, 0xeb9d, 0xec6b, 0xed51, 0xee51, 0xef6a, 0xf09c, 0xf1e7, 0xf34b,
    0xf4c8, 0xf65d, 0xf80a, 0xf9cf, 0xfbab, 0xfd9e, 0xffa7, 0x01c6, 0x03f9, 0x0640, 0x089b, 0x0b07,
    0x0d85, 0x1014, 0x12b2, 0x155e, 0x1817, 0x1adc, 0x1dab, 0x2084, 0x2365, 0x264d, 0x293a, 0x2c2c,
    0x2f20, 0x3215, 0x3509, 0x37fc, 0x3aec, 0x3dd7, 0x40bc, 0x4399, 0x466d, 0x4937, 0x4bf4, 0x4ea4,
    0x5145, 0x53d6, 0x5655, 0x58c1, 0x5b18, 0x5d5b, 0x5f86, 0x619a, 0x6394, 0x6574, 0x6739, 0x68e2,
    0x6a6e, 0x6bdc, 0x6d2b, 0x6e5b, 0x6f6b, 0x705a, 0x7128, 0x71d4, 0x725f, 0x72c7, 0x730c, 0x732f,
};

// ----- 160 : 147 ---------------
//makeFIR(32, 160, 147, .965, 10);
// Meets +/- 2dB frequency response tolerance of GIS-304-108 at 20 kHz (resampling from 44.1k to 48k).
// Stop-band rejection is >85 dB (20 to 20kHz), except where aliasing occurs above 20kHz i.e. inaudible.
// Previous settings were makeFIR(24, 160, 147, .9, 8), however this resulted in -6.5 dB roll-off
// at 20kHz.
static const int16_t fir_160_147[ (160 * FIR_TAP_32) / 2] = {
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0001, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfffa, 0xfffa, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9,
    0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9,
    0xfff9, 0xfff9, 0xfff9, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xfffe, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001,
    0x0001, 0x0001, 0x0001, 0x0001, 0x0002, 0x0002, 0x0002, 0x0002, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0004, 0x0004, 0x0004, 0x0004, 0x0005, 0x0005, 0x0005, 0x0005, 0x0006, 0x0006, 0x0006,
    0x0006, 0x0007, 0x0007, 0x0007, 0x0007, 0x0008, 0x0008, 0x0008, 0x0008, 0x0009, 0x0009, 0x0009,
    0x0009, 0x0009, 0x000a, 0x000a, 0x000a, 0x000a, 0x000b, 0x000b, 0x000b, 0x000b, 0x000c, 0x000c,
    0x000c, 0x000c, 0x000c, 0x000d, 0x000d, 0x000d, 0x000d, 0x000d, 0x000e, 0x000e, 0x000e, 0x000e,
    0x000e, 0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010,
    0x0010, 0x0010, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011,
    0x0011, 0x0011, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012,
    0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011, 0x0011,
    0x0011, 0x0011, 0x0011, 0x0011, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x000f, 0x000f,
    0x000f, 0x000f, 0x000f, 0x000e, 0x000e, 0x000e, 0x000e, 0x000d, 0x000d, 0x000d, 0x000d, 0x000c,
    0x000c, 0x000c, 0x000b, 0x000b, 0x000b, 0x000a, 0x000a, 0x000a, 0x0009, 0x0009, 0x0009, 0x0008,
    0x0008, 0x0008, 0x0007, 0x0007, 0x0006, 0x0006, 0x0005, 0x0005, 0x0005, 0x0004, 0x0004, 0x0003,
    0x0003, 0x0002, 0x0002, 0x0001, 0x0001, 0x0000, 0x0000, 0xffff, 0xffff, 0xfffe, 0xfffe, 0xfffd,
    0xfffd, 0xfffc, 0xfffc, 0xfffb, 0xfffb, 0xfffa, 0xfffa, 0xfff9, 0xfff9, 0xfff8, 0xfff7, 0xfff7,
    0xfff6, 0xfff6, 0xfff5, 0xfff5, 0xfff4, 0xfff4, 0xfff3, 0xfff2, 0xfff2, 0xfff1, 0xfff1, 0xfff0,
    0xfff0, 0xffef, 0xffef, 0xffee, 0xffed, 0xffed, 0xffec, 0xffec, 0xffeb, 0xffeb, 0xffea, 0xffea,
    0xffe9, 0xffe9, 0xffe8, 0xffe7, 0xffe7, 0xffe6, 0xffe6, 0xffe5, 0xffe5, 0xffe4, 0xffe4, 0xffe4,
    0xffe3, 0xffe3, 0xffe2, 0xffe2, 0xffe1, 0xffe1, 0xffe0, 0xffe0, 0xffe0, 0xffdf, 0xffdf, 0xffde,
    0xffde, 0xffde, 0xffdd, 0xffdd, 0xffdd, 0xffdc, 0xffdc, 0xffdc, 0xffdb, 0xffdb, 0xffdb, 0xffdb,
    0xffda, 0xffda, 0xffda, 0xffda, 0xffd9, 0xffd9, 0xffd9, 0xffd9, 0xffd9, 0xffd9, 0xffd9, 0xffd8,
    0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8, 0xffd8,
    0xffd9, 0xffd9, 0xffd9, 0xffd9, 0xffd9, 0xffd9, 0xffda, 0xffda, 0xffda, 0xffda, 0xffdb, 0xffdb,
    0xffdb, 0xffdb, 0xffdc, 0xffdc, 0xffdc, 0xffdd, 0xffdd, 0xffde, 0xffde, 0xffdf, 0xffdf, 0xffdf,
    0xffe0, 0xffe0, 0xffe1, 0xffe2, 0xffe2, 0xffe3, 0xffe3, 0xffe4, 0xffe5, 0xffe5, 0xffe6, 0xffe7,
    0xffe7, 0xffe8, 0xffe9, 0xffe9, 0xffea, 0xffeb, 0xffec, 0xffed, 0xffed, 0xffee, 0xffef, 0xfff0,
    0xfff1, 0xfff2, 0xfff3, 0xfff3, 0xfff4, 0xfff5, 0xfff6, 0xfff7, 0xfff8, 0xfff9, 0xfffa, 0xfffb,
    0xfffc, 0xfffd, 0xfffe, 0xffff, 0x0000, 0x0001, 0x0002, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008,
    0x0009, 0x000a, 0x000b, 0x000c, 0x000e, 0x000f, 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0016,
    0x0017, 0x0018, 0x0019, 0x001a, 0x001b, 0x001d, 0x001e, 0x001f, 0x0020, 0x0021, 0x0022, 0x0023,
    0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002f, 0x0030, 0x0031,
    0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d,
    0x003d, 0x003e, 0x003f, 0x0040, 0x0041, 0x0042, 0x0042, 0x0043, 0x0044, 0x0045, 0x0045, 0x0046,
    0x0047, 0x0047, 0x0048, 0x0049, 0x0049, 0x004a, 0x004a, 0x004b, 0x004b, 0x004c, 0x004c, 0x004d,
    0x004d, 0x004d, 0x004e, 0x004e, 0x004e, 0x004f, 0x004f, 0x004f, 0x004f, 0x004f, 0x004f, 0x0050,
    0x0050, 0x0050, 0x0050, 0x0050, 0x0050, 0x004f, 0x004f, 0x004f, 0x004f, 0x004f, 0x004e, 0x004e,
    0x004e, 0x004e, 0x004d, 0x004d, 0x004c, 0x004c, 0x004b, 0x004b, 0x004a, 0x004a, 0x0049, 0x0048,
    0x0048, 0x0047, 0x0046, 0x0045, 0x0044, 0x0044, 0x0043, 0x0042, 0x0041, 0x0040, 0x003f, 0x003e,
    0x003d, 0x003b, 0x003a, 0x0039, 0x0038, 0x0037, 0x0035, 0x0034, 0x0033, 0x0031, 0x0030, 0x002f,
    0x002d, 0x002c, 0x002a, 0x0029, 0x0027, 0x0025, 0x0024, 0x0022, 0x0021, 0x001f, 0x001d, 0x001b,
    0x001a, 0x0018, 0x0016, 0x0014, 0x0012, 0x0011, 0x000f, 0x000d, 0x000b, 0x0009, 0x0007, 0x0005,
    0x0003, 0x0001, 0xffff, 0xfffd, 0xfffb, 0xfff9, 0xfff7, 0xfff5, 0xfff2, 0xfff0, 0xffee, 0xffec,
    0xffea, 0xffe8, 0xffe6, 0xffe3, 0xffe1, 0xffdf, 0xffdd, 0xffdb, 0xffd9, 0xffd6, 0xffd4, 0xffd2,
    0xffd0, 0xffce, 0xffcc, 0xffc9, 0xffc7, 0xffc5, 0xffc3, 0xffc1, 0xffbf, 0xffbc, 0xffba, 0xffb8,
    0xffb6, 0xffb4, 0xffb2, 0xffb0, 0xffae, 0xffac, 0xffaa, 0xffa8, 0xffa6, 0xffa4, 0xffa2, 0xffa0,
    0xff9e, 0xff9c, 0xff9a, 0xff99, 0xff97, 0xff95, 0xff93, 0xff92, 0xff90, 0xff8e, 0xff8d, 0xff8b,
    0xff8a, 0xff88, 0xff87, 0xff85, 0xff84, 0xff82, 0xff81, 0xff80, 0xff7e, 0xff7d, 0xff7c, 0xff7b,
    0xff7a, 0xff79, 0xff78, 0xff77, 0xff76, 0xff75, 0xff74, 0xff73, 0xff73, 0xff72, 0xff71, 0xff71,
    0xff70, 0xff70, 0xff6f, 0xff6f, 0xff6e, 0xff6e, 0xff6e, 0xff6e, 0xff6e, 0xff6d, 0xff6d, 0xff6e,
    0xff6e, 0xff6e, 0xff6e, 0xff6e, 0xff6f, 0xff6f, 0xff6f, 0xff70, 0xff70, 0xff71, 0xff72, 0xff72,
    0xff73, 0xff74, 0xff75, 0xff76, 0xff77, 0xff78, 0xff79, 0xff7a, 0xff7b, 0xff7d, 0xff7e, 0xff7f,
    0xff81, 0xff82, 0xff84, 0xff86, 0xff87, 0xff89, 0xff8b, 0xff8d, 0xff8f, 0xff91, 0xff93, 0xff95,
    0xff97, 0xff99, 0xff9b, 0xff9e, 0xffa0, 0xffa2, 0xffa5, 0xffa7, 0xffaa, 0xffad, 0xffaf, 0xffb2,
    0xffb5, 0xffb7, 0xffba, 0xffbd, 0xffc0, 0xffc3, 0xffc6, 0xffc9, 0xffcc, 0xffcf, 0xffd3, 0xffd6,
    0xffd9, 0xffdc, 0xffe0, 0xffe3, 0xffe6, 0xffea, 0xffed, 0xfff1, 0xfff4, 0xfff8, 0xfffb, 0xffff,
    0x0003, 0x0006, 0x000a, 0x000e, 0x0011, 0x0015, 0x0019, 0x001d, 0x0020, 0x0024, 0x0028, 0x002c,
    0x0030, 0x0033, 0x0037, 0x003b, 0x003f, 0x0043, 0x0046, 0x004a, 0x004e, 0x0052, 0x0056, 0x005a,
    0x005d, 0x0061, 0x0065, 0x0069, 0x006d, 0x0070, 0x0074, 0x0078, 0x007b, 0x007f, 0x0083, 0x0086,
    0x008a, 0x008d, 0x0091, 0x0094, 0x0098, 0x009b, 0x009f, 0x00a2, 0x00a5, 0x00a9, 0x00ac, 0x00af,
    0x00b2, 0x00b5, 0x00b8, 0x00bb, 0x00be, 0x00c1, 0x00c4, 0x00c7, 0x00ca, 0x00cc, 0x00cf, 0x00d2,
    0x00d4, 0x00d6, 0x00d9, 0x00db, 0x00dd, 0x00e0, 0x00e2, 0x00e4, 0x00e6, 0x00e7, 0x00e9, 0x00eb,
    0x00ed, 0x00ee, 0x00f0, 0x00f1, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7, 0x00f8, 0x00f9, 0x00fa,
    0x00fa, 0x00fb, 0x00fb, 0x00fc, 0x00fc, 0x00fc, 0x00fc, 0x00fc, 0x00fc, 0x00fc, 0x00fc, 0x00fc,
    0x00fb, 0x00fb, 0x00fa, 0x00f9, 0x00f8, 0x00f7, 0x00f6, 0x00f5, 0x00f4, 0x00f3, 0x00f1, 0x00f0,
    0x00ee, 0x00ec, 0x00eb, 0x00e9, 0x00e7, 0x00e5, 0x00e2, 0x00e0, 0x00de, 0x00db, 0x00d9, 0x00d6,
    0x00d3, 0x00d0, 0x00cd, 0x00ca, 0x00c7, 0x00c4, 0x00c0, 0x00bd, 0x00b9, 0x00b6, 0x00b2, 0x00ae,
    0x00aa, 0x00a6, 0x00a2, 0x009e, 0x009a, 0x0095, 0x0091, 0x008d, 0x0088, 0x0083, 0x007f, 0x007a,
    0x0075, 0x0070, 0x006b, 0x0066, 0x0061, 0x005c, 0x0056, 0x0051, 0x004c, 0x0046, 0x0041, 0x003b,
    0x0035, 0x0030, 0x002a, 0x0024, 0x001e, 0x0018, 0x0013, 0x000d, 0x0007, 0x0000, 0xfffa, 0xfff4,
    0xffee, 0xffe8, 0xffe2, 0xffdb, 0xffd5, 0xffcf, 0xffc9, 0xffc2, 0xffbc, 0xffb6, 0xffaf, 0xffa9,
    0xffa2, 0xff9c, 0xff96, 0xff8f, 0xff89, 0xff83, 0xff7c, 0xff76, 0xff70, 0xff69, 0xff63, 0xff5d,
    0xff56, 0xff50, 0xff4a, 0xff44, 0xff3e, 0xff38, 0xff32, 0xff2c, 0xff26, 0xff20, 0xff1a, 0xff14,
    0xff0e, 0xff08, 0xff03, 0xfefd, 0xfef8, 0xfef2, 0xfeed, 0xfee7, 0xfee2, 0xfedd, 0xfed8, 0xfed3,
    0xfece, 0xfec9, 0xfec4, 0xfec0, 0xfebb, 0xfeb6, 0xfeb2, 0xfeae, 0xfeaa, 0xfea5, 0xfea1, 0xfe9e,
    0xfe9a, 0xfe96, 0xfe93, 0xfe8f, 0xfe8c, 0xfe89, 0xfe85, 0xfe82, 0xfe80, 0xfe7d, 0xfe7a, 0xfe78,
    0xfe76, 0xfe73, 0xfe71, 0xfe6f, 0xfe6e, 0xfe6c, 0xfe6a, 0xfe69, 0xfe68, 0xfe67, 0xfe66, 0xfe65,
    0xfe64, 0xfe64, 0xfe63, 0xfe63, 0xfe63, 0xfe63, 0xfe64, 0xfe64, 0xfe65, 0xfe65, 0xfe66, 0xfe67,
    0xfe68, 0xfe6a, 0xfe6b, 0xfe6d, 0xfe6f, 0xfe71, 0xfe73, 0xfe75, 0xfe78, 0xfe7a, 0xfe7d, 0xfe80,
    0xfe83, 0xfe86, 0xfe8a, 0xfe8d, 0xfe91, 0xfe95, 0xfe99, 0xfe9d, 0xfea1, 0xfea6, 0xfeab, 0xfeaf,
    0xfeb4, 0xfeb9, 0xfebf, 0xfec4, 0xfeca, 0xfecf, 0xfed5, 0xfedb, 0xfee1, 0xfee7, 0xfeee, 0xfef4,
    0xfefb, 0xff02, 0xff09, 0xff10, 0xff17, 0xff1e, 0xff26, 0xff2d, 0xff35, 0xff3d, 0xff45, 0xff4d,
    0xff55, 0xff5d, 0xff66, 0xff6e, 0xff77, 0xff7f, 0xff88, 0xff91, 0xff9a, 0xffa3, 0xffac, 0xffb5,
    0xffbe, 0xffc8, 0xffd1, 0xffdb, 0xffe4, 0xffee, 0xfff7, 0x0001, 0x000b, 0x0015, 0x001f, 0x0029,
    0x0033, 0x003d, 0x0047, 0x0051, 0x005b, 0x0065, 0x006f, 0x0079, 0x0083, 0x008d, 0x0097, 0x00a2,
    0x00ac, 0x00b6, 0x00c0, 0x00ca, 0x00d4, 0x00de, 0x00e8, 0x00f2, 0x00fc, 0x0106, 0x0110, 0x011a,
    0x0124, 0x012e, 0x0137, 0x0141, 0x014a, 0x0154, 0x015d, 0x0167, 0x0170, 0x0179, 0x0182, 0x018b,
    0x0194, 0x019d, 0x01a6, 0x01ae, 0x01b7, 0x01bf, 0x01c7, 0x01cf, 0x01d7, 0x01df, 0x01e7, 0x01ee,
    0x01f6, 0x01fd, 0x0204, 0x020b, 0x0212, 0x0219, 0x021f, 0x0225, 0x022c, 0x0232, 0x0237, 0x023d,
    0x0243, 0x0248, 0x024d, 0x0252, 0x0257, 0x025b, 0x025f, 0x0264, 0x0268, 0x026b, 0x026f, 0x0272,
    0x0275, 0x0278, 0x027b, 0x027d, 0x027f, 0x0281, 0x0283, 0x0285, 0x0286, 0x0287, 0x0288, 0x0288,
    0x0289, 0x0289, 0x0289, 0x0288, 0x0288, 0x0287, 0x0286, 0x0285, 0x0283, 0x0281, 0x027f, 0x027d,
    0x027a, 0x0278, 0x0275, 0x0271, 0x026e, 0x026a, 0x0266, 0x0262, 0x025d, 0x0258, 0x0253, 0x024e,
    0x0249, 0x0243, 0x023d, 0x0237, 0x0230, 0x022a, 0x0223, 0x021b, 0x0214, 0x020c, 0x0205, 0x01fc,
    0x01f4, 0x01eb, 0x01e3, 0x01da, 0x01d0, 0x01c7, 0x01bd, 0x01b3, 0x01a9, 0x019f, 0x0194, 0x018a,
    0x017f, 0x0174, 0x0168, 0x015d, 0x0151, 0x0145, 0x0139, 0x012d, 0x0121, 0x0114, 0x0107, 0x00fa,
    0x00ed, 0x00e0, 0x00d3, 0x00c5, 0x00b8, 0x00aa, 0x009c, 0x008e, 0x0080, 0x0071, 0x0063, 0x0054,
    0x0046, 0x0037, 0x0028, 0x0019, 0x000a, 0xfffb, 0xffec, 0xffdd, 0xffcd, 0xffbe, 0xffaf, 0xff9f,
    0xff90, 0xff80, 0xff71, 0xff61, 0xff51, 0xff42, 0xff32, 0xff22, 0xff13, 0xff03, 0xfef3, 0xfee4,
    0xfed4, 0xfec5, 0xfeb5, 0xfea6, 0xfe96, 0xfe87, 0xfe78, 0xfe68, 0xfe59, 0xfe4a, 0xfe3b, 0xfe2c,
    0xfe1d, 0xfe0e, 0xfe00, 0xfdf1, 0xfde3, 0xfdd5, 0xfdc7, 0xfdb9, 0xfdab, 0xfd9d, 0xfd8f, 0xfd82,
    0xfd75, 0xfd68, 0xfd5b, 0xfd4e, 0xfd42, 0xfd35, 0xfd29, 0xfd1d, 0xfd12, 0xfd06, 0xfcfb, 0xfcf0,
    0xfce5, 0xfcdb, 0xfcd0, 0xfcc6, 0xfcbc, 0xfcb3, 0xfca9, 0xfca0, 0xfc98, 0xfc8f, 0xfc87, 0xfc7f,
    0xfc77, 0xfc70, 0xfc69, 0xfc62, 0xfc5c, 0xfc56, 0xfc50, 0xfc4a, 0xfc45, 0xfc40, 0xfc3c, 0xfc38,
    0xfc34, 0xfc30, 0xfc2d, 0xfc2a, 0xfc28, 0xfc26, 0xfc24, 0xfc22, 0xfc21, 0xfc21, 0xfc20, 0xfc20,
    0xfc21, 0xfc21, 0xfc22, 0xfc24, 0xfc26, 0xfc28, 0xfc2b, 0xfc2d, 0xfc31, 0xfc34, 0xfc39, 0xfc3d,
    0xfc42, 0xfc47, 0xfc4d, 0xfc53, 0xfc59, 0xfc60, 0xfc67, 0xfc6e, 0xfc76, 0xfc7e, 0xfc87, 0xfc90,
    0xfc99, 0xfca2, 0xfcac, 0xfcb7, 0xfcc2, 0xfccd, 0xfcd8, 0xfce4, 0xfcf0, 0xfcfd, 0xfd0a, 0xfd17,
    0xfd24, 0xfd32, 0xfd40, 0xfd4f, 0xfd5e, 0xfd6d, 0xfd7c, 0xfd8c, 0xfd9c, 0xfdad, 0xfdbd, 0xfdce,
    0xfde0, 0xfdf1, 0xfe03, 0xfe15, 0xfe28, 0xfe3a, 0xfe4d, 0xfe61, 0xfe74, 0xfe88, 0xfe9c, 0xfeb0,
    0xfec4, 0xfed9, 0xfeed, 0xff02, 0xff18, 0xff2d, 0xff43, 0xff58, 0xff6e, 0xff84, 0xff9b, 0xffb1,
    0xffc7, 0xffde, 0xfff5, 0x000c, 0x0023, 0x003a, 0x0051, 0x0068, 0x0080, 0x0097, 0x00af, 0x00c6,
    0x00de, 0x00f5, 0x010d, 0x0125, 0x013c, 0x0154, 0x016c, 0x0183, 0x019b, 0x01b3, 0x01ca, 0x01e2,
    0x01f9, 0x0210, 0x0228, 0x023f, 0x0256, 0x026d, 0x0284, 0x029b, 0x02b1, 0x02c8, 0x02de, 0x02f4,
    0x030a, 0x0320, 0x0336, 0x034b, 0x0361, 0x0376, 0x038b, 0x039f, 0x03b4, 0x03c8, 0x03dc, 0x03ef,
    0x0403, 0x0416, 0x0428, 0x043b, 0x044d, 0x045f, 0x0470, 0x0482, 0x0492, 0x04a3, 0x04b3, 0x04c3,
    0x04d2, 0x04e1, 0x04f0, 0x04fe, 0x050c, 0x051a, 0x0527, 0x0533, 0x0540, 0x054b, 0x0557, 0x0562,
    0x056c, 0x0576, 0x0580, 0x0589, 0x0591, 0x059a, 0x05a1, 0x05a8, 0x05af, 0x05b5, 0x05bb, 0x05c0,
    0x05c5, 0x05c9, 0x05cc, 0x05cf, 0x05d2, 0x05d4, 0x05d5, 0x05d6, 0x05d7, 0x05d7, 0x05d6, 0x05d5,
    0x05d3, 0x05d1, 0x05ce, 0x05ca, 0x05c6, 0x05c2, 0x05bc, 0x05b7, 0x05b0, 0x05aa, 0x05a2, 0x059a,
    0x0592, 0x0589, 0x057f, 0x0575, 0x056a, 0x055f, 0x0553, 0x0546, 0x0539, 0x052c, 0x051d, 0x050f,
    0x0500, 0x04f0, 0x04e0, 0x04cf, 0x04bd, 0x04ab, 0x0499, 0x0486, 0x0473, 0x045f, 0x044a, 0x0435,
    0x0420, 0x040a, 0x03f3, 0x03dc, 0x03c5, 0x03ad, 0x0395, 0x037c, 0x0362, 0x0349, 0x032f, 0x0314,
    0x02f9, 0x02de, 0x02c2, 0x02a6, 0x0289, 0x026c, 0x024f, 0x0231, 0x0213, 0x01f5, 0x01d6, 0x01b7,
    0x0197, 0x0178, 0x0158, 0x0137, 0x0117, 0x00f6, 0x00d5, 0x00b4, 0x0092, 0x0070, 0x004e, 0x002c,
    0x000a, 0xffe7, 0xffc5, 0xffa2, 0xff7f, 0xff5c, 0xff39, 0xff15, 0xfef2, 0xfece, 0xfeab, 0xfe87,
    0xfe63, 0xfe40, 0xfe1c, 0xfdf8, 0xfdd5, 0xfdb1, 0xfd8d, 0xfd6a, 0xfd46, 0xfd23, 0xfcff, 0xfcdc,
    0xfcb9, 0xfc96, 0xfc73, 0xfc50, 0xfc2e, 0xfc0b, 0xfbe9, 0xfbc7, 0xfba5, 0xfb84, 0xfb63, 0xfb42,
    0xfb21, 0xfb00, 0xfae0, 0xfac1, 0xfaa1, 0xfa82, 0xfa63, 0xfa45, 0xfa27, 0xfa09, 0xf9ec, 0xf9cf,
    0xf9b3, 0xf997, 0xf97b, 0xf960, 0xf946, 0xf92c, 0xf912, 0xf8f9, 0xf8e1, 0xf8c9, 0xf8b1, 0xf89b,
    0xf884, 0xf86f, 0xf85a, 0xf845, 0xf831, 0xf81e, 0xf80b, 0xf7fa, 0xf7e8, 0xf7d8, 0xf7c8, 0xf7b9,
    0xf7aa, 0xf79c, 0xf78f, 0xf783, 0xf777, 0xf76c, 0xf762, 0xf758, 0xf750, 0xf748, 0xf741, 0xf73a,
    0xf735, 0xf730, 0xf72c, 0xf729, 0xf727, 0xf725, 0xf724, 0xf724, 0xf725, 0xf727, 0xf72a, 0xf72d,
    0xf731, 0xf737, 0xf73d, 0xf743, 0xf74b, 0xf754, 0xf75d, 0xf767, 0xf772, 0xf77e, 0xf78b, 0xf799,
    0xf7a7, 0xf7b7, 0xf7c7, 0xf7d8, 0xf7ea, 0xf7fd, 0xf811, 0xf825, 0xf83a, 0xf851, 0xf868, 0xf87f,
    0xf898, 0xf8b2, 0xf8cc, 0xf8e7, 0xf903, 0xf920, 0xf93d, 0xf95c, 0xf97b, 0xf99b, 0xf9bb, 0xf9dd,
    0xf9ff, 0xfa22, 0xfa45, 0xfa6a, 0xfa8f, 0xfab5, 0xfadb, 0xfb02, 0xfb2a, 0xfb53, 0xfb7c, 0xfba6,
    0xfbd0, 0xfbfb, 0xfc27, 0xfc53, 0xfc80, 0xfcad, 0xfcdb, 0xfd0a, 0xfd39, 0xfd69, 0xfd99, 0xfdc9,
    0xfdfa, 0xfe2c, 0xfe5e, 0xfe90, 0xfec3, 0xfef6, 0xff2a, 0xff5d, 0xff92, 0xffc6, 0xfffb, 0x0030,
    0x0066, 0x009c, 0x00d1, 0x0108, 0x013e, 0x0175, 0x01ab, 0x01e2, 0x0219, 0x0250, 0x0287, 0x02bf,
    0x02f6, 0x032d, 0x0365, 0x039c, 0x03d3, 0x040b, 0x0442, 0x0479, 0x04b0, 0x04e7, 0x051e, 0x0554,
    0x058b, 0x05c1, 0x05f7, 0x062c, 0x0662, 0x0697, 0x06cc, 0x0700, 0x0734, 0x0768, 0x079c, 0x07ce,
    0x0801, 0x0833, 0x0864, 0x0896, 0x08c6, 0x08f6, 0x0925, 0x0954, 0x0983, 0x09b0, 0x09dd, 0x0a09,
    0x0a35, 0x0a60, 0x0a8a, 0x0ab4, 0x0adc, 0x0b04, 0x0b2c, 0x0b52, 0x0b77, 0x0b9c, 0x0bc0, 0x0be3,
    0x0c05, 0x0c26, 0x0c46, 0x0c65, 0x0c83, 0x0ca0, 0x0cbd, 0x0cd8, 0x0cf2, 0x0d0b, 0x0d23, 0x0d3a,
    0x0d50, 0x0d64, 0x0d78, 0x0d8a, 0x0d9c, 0x0dac, 0x0dbb, 0x0dc9, 0x0dd5, 0x0de0, 0x0deb, 0x0df3,
    0x0dfb, 0x0e01, 0x0e07, 0x0e0a, 0x0e0d, 0x0e0e, 0x0e0e, 0x0e0d, 0x0e0a, 0x0e06, 0x0e00, 0x0dfa,
    0x0df2, 0x0de8, 0x0ddd, 0x0dd1, 0x0dc4, 0x0db5, 0x0da5, 0x0d93, 0x0d80, 0x0d6c, 0x0d56, 0x0d3f,
    0x0d26, 0x0d0c, 0x0cf1, 0x0cd4, 0x0cb6, 0x0c97, 0x0c76, 0x0c54, 0x0c30, 0x0c0b, 0x0be5, 0x0bbd,
    0x0b94, 0x0b6a, 0x0b3e, 0x0b11, 0x0ae3, 0x0ab3, 0x0a82, 0x0a50, 0x0a1d, 0x09e8, 0x09b2, 0x097a,
    0x0942, 0x0908, 0x08cd, 0x0890, 0x0853, 0x0814, 0x07d4, 0x0793, 0x0751, 0x070e, 0x06c9, 0x0684,
    0x063d, 0x05f6, 0x05ad, 0x0563, 0x0519, 0x04cd, 0x0480, 0x0433, 0x03e4, 0x0395, 0x0345, 0x02f4,
    0x02a2, 0x024f, 0x01fb, 0x01a7, 0x0152, 0x00fc, 0x00a6, 0x004e, 0xfff7, 0xff9e, 0xff45, 0xfeec,
    0xfe92, 0xfe37, 0xfddc, 0xfd80, 0xfd25, 0xfcc8, 0xfc6c, 0xfc0f, 0xfbb1, 0xfb54, 0xfaf6, 0xfa98,
    0xfa3a, 0xf9db, 0xf97d, 0xf91e, 0xf8c0, 0xf861, 0xf802, 0xf7a4, 0xf745, 0xf6e7, 0xf689, 0xf62b,
    0xf5cd, 0xf56f, 0xf512, 0xf4b5, 0xf458, 0xf3fc, 0xf3a0, 0xf345, 0xf2ea, 0xf290, 0xf236, 0xf1dd,
    0xf184, 0xf12c, 0xf0d5, 0xf07f, 0xf029, 0xefd4, 0xef80, 0xef2d, 0xeedb, 0xee8a, 0xee39, 0xedea,
    0xed9c, 0xed4f, 0xed03, 0xecb8, 0xec6e, 0xec26, 0xebdf, 0xeb99, 0xeb55, 0xeb11, 0xead0, 0xea90,
    0xea51, 0xea13, 0xe9d8, 0xe99e, 0xe965, 0xe92e, 0xe8f9, 0xe8c5, 0xe893, 0xe863, 0xe835, 0xe809,
    0xe7de, 0xe7b5, 0xe78f, 0xe76a, 0xe747, 0xe726, 0xe707, 0xe6eb, 0xe6d0, 0xe6b7, 0xe6a1, 0xe68d,
    0xe67b, 0xe66b, 0xe65e, 0xe652, 0xe649, 0xe643, 0xe63e, 0xe63d, 0xe63d, 0xe640, 0xe645, 0xe64d,
    0xe657, 0xe664, 0xe673, 0xe685, 0xe699, 0xe6b0, 0xe6ca, 0xe6e6, 0xe705, 0xe726, 0xe74a, 0xe770,
    0xe79a, 0xe7c6, 0xe7f5, 0xe826, 0xe85a, 0xe891, 0xe8cb, 0xe907, 0xe946, 0xe988, 0xe9cd, 0xea14,
    0xea5e, 0xeaab, 0xeafb, 0xeb4d, 0xeba3, 0xebfb, 0xec56, 0xecb4, 0xed14, 0xed78, 0xedde, 0xee47,
    0xeeb3, 0xef21, 0xef93, 0xf007, 0xf07e, 0xf0f7, 0xf174, 0xf1f3, 0xf275, 0xf2f9, 0xf381, 0xf40b,
    0xf497, 0xf527, 0xf5b9, 0xf64e, 0xf6e5, 0xf77f, 0xf81c, 0xf8bb, 0xf95d, 0xfa01, 0xfaa8, 0xfb52,
    0xfbfe, 0xfcac, 0xfd5d, 0xfe10, 0xfec6, 0xff7e, 0x0039, 0x00f6, 0x01b5, 0x0276, 0x033a, 0x0400,
    0x04c8, 0x0592, 0x065f, 0x072d, 0x07fe, 0x08d0, 0x09a5, 0x0a7c, 0x0b54, 0x0c2f, 0x0d0b, 0x0de9,
    0x0ec9, 0x0fab, 0x108f, 0x1174, 0x125b, 0x1344, 0x142e, 0x1519, 0x1606, 0x16f5, 0x17e5, 0x18d6,
    0x19c9, 0x1abd, 0x1bb2, 0x1ca9, 0x1da0, 0x1e99, 0x1f93, 0x208e, 0x218a, 0x2286, 0x2384, 0x2482,
    0x2582, 0x2681, 0x2782, 0x2883, 0x2985, 0x2a88, 0x2b8b, 0x2c8e, 0x2d92, 0x2e96, 0x2f9b, 0x309f,
    0x31a4, 0x32a9, 0x33ae, 0x34b4, 0x35b9, 0x36be, 0x37c3, 0x38c8, 0x39cc, 0x3ad1, 0x3bd5, 0x3cd8,
    0x3ddc, 0x3edf, 0x3fe1, 0x40e3, 0x41e4, 0x42e4, 0x43e4, 0x44e2, 0x45e1, 0x46de, 0x47da, 0x48d5,
    0x49cf, 0x4ac8, 0x4bc0, 0x4cb7, 0x4dac, 0x4ea0, 0x4f93, 0x5085, 0x5175, 0x5263, 0x5350, 0x543b,
    0x5525, 0x560d, 0x56f3, 0x57d7, 0x58ba, 0x599a, 0x5a79, 0x5b56, 0x5c30, 0x5d09, 0x5ddf, 0x5eb4,
    0x5f86, 0x6055, 0x6123, 0x61ee, 0x62b7, 0x637d, 0x6441, 0x6502, 0x65c1, 0x667d, 0x6736, 0x67ed,
    0x68a1, 0x6952, 0x6a00, 0x6aac, 0x6b55, 0x6bfa, 0x6c9d, 0x6d3d, 0x6dda, 0x6e74, 0x6f0a, 0x6f9e,
    0x702e, 0x70bb, 0x7145, 0x71cc, 0x724f, 0x72cf, 0x734c, 0x73c5, 0x743b, 0x74ae, 0x751d, 0x7589,
    0x75f1, 0x7656, 0x76b7, 0x7714, 0x776e, 0x77c5, 0x7817, 0x7866, 0x78b2, 0x78fa, 0x793e, 0x797e,
    0x79bb, 0x79f4, 0x7a29, 0x7a5b, 0x7a88, 0x7ab2, 0x7ad8, 0x7afb, 0x7b19, 0x7b34, 0x7b4b, 0x7b5e,
    0x7b6e, 0x7b79, 0x7b81, 0x7b85,
    };


// ----- 320 : 147 ---------------
// makeFIR(24, 320, 147, .9, 8);
static const int16_t fir_320_147[ 320 * FIR_TAP_24 / 2] = {
    0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004,
    0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004,
    0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004,
    0x0004, 0x0004, 0x0004, 0x0004, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005,
    0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004,
    0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002,
    0x0002, 0x0002, 0x0002, 0x0002, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0001, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff7, 0xfff7, 0xfff7, 0xfff7, 0xfff7,
    0xfff7, 0xfff7, 0xfff6, 0xfff6, 0xfff6, 0xfff6, 0xfff6, 0xfff6, 0xfff6, 0xfff5, 0xfff5, 0xfff5,
    0xfff5, 0xfff5, 0xfff5, 0xfff5, 0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff3,
    0xfff3, 0xfff3, 0xfff3, 0xfff3, 0xfff3, 0xfff3, 0xfff2, 0xfff2, 0xfff2, 0xfff2, 0xfff2, 0xfff2,
    0xfff2, 0xfff1, 0xfff1, 0xfff1, 0xfff1, 0xfff1, 0xfff1, 0xfff1, 0xfff0, 0xfff0, 0xfff0, 0xfff0,
    0xfff0, 0xfff0, 0xfff0, 0xfff0, 0xffef, 0xffef, 0xffef, 0xffef, 0xffef, 0xffef, 0xffef, 0xffee,
    0xffee, 0xffee, 0xffee, 0xffee, 0xffee, 0xffee, 0xffee, 0xffed, 0xffed, 0xffed, 0xffed, 0xffed,
    0xffed, 0xffed, 0xffed, 0xffed, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec,
    0xffec, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb,
    0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea,
    0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9,
    0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8,
    0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8,
    0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8,
    0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9,
    0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffea, 0xffea,
    0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffeb, 0xffeb, 0xffeb, 0xffeb,
    0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec,
    0xffed, 0xffed, 0xffed, 0xffed, 0xffed, 0xffed, 0xffee, 0xffee, 0xffee, 0xffee, 0xffee, 0xffee,
    0xffef, 0xffef, 0xffef, 0xffef, 0xffef, 0xfff0, 0xfff0, 0xfff0, 0xfff0, 0xfff0, 0xfff1, 0xfff1,
    0xfff1, 0xfff1, 0xfff2, 0xfff2, 0xfff2, 0xfff2, 0xfff3, 0xfff3, 0xfff3, 0xfff3, 0xfff3, 0xfff4,
    0xfff4, 0xfff4, 0xfff4, 0xfff5, 0xfff5, 0xfff5, 0xfff6, 0xfff6, 0xfff6, 0xfff6, 0xfff7, 0xfff7,
    0xfff7, 0xfff7, 0xfff8, 0xfff8, 0xfff8, 0xfff9, 0xfff9, 0xfff9, 0xfffa, 0xfffa, 0xfffa, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffc, 0xfffc, 0xfffc, 0xfffd, 0xfffd, 0xfffd, 0xfffe, 0xfffe, 0xfffe,
    0xffff, 0xffff, 0xffff, 0x0000, 0x0000, 0x0001, 0x0001, 0x0001, 0x0002, 0x0002, 0x0002, 0x0003,
    0x0003, 0x0003, 0x0004, 0x0004, 0x0005, 0x0005, 0x0005, 0x0006, 0x0006, 0x0006, 0x0007, 0x0007,
    0x0008, 0x0008, 0x0008, 0x0009, 0x0009, 0x000a, 0x000a, 0x000a, 0x000b, 0x000b, 0x000c, 0x000c,
    0x000c, 0x000d, 0x000d, 0x000e, 0x000e, 0x000e, 0x000f, 0x000f, 0x0010, 0x0010, 0x0011, 0x0011,
    0x0011, 0x0012, 0x0012, 0x0013, 0x0013, 0x0014, 0x0014, 0x0014, 0x0015, 0x0015, 0x0016, 0x0016,
    0x0017, 0x0017, 0x0017, 0x0018, 0x0018, 0x0019, 0x0019, 0x001a, 0x001a, 0x001a, 0x001b, 0x001b,
    0x001c, 0x001c, 0x001d, 0x001d, 0x001e, 0x001e, 0x001e, 0x001f, 0x001f, 0x0020, 0x0020, 0x0021,
    0x0021, 0x0021, 0x0022, 0x0022, 0x0023, 0x0023, 0x0024, 0x0024, 0x0024, 0x0025, 0x0025, 0x0026,
    0x0026, 0x0027, 0x0027, 0x0027, 0x0028, 0x0028, 0x0029, 0x0029, 0x002a, 0x002a, 0x002a, 0x002b,
    0x002b, 0x002c, 0x002c, 0x002c, 0x002d, 0x002d, 0x002e, 0x002e, 0x002e, 0x002f, 0x002f, 0x0030,
    0x0030, 0x0030, 0x0031, 0x0031, 0x0032, 0x0032, 0x0032, 0x0033, 0x0033, 0x0033, 0x0034, 0x0034,
    0x0035, 0x0035, 0x0035, 0x0036, 0x0036, 0x0036, 0x0037, 0x0037, 0x0037, 0x0038, 0x0038, 0x0038,
    0x0039, 0x0039, 0x0039, 0x003a, 0x003a, 0x003a, 0x003b, 0x003b, 0x003b, 0x003c, 0x003c, 0x003c,
    0x003c, 0x003d, 0x003d, 0x003d, 0x003e, 0x003e, 0x003e, 0x003e, 0x003f, 0x003f, 0x003f, 0x003f,
    0x0040, 0x0040, 0x0040, 0x0040, 0x0040, 0x0041, 0x0041, 0x0041, 0x0041, 0x0042, 0x0042, 0x0042,
    0x0042, 0x0042, 0x0042, 0x0043, 0x0043, 0x0043, 0x0043, 0x0043, 0x0043, 0x0044, 0x0044, 0x0044,
    0x0044, 0x0044, 0x0044, 0x0044, 0x0044, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045,
    0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045,
    0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045, 0x0045,
    0x0045, 0x0045, 0x0045, 0x0045, 0x0044, 0x0044, 0x0044, 0x0044, 0x0044, 0x0044, 0x0044, 0x0043,
    0x0043, 0x0043, 0x0043, 0x0043, 0x0043, 0x0042, 0x0042, 0x0042, 0x0042, 0x0042, 0x0041, 0x0041,
    0x0041, 0x0041, 0x0040, 0x0040, 0x0040, 0x0040, 0x003f, 0x003f, 0x003f, 0x003e, 0x003e, 0x003e,
    0x003d, 0x003d, 0x003d, 0x003c, 0x003c, 0x003c, 0x003b, 0x003b, 0x003b, 0x003a, 0x003a, 0x0039,
    0x0039, 0x0039, 0x0038, 0x0038, 0x0037, 0x0037, 0x0036, 0x0036, 0x0035, 0x0035, 0x0034, 0x0034,
    0x0034, 0x0033, 0x0032, 0x0032, 0x0031, 0x0031, 0x0030, 0x0030, 0x002f, 0x002f, 0x002e, 0x002e,
    0x002d, 0x002c, 0x002c, 0x002b, 0x002b, 0x002a, 0x0029, 0x0029, 0x0028, 0x0028, 0x0027, 0x0026,
    0x0026, 0x0025, 0x0024, 0x0024, 0x0023, 0x0022, 0x0022, 0x0021, 0x0020, 0x001f, 0x001f, 0x001e,
    0x001d, 0x001d, 0x001c, 0x001b, 0x001a, 0x0019, 0x0019, 0x0018, 0x0017, 0x0016, 0x0016, 0x0015,
    0x0014, 0x0013, 0x0012, 0x0012, 0x0011, 0x0010, 0x000f, 0x000e, 0x000d, 0x000d, 0x000c, 0x000b,
    0x000a, 0x0009, 0x0008, 0x0007, 0x0006, 0x0006, 0x0005, 0x0004, 0x0003, 0x0002, 0x0001, 0x0000,
    0xffff, 0xfffe, 0xfffd, 0xfffc, 0xfffb, 0xfffa, 0xfff9, 0xfff9, 0xfff8, 0xfff7, 0xfff6, 0xfff5,
    0xfff4, 0xfff3, 0xfff2, 0xfff1, 0xfff0, 0xffef, 0xffee, 0xffed, 0xffec, 0xffeb, 0xffea, 0xffe9,
    0xffe8, 0xffe7, 0xffe6, 0xffe5, 0xffe4, 0xffe3, 0xffe2, 0xffe1, 0xffe0, 0xffde, 0xffdd, 0xffdc,
    0xffdb, 0xffda, 0xffd9, 0xffd8, 0xffd7, 0xffd6, 0xffd5, 0xffd4, 0xffd3, 0xffd2, 0xffd1, 0xffd0,
    0xffcf, 0xffce, 0xffcd, 0xffcc, 0xffcb, 0xffc9, 0xffc8, 0xffc7, 0xffc6, 0xffc5, 0xffc4, 0xffc3,
    0xffc2, 0xffc1, 0xffc0, 0xffbf, 0xffbe, 0xffbd, 0xffbc, 0xffbb, 0xffba, 0xffb8, 0xffb7, 0xffb6,
    0xffb5, 0xffb4, 0xffb3, 0xffb2, 0xffb1, 0xffb0, 0xffaf, 0xffae, 0xffad, 0xffac, 0xffab, 0xffaa,
    0xffa9, 0xffa8, 0xffa7, 0xffa6, 0xffa5, 0xffa4, 0xffa3, 0xffa2, 0xffa1, 0xffa0, 0xff9f, 0xff9e,
    0xff9d, 0xff9c, 0xff9b, 0xff9a, 0xff99, 0xff98, 0xff97, 0xff96, 0xff95, 0xff94, 0xff93, 0xff92,
    0xff91, 0xff90, 0xff8f, 0xff8e, 0xff8d, 0xff8d, 0xff8c, 0xff8b, 0xff8a, 0xff89, 0xff88, 0xff87,
    0xff86, 0xff85, 0xff85, 0xff84, 0xff83, 0xff82, 0xff81, 0xff80, 0xff80, 0xff7f, 0xff7e, 0xff7d,
    0xff7c, 0xff7c, 0xff7b, 0xff7a, 0xff79, 0xff79, 0xff78, 0xff77, 0xff76, 0xff76, 0xff75, 0xff74,
    0xff74, 0xff73, 0xff72, 0xff72, 0xff71, 0xff70, 0xff70, 0xff6f, 0xff6e, 0xff6e, 0xff6d, 0xff6d,
    0xff6c, 0xff6c, 0xff6b, 0xff6a, 0xff6a, 0xff69, 0xff69, 0xff68, 0xff68, 0xff67, 0xff67, 0xff66,
    0xff66, 0xff66, 0xff65, 0xff65, 0xff64, 0xff64, 0xff63, 0xff63, 0xff63, 0xff62, 0xff62, 0xff62,
    0xff61, 0xff61, 0xff61, 0xff61, 0xff60, 0xff60, 0xff60, 0xff60, 0xff5f, 0xff5f, 0xff5f, 0xff5f,
    0xff5f, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e,
    0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e, 0xff5e,
    0xff5f, 0xff5f, 0xff5f, 0xff5f, 0xff5f, 0xff60, 0xff60, 0xff60, 0xff60, 0xff61, 0xff61, 0xff61,
    0xff62, 0xff62, 0xff62, 0xff63, 0xff63, 0xff63, 0xff64, 0xff64, 0xff65, 0xff65, 0xff66, 0xff66,
    0xff67, 0xff67, 0xff68, 0xff68, 0xff69, 0xff6a, 0xff6a, 0xff6b, 0xff6b, 0xff6c, 0xff6d, 0xff6d,
    0xff6e, 0xff6f, 0xff6f, 0xff70, 0xff71, 0xff72, 0xff73, 0xff73, 0xff74, 0xff75, 0xff76, 0xff77,
    0xff78, 0xff78, 0xff79, 0xff7a, 0xff7b, 0xff7c, 0xff7d, 0xff7e, 0xff7f, 0xff80, 0xff81, 0xff82,
    0xff83, 0xff84, 0xff85, 0xff86, 0xff88, 0xff89, 0xff8a, 0xff8b, 0xff8c, 0xff8d, 0xff8f, 0xff90,
    0xff91, 0xff92, 0xff94, 0xff95, 0xff96, 0xff97, 0xff99, 0xff9a, 0xff9b, 0xff9d, 0xff9e, 0xff9f,
    0xffa1, 0xffa2, 0xffa4, 0xffa5, 0xffa7, 0xffa8, 0xffaa, 0xffab, 0xffad, 0xffae, 0xffb0, 0xffb1,
    0xffb3, 0xffb4, 0xffb6, 0xffb7, 0xffb9, 0xffbb, 0xffbc, 0xffbe, 0xffc0, 0xffc1, 0xffc3, 0xffc5,
    0xffc6, 0xffc8, 0xffca, 0xffcc, 0xffcd, 0xffcf, 0xffd1, 0xffd3, 0xffd5, 0xffd6, 0xffd8, 0xffda,
    0xffdc, 0xffde, 0xffe0, 0xffe1, 0xffe3, 0xffe5, 0xffe7, 0xffe9, 0xffeb, 0xffed, 0xffef, 0xfff1,
    0xfff3, 0xfff5, 0xfff7, 0xfff9, 0xfffb, 0xfffd, 0xffff, 0x0001, 0x0003, 0x0005, 0x0007, 0x0009,
    0x000b, 0x000d, 0x000f, 0x0011, 0x0013, 0x0015, 0x0018, 0x001a, 0x001c, 0x001e, 0x0020, 0x0022,
    0x0024, 0x0026, 0x0029, 0x002b, 0x002d, 0x002f, 0x0031, 0x0033, 0x0036, 0x0038, 0x003a, 0x003c,
    0x003e, 0x0041, 0x0043, 0x0045, 0x0047, 0x0049, 0x004c, 0x004e, 0x0050, 0x0052, 0x0055, 0x0057,
    0x0059, 0x005b, 0x005e, 0x0060, 0x0062, 0x0064, 0x0066, 0x0069, 0x006b, 0x006d, 0x006f, 0x0072,
    0x0074, 0x0076, 0x0078, 0x007b, 0x007d, 0x007f, 0x0081, 0x0084, 0x0086, 0x0088, 0x008a, 0x008c,
    0x008f, 0x0091, 0x0093, 0x0095, 0x0097, 0x009a, 0x009c, 0x009e, 0x00a0, 0x00a2, 0x00a5, 0x00a7,
    0x00a9, 0x00ab, 0x00ad, 0x00af, 0x00b2, 0x00b4, 0x00b6, 0x00b8, 0x00ba, 0x00bc, 0x00be, 0x00c0,
    0x00c2, 0x00c4, 0x00c7, 0x00c9, 0x00cb, 0x00cd, 0x00cf, 0x00d1, 0x00d3, 0x00d5, 0x00d7, 0x00d9,
    0x00db, 0x00dd, 0x00df, 0x00e1, 0x00e3, 0x00e4, 0x00e6, 0x00e8, 0x00ea, 0x00ec, 0x00ee, 0x00f0,
    0x00f2, 0x00f3, 0x00f5, 0x00f7, 0x00f9, 0x00fb, 0x00fc, 0x00fe, 0x0100, 0x0101, 0x0103, 0x0105,
    0x0107, 0x0108, 0x010a, 0x010b, 0x010d, 0x010f, 0x0110, 0x0112, 0x0113, 0x0115, 0x0116, 0x0118,
    0x0119, 0x011b, 0x011c, 0x011e, 0x011f, 0x0120, 0x0122, 0x0123, 0x0124, 0x0126, 0x0127, 0x0128,
    0x012a, 0x012b, 0x012c, 0x012d, 0x012e, 0x012f, 0x0131, 0x0132, 0x0133, 0x0134, 0x0135, 0x0136,
    0x0137, 0x0138, 0x0139, 0x013a, 0x013b, 0x013c, 0x013c, 0x013d, 0x013e, 0x013f, 0x0140, 0x0140,
    0x0141, 0x0142, 0x0143, 0x0143, 0x0144, 0x0144, 0x0145, 0x0146, 0x0146, 0x0147, 0x0147, 0x0148,
    0x0148, 0x0148, 0x0149, 0x0149, 0x0149, 0x014a, 0x014a, 0x014a, 0x014a, 0x014b, 0x014b, 0x014b,
    0x014b, 0x014b, 0x014b, 0x014b, 0x014b, 0x014b, 0x014b, 0x014b, 0x014b, 0x014b, 0x014b, 0x014b,
    0x014a, 0x014a, 0x014a, 0x014a, 0x0149, 0x0149, 0x0149, 0x0148, 0x0148, 0x0147, 0x0147, 0x0146,
    0x0146, 0x0145, 0x0144, 0x0144, 0x0143, 0x0142, 0x0142, 0x0141, 0x0140, 0x013f, 0x013f, 0x013e,
    0x013d, 0x013c, 0x013b, 0x013a, 0x0139, 0x0138, 0x0137, 0x0136, 0x0134, 0x0133, 0x0132, 0x0131,
    0x0130, 0x012e, 0x012d, 0x012c, 0x012a, 0x0129, 0x0127, 0x0126, 0x0124, 0x0123, 0x0121, 0x0120,
    0x011e, 0x011d, 0x011b, 0x0119, 0x0117, 0x0116, 0x0114, 0x0112, 0x0110, 0x010e, 0x010d, 0x010b,
    0x0109, 0x0107, 0x0105, 0x0103, 0x0100, 0x00fe, 0x00fc, 0x00fa, 0x00f8, 0x00f6, 0x00f3, 0x00f1,
    0x00ef, 0x00ed, 0x00ea, 0x00e8, 0x00e5, 0x00e3, 0x00e0, 0x00de, 0x00db, 0x00d9, 0x00d6, 0x00d4,
    0x00d1, 0x00ce, 0x00cc, 0x00c9, 0x00c6, 0x00c4, 0x00c1, 0x00be, 0x00bb, 0x00b8, 0x00b5, 0x00b2,
    0x00af, 0x00ad, 0x00aa, 0x00a7, 0x00a3, 0x00a0, 0x009d, 0x009a, 0x0097, 0x0094, 0x0091, 0x008e,
    0x008a, 0x0087, 0x0084, 0x0080, 0x007d, 0x007a, 0x0076, 0x0073, 0x0070, 0x006c, 0x0069, 0x0065,
    0x0062, 0x005e, 0x005b, 0x0057, 0x0054, 0x0050, 0x004c, 0x0049, 0x0045, 0x0042, 0x003e, 0x003a,
    0x0036, 0x0033, 0x002f, 0x002b, 0x0027, 0x0024, 0x0020, 0x001c, 0x0018, 0x0014, 0x0010, 0x000c,
    0x0009, 0x0005, 0x0001, 0xfffd, 0xfff9, 0xfff5, 0xfff1, 0xffed, 0xffe9, 0xffe5, 0xffe1, 0xffdd,
    0xffd9, 0xffd4, 0xffd0, 0xffcc, 0xffc8, 0xffc4, 0xffc0, 0xffbc, 0xffb8, 0xffb3, 0xffaf, 0xffab,
    0xffa7, 0xffa3, 0xff9f, 0xff9a, 0xff96, 0xff92, 0xff8e, 0xff8a, 0xff85, 0xff81, 0xff7d, 0xff79,
    0xff74, 0xff70, 0xff6c, 0xff68, 0xff63, 0xff5f, 0xff5b, 0xff57, 0xff52, 0xff4e, 0xff4a, 0xff45,
    0xff41, 0xff3d, 0xff39, 0xff34, 0xff30, 0xff2c, 0xff28, 0xff23, 0xff1f, 0xff1b, 0xff17, 0xff12,
    0xff0e, 0xff0a, 0xff06, 0xff02, 0xfefd, 0xfef9, 0xfef5, 0xfef1, 0xfeed, 0xfee8, 0xfee4, 0xfee0,
    0xfedc, 0xfed8, 0xfed4, 0xfed0, 0xfecc, 0xfec8, 0xfec3, 0xfebf, 0xfebb, 0xfeb7, 0xfeb3, 0xfeaf,
    0xfeab, 0xfea7, 0xfea3, 0xfe9f, 0xfe9b, 0xfe97, 0xfe94, 0xfe90, 0xfe8c, 0xfe88, 0xfe84, 0xfe80,
    0xfe7c, 0xfe79, 0xfe75, 0xfe71, 0xfe6d, 0xfe6a, 0xfe66, 0xfe62, 0xfe5f, 0xfe5b, 0xfe58, 0xfe54,
    0xfe50, 0xfe4d, 0xfe49, 0xfe46, 0xfe42, 0xfe3f, 0xfe3c, 0xfe38, 0xfe35, 0xfe31, 0xfe2e, 0xfe2b,
    0xfe28, 0xfe24, 0xfe21, 0xfe1e, 0xfe1b, 0xfe18, 0xfe15, 0xfe12, 0xfe0f, 0xfe0c, 0xfe09, 0xfe06,
    0xfe03, 0xfe00, 0xfdfd, 0xfdfa, 0xfdf8, 0xfdf5, 0xfdf2, 0xfdef, 0xfded, 0xfdea, 0xfde8, 0xfde5,
    0xfde3, 0xfde0, 0xfdde, 0xfddb, 0xfdd9, 0xfdd7, 0xfdd4, 0xfdd2, 0xfdd0, 0xfdce, 0xfdcc, 0xfdca,
    0xfdc8, 0xfdc6, 0xfdc4, 0xfdc2, 0xfdc0, 0xfdbe, 0xfdbc, 0xfdba, 0xfdb9, 0xfdb7, 0xfdb5, 0xfdb4,
    0xfdb2, 0xfdb1, 0xfdaf, 0xfdae, 0xfdac, 0xfdab, 0xfdaa, 0xfda9, 0xfda7, 0xfda6, 0xfda5, 0xfda4,
    0xfda3, 0xfda2, 0xfda1, 0xfda0, 0xfd9f, 0xfd9f, 0xfd9e, 0xfd9d, 0xfd9d, 0xfd9c, 0xfd9b, 0xfd9b,
    0xfd9a, 0xfd9a, 0xfd9a, 0xfd99, 0xfd99, 0xfd99, 0xfd99, 0xfd99, 0xfd99, 0xfd99, 0xfd99, 0xfd99,
    0xfd99, 0xfd99, 0xfd99, 0xfd9a, 0xfd9a, 0xfd9a, 0xfd9b, 0xfd9b, 0xfd9c, 0xfd9c, 0xfd9d, 0xfd9e,
    0xfd9f, 0xfd9f, 0xfda0, 0xfda1, 0xfda2, 0xfda3, 0xfda4, 0xfda5, 0xfda6, 0xfda8, 0xfda9, 0xfdaa,
    0xfdac, 0xfdad, 0xfdaf, 0xfdb0, 0xfdb2, 0xfdb3, 0xfdb5, 0xfdb7, 0xfdb9, 0xfdbb, 0xfdbc, 0xfdbe,
    0xfdc0, 0xfdc3, 0xfdc5, 0xfdc7, 0xfdc9, 0xfdcb, 0xfdce, 0xfdd0, 0xfdd3, 0xfdd5, 0xfdd8, 0xfdda,
    0xfddd, 0xfde0, 0xfde2, 0xfde5, 0xfde8, 0xfdeb, 0xfdee, 0xfdf1, 0xfdf4, 0xfdf7, 0xfdfa, 0xfdfe,
    0xfe01, 0xfe04, 0xfe08, 0xfe0b, 0xfe0f, 0xfe12, 0xfe16, 0xfe19, 0xfe1d, 0xfe21, 0xfe25, 0xfe29,
    0xfe2d, 0xfe31, 0xfe35, 0xfe39, 0xfe3d, 0xfe41, 0xfe45, 0xfe49, 0xfe4e, 0xfe52, 0xfe56, 0xfe5b,
    0xfe5f, 0xfe64, 0xfe69, 0xfe6d, 0xfe72, 0xfe77, 0xfe7b, 0xfe80, 0xfe85, 0xfe8a, 0xfe8f, 0xfe94,
    0xfe99, 0xfe9e, 0xfea3, 0xfea9, 0xfeae, 0xfeb3, 0xfeb9, 0xfebe, 0xfec3, 0xfec9, 0xfece, 0xfed4,
    0xfeda, 0xfedf, 0xfee5, 0xfeeb, 0xfef0, 0xfef6, 0xfefc, 0xff02, 0xff08, 0xff0e, 0xff14, 0xff1a,
    0xff20, 0xff26, 0xff2c, 0xff32, 0xff39, 0xff3f, 0xff45, 0xff4b, 0xff52, 0xff58, 0xff5f, 0xff65,
    0xff6c, 0xff72, 0xff79, 0xff7f, 0xff86, 0xff8d, 0xff93, 0xff9a, 0xffa1, 0xffa7, 0xffae, 0xffb5,
    0xffbc, 0xffc3, 0xffca, 0xffd1, 0xffd8, 0xffdf, 0xffe6, 0xffed, 0xfff4, 0xfffb, 0x0002, 0x0009,
    0x0010, 0x0017, 0x001f, 0x0026, 0x002d, 0x0034, 0x003c, 0x0043, 0x004a, 0x0052, 0x0059, 0x0060,
    0x0068, 0x006f, 0x0076, 0x007e, 0x0085, 0x008d, 0x0094, 0x009c, 0x00a3, 0x00ab, 0x00b2, 0x00ba,
    0x00c1, 0x00c9, 0x00d0, 0x00d8, 0x00df, 0x00e7, 0x00ee, 0x00f6, 0x00fd, 0x0105, 0x010d, 0x0114,
    0x011c, 0x0123, 0x012b, 0x0132, 0x013a, 0x0141, 0x0149, 0x0151, 0x0158, 0x0160, 0x0167, 0x016f,
    0x0176, 0x017e, 0x0185, 0x018d, 0x0194, 0x019c, 0x01a3, 0x01ab, 0x01b2, 0x01ba, 0x01c1, 0x01c9,
    0x01d0, 0x01d7, 0x01df, 0x01e6, 0x01ed, 0x01f5, 0x01fc, 0x0203, 0x020b, 0x0212, 0x0219, 0x0220,
    0x0228, 0x022f, 0x0236, 0x023d, 0x0244, 0x024b, 0x0252, 0x0259, 0x0260, 0x0267, 0x026e, 0x0275,
    0x027c, 0x0283, 0x028a, 0x0290, 0x0297, 0x029e, 0x02a5, 0x02ab, 0x02b2, 0x02b8, 0x02bf, 0x02c5,
    0x02cc, 0x02d2, 0x02d9, 0x02df, 0x02e5, 0x02ec, 0x02f2, 0x02f8, 0x02fe, 0x0304, 0x030a, 0x0310,
    0x0316, 0x031c, 0x0322, 0x0328, 0x032e, 0x0334, 0x0339, 0x033f, 0x0345, 0x034a, 0x0350, 0x0355,
    0x035a, 0x0360, 0x0365, 0x036a, 0x036f, 0x0375, 0x037a, 0x037f, 0x0384, 0x0388, 0x038d, 0x0392,
    0x0397, 0x039b, 0x03a0, 0x03a5, 0x03a9, 0x03ad, 0x03b2, 0x03b6, 0x03ba, 0x03be, 0x03c3, 0x03c7,
    0x03cb, 0x03ce, 0x03d2, 0x03d6, 0x03da, 0x03dd, 0x03e1, 0x03e4, 0x03e8, 0x03eb, 0x03ee, 0x03f2,
    0x03f5, 0x03f8, 0x03fb, 0x03fe, 0x0400, 0x0403, 0x0406, 0x0409, 0x040b, 0x040e, 0x0410, 0x0412,
    0x0414, 0x0417, 0x0419, 0x041b, 0x041d, 0x041e, 0x0420, 0x0422, 0x0424, 0x0425, 0x0426, 0x0428,
    0x0429, 0x042a, 0x042b, 0x042c, 0x042d, 0x042e, 0x042f, 0x0430, 0x0430, 0x0431, 0x0431, 0x0432,
    0x0432, 0x0432, 0x0432, 0x0432, 0x0432, 0x0432, 0x0432, 0x0431, 0x0431, 0x0430, 0x0430, 0x042f,
    0x042e, 0x042e, 0x042d, 0x042c, 0x042a, 0x0429, 0x0428, 0x0426, 0x0425, 0x0423, 0x0422, 0x0420,
    0x041e, 0x041c, 0x041a, 0x0418, 0x0416, 0x0413, 0x0411, 0x040f, 0x040c, 0x0409, 0x0407, 0x0404,
    0x0401, 0x03fe, 0x03fb, 0x03f7, 0x03f4, 0x03f1, 0x03ed, 0x03ea, 0x03e6, 0x03e2, 0x03de, 0x03da,
    0x03d6, 0x03d2, 0x03ce, 0x03ca, 0x03c5, 0x03c1, 0x03bc, 0x03b7, 0x03b3, 0x03ae, 0x03a9, 0x03a4,
    0x039f, 0x0399, 0x0394, 0x038f, 0x0389, 0x0384, 0x037e, 0x0378, 0x0372, 0x036c, 0x0366, 0x0360,
    0x035a, 0x0354, 0x034d, 0x0347, 0x0340, 0x033a, 0x0333, 0x032c, 0x0325, 0x031e, 0x0317, 0x0310,
    0x0309, 0x0301, 0x02fa, 0x02f2, 0x02eb, 0x02e3, 0x02db, 0x02d4, 0x02cc, 0x02c4, 0x02bc, 0x02b4,
    0x02ab, 0x02a3, 0x029b, 0x0292, 0x028a, 0x0281, 0x0278, 0x0270, 0x0267, 0x025e, 0x0255, 0x024c,
    0x0243, 0x0239, 0x0230, 0x0227, 0x021d, 0x0214, 0x020a, 0x0200, 0x01f7, 0x01ed, 0x01e3, 0x01d9,
    0x01cf, 0x01c5, 0x01bb, 0x01b1, 0x01a7, 0x019c, 0x0192, 0x0187, 0x017d, 0x0172, 0x0168, 0x015d,
    0x0152, 0x0147, 0x013c, 0x0132, 0x0127, 0x011b, 0x0110, 0x0105, 0x00fa, 0x00ef, 0x00e3, 0x00d8,
    0x00cd, 0x00c1, 0x00b6, 0x00aa, 0x009e, 0x0093, 0x0087, 0x007b, 0x006f, 0x0064, 0x0058, 0x004c,
    0x0040, 0x0034, 0x0028, 0x001c, 0x0010, 0x0003, 0xfff7, 0xffeb, 0xffdf, 0xffd2, 0xffc6, 0xffba,
    0xffad, 0xffa1, 0xff94, 0xff88, 0xff7b, 0xff6f, 0xff62, 0xff56, 0xff49, 0xff3c, 0xff30, 0xff23,
    0xff16, 0xff0a, 0xfefd, 0xfef0, 0xfee3, 0xfed7, 0xfeca, 0xfebd, 0xfeb0, 0xfea3, 0xfe96, 0xfe8a,
    0xfe7d, 0xfe70, 0xfe63, 0xfe56, 0xfe49, 0xfe3c, 0xfe2f, 0xfe23, 0xfe16, 0xfe09, 0xfdfc, 0xfdef,
    0xfde2, 0xfdd5, 0xfdc8, 0xfdbc, 0xfdaf, 0xfda2, 0xfd95, 0xfd88, 0xfd7c, 0xfd6f, 0xfd62, 0xfd55,
    0xfd48, 0xfd3c, 0xfd2f, 0xfd22, 0xfd16, 0xfd09, 0xfcfc, 0xfcf0, 0xfce3, 0xfcd7, 0xfcca, 0xfcbe,
    0xfcb1, 0xfca5, 0xfc99, 0xfc8c, 0xfc80, 0xfc74, 0xfc67, 0xfc5b, 0xfc4f, 0xfc43, 0xfc37, 0xfc2b,
    0xfc1f, 0xfc13, 0xfc07, 0xfbfb, 0xfbef, 0xfbe4, 0xfbd8, 0xfbcc, 0xfbc1, 0xfbb5, 0xfbaa, 0xfb9e,
    0xfb93, 0xfb87, 0xfb7c, 0xfb71, 0xfb66, 0xfb5b, 0xfb50, 0xfb45, 0xfb3a, 0xfb2f, 0xfb24, 0xfb19,
    0xfb0f, 0xfb04, 0xfafa, 0xfaef, 0xfae5, 0xfadb, 0xfad1, 0xfac7, 0xfabc, 0xfab3, 0xfaa9, 0xfa9f,
    0xfa95, 0xfa8c, 0xfa82, 0xfa79, 0xfa6f, 0xfa66, 0xfa5d, 0xfa54, 0xfa4b, 0xfa42, 0xfa39, 0xfa30,
    0xfa27, 0xfa1f, 0xfa16, 0xfa0e, 0xfa06, 0xf9fe, 0xf9f6, 0xf9ee, 0xf9e6, 0xf9de, 0xf9d6, 0xf9cf,
    0xf9c7, 0xf9c0, 0xf9b9, 0xf9b2, 0xf9ab, 0xf9a4, 0xf99d, 0xf996, 0xf990, 0xf989, 0xf983, 0xf97d,
    0xf976, 0xf970, 0xf96b, 0xf965, 0xf95f, 0xf95a, 0xf954, 0xf94f, 0xf94a, 0xf945, 0xf940, 0xf93b,
    0xf936, 0xf932, 0xf92d, 0xf929, 0xf925, 0xf921, 0xf91d, 0xf919, 0xf916, 0xf912, 0xf90f, 0xf90b,
    0xf908, 0xf905, 0xf903, 0xf900, 0xf8fd, 0xf8fb, 0xf8f8, 0xf8f6, 0xf8f4, 0xf8f2, 0xf8f1, 0xf8ef,
    0xf8ee, 0xf8ec, 0xf8eb, 0xf8ea, 0xf8e9, 0xf8e8, 0xf8e8, 0xf8e7, 0xf8e7, 0xf8e7, 0xf8e7, 0xf8e7,
    0xf8e7, 0xf8e8, 0xf8e8, 0xf8e9, 0xf8ea, 0xf8eb, 0xf8ec, 0xf8ed, 0xf8ef, 0xf8f0, 0xf8f2, 0xf8f4,
    0xf8f6, 0xf8f8, 0xf8fa, 0xf8fd, 0xf900, 0xf902, 0xf905, 0xf908, 0xf90c, 0xf90f, 0xf913, 0xf916,
    0xf91a, 0xf91e, 0xf922, 0xf927, 0xf92b, 0xf930, 0xf935, 0xf93a, 0xf93f, 0xf944, 0xf949, 0xf94f,
    0xf955, 0xf95a, 0xf961, 0xf967, 0xf96d, 0xf973, 0xf97a, 0xf981, 0xf988, 0xf98f, 0xf996, 0xf99e,
    0xf9a5, 0xf9ad, 0xf9b5, 0xf9bd, 0xf9c5, 0xf9cd, 0xf9d6, 0xf9de, 0xf9e7, 0xf9f0, 0xf9f9, 0xfa02,
    0xfa0c, 0xfa15, 0xfa1f, 0xfa29, 0xfa33, 0xfa3d, 0xfa47, 0xfa51, 0xfa5c, 0xfa67, 0xfa72, 0xfa7d,
    0xfa88, 0xfa93, 0xfa9e, 0xfaaa, 0xfab6, 0xfac2, 0xface, 0xfada, 0xfae6, 0xfaf3, 0xfaff, 0xfb0c,
    0xfb19, 0xfb26, 0xfb33, 0xfb40, 0xfb4d, 0xfb5b, 0xfb69, 0xfb77, 0xfb84, 0xfb93, 0xfba1, 0xfbaf,
    0xfbbe, 0xfbcc, 0xfbdb, 0xfbea, 0xfbf9, 0xfc08, 0xfc17, 0xfc27, 0xfc36, 0xfc46, 0xfc55, 0xfc65,
    0xfc75, 0xfc85, 0xfc96, 0xfca6, 0xfcb7, 0xfcc7, 0xfcd8, 0xfce9, 0xfcfa, 0xfd0b, 0xfd1c, 0xfd2d,
    0xfd3e, 0xfd50, 0xfd61, 0xfd73, 0xfd85, 0xfd97, 0xfda9, 0xfdbb, 0xfdcd, 0xfddf, 0xfdf2, 0xfe04,
    0xfe17, 0xfe2a, 0xfe3c, 0xfe4f, 0xfe62, 0xfe75, 0xfe88, 0xfe9c, 0xfeaf, 0xfec2, 0xfed6, 0xfeea,
    0xfefd, 0xff11, 0xff25, 0xff39, 0xff4d, 0xff61, 0xff75, 0xff89, 0xff9d, 0xffb1, 0xffc6, 0xffda,
    0xffef, 0x0003, 0x0018, 0x002d, 0x0042, 0x0056, 0x006b, 0x0080, 0x0095, 0x00aa, 0x00bf, 0x00d4,
    0x00ea, 0x00ff, 0x0114, 0x0129, 0x013f, 0x0154, 0x016a, 0x017f, 0x0194, 0x01aa, 0x01c0, 0x01d5,
    0x01eb, 0x0200, 0x0216, 0x022c, 0x0241, 0x0257, 0x026d, 0x0283, 0x0298, 0x02ae, 0x02c4, 0x02da,
    0x02f0, 0x0306, 0x031b, 0x0331, 0x0347, 0x035d, 0x0373, 0x0388, 0x039e, 0x03b4, 0x03ca, 0x03e0,
    0x03f5, 0x040b, 0x0421, 0x0437, 0x044c, 0x0462, 0x0478, 0x048d, 0x04a3, 0x04b9, 0x04ce, 0x04e4,
    0x04f9, 0x050e, 0x0524, 0x0539, 0x054f, 0x0564, 0x0579, 0x058e, 0x05a3, 0x05b8, 0x05cd, 0x05e2,
    0x05f7, 0x060c, 0x0621, 0x0636, 0x064a, 0x065f, 0x0674, 0x0688, 0x069c, 0x06b1, 0x06c5, 0x06d9,
    0x06ed, 0x0701, 0x0715, 0x0729, 0x073d, 0x0751, 0x0764, 0x0778, 0x078b, 0x079e, 0x07b2, 0x07c5,
    0x07d8, 0x07eb, 0x07fd, 0x0810, 0x0823, 0x0835, 0x0847, 0x085a, 0x086c, 0x087e, 0x0890, 0x08a2,
    0x08b3, 0x08c5, 0x08d6, 0x08e8, 0x08f9, 0x090a, 0x091b, 0x092b, 0x093c, 0x094c, 0x095d, 0x096d,
    0x097d, 0x098d, 0x099d, 0x09ac, 0x09bc, 0x09cb, 0x09da, 0x09e9, 0x09f8, 0x0a07, 0x0a16, 0x0a24,
    0x0a32, 0x0a40, 0x0a4e, 0x0a5c, 0x0a69, 0x0a77, 0x0a84, 0x0a91, 0x0a9e, 0x0aab, 0x0ab7, 0x0ac3,
    0x0ad0, 0x0adb, 0x0ae7, 0x0af3, 0x0afe, 0x0b09, 0x0b14, 0x0b1f, 0x0b2a, 0x0b34, 0x0b3f, 0x0b49,
    0x0b53, 0x0b5c, 0x0b66, 0x0b6f, 0x0b78, 0x0b81, 0x0b89, 0x0b92, 0x0b9a, 0x0ba2, 0x0baa, 0x0bb1,
    0x0bb9, 0x0bc0, 0x0bc7, 0x0bcd, 0x0bd4, 0x0bda, 0x0be0, 0x0be6, 0x0beb, 0x0bf1, 0x0bf6, 0x0bfb,
    0x0bff, 0x0c04, 0x0c08, 0x0c0c, 0x0c10, 0x0c13, 0x0c17, 0x0c1a, 0x0c1c, 0x0c1f, 0x0c21, 0x0c23,
    0x0c25, 0x0c27, 0x0c28, 0x0c29, 0x0c2a, 0x0c2b, 0x0c2b, 0x0c2b, 0x0c2b, 0x0c2a, 0x0c2a, 0x0c29,
    0x0c28, 0x0c26, 0x0c25, 0x0c23, 0x0c21, 0x0c1e, 0x0c1c, 0x0c19, 0x0c15, 0x0c12, 0x0c0e, 0x0c0a,
    0x0c06, 0x0c02, 0x0bfd, 0x0bf8, 0x0bf3, 0x0bed, 0x0be7, 0x0be1, 0x0bdb, 0x0bd4, 0x0bce, 0x0bc6,
    0x0bbf, 0x0bb7, 0x0bb0, 0x0ba7, 0x0b9f, 0x0b96, 0x0b8d, 0x0b84, 0x0b7b, 0x0b71, 0x0b67, 0x0b5d,
    0x0b52, 0x0b47, 0x0b3c, 0x0b31, 0x0b25, 0x0b19, 0x0b0d, 0x0b01, 0x0af4, 0x0ae7, 0x0ada, 0x0acc,
    0x0abf, 0x0ab1, 0x0aa2, 0x0a94, 0x0a85, 0x0a76, 0x0a67, 0x0a57, 0x0a47, 0x0a37, 0x0a27, 0x0a16,
    0x0a05, 0x09f4, 0x09e2, 0x09d1, 0x09bf, 0x09ad, 0x099a, 0x0987, 0x0974, 0x0961, 0x094e, 0x093a,
    0x0926, 0x0912, 0x08fd, 0x08e8, 0x08d3, 0x08be, 0x08a8, 0x0893, 0x087d, 0x0866, 0x0850, 0x0839,
    0x0822, 0x080b, 0x07f3, 0x07dc, 0x07c4, 0x07ab, 0x0793, 0x077a, 0x0761, 0x0748, 0x072f, 0x0715,
    0x06fb, 0x06e1, 0x06c7, 0x06ac, 0x0691, 0x0676, 0x065b, 0x0640, 0x0624, 0x0608, 0x05ec, 0x05d0,
    0x05b3, 0x0596, 0x0579, 0x055c, 0x053f, 0x0521, 0x0503, 0x04e5, 0x04c7, 0x04a9, 0x048a, 0x046b,
    0x044c, 0x042d, 0x040d, 0x03ee, 0x03ce, 0x03ae, 0x038e, 0x036d, 0x034d, 0x032c, 0x030b, 0x02ea,
    0x02c9, 0x02a7, 0x0286, 0x0264, 0x0242, 0x0220, 0x01fe, 0x01db, 0x01b9, 0x0196, 0x0173, 0x0150,
    0x012d, 0x010a, 0x00e6, 0x00c2, 0x009f, 0x007b, 0x0057, 0x0032, 0x000e, 0xffea, 0xffc5, 0xffa0,
    0xff7c, 0xff57, 0xff32, 0xff0c, 0xfee7, 0xfec2, 0xfe9c, 0xfe77, 0xfe51, 0xfe2b, 0xfe05, 0xfddf,
    0xfdb9, 0xfd93, 0xfd6c, 0xfd46, 0xfd20, 0xfcf9, 0xfcd2, 0xfcac, 0xfc85, 0xfc5e, 0xfc37, 0xfc10,
    0xfbe9, 0xfbc2, 0xfb9b, 0xfb74, 0xfb4d, 0xfb25, 0xfafe, 0xfad7, 0xfaaf, 0xfa88, 0xfa60, 0xfa39,
    0xfa11, 0xf9ea, 0xf9c2, 0xf99a, 0xf973, 0xf94b, 0xf924, 0xf8fc, 0xf8d4, 0xf8ad, 0xf885, 0xf85d,
    0xf836, 0xf80e, 0xf7e6, 0xf7bf, 0xf797, 0xf770, 0xf748, 0xf721, 0xf6f9, 0xf6d2, 0xf6ab, 0xf683,
    0xf65c, 0xf635, 0xf60d, 0xf5e6, 0xf5bf, 0xf598, 0xf571, 0xf54a, 0xf524, 0xf4fd, 0xf4d6, 0xf4b0,
    0xf489, 0xf463, 0xf43d, 0xf416, 0xf3f0, 0xf3ca, 0xf3a4, 0xf37f, 0xf359, 0xf333, 0xf30e, 0xf2e9,
    0xf2c3, 0xf29e, 0xf279, 0xf254, 0xf230, 0xf20b, 0xf1e7, 0xf1c3, 0xf19f, 0xf17b, 0xf157, 0xf133,
    0xf110, 0xf0ec, 0xf0c9, 0xf0a6, 0xf084, 0xf061, 0xf03f, 0xf01c, 0xeffa, 0xefd8, 0xefb7, 0xef95,
    0xef74, 0xef53, 0xef32, 0xef12, 0xeef1, 0xeed1, 0xeeb1, 0xee91, 0xee72, 0xee53, 0xee34, 0xee15,
    0xedf6, 0xedd8, 0xedba, 0xed9c, 0xed7f, 0xed61, 0xed44, 0xed28, 0xed0b, 0xecef, 0xecd3, 0xecb7,
    0xec9c, 0xec81, 0xec66, 0xec4b, 0xec31, 0xec17, 0xebfe, 0xebe4, 0xebcb, 0xebb3, 0xeb9a, 0xeb82,
    0xeb6a, 0xeb53, 0xeb3c, 0xeb25, 0xeb0f, 0xeaf9, 0xeae3, 0xeacd, 0xeab8, 0xeaa4, 0xea8f, 0xea7b,
    0xea68, 0xea54, 0xea41, 0xea2f, 0xea1c, 0xea0b, 0xe9f9, 0xe9e8, 0xe9d7, 0xe9c7, 0xe9b7, 0xe9a7,
    0xe998, 0xe989, 0xe97b, 0xe96d, 0xe960, 0xe952, 0xe946, 0xe939, 0xe92d, 0xe922, 0xe917, 0xe90c,
    0xe902, 0xe8f8, 0xe8ee, 0xe8e5, 0xe8dd, 0xe8d5, 0xe8cd, 0xe8c6, 0xe8bf, 0xe8b9, 0xe8b3, 0xe8ad,
    0xe8a8, 0xe8a4, 0xe8a0, 0xe89c, 0xe899, 0xe896, 0xe894, 0xe892, 0xe891, 0xe890, 0xe890, 0xe890,
    0xe890, 0xe891, 0xe893, 0xe895, 0xe898, 0xe89b, 0xe89e, 0xe8a2, 0xe8a7, 0xe8ac, 0xe8b1, 0xe8b7,
    0xe8be, 0xe8c5, 0xe8cc, 0xe8d4, 0xe8dd, 0xe8e6, 0xe8ef, 0xe8fa, 0xe904, 0xe90f, 0xe91b, 0xe927,
    0xe934, 0xe941, 0xe94f, 0xe95d, 0xe96c, 0xe97b, 0xe98b, 0xe99b, 0xe9ac, 0xe9be, 0xe9d0, 0xe9e2,
    0xe9f6, 0xea09, 0xea1d, 0xea32, 0xea47, 0xea5d, 0xea74, 0xea8a, 0xeaa2, 0xeaba, 0xead3, 0xeaec,
    0xeb05, 0xeb20, 0xeb3a, 0xeb56, 0xeb72, 0xeb8e, 0xebab, 0xebc9, 0xebe7, 0xec05, 0xec25, 0xec44,
    0xec65, 0xec86, 0xeca7, 0xecc9, 0xecec, 0xed0f, 0xed33, 0xed57, 0xed7c, 0xeda1, 0xedc7, 0xedee,
    0xee15, 0xee3d, 0xee65, 0xee8e, 0xeeb7, 0xeee1, 0xef0c, 0xef37, 0xef62, 0xef8f, 0xefbb, 0xefe9,
    0xf017, 0xf045, 0xf074, 0xf0a4, 0xf0d4, 0xf105, 0xf136, 0xf168, 0xf19a, 0xf1cd, 0xf201, 0xf235,
    0xf269, 0xf29f, 0xf2d4, 0xf30b, 0xf342, 0xf379, 0xf3b1, 0xf3ea, 0xf423, 0xf45c, 0xf497, 0xf4d1,
    0xf50d, 0xf549, 0xf585, 0xf5c2, 0xf5ff, 0xf63d, 0xf67c, 0xf6bb, 0xf6fb, 0xf73b, 0xf77c, 0xf7bd,
    0xf7ff, 0xf841, 0xf884, 0xf8c8, 0xf90c, 0xf950, 0xf995, 0xf9db, 0xfa21, 0xfa67, 0xfaae, 0xfaf6,
    0xfb3e, 0xfb87, 0xfbd0, 0xfc19, 0xfc64, 0xfcae, 0xfcf9, 0xfd45, 0xfd91, 0xfdde, 0xfe2b, 0xfe79,
    0xfec7, 0xff16, 0xff65, 0xffb4, 0x0004, 0x0055, 0x00a6, 0x00f8, 0x014a, 0x019c, 0x01ef, 0x0243,
    0x0296, 0x02eb, 0x0340, 0x0395, 0x03eb, 0x0441, 0x0497, 0x04ee, 0x0546, 0x059e, 0x05f6, 0x064f,
    0x06a8, 0x0702, 0x075c, 0x07b7, 0x0811, 0x086d, 0x08c9, 0x0925, 0x0981, 0x09de, 0x0a3c, 0x0a9a,
    0x0af8, 0x0b56, 0x0bb5, 0x0c15, 0x0c74, 0x0cd4, 0x0d35, 0x0d96, 0x0df7, 0x0e58, 0x0eba, 0x0f1d,
    0x0f7f, 0x0fe2, 0x1046, 0x10a9, 0x110d, 0x1172, 0x11d6, 0x123b, 0x12a1, 0x1307, 0x136c, 0x13d3,
    0x1439, 0x14a0, 0x1508, 0x156f, 0x15d7, 0x163f, 0x16a7, 0x1710, 0x1779, 0x17e2, 0x184c, 0x18b5,
    0x191f, 0x198a, 0x19f4, 0x1a5f, 0x1aca, 0x1b35, 0x1ba1, 0x1c0c, 0x1c78, 0x1ce5, 0x1d51, 0x1dbe,
    0x1e2a, 0x1e97, 0x1f05, 0x1f72, 0x1fe0, 0x204d, 0x20bb, 0x212a, 0x2198, 0x2206, 0x2275, 0x22e4,
    0x2353, 0x23c2, 0x2431, 0x24a1, 0x2511, 0x2580, 0x25f0, 0x2660, 0x26d0, 0x2740, 0x27b1, 0x2821,
    0x2892, 0x2902, 0x2973, 0x29e4, 0x2a55, 0x2ac6, 0x2b37, 0x2ba8, 0x2c19, 0x2c8a, 0x2cfc, 0x2d6d,
    0x2dde, 0x2e50, 0x2ec1, 0x2f33, 0x2fa4, 0x3016, 0x3087, 0x30f9, 0x316a, 0x31dc, 0x324e, 0x32bf,
    0x3331, 0x33a2, 0x3414, 0x3485, 0x34f7, 0x3568, 0x35d9, 0x364b, 0x36bc, 0x372d, 0x379e, 0x380f,
    0x3880, 0x38f1, 0x3962, 0x39d3, 0x3a43, 0x3ab4, 0x3b24, 0x3b95, 0x3c05, 0x3c75, 0x3ce5, 0x3d55,
    0x3dc5, 0x3e34, 0x3ea4, 0x3f13, 0x3f82, 0x3ff1, 0x4060, 0x40cf, 0x413d, 0x41ab, 0x4219, 0x4287,
    0x42f5, 0x4363, 0x43d0, 0x443d, 0x44aa, 0x4517, 0x4583, 0x45ef, 0x465b, 0x46c7, 0x4733, 0x479e,
    0x4809, 0x4874, 0x48de, 0x4948, 0x49b2, 0x4a1c, 0x4a86, 0x4aef, 0x4b57, 0x4bc0, 0x4c28, 0x4c90,
    0x4cf8, 0x4d5f, 0x4dc6, 0x4e2d, 0x4e93, 0x4ef9, 0x4f5f, 0x4fc4, 0x5029, 0x508e, 0x50f2, 0x5156,
    0x51b9, 0x521c, 0x527f, 0x52e1, 0x5343, 0x53a5, 0x5406, 0x5467, 0x54c8, 0x5528, 0x5587, 0x55e6,
    0x5645, 0x56a3, 0x5701, 0x575f, 0x57bc, 0x5818, 0x5874, 0x58d0, 0x592b, 0x5986, 0x59e0, 0x5a3a,
    0x5a93, 0x5aec, 0x5b45, 0x5b9d, 0x5bf4, 0x5c4b, 0x5ca1, 0x5cf7, 0x5d4d, 0x5da1, 0x5df6, 0x5e4a,
    0x5e9d, 0x5ef0, 0x5f42, 0x5f94, 0x5fe5, 0x6036, 0x6086, 0x60d5, 0x6124, 0x6173, 0x61c1, 0x620e,
    0x625b, 0x62a7, 0x62f2, 0x633d, 0x6388, 0x63d2, 0x641b, 0x6463, 0x64ac, 0x64f3, 0x653a, 0x6580,
    0x65c6, 0x660b, 0x664f, 0x6693, 0x66d6, 0x6718, 0x675a, 0x679c, 0x67dc, 0x681c, 0x685b, 0x689a,
    0x68d8, 0x6915, 0x6952, 0x698e, 0x69ca, 0x6a04, 0x6a3e, 0x6a78, 0x6ab1, 0x6ae9, 0x6b20, 0x6b57,
    0x6b8d, 0x6bc2, 0x6bf6, 0x6c2a, 0x6c5e, 0x6c90, 0x6cc2, 0x6cf3, 0x6d24, 0x6d53, 0x6d82, 0x6db1,
    0x6dde, 0x6e0b, 0x6e37, 0x6e63, 0x6e8d, 0x6eb7, 0x6ee1, 0x6f09, 0x6f31, 0x6f58, 0x6f7e, 0x6fa4,
    0x6fc9, 0x6fed, 0x7010, 0x7033, 0x7055, 0x7076, 0x7096, 0x70b6, 0x70d5, 0x70f3, 0x7110, 0x712d,
    0x7149, 0x7164, 0x717e, 0x7198, 0x71b1, 0x71c9, 0x71e0, 0x71f7, 0x720c, 0x7221, 0x7236, 0x7249,
    0x725c, 0x726e, 0x727f, 0x728f, 0x729f, 0x72ae, 0x72bc, 0x72c9, 0x72d5, 0x72e1, 0x72ec, 0x72f6,
    0x7300, 0x7308, 0x7310, 0x7317, 0x731d, 0x7323, 0x7327, 0x732b, 0x732e, 0x7331, 0x7332, 0x7333,
};

// ----- 147 : 160 ---------------
// makeFIR(32, 147, 160, .96, 8);
// Meets +/- 2dB frequency response tolerance of GIS-304-108 at 20 kHz (resampling from 48k to 44.1k).
// Stop-band rejection is >85 dB (with input from 20 to 20kHz). With input from 20kHz-24kHz, aliasing
// has <85dB rejection, but aliasing component is >16kHz and inaudible.
// Previous settings were makeFIR(24, 147, 160, .95, 8), however this resulted in -3.2 dB roll-off
// at 20kHz.
static const int16_t fir_147_160[ (147 * FIR_TAP_32) / 2] = {
    0x0001, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0003, 0x0003, 0x0003, 0x0003, 0x0003,
    0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0006, 0x0006,
    0x0006, 0x0006, 0x0006, 0x0007, 0x0007, 0x0007, 0x0007, 0x0008, 0x0008, 0x0008, 0x0008, 0x0008,
    0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x000a, 0x000a, 0x000a, 0x000a, 0x000a, 0x000b, 0x000b,
    0x000b, 0x000b, 0x000b, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c, 0x000d, 0x000d, 0x000d,
    0x000d, 0x000d, 0x000d, 0x000e, 0x000e, 0x000e, 0x000e, 0x000e, 0x000e, 0x000e, 0x000f, 0x000f,
    0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x0010, 0x0010, 0x0010,
    0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010,
    0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x000f, 0x000f, 0x000f, 0x000f,
    0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x000e, 0x000e, 0x000e, 0x000e, 0x000e, 0x000e, 0x000d,
    0x000d, 0x000d, 0x000d, 0x000d, 0x000c, 0x000c, 0x000c, 0x000c, 0x000b, 0x000b, 0x000b, 0x000b,
    0x000a, 0x000a, 0x000a, 0x0009, 0x0009, 0x0009, 0x0009, 0x0008, 0x0008, 0x0008, 0x0007, 0x0007,
    0x0006, 0x0006, 0x0006, 0x0005, 0x0005, 0x0005, 0x0004, 0x0004, 0x0003, 0x0003, 0x0002, 0x0002,
    0x0002, 0x0001, 0x0001, 0x0000, 0x0000, 0xffff, 0xffff, 0xfffe, 0xfffe, 0xfffd, 0xfffd, 0xfffc,
    0xfffc, 0xfffb, 0xfffb, 0xfffa, 0xfffa, 0xfff9, 0xfff9, 0xfff8, 0xfff8, 0xfff7, 0xfff7, 0xfff6,
    0xfff6, 0xfff5, 0xfff5, 0xfff4, 0xfff4, 0xfff3, 0xfff3, 0xfff2, 0xfff2, 0xfff1, 0xfff1, 0xfff0,
    0xffef, 0xffef, 0xffee, 0xffee, 0xffed, 0xffed, 0xffec, 0xffec, 0xffeb, 0xffeb, 0xffea, 0xffea,
    0xffe9, 0xffe9, 0xffe8, 0xffe8, 0xffe7, 0xffe7, 0xffe6, 0xffe6, 0xffe5, 0xffe5, 0xffe5, 0xffe4,
    0xffe4, 0xffe3, 0xffe3, 0xffe2, 0xffe2, 0xffe2, 0xffe1, 0xffe1, 0xffe0, 0xffe0, 0xffe0, 0xffdf,
    0xffdf, 0xffdf, 0xffde, 0xffde, 0xffde, 0xffdd, 0xffdd, 0xffdd, 0xffdd, 0xffdc, 0xffdc, 0xffdc,
    0xffdc, 0xffdc, 0xffdb, 0xffdb, 0xffdb, 0xffdb, 0xffdb, 0xffdb, 0xffda, 0xffda, 0xffda, 0xffda,
    0xffda, 0xffda, 0xffda, 0xffda, 0xffda, 0xffda, 0xffda, 0xffda, 0xffda, 0xffda, 0xffda, 0xffda,
    0xffdb, 0xffdb, 0xffdb, 0xffdb, 0xffdb, 0xffdb, 0xffdc, 0xffdc, 0xffdc, 0xffdc, 0xffdd, 0xffdd,
    0xffdd, 0xffdd, 0xffde, 0xffde, 0xffdf, 0xffdf, 0xffdf, 0xffe0, 0xffe0, 0xffe1, 0xffe1, 0xffe2,
    0xffe2, 0xffe3, 0xffe3, 0xffe4, 0xffe4, 0xffe5, 0xffe5, 0xffe6, 0xffe7, 0xffe7, 0xffe8, 0xffe9,
    0xffe9, 0xffea, 0xffeb, 0xffeb, 0xffec, 0xffed, 0xffee, 0xffef, 0xffef, 0xfff0, 0xfff1, 0xfff2,
    0xfff3, 0xfff4, 0xfff4, 0xfff5, 0xfff6, 0xfff7, 0xfff8, 0xfff9, 0xfffa, 0xfffb, 0xfffc, 0xfffd,
    0xfffe, 0xffff, 0x0000, 0x0001, 0x0002, 0x0003, 0x0004, 0x0005, 0x0006, 0x0007, 0x0008, 0x0009,
    0x000a, 0x000b, 0x000d, 0x000e, 0x000f, 0x0010, 0x0011, 0x0012, 0x0013, 0x0014, 0x0015, 0x0016,
    0x0018, 0x0019, 0x001a, 0x001b, 0x001c, 0x001d, 0x001e, 0x001f, 0x0021, 0x0022, 0x0023, 0x0024,
    0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002e, 0x002f, 0x0030, 0x0031,
    0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c,
    0x003d, 0x003e, 0x003e, 0x003f, 0x0040, 0x0041, 0x0042, 0x0042, 0x0043, 0x0044, 0x0044, 0x0045,
    0x0046, 0x0046, 0x0047, 0x0047, 0x0048, 0x0048, 0x0049, 0x0049, 0x004a, 0x004a, 0x004b, 0x004b,
    0x004b, 0x004c, 0x004c, 0x004c, 0x004d, 0x004d, 0x004d, 0x004d, 0x004d, 0x004d, 0x004e, 0x004e,
    0x004e, 0x004e, 0x004e, 0x004e, 0x004d, 0x004d, 0x004d, 0x004d, 0x004d, 0x004d, 0x004c, 0x004c,
    0x004c, 0x004b, 0x004b, 0x004a, 0x004a, 0x004a, 0x0049, 0x0048, 0x0048, 0x0047, 0x0047, 0x0046,
    0x0045, 0x0044, 0x0044, 0x0043, 0x0042, 0x0041, 0x0040, 0x003f, 0x003e, 0x003d, 0x003c, 0x003b,
    0x003a, 0x0039, 0x0038, 0x0037, 0x0036, 0x0034, 0x0033, 0x0032, 0x0031, 0x002f, 0x002e, 0x002c,
    0x002b, 0x002a, 0x0028, 0x0027, 0x0025, 0x0023, 0x0022, 0x0020, 0x001f, 0x001d, 0x001b, 0x001a,
    0x0018, 0x0016, 0x0014, 0x0013, 0x0011, 0x000f, 0x000d, 0x000b, 0x0009, 0x0007, 0x0006, 0x0004,
    0x0002, 0x0000, 0xfffe, 0xfffc, 0xfffa, 0xfff8, 0xfff6, 0xfff4, 0xfff2, 0xffef, 0xffed, 0xffeb,
    0xffe9, 0xffe7, 0xffe5, 0xffe3, 0xffe1, 0xffdf, 0xffdd, 0xffda, 0xffd8, 0xffd6, 0xffd4, 0xffd2,
    0xffd0, 0xffce, 0xffcc, 0xffc9, 0xffc7, 0xffc5, 0xffc3, 0xffc1, 0xffbf, 0xffbd, 0xffbb, 0xffb9,
    0xffb7, 0xffb5, 0xffb3, 0xffb1, 0xffaf, 0xffad, 0xffab, 0xffa9, 0xffa7, 0xffa5, 0xffa3, 0xffa1,
    0xffa0, 0xff9e, 0xff9c, 0xff9a, 0xff99, 0xff97, 0xff95, 0xff94, 0xff92, 0xff90, 0xff8f, 0xff8d,
    0xff8c, 0xff8a, 0xff89, 0xff87, 0xff86, 0xff85, 0xff83, 0xff82, 0xff81, 0xff80, 0xff7f, 0xff7e,
    0xff7c, 0xff7b, 0xff7a, 0xff7a, 0xff79, 0xff78, 0xff77, 0xff76, 0xff76, 0xff75, 0xff74, 0xff74,
    0xff73, 0xff73, 0xff72, 0xff72, 0xff72, 0xff71, 0xff71, 0xff71, 0xff71, 0xff71, 0xff71, 0xff71,
    0xff71, 0xff71, 0xff71, 0xff72, 0xff72, 0xff72, 0xff73, 0xff73, 0xff74, 0xff74, 0xff75, 0xff76,
    0xff76, 0xff77, 0xff78, 0xff79, 0xff7a, 0xff7b, 0xff7c, 0xff7d, 0xff7e, 0xff80, 0xff81, 0xff82,
    0xff84, 0xff85, 0xff87, 0xff88, 0xff8a, 0xff8c, 0xff8e, 0xff8f, 0xff91, 0xff93, 0xff95, 0xff97,
    0xff99, 0xff9b, 0xff9e, 0xffa0, 0xffa2, 0xffa4, 0xffa7, 0xffa9, 0xffac, 0xffae, 0xffb1, 0xffb3,
    0xffb6, 0xffb9, 0xffbc, 0xffbe, 0xffc1, 0xffc4, 0xffc7, 0xffca, 0xffcd, 0xffd0, 0xffd3, 0xffd6,
    0xffd9, 0xffdd, 0xffe0, 0xffe3, 0xffe6, 0xffea, 0xffed, 0xfff0, 0xfff4, 0xfff7, 0xfffb, 0xfffe,
    0x0002, 0x0005, 0x0009, 0x000c, 0x0010, 0x0014, 0x0017, 0x001b, 0x001f, 0x0022, 0x0026, 0x002a,
    0x002d, 0x0031, 0x0035, 0x0038, 0x003c, 0x0040, 0x0044, 0x0047, 0x004b, 0x004f, 0x0052, 0x0056,
    0x005a, 0x005d, 0x0061, 0x0065, 0x0068, 0x006c, 0x0070, 0x0073, 0x0077, 0x007a, 0x007e, 0x0081,
    0x0085, 0x0088, 0x008c, 0x008f, 0x0093, 0x0096, 0x0099, 0x009c, 0x00a0, 0x00a3, 0x00a6, 0x00a9,
    0x00ac, 0x00af, 0x00b2, 0x00b5, 0x00b8, 0x00bb, 0x00be, 0x00c0, 0x00c3, 0x00c6, 0x00c8, 0x00cb,
    0x00cd, 0x00cf, 0x00d2, 0x00d4, 0x00d6, 0x00d8, 0x00da, 0x00dc, 0x00de, 0x00e0, 0x00e2, 0x00e4,
    0x00e5, 0x00e7, 0x00e8, 0x00ea, 0x00eb, 0x00ec, 0x00ee, 0x00ef, 0x00f0, 0x00f1, 0x00f1, 0x00f2,
    0x00f3, 0x00f4, 0x00f4, 0x00f4, 0x00f5, 0x00f5, 0x00f5, 0x00f5, 0x00f5, 0x00f5, 0x00f5, 0x00f5,
    0x00f4, 0x00f4, 0x00f3, 0x00f3, 0x00f2, 0x00f1, 0x00f0, 0x00ef, 0x00ee, 0x00ed, 0x00eb, 0x00ea,
    0x00e9, 0x00e7, 0x00e5, 0x00e3, 0x00e2, 0x00e0, 0x00de, 0x00db, 0x00d9, 0x00d7, 0x00d4, 0x00d2,
    0x00cf, 0x00cc, 0x00ca, 0x00c7, 0x00c4, 0x00c1, 0x00bd, 0x00ba, 0x00b7, 0x00b3, 0x00b0, 0x00ac,
    0x00a9, 0x00a5, 0x00a1, 0x009d, 0x0099, 0x0095, 0x0091, 0x008c, 0x0088, 0x0084, 0x007f, 0x007b,
    0x0076, 0x0071, 0x006d, 0x0068, 0x0063, 0x005e, 0x0059, 0x0054, 0x004f, 0x0049, 0x0044, 0x003f,
    0x0039, 0x0034, 0x002e, 0x0029, 0x0023, 0x001e, 0x0018, 0x0012, 0x000d, 0x0007, 0x0001, 0xfffb,
    0xfff5, 0xffef, 0xffe9, 0xffe3, 0xffdd, 0xffd7, 0xffd1, 0xffcb, 0xffc5, 0xffbf, 0xffb9, 0xffb3,
    0xffad, 0xffa7, 0xffa0, 0xff9a, 0xff94, 0xff8e, 0xff88, 0xff82, 0xff7c, 0xff76, 0xff70, 0xff6a,
    0xff63, 0xff5d, 0xff57, 0xff52, 0xff4c, 0xff46, 0xff40, 0xff3a, 0xff34, 0xff2f, 0xff29, 0xff23,
    0xff1e, 0xff18, 0xff13, 0xff0d, 0xff08, 0xff02, 0xfefd, 0xfef8, 0xfef3, 0xfeee, 0xfee9, 0xfee4,
    0xfedf, 0xfeda, 0xfed6, 0xfed1, 0xfecc, 0xfec8, 0xfec4, 0xfebf, 0xfebb, 0xfeb7, 0xfeb3, 0xfeaf,
    0xfeac, 0xfea8, 0xfea4, 0xfea1, 0xfe9e, 0xfe9a, 0xfe97, 0xfe94, 0xfe91, 0xfe8f, 0xfe8c, 0xfe8a,
    0xfe87, 0xfe85, 0xfe83, 0xfe81, 0xfe7f, 0xfe7d, 0xfe7b, 0xfe7a, 0xfe79, 0xfe77, 0xfe76, 0xfe75,
    0xfe74, 0xfe74, 0xfe73, 0xfe73, 0xfe73, 0xfe73, 0xfe73, 0xfe73, 0xfe73, 0xfe74, 0xfe74, 0xfe75,
    0xfe76, 0xfe77, 0xfe78, 0xfe7a, 0xfe7b, 0xfe7d, 0xfe7f, 0xfe81, 0xfe83, 0xfe85, 0xfe87, 0xfe8a,
    0xfe8d, 0xfe90, 0xfe93, 0xfe96, 0xfe99, 0xfe9d, 0xfea0, 0xfea4, 0xfea8, 0xfeac, 0xfeb0, 0xfeb5,
    0xfeb9, 0xfebe, 0xfec3, 0xfec7, 0xfecd, 0xfed2, 0xfed7, 0xfedd, 0xfee2, 0xfee8, 0xfeee, 0xfef4,
    0xfefa, 0xff00, 0xff07, 0xff0d, 0xff14, 0xff1b, 0xff22, 0xff29, 0xff30, 0xff37, 0xff3e, 0xff46,
    0xff4e, 0xff55, 0xff5d, 0xff65, 0xff6d, 0xff75, 0xff7d, 0xff86, 0xff8e, 0xff96, 0xff9f, 0xffa7,
    0xffb0, 0xffb9, 0xffc2, 0xffcb, 0xffd4, 0xffdd, 0xffe6, 0xffef, 0xfff8, 0x0002, 0x000b, 0x0014,
    0x001e, 0x0027, 0x0031, 0x003a, 0x0044, 0x004d, 0x0057, 0x0060, 0x006a, 0x0074, 0x007d, 0x0087,
    0x0091, 0x009a, 0x00a4, 0x00ae, 0x00b7, 0x00c1, 0x00cb, 0x00d4, 0x00de, 0x00e7, 0x00f1, 0x00fa,
    0x0103, 0x010d, 0x0116, 0x011f, 0x0129, 0x0132, 0x013b, 0x0144, 0x014d, 0x0156, 0x015e, 0x0167,
    0x0170, 0x0178, 0x0181, 0x0189, 0x0191, 0x0199, 0x01a2, 0x01a9, 0x01b1, 0x01b9, 0x01c1, 0x01c8,
    0x01cf, 0x01d6, 0x01de, 0x01e4, 0x01eb, 0x01f2, 0x01f8, 0x01ff, 0x0205, 0x020b, 0x0211, 0x0217,
    0x021c, 0x0221, 0x0227, 0x022c, 0x0231, 0x0235, 0x023a, 0x023e, 0x0242, 0x0246, 0x024a, 0x024d,
    0x0251, 0x0254, 0x0257, 0x025a, 0x025c, 0x025e, 0x0261, 0x0263, 0x0264, 0x0266, 0x0267, 0x0268,
    0x0269, 0x026a, 0x026a, 0x026a, 0x026a, 0x026a, 0x0269, 0x0269, 0x0268, 0x0266, 0x0265, 0x0263,
    0x0262, 0x025f, 0x025d, 0x025b, 0x0258, 0x0255, 0x0251, 0x024e, 0x024a, 0x0246, 0x0242, 0x023e,
    0x0239, 0x0234, 0x022f, 0x022a, 0x0224, 0x021e, 0x0218, 0x0212, 0x020c, 0x0205, 0x01fe, 0x01f7,
    0x01ef, 0x01e8, 0x01e0, 0x01d8, 0x01d0, 0x01c7, 0x01bf, 0x01b6, 0x01ad, 0x01a4, 0x019a, 0x0191,
    0x0187, 0x017d, 0x0173, 0x0168, 0x015e, 0x0153, 0x0148, 0x013d, 0x0131, 0x0126, 0x011a, 0x010f,
    0x0103, 0x00f7, 0x00ea, 0x00de, 0x00d1, 0x00c5, 0x00b8, 0x00ab, 0x009e, 0x0091, 0x0083, 0x0076,
    0x0069, 0x005b, 0x004d, 0x003f, 0x0031, 0x0023, 0x0015, 0x0007, 0xfff9, 0xffeb, 0xffdc, 0xffce,
    0xffbf, 0xffb1, 0xffa2, 0xff94, 0xff85, 0xff76, 0xff68, 0xff59, 0xff4a, 0xff3b, 0xff2d, 0xff1e,
    0xff0f, 0xff00, 0xfef2, 0xfee3, 0xfed4, 0xfec6, 0xfeb7, 0xfea9, 0xfe9a, 0xfe8c, 0xfe7e, 0xfe6f,
    0xfe61, 0xfe53, 0xfe45, 0xfe37, 0xfe29, 0xfe1b, 0xfe0e, 0xfe00, 0xfdf3, 0xfde6, 0xfdd9, 0xfdcc,
    0xfdbf, 0xfdb2, 0xfda5, 0xfd99, 0xfd8d, 0xfd81, 0xfd75, 0xfd69, 0xfd5d, 0xfd52, 0xfd47, 0xfd3c,
    0xfd31, 0xfd27, 0xfd1c, 0xfd12, 0xfd08, 0xfcff, 0xfcf5, 0xfcec, 0xfce3, 0xfcda, 0xfcd2, 0xfcca,
    0xfcc2, 0xfcba, 0xfcb3, 0xfcab, 0xfca4, 0xfc9e, 0xfc98, 0xfc91, 0xfc8c, 0xfc86, 0xfc81, 0xfc7c,
    0xfc78, 0xfc73, 0xfc6f, 0xfc6c, 0xfc69, 0xfc66, 0xfc63, 0xfc60, 0xfc5e, 0xfc5d, 0xfc5b, 0xfc5a,
    0xfc5a, 0xfc59, 0xfc59, 0xfc59, 0xfc5a, 0xfc5b, 0xfc5c, 0xfc5e, 0xfc60, 0xfc62, 0xfc65, 0xfc68,
    0xfc6c, 0xfc6f, 0xfc73, 0xfc78, 0xfc7d, 0xfc82, 0xfc87, 0xfc8d, 0xfc93, 0xfc9a, 0xfca1, 0xfca8,
    0xfcb0, 0xfcb8, 0xfcc0, 0xfcc8, 0xfcd1, 0xfcdb, 0xfce4, 0xfcee, 0xfcf9, 0xfd03, 0xfd0e, 0xfd19,
    0xfd25, 0xfd31, 0xfd3d, 0xfd4a, 0xfd57, 0xfd64, 0xfd71, 0xfd7f, 0xfd8d, 0xfd9b, 0xfdaa, 0xfdb9,
    0xfdc8, 0xfdd8, 0xfde8, 0xfdf8, 0xfe08, 0xfe19, 0xfe29, 0xfe3b, 0xfe4c, 0xfe5d, 0xfe6f, 0xfe81,
    0xfe94, 0xfea6, 0xfeb9, 0xfecc, 0xfedf, 0xfef2, 0xff06, 0xff19, 0xff2d, 0xff41, 0xff56, 0xff6a,
    0xff7e, 0xff93, 0xffa8, 0xffbd, 0xffd2, 0xffe7, 0xfffc, 0x0012, 0x0027, 0x003d, 0x0053, 0x0068,
    0x007e, 0x0094, 0x00aa, 0x00c0, 0x00d6, 0x00ec, 0x0102, 0x0118, 0x012e, 0x0144, 0x015a, 0x0170,
    0x0186, 0x019c, 0x01b2, 0x01c8, 0x01de, 0x01f4, 0x0209, 0x021f, 0x0234, 0x024a, 0x025f, 0x0274,
    0x0289, 0x029e, 0x02b3, 0x02c7, 0x02dc, 0x02f0, 0x0304, 0x0318, 0x032c, 0x0340, 0x0353, 0x0366,
    0x0379, 0x038c, 0x039e, 0x03b0, 0x03c2, 0x03d4, 0x03e5, 0x03f6, 0x0407, 0x0418, 0x0428, 0x0438,
    0x0447, 0x0457, 0x0466, 0x0474, 0x0483, 0x0490, 0x049e, 0x04ab, 0x04b8, 0x04c5, 0x04d1, 0x04dc,
    0x04e8, 0x04f3, 0x04fd, 0x0507, 0x0511, 0x051a, 0x0523, 0x052b, 0x0533, 0x053b, 0x0542, 0x0549,
    0x054f, 0x0554, 0x055a, 0x055e, 0x0563, 0x0566, 0x056a, 0x056d, 0x056f, 0x0571, 0x0572, 0x0573,
    0x0573, 0x0573, 0x0573, 0x0571, 0x0570, 0x056e, 0x056b, 0x0568, 0x0564, 0x0560, 0x055b, 0x0556,
    0x0550, 0x054a, 0x0543, 0x053c, 0x0534, 0x052b, 0x0522, 0x0519, 0x050f, 0x0505, 0x04fa, 0x04ee,
    0x04e2, 0x04d6, 0x04c9, 0x04bb, 0x04ad, 0x049e, 0x048f, 0x0480, 0x0470, 0x045f, 0x044e, 0x043d,
    0x042b, 0x0419, 0x0406, 0x03f2, 0x03df, 0x03ca, 0x03b6, 0x03a0, 0x038b, 0x0375, 0x035e, 0x0347,
    0x0330, 0x0319, 0x0300, 0x02e8, 0x02cf, 0x02b6, 0x029c, 0x0282, 0x0268, 0x024d, 0x0232, 0x0217,
    0x01fb, 0x01df, 0x01c3, 0x01a6, 0x0189, 0x016c, 0x014e, 0x0131, 0x0113, 0x00f4, 0x00d6, 0x00b7,
    0x0098, 0x0079, 0x005a, 0x003a, 0x001b, 0xfffb, 0xffdb, 0xffbb, 0xff9a, 0xff7a, 0xff59, 0xff39,
    0xff18, 0xfef7, 0xfed6, 0xfeb5, 0xfe95, 0xfe74, 0xfe53, 0xfe32, 0xfe11, 0xfdf0, 0xfdcf, 0xfdae,
    0xfd8d, 0xfd6c, 0xfd4b, 0xfd2b, 0xfd0a, 0xfcea, 0xfcca, 0xfca9, 0xfc89, 0xfc6a, 0xfc4a, 0xfc2b,
    0xfc0b, 0xfbec, 0xfbcd, 0xfbaf, 0xfb90, 0xfb72, 0xfb55, 0xfb37, 0xfb1a, 0xfafd, 0xfae0, 0xfac4,
    0xfaa8, 0xfa8c, 0xfa71, 0xfa56, 0xfa3c, 0xfa22, 0xfa08, 0xf9ef, 0xf9d6, 0xf9be, 0xf9a6, 0xf98f,
    0xf978, 0xf961, 0xf94c, 0xf936, 0xf921, 0xf90d, 0xf8f9, 0xf8e6, 0xf8d3, 0xf8c1, 0xf8af, 0xf89e,
    0xf88e, 0xf87e, 0xf86f, 0xf861, 0xf853, 0xf845, 0xf839, 0xf82d, 0xf822, 0xf817, 0xf80d, 0xf804,
    0xf7fb, 0xf7f3, 0xf7ec, 0xf7e6, 0xf7e0, 0xf7db, 0xf7d7, 0xf7d3, 0xf7d0, 0xf7ce, 0xf7cd, 0xf7cc,
    0xf7cc, 0xf7cd, 0xf7cf, 0xf7d2, 0xf7d5, 0xf7d9, 0xf7de, 0xf7e3, 0xf7e9, 0xf7f1, 0xf7f8, 0xf801,
    0xf80b, 0xf815, 0xf820, 0xf82c, 0xf838, 0xf846, 0xf854, 0xf863, 0xf873, 0xf883, 0xf894, 0xf8a6,
    0xf8b9, 0xf8cd, 0xf8e1, 0xf8f6, 0xf90c, 0xf923, 0xf93a, 0xf952, 0xf96b, 0xf985, 0xf99f, 0xf9ba,
    0xf9d6, 0xf9f3, 0xfa10, 0xfa2e, 0xfa4d, 0xfa6c, 0xfa8c, 0xfaad, 0xface, 0xfaf0, 0xfb12, 0xfb36,
    0xfb5a, 0xfb7e, 0xfba3, 0xfbc9, 0xfbef, 0xfc16, 0xfc3e, 0xfc66, 0xfc8e, 0xfcb7, 0xfce1, 0xfd0b,
    0xfd36, 0xfd61, 0xfd8c, 0xfdb8, 0xfde5, 0xfe12, 0xfe3f, 0xfe6d, 0xfe9b, 0xfeca, 0xfef8, 0xff28,
    0xff57, 0xff87, 0xffb7, 0xffe8, 0x0018, 0x0049, 0x007b, 0x00ac, 0x00de, 0x010f, 0x0141, 0x0174,
    0x01a6, 0x01d8, 0x020b, 0x023d, 0x0270, 0x02a3, 0x02d6, 0x0308, 0x033b, 0x036e, 0x03a1, 0x03d3,
    0x0406, 0x0438, 0x046b, 0x049d, 0x04cf, 0x0501, 0x0533, 0x0564, 0x0596, 0x05c7, 0x05f8, 0x0628,
    0x0658, 0x0688, 0x06b8, 0x06e7, 0x0716, 0x0745, 0x0773, 0x07a0, 0x07cd, 0x07fa, 0x0826, 0x0852,
    0x087d, 0x08a8, 0x08d2, 0x08fc, 0x0925, 0x094d, 0x0975, 0x099c, 0x09c2, 0x09e8, 0x0a0d, 0x0a31,
    0x0a55, 0x0a77, 0x0a99, 0x0abb, 0x0adb, 0x0afb, 0x0b1a, 0x0b38, 0x0b55, 0x0b71, 0x0b8c, 0x0ba7,
    0x0bc0, 0x0bd9, 0x0bf0, 0x0c07, 0x0c1d, 0x0c31, 0x0c45, 0x0c58, 0x0c69, 0x0c7a, 0x0c89, 0x0c98,
    0x0ca5, 0x0cb2, 0x0cbd, 0x0cc7, 0x0cd0, 0x0cd8, 0x0cde, 0x0ce4, 0x0ce8, 0x0cec, 0x0cee, 0x0cef,
    0x0cee, 0x0ced, 0x0cea, 0x0ce6, 0x0ce1, 0x0cdb, 0x0cd3, 0x0cca, 0x0cc0, 0x0cb5, 0x0ca8, 0x0c9a,
    0x0c8b, 0x0c7b, 0x0c6a, 0x0c57, 0x0c43, 0x0c2d, 0x0c17, 0x0bff, 0x0be6, 0x0bcc, 0x0bb0, 0x0b93,
    0x0b75, 0x0b56, 0x0b35, 0x0b13, 0x0af0, 0x0acc, 0x0aa6, 0x0a80, 0x0a58, 0x0a2f, 0x0a04, 0x09d9,
    0x09ac, 0x097e, 0x094f, 0x091f, 0x08ed, 0x08bb, 0x0887, 0x0852, 0x081c, 0x07e5, 0x07ad, 0x0774,
    0x0739, 0x06fe, 0x06c2, 0x0684, 0x0646, 0x0607, 0x05c6, 0x0585, 0x0543, 0x0500, 0x04bb, 0x0476,
    0x0431, 0x03ea, 0x03a2, 0x035a, 0x0311, 0x02c7, 0x027c, 0x0231, 0x01e4, 0x0197, 0x014a, 0x00fc,
    0x00ad, 0x005d, 0x000d, 0xffbd, 0xff6c, 0xff1a, 0xfec8, 0xfe75, 0xfe22, 0xfdcf, 0xfd7b, 0xfd27,
    0xfcd2, 0xfc7e, 0xfc29, 0xfbd3, 0xfb7e, 0xfb28, 0xfad2, 0xfa7c, 0xfa26, 0xf9cf, 0xf979, 0xf923,
    0xf8cd, 0xf876, 0xf820, 0xf7ca, 0xf774, 0xf71e, 0xf6c8, 0xf673, 0xf61e, 0xf5c9, 0xf574, 0xf520,
    0xf4cc, 0xf479, 0xf425, 0xf3d3, 0xf381, 0xf32f, 0xf2de, 0xf28e, 0xf23e, 0xf1ef, 0xf1a1, 0xf153,
    0xf106, 0xf0ba, 0xf06e, 0xf024, 0xefda, 0xef92, 0xef4a, 0xef03, 0xeebd, 0xee78, 0xee35, 0xedf2,
    0xedb1, 0xed71, 0xed32, 0xecf4, 0xecb7, 0xec7c, 0xec42, 0xec09, 0xebd2, 0xeb9c, 0xeb68, 0xeb35,
    0xeb04, 0xead4, 0xeaa5, 0xea79, 0xea4d, 0xea24, 0xe9fc, 0xe9d6, 0xe9b2, 0xe98f, 0xe96e, 0xe94f,
    0xe932, 0xe916, 0xe8fd, 0xe8e5, 0xe8d0, 0xe8bc, 0xe8aa, 0xe89a, 0xe88d, 0xe881, 0xe877, 0xe870,
    0xe86a, 0xe867, 0xe866, 0xe867, 0xe86a, 0xe870, 0xe877, 0xe881, 0xe88d, 0xe89c, 0xe8ad, 0xe8c0,
    0xe8d5, 0xe8ed, 0xe907, 0xe923, 0xe942, 0xe964, 0xe987, 0xe9ad, 0xe9d6, 0xea01, 0xea2e, 0xea5e,
    0xea91, 0xeac5, 0xeafd, 0xeb37, 0xeb73, 0xebb2, 0xebf3, 0xec37, 0xec7d, 0xecc6, 0xed12, 0xed60,
    0xedb0, 0xee03, 0xee59, 0xeeb1, 0xef0c, 0xef69, 0xefc9, 0xf02b, 0xf090, 0xf0f7, 0xf161, 0xf1cd,
    0xf23c, 0xf2ad, 0xf321, 0xf397, 0xf410, 0xf48b, 0xf509, 0xf589, 0xf60c, 0xf691, 0xf718, 0xf7a2,
    0xf82e, 0xf8bc, 0xf94d, 0xf9e0, 0xfa76, 0xfb0e, 0xfba8, 0xfc44, 0xfce3, 0xfd83, 0xfe26, 0xfecc,
    0xff73, 0x001c, 0x00c8, 0x0176, 0x0225, 0x02d7, 0x038b, 0x0441, 0x04f9, 0x05b2, 0x066e, 0x072b,
    0x07eb, 0x08ac, 0x096f, 0x0a34, 0x0afa, 0x0bc2, 0x0c8c, 0x0d58, 0x0e25, 0x0ef3, 0x0fc4, 0x1095,
    0x1169, 0x123d, 0x1313, 0x13eb, 0x14c3, 0x159d, 0x1678, 0x1755, 0x1833, 0x1911, 0x19f1, 0x1ad2,
    0x1bb4, 0x1c97, 0x1d7b, 0x1e5f, 0x1f45, 0x202b, 0x2113, 0x21fa, 0x22e3, 0x23cc, 0x24b6, 0x25a0,
    0x268b, 0x2777, 0x2862, 0x294f, 0x2a3b, 0x2b28, 0x2c15, 0x2d02, 0x2def, 0x2edd, 0x2fca, 0x30b8,
    0x31a5, 0x3292, 0x3380, 0x346d, 0x355a, 0x3646, 0x3733, 0x381e, 0x390a, 0x39f5, 0x3ae0, 0x3bca,
    0x3cb3, 0x3d9c, 0x3e84, 0x3f6b, 0x4052, 0x4138, 0x421c, 0x4300, 0x43e3, 0x44c5, 0x45a6, 0x4686,
    0x4764, 0x4842, 0x491e, 0x49f9, 0x4ad3, 0x4bab, 0x4c81, 0x4d57, 0x4e2a, 0x4efd, 0x4fcd, 0x509c,
    0x5169, 0x5235, 0x52ff, 0x53c7, 0x548d, 0x5551, 0x5613, 0x56d3, 0x5792, 0x584e, 0x5908, 0x59c0,
    0x5a75, 0x5b29, 0x5bda, 0x5c89, 0x5d36, 0x5de0, 0x5e88, 0x5f2d, 0x5fd0, 0x6070, 0x610e, 0x61a9,
    0x6242, 0x62d8, 0x636b, 0x63fc, 0x648a, 0x6515, 0x659d, 0x6623, 0x66a5, 0x6725, 0x67a2, 0x681b,
    0x6892, 0x6906, 0x6977, 0x69e5, 0x6a4f, 0x6ab7, 0x6b1c, 0x6b7d, 0x6bdb, 0x6c36, 0x6c8e, 0x6ce2,
    0x6d34, 0x6d82, 0x6dcd, 0x6e14, 0x6e59, 0x6e99, 0x6ed7, 0x6f11, 0x6f48, 0x6f7c, 0x6fac, 0x6fd9,
    0x7002, 0x7028, 0x704a, 0x7069, 0x7085, 0x709d, 0x70b2, 0x70c3, 0x70d1, 0x70db, 0x70e2, 0x70e6,
};


// ----- 320 : 441 ---------------
// makeFIR(24, 320, 441, .9, 8);
static const int16_t fir_320_441[ 320 * FIR_TAP_24 / 2] = {
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfff9, 0xfff9,
    0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9,
    0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9,
    0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8,
    0xfff8, 0xfff8, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9,
    0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9,
    0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffb, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffd, 0xfffd, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001, 0x0001,
    0x0001, 0x0001, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0002, 0x0003, 0x0003,
    0x0003, 0x0003, 0x0003, 0x0003, 0x0003, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004, 0x0004,
    0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0005, 0x0006, 0x0006, 0x0006, 0x0006, 0x0006,
    0x0006, 0x0007, 0x0007, 0x0007, 0x0007, 0x0007, 0x0007, 0x0007, 0x0008, 0x0008, 0x0008, 0x0008,
    0x0008, 0x0008, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x0009, 0x000a, 0x000a, 0x000a, 0x000a,
    0x000a, 0x000a, 0x000b, 0x000b, 0x000b, 0x000b, 0x000b, 0x000c, 0x000c, 0x000c, 0x000c, 0x000c,
    0x000c, 0x000d, 0x000d, 0x000d, 0x000d, 0x000d, 0x000d, 0x000e, 0x000e, 0x000e, 0x000e, 0x000e,
    0x000f, 0x000f, 0x000f, 0x000f, 0x000f, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0010, 0x0011,
    0x0011, 0x0011, 0x0011, 0x0011, 0x0012, 0x0012, 0x0012, 0x0012, 0x0012, 0x0013, 0x0013, 0x0013,
    0x0013, 0x0013, 0x0014, 0x0014, 0x0014, 0x0014, 0x0014, 0x0015, 0x0015, 0x0015, 0x0015, 0x0015,
    0x0015, 0x0016, 0x0016, 0x0016, 0x0016, 0x0016, 0x0017, 0x0017, 0x0017, 0x0017, 0x0017, 0x0018,
    0x0018, 0x0018, 0x0018, 0x0018, 0x0019, 0x0019, 0x0019, 0x0019, 0x0019, 0x001a, 0x001a, 0x001a,
    0x001a, 0x001a, 0x001b, 0x001b, 0x001b, 0x001b, 0x001b, 0x001c, 0x001c, 0x001c, 0x001c, 0x001c,
    0x001d, 0x001d, 0x001d, 0x001d, 0x001d, 0x001d, 0x001e, 0x001e, 0x001e, 0x001e, 0x001e, 0x001f,
    0x001f, 0x001f, 0x001f, 0x001f, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0020, 0x0021, 0x0021,
    0x0021, 0x0021, 0x0021, 0x0022, 0x0022, 0x0022, 0x0022, 0x0022, 0x0022, 0x0023, 0x0023, 0x0023,
    0x0023, 0x0023, 0x0024, 0x0024, 0x0024, 0x0024, 0x0024, 0x0024, 0x0025, 0x0025, 0x0025, 0x0025,
    0x0025, 0x0025, 0x0026, 0x0026, 0x0026, 0x0026, 0x0026, 0x0026, 0x0026, 0x0027, 0x0027, 0x0027,
    0x0027, 0x0027, 0x0027, 0x0028, 0x0028, 0x0028, 0x0028, 0x0028, 0x0028, 0x0028, 0x0029, 0x0029,
    0x0029, 0x0029, 0x0029, 0x0029, 0x0029, 0x0029, 0x002a, 0x002a, 0x002a, 0x002a, 0x002a, 0x002a,
    0x002a, 0x002a, 0x002b, 0x002b, 0x002b, 0x002b, 0x002b, 0x002b, 0x002b, 0x002b, 0x002b, 0x002c,
    0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002d, 0x002d,
    0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d,
    0x002d, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e,
    0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e,
    0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e,
    0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e, 0x002e,
    0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d, 0x002d,
    0x002d, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002c, 0x002b,
    0x002b, 0x002b, 0x002b, 0x002b, 0x002b, 0x002b, 0x002a, 0x002a, 0x002a, 0x002a, 0x002a, 0x002a,
    0x002a, 0x0029, 0x0029, 0x0029, 0x0029, 0x0029, 0x0029, 0x0028, 0x0028, 0x0028, 0x0028, 0x0028,
    0x0027, 0x0027, 0x0027, 0x0027, 0x0027, 0x0026, 0x0026, 0x0026, 0x0026, 0x0025, 0x0025, 0x0025,
    0x0025, 0x0025, 0x0024, 0x0024, 0x0024, 0x0024, 0x0023, 0x0023, 0x0023, 0x0023, 0x0022, 0x0022,
    0x0022, 0x0021, 0x0021, 0x0021, 0x0021, 0x0020, 0x0020, 0x0020, 0x001f, 0x001f, 0x001f, 0x001f,
    0x001e, 0x001e, 0x001e, 0x001d, 0x001d, 0x001d, 0x001c, 0x001c, 0x001c, 0x001b, 0x001b, 0x001b,
    0x001a, 0x001a, 0x001a, 0x0019, 0x0019, 0x0018, 0x0018, 0x0018, 0x0017, 0x0017, 0x0017, 0x0016,
    0x0016, 0x0015, 0x0015, 0x0015, 0x0014, 0x0014, 0x0013, 0x0013, 0x0013, 0x0012, 0x0012, 0x0011,
    0x0011, 0x0011, 0x0010, 0x0010, 0x000f, 0x000f, 0x000e, 0x000e, 0x000d, 0x000d, 0x000d, 0x000c,
    0x000c, 0x000b, 0x000b, 0x000a, 0x000a, 0x0009, 0x0009, 0x0008, 0x0008, 0x0007, 0x0007, 0x0006,
    0x0006, 0x0005, 0x0005, 0x0004, 0x0004, 0x0003, 0x0003, 0x0002, 0x0002, 0x0001, 0x0001, 0x0000,
    0x0000, 0xffff, 0xffff, 0xfffe, 0xfffe, 0xfffd, 0xfffd, 0xfffc, 0xfffb, 0xfffb, 0xfffa, 0xfffa,
    0xfff9, 0xfff9, 0xfff8, 0xfff7, 0xfff7, 0xfff6, 0xfff6, 0xfff5, 0xfff5, 0xfff4, 0xfff3, 0xfff3,
    0xfff2, 0xfff2, 0xfff1, 0xfff1, 0xfff0, 0xffef, 0xffef, 0xffee, 0xffed, 0xffed, 0xffec, 0xffec,
    0xffeb, 0xffea, 0xffea, 0xffe9, 0xffe9, 0xffe8, 0xffe7, 0xffe7, 0xffe6, 0xffe5, 0xffe5, 0xffe4,
    0xffe4, 0xffe3, 0xffe2, 0xffe2, 0xffe1, 0xffe0, 0xffe0, 0xffdf, 0xffde, 0xffde, 0xffdd, 0xffdc,
    0xffdc, 0xffdb, 0xffda, 0xffda, 0xffd9, 0xffd8, 0xffd8, 0xffd7, 0xffd6, 0xffd6, 0xffd5, 0xffd4,
    0xffd4, 0xffd3, 0xffd2, 0xffd2, 0xffd1, 0xffd0, 0xffd0, 0xffcf, 0xffce, 0xffce, 0xffcd, 0xffcc,
    0xffcc, 0xffcb, 0xffca, 0xffc9, 0xffc9, 0xffc8, 0xffc7, 0xffc7, 0xffc6, 0xffc5, 0xffc5, 0xffc4,
    0xffc3, 0xffc3, 0xffc2, 0xffc1, 0xffc0, 0xffc0, 0xffbf, 0xffbe, 0xffbe, 0xffbd, 0xffbc, 0xffbc,
    0xffbb, 0xffba, 0xffb9, 0xffb9, 0xffb8, 0xffb7, 0xffb7, 0xffb6, 0xffb5, 0xffb5, 0xffb4, 0xffb3,
    0xffb3, 0xffb2, 0xffb1, 0xffb0, 0xffb0, 0xffaf, 0xffae, 0xffae, 0xffad, 0xffac, 0xffac, 0xffab,
    0xffaa, 0xffaa, 0xffa9, 0xffa8, 0xffa7, 0xffa7, 0xffa6, 0xffa5, 0xffa5, 0xffa4, 0xffa3, 0xffa3,
    0xffa2, 0xffa1, 0xffa1, 0xffa0, 0xff9f, 0xff9f, 0xff9e, 0xff9d, 0xff9d, 0xff9c, 0xff9b, 0xff9b,
    0xff9a, 0xff99, 0xff99, 0xff98, 0xff97, 0xff97, 0xff96, 0xff96, 0xff95, 0xff94, 0xff94, 0xff93,
    0xff92, 0xff92, 0xff91, 0xff90, 0xff90, 0xff8f, 0xff8f, 0xff8e, 0xff8d, 0xff8d, 0xff8c, 0xff8c,
    0xff8b, 0xff8a, 0xff8a, 0xff89, 0xff89, 0xff88, 0xff87, 0xff87, 0xff86, 0xff86, 0xff85, 0xff85,
    0xff84, 0xff83, 0xff83, 0xff82, 0xff82, 0xff81, 0xff81, 0xff80, 0xff80, 0xff7f, 0xff7f, 0xff7e,
    0xff7d, 0xff7d, 0xff7c, 0xff7c, 0xff7b, 0xff7b, 0xff7a, 0xff7a, 0xff79, 0xff79, 0xff78, 0xff78,
    0xff78, 0xff77, 0xff77, 0xff76, 0xff76, 0xff75, 0xff75, 0xff74, 0xff74, 0xff73, 0xff73, 0xff73,
    0xff72, 0xff72, 0xff71, 0xff71, 0xff71, 0xff70, 0xff70, 0xff6f, 0xff6f, 0xff6f, 0xff6e, 0xff6e,
    0xff6e, 0xff6d, 0xff6d, 0xff6d, 0xff6c, 0xff6c, 0xff6c, 0xff6b, 0xff6b, 0xff6b, 0xff6a, 0xff6a,
    0xff6a, 0xff69, 0xff69, 0xff69, 0xff69, 0xff68, 0xff68, 0xff68, 0xff68, 0xff67, 0xff67, 0xff67,
    0xff67, 0xff66, 0xff66, 0xff66, 0xff66, 0xff66, 0xff66, 0xff65, 0xff65, 0xff65, 0xff65, 0xff65,
    0xff65, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff63,
    0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff63,
    0xff63, 0xff63, 0xff63, 0xff63, 0xff63, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64, 0xff64,
    0xff64, 0xff64, 0xff64, 0xff65, 0xff65, 0xff65, 0xff65, 0xff65, 0xff65, 0xff66, 0xff66, 0xff66,
    0xff66, 0xff67, 0xff67, 0xff67, 0xff67, 0xff67, 0xff68, 0xff68, 0xff68, 0xff69, 0xff69, 0xff69,
    0xff6a, 0xff6a, 0xff6a, 0xff6b, 0xff6b, 0xff6b, 0xff6c, 0xff6c, 0xff6c, 0xff6d, 0xff6d, 0xff6e,
    0xff6e, 0xff6e, 0xff6f, 0xff6f, 0xff70, 0xff70, 0xff71, 0xff71, 0xff72, 0xff72, 0xff73, 0xff73,
    0xff74, 0xff74, 0xff75, 0xff75, 0xff76, 0xff76, 0xff77, 0xff78, 0xff78, 0xff79, 0xff79, 0xff7a,
    0xff7b, 0xff7b, 0xff7c, 0xff7d, 0xff7d, 0xff7e, 0xff7f, 0xff7f, 0xff80, 0xff81, 0xff81, 0xff82,
    0xff83, 0xff84, 0xff84, 0xff85, 0xff86, 0xff87, 0xff87, 0xff88, 0xff89, 0xff8a, 0xff8b, 0xff8b,
    0xff8c, 0xff8d, 0xff8e, 0xff8f, 0xff90, 0xff91, 0xff92, 0xff92, 0xff93, 0xff94, 0xff95, 0xff96,
    0xff97, 0xff98, 0xff99, 0xff9a, 0xff9b, 0xff9c, 0xff9d, 0xff9e, 0xff9f, 0xffa0, 0xffa1, 0xffa2,
    0xffa3, 0xffa4, 0xffa5, 0xffa6, 0xffa7, 0xffa9, 0xffaa, 0xffab, 0xffac, 0xffad, 0xffae, 0xffaf,
    0xffb0, 0xffb2, 0xffb3, 0xffb4, 0xffb5, 0xffb6, 0xffb8, 0xffb9, 0xffba, 0xffbb, 0xffbc, 0xffbe,
    0xffbf, 0xffc0, 0xffc1, 0xffc3, 0xffc4, 0xffc5, 0xffc7, 0xffc8, 0xffc9, 0xffcb, 0xffcc, 0xffcd,
    0xffcf, 0xffd0, 0xffd1, 0xffd3, 0xffd4, 0xffd5, 0xffd7, 0xffd8, 0xffda, 0xffdb, 0xffdd, 0xffde,
    0xffdf, 0xffe1, 0xffe2, 0xffe4, 0xffe5, 0xffe7, 0xffe8, 0xffea, 0xffeb, 0xffed, 0xffee, 0xfff0,
    0xfff1, 0xfff3, 0xfff4, 0xfff6, 0xfff7, 0xfff9, 0xfffa, 0xfffc, 0xfffe, 0xffff, 0x0001, 0x0002,
    0x0004, 0x0006, 0x0007, 0x0009, 0x000a, 0x000c, 0x000e, 0x000f, 0x0011, 0x0013, 0x0014, 0x0016,
    0x0018, 0x0019, 0x001b, 0x001d, 0x001e, 0x0020, 0x0022, 0x0024, 0x0025, 0x0027, 0x0029, 0x002a,
    0x002c, 0x002e, 0x0030, 0x0031, 0x0033, 0x0035, 0x0037, 0x0038, 0x003a, 0x003c, 0x003e, 0x0040,
    0x0041, 0x0043, 0x0045, 0x0047, 0x0049, 0x004a, 0x004c, 0x004e, 0x0050, 0x0052, 0x0053, 0x0055,
    0x0057, 0x0059, 0x005b, 0x005d, 0x005e, 0x0060, 0x0062, 0x0064, 0x0066, 0x0068, 0x006a, 0x006b,
    0x006d, 0x006f, 0x0071, 0x0073, 0x0075, 0x0077, 0x0079, 0x007a, 0x007c, 0x007e, 0x0080, 0x0082,
    0x0084, 0x0086, 0x0088, 0x0089, 0x008b, 0x008d, 0x008f, 0x0091, 0x0093, 0x0095, 0x0097, 0x0099,
    0x009b, 0x009c, 0x009e, 0x00a0, 0x00a2, 0x00a4, 0x00a6, 0x00a8, 0x00aa, 0x00ac, 0x00ad, 0x00af,
    0x00b1, 0x00b3, 0x00b5, 0x00b7, 0x00b9, 0x00bb, 0x00bd, 0x00be, 0x00c0, 0x00c2, 0x00c4, 0x00c6,
    0x00c8, 0x00ca, 0x00cc, 0x00cd, 0x00cf, 0x00d1, 0x00d3, 0x00d5, 0x00d7, 0x00d9, 0x00da, 0x00dc,
    0x00de, 0x00e0, 0x00e2, 0x00e4, 0x00e6, 0x00e7, 0x00e9, 0x00eb, 0x00ed, 0x00ef, 0x00f0, 0x00f2,
    0x00f4, 0x00f6, 0x00f8, 0x00f9, 0x00fb, 0x00fd, 0x00ff, 0x0101, 0x0102, 0x0104, 0x0106, 0x0108,
    0x0109, 0x010b, 0x010d, 0x010e, 0x0110, 0x0112, 0x0114, 0x0115, 0x0117, 0x0119, 0x011a, 0x011c,
    0x011e, 0x011f, 0x0121, 0x0123, 0x0124, 0x0126, 0x0128, 0x0129, 0x012b, 0x012c, 0x012e, 0x0130,
    0x0131, 0x0133, 0x0134, 0x0136, 0x0137, 0x0139, 0x013b, 0x013c, 0x013e, 0x013f, 0x0141, 0x0142,
    0x0144, 0x0145, 0x0146, 0x0148, 0x0149, 0x014b, 0x014c, 0x014e, 0x014f, 0x0150, 0x0152, 0x0153,
    0x0155, 0x0156, 0x0157, 0x0159, 0x015a, 0x015b, 0x015d, 0x015e, 0x015f, 0x0160, 0x0162, 0x0163,
    0x0164, 0x0165, 0x0166, 0x0168, 0x0169, 0x016a, 0x016b, 0x016c, 0x016d, 0x016f, 0x0170, 0x0171,
    0x0172, 0x0173, 0x0174, 0x0175, 0x0176, 0x0177, 0x0178, 0x0179, 0x017a, 0x017b, 0x017c, 0x017d,
    0x017e, 0x017f, 0x0180, 0x0181, 0x0181, 0x0182, 0x0183, 0x0184, 0x0185, 0x0186, 0x0186, 0x0187,
    0x0188, 0x0189, 0x0189, 0x018a, 0x018b, 0x018b, 0x018c, 0x018d, 0x018d, 0x018e, 0x018f, 0x018f,
    0x0190, 0x0190, 0x0191, 0x0191, 0x0192, 0x0192, 0x0193, 0x0193, 0x0194, 0x0194, 0x0195, 0x0195,
    0x0195, 0x0196, 0x0196, 0x0196, 0x0197, 0x0197, 0x0197, 0x0197, 0x0198, 0x0198, 0x0198, 0x0198,
    0x0198, 0x0199, 0x0199, 0x0199, 0x0199, 0x0199, 0x0199, 0x0199, 0x0199, 0x0199, 0x0199, 0x0199,
    0x0199, 0x0199, 0x0199, 0x0199, 0x0199, 0x0198, 0x0198, 0x0198, 0x0198, 0x0198, 0x0197, 0x0197,
    0x0197, 0x0197, 0x0196, 0x0196, 0x0196, 0x0195, 0x0195, 0x0194, 0x0194, 0x0193, 0x0193, 0x0192,
    0x0192, 0x0191, 0x0191, 0x0190, 0x0190, 0x018f, 0x018e, 0x018e, 0x018d, 0x018c, 0x018c, 0x018b,
    0x018a, 0x0189, 0x0189, 0x0188, 0x0187, 0x0186, 0x0185, 0x0184, 0x0183, 0x0182, 0x0181, 0x0181,
    0x017f, 0x017e, 0x017d, 0x017c, 0x017b, 0x017a, 0x0179, 0x0178, 0x0177, 0x0176, 0x0174, 0x0173,
    0x0172, 0x0171, 0x016f, 0x016e, 0x016d, 0x016b, 0x016a, 0x0168, 0x0167, 0x0166, 0x0164, 0x0163,
    0x0161, 0x0160, 0x015e, 0x015d, 0x015b, 0x0159, 0x0158, 0x0156, 0x0154, 0x0153, 0x0151, 0x014f,
    0x014d, 0x014c, 0x014a, 0x0148, 0x0146, 0x0144, 0x0143, 0x0141, 0x013f, 0x013d, 0x013b, 0x0139,
    0x0137, 0x0135, 0x0133, 0x0131, 0x012f, 0x012c, 0x012a, 0x0128, 0x0126, 0x0124, 0x0122, 0x011f,
    0x011d, 0x011b, 0x0119, 0x0116, 0x0114, 0x0112, 0x010f, 0x010d, 0x010a, 0x0108, 0x0105, 0x0103,
    0x0100, 0x00fe, 0x00fb, 0x00f9, 0x00f6, 0x00f4, 0x00f1, 0x00ee, 0x00ec, 0x00e9, 0x00e6, 0x00e4,
    0x00e1, 0x00de, 0x00db, 0x00d9, 0x00d6, 0x00d3, 0x00d0, 0x00cd, 0x00ca, 0x00c8, 0x00c5, 0x00c2,
    0x00bf, 0x00bc, 0x00b9, 0x00b6, 0x00b3, 0x00b0, 0x00ad, 0x00a9, 0x00a6, 0x00a3, 0x00a0, 0x009d,
    0x009a, 0x0097, 0x0093, 0x0090, 0x008d, 0x008a, 0x0086, 0x0083, 0x0080, 0x007c, 0x0079, 0x0076,
    0x0072, 0x006f, 0x006b, 0x0068, 0x0065, 0x0061, 0x005e, 0x005a, 0x0057, 0x0053, 0x0050, 0x004c,
    0x0048, 0x0045, 0x0041, 0x003e, 0x003a, 0x0036, 0x0033, 0x002f, 0x002b, 0x0028, 0x0024, 0x0020,
    0x001d, 0x0019, 0x0015, 0x0011, 0x000d, 0x000a, 0x0006, 0x0002, 0xfffe, 0xfffa, 0xfff6, 0xfff2,
    0xffef, 0xffeb, 0xffe7, 0xffe3, 0xffdf, 0xffdb, 0xffd7, 0xffd3, 0xffcf, 0xffcb, 0xffc7, 0xffc3,
    0xffbf, 0xffbb, 0xffb7, 0xffb3, 0xffaf, 0xffab, 0xffa6, 0xffa2, 0xff9e, 0xff9a, 0xff96, 0xff92,
    0xff8e, 0xff89, 0xff85, 0xff81, 0xff7d, 0xff79, 0xff75, 0xff70, 0xff6c, 0xff68, 0xff64, 0xff5f,
    0xff5b, 0xff57, 0xff53, 0xff4e, 0xff4a, 0xff46, 0xff42, 0xff3d, 0xff39, 0xff35, 0xff30, 0xff2c,
    0xff28, 0xff24, 0xff1f, 0xff1b, 0xff17, 0xff12, 0xff0e, 0xff0a, 0xff05, 0xff01, 0xfefd, 0xfef8,
    0xfef4, 0xfeef, 0xfeeb, 0xfee7, 0xfee2, 0xfede, 0xfeda, 0xfed5, 0xfed1, 0xfecd, 0xfec8, 0xfec4,
    0xfebf, 0xfebb, 0xfeb7, 0xfeb2, 0xfeae, 0xfeaa, 0xfea5, 0xfea1, 0xfe9d, 0xfe98, 0xfe94, 0xfe90,
    0xfe8b, 0xfe87, 0xfe82, 0xfe7e, 0xfe7a, 0xfe75, 0xfe71, 0xfe6d, 0xfe68, 0xfe64, 0xfe60, 0xfe5b,
    0xfe57, 0xfe53, 0xfe4f, 0xfe4a, 0xfe46, 0xfe42, 0xfe3d, 0xfe39, 0xfe35, 0xfe31, 0xfe2c, 0xfe28,
    0xfe24, 0xfe20, 0xfe1b, 0xfe17, 0xfe13, 0xfe0f, 0xfe0b, 0xfe06, 0xfe02, 0xfdfe, 0xfdfa, 0xfdf6,
    0xfdf2, 0xfdee, 0xfde9, 0xfde5, 0xfde1, 0xfddd, 0xfdd9, 0xfdd5, 0xfdd1, 0xfdcd, 0xfdc9, 0xfdc5,
    0xfdc1, 0xfdbd, 0xfdb9, 0xfdb5, 0xfdb1, 0xfdad, 0xfda9, 0xfda5, 0xfda1, 0xfd9e, 0xfd9a, 0xfd96,
    0xfd92, 0xfd8e, 0xfd8a, 0xfd87, 0xfd83, 0xfd7f, 0xfd7b, 0xfd78, 0xfd74, 0xfd70, 0xfd6d, 0xfd69,
    0xfd65, 0xfd62, 0xfd5e, 0xfd5b, 0xfd57, 0xfd53, 0xfd50, 0xfd4c, 0xfd49, 0xfd45, 0xfd42, 0xfd3f,
    0xfd3b, 0xfd38, 0xfd34, 0xfd31, 0xfd2e, 0xfd2a, 0xfd27, 0xfd24, 0xfd21, 0xfd1d, 0xfd1a, 0xfd17,
    0xfd14, 0xfd11, 0xfd0e, 0xfd0b, 0xfd08, 0xfd05, 0xfd02, 0xfcff, 0xfcfc, 0xfcf9, 0xfcf6, 0xfcf3,
    0xfcf0, 0xfced, 0xfcea, 0xfce8, 0xfce5, 0xfce2, 0xfcdf, 0xfcdd, 0xfcda, 0xfcd8, 0xfcd5, 0xfcd2,
    0xfcd0, 0xfccd, 0xfccb, 0xfcc8, 0xfcc6, 0xfcc4, 0xfcc1, 0xfcbf, 0xfcbd, 0xfcba, 0xfcb8, 0xfcb6,
    0xfcb4, 0xfcb1, 0xfcaf, 0xfcad, 0xfcab, 0xfca9, 0xfca7, 0xfca5, 0xfca3, 0xfca1, 0xfc9f, 0xfc9e,
    0xfc9c, 0xfc9a, 0xfc98, 0xfc97, 0xfc95, 0xfc93, 0xfc92, 0xfc90, 0xfc8f, 0xfc8d, 0xfc8c, 0xfc8a,
    0xfc89, 0xfc87, 0xfc86, 0xfc85, 0xfc83, 0xfc82, 0xfc81, 0xfc80, 0xfc7f, 0xfc7e, 0xfc7d, 0xfc7c,
    0xfc7b, 0xfc7a, 0xfc79, 0xfc78, 0xfc77, 0xfc76, 0xfc75, 0xfc75, 0xfc74, 0xfc73, 0xfc73, 0xfc72,
    0xfc72, 0xfc71, 0xfc71, 0xfc70, 0xfc70, 0xfc70, 0xfc6f, 0xfc6f, 0xfc6f, 0xfc6f, 0xfc6e, 0xfc6e,
    0xfc6e, 0xfc6e, 0xfc6e, 0xfc6e, 0xfc6e, 0xfc6f, 0xfc6f, 0xfc6f, 0xfc6f, 0xfc6f, 0xfc70, 0xfc70,
    0xfc71, 0xfc71, 0xfc72, 0xfc72, 0xfc73, 0xfc73, 0xfc74, 0xfc75, 0xfc75, 0xfc76, 0xfc77, 0xfc78,
    0xfc79, 0xfc7a, 0xfc7b, 0xfc7c, 0xfc7d, 0xfc7e, 0xfc7f, 0xfc81, 0xfc82, 0xfc83, 0xfc84, 0xfc86,
    0xfc87, 0xfc89, 0xfc8a, 0xfc8c, 0xfc8d, 0xfc8f, 0xfc91, 0xfc93, 0xfc94, 0xfc96, 0xfc98, 0xfc9a,
    0xfc9c, 0xfc9e, 0xfca0, 0xfca2, 0xfca4, 0xfca6, 0xfca9, 0xfcab, 0xfcad, 0xfcaf, 0xfcb2, 0xfcb4,
    0xfcb7, 0xfcb9, 0xfcbc, 0xfcbf, 0xfcc1, 0xfcc4, 0xfcc7, 0xfcc9, 0xfccc, 0xfccf, 0xfcd2, 0xfcd5,
    0xfcd8, 0xfcdb, 0xfcde, 0xfce2, 0xfce5, 0xfce8, 0xfceb, 0xfcef, 0xfcf2, 0xfcf5, 0xfcf9, 0xfcfc,
    0xfd00, 0xfd04, 0xfd07, 0xfd0b, 0xfd0f, 0xfd12, 0xfd16, 0xfd1a, 0xfd1e, 0xfd22, 0xfd26, 0xfd2a,
    0xfd2e, 0xfd32, 0xfd36, 0xfd3b, 0xfd3f, 0xfd43, 0xfd47, 0xfd4c, 0xfd50, 0xfd55, 0xfd59, 0xfd5e,
    0xfd62, 0xfd67, 0xfd6c, 0xfd71, 0xfd75, 0xfd7a, 0xfd7f, 0xfd84, 0xfd89, 0xfd8e, 0xfd93, 0xfd98,
    0xfd9d, 0xfda2, 0xfda7, 0xfdad, 0xfdb2, 0xfdb7, 0xfdbd, 0xfdc2, 0xfdc8, 0xfdcd, 0xfdd3, 0xfdd8,
    0xfdde, 0xfde3, 0xfde9, 0xfdef, 0xfdf5, 0xfdfb, 0xfe00, 0xfe06, 0xfe0c, 0xfe12, 0xfe18, 0xfe1e,
    0xfe24, 0xfe2b, 0xfe31, 0xfe37, 0xfe3d, 0xfe43, 0xfe4a, 0xfe50, 0xfe57, 0xfe5d, 0xfe63, 0xfe6a,
    0xfe71, 0xfe77, 0xfe7e, 0xfe84, 0xfe8b, 0xfe92, 0xfe99, 0xfe9f, 0xfea6, 0xfead, 0xfeb4, 0xfebb,
    0xfec2, 0xfec9, 0xfed0, 0xfed7, 0xfede, 0xfee6, 0xfeed, 0xfef4, 0xfefb, 0xff03, 0xff0a, 0xff11,
    0xff19, 0xff20, 0xff27, 0xff2f, 0xff36, 0xff3e, 0xff46, 0xff4d, 0xff55, 0xff5d, 0xff64, 0xff6c,
    0xff74, 0xff7b, 0xff83, 0xff8b, 0xff93, 0xff9b, 0xffa3, 0xffab, 0xffb3, 0xffbb, 0xffc3, 0xffcb,
    0xffd3, 0xffdb, 0xffe3, 0xffeb, 0xfff4, 0xfffc, 0x0004, 0x000c, 0x0015, 0x001d, 0x0025, 0x002e,
    0x0036, 0x003f, 0x0047, 0x004f, 0x0058, 0x0060, 0x0069, 0x0071, 0x007a, 0x0083, 0x008b, 0x0094,
    0x009c, 0x00a5, 0x00ae, 0x00b6, 0x00bf, 0x00c8, 0x00d1, 0x00d9, 0x00e2, 0x00eb, 0x00f4, 0x00fd,
    0x0105, 0x010e, 0x0117, 0x0120, 0x0129, 0x0132, 0x013b, 0x0144, 0x014d, 0x0156, 0x015f, 0x0168,
    0x0171, 0x017a, 0x0183, 0x018c, 0x0195, 0x019e, 0x01a7, 0x01b0, 0x01b9, 0x01c2, 0x01cb, 0x01d4,
    0x01dd, 0x01e6, 0x01ef, 0x01f9, 0x0202, 0x020b, 0x0214, 0x021d, 0x0226, 0x022f, 0x0239, 0x0242,
    0x024b, 0x0254, 0x025d, 0x0266, 0x026f, 0x0279, 0x0282, 0x028b, 0x0294, 0x029d, 0x02a6, 0x02af,
    0x02b9, 0x02c2, 0x02cb, 0x02d4, 0x02dd, 0x02e6, 0x02ef, 0x02f8, 0x0302, 0x030b, 0x0314, 0x031d,
    0x0326, 0x032f, 0x0338, 0x0341, 0x034a, 0x0353, 0x035c, 0x0365, 0x036e, 0x0377, 0x0380, 0x0389,
    0x0392, 0x039b, 0x03a4, 0x03ad, 0x03b6, 0x03be, 0x03c7, 0x03d0, 0x03d9, 0x03e2, 0x03eb, 0x03f3,
    0x03fc, 0x0405, 0x040e, 0x0416, 0x041f, 0x0428, 0x0430, 0x0439, 0x0441, 0x044a, 0x0453, 0x045b,
    0x0464, 0x046c, 0x0475, 0x047d, 0x0485, 0x048e, 0x0496, 0x049e, 0x04a7, 0x04af, 0x04b7, 0x04c0,
    0x04c8, 0x04d0, 0x04d8, 0x04e0, 0x04e8, 0x04f0, 0x04f8, 0x0500, 0x0508, 0x0510, 0x0518, 0x0520,
    0x0528, 0x052f, 0x0537, 0x053f, 0x0547, 0x054e, 0x0556, 0x055d, 0x0565, 0x056c, 0x0574, 0x057b,
    0x0583, 0x058a, 0x0591, 0x0599, 0x05a0, 0x05a7, 0x05ae, 0x05b5, 0x05bc, 0x05c3, 0x05ca, 0x05d1,
    0x05d8, 0x05df, 0x05e6, 0x05ec, 0x05f3, 0x05fa, 0x0600, 0x0607, 0x060d, 0x0614, 0x061a, 0x0620,
    0x0627, 0x062d, 0x0633, 0x0639, 0x0640, 0x0646, 0x064c, 0x0652, 0x0657, 0x065d, 0x0663, 0x0669,
    0x066f, 0x0674, 0x067a, 0x067f, 0x0685, 0x068a, 0x0690, 0x0695, 0x069a, 0x069f, 0x06a4, 0x06a9,
    0x06ae, 0x06b3, 0x06b8, 0x06bd, 0x06c2, 0x06c7, 0x06cb, 0x06d0, 0x06d4, 0x06d9, 0x06dd, 0x06e1,
    0x06e6, 0x06ea, 0x06ee, 0x06f2, 0x06f6, 0x06fa, 0x06fe, 0x0702, 0x0706, 0x0709, 0x070d, 0x0710,
    0x0714, 0x0717, 0x071b, 0x071e, 0x0721, 0x0724, 0x0727, 0x072a, 0x072d, 0x0730, 0x0733, 0x0736,
    0x0738, 0x073b, 0x073d, 0x0740, 0x0742, 0x0744, 0x0747, 0x0749, 0x074b, 0x074d, 0x074f, 0x0750,
    0x0752, 0x0754, 0x0755, 0x0757, 0x0758, 0x075a, 0x075b, 0x075c, 0x075d, 0x075e, 0x075f, 0x0760,
    0x0761, 0x0762, 0x0763, 0x0763, 0x0764, 0x0764, 0x0764, 0x0765, 0x0765, 0x0765, 0x0765, 0x0765,
    0x0765, 0x0765, 0x0764, 0x0764, 0x0763, 0x0763, 0x0762, 0x0761, 0x0761, 0x0760, 0x075f, 0x075e,
    0x075d, 0x075b, 0x075a, 0x0759, 0x0757, 0x0756, 0x0754, 0x0752, 0x0750, 0x074e, 0x074c, 0x074a,
    0x0748, 0x0746, 0x0744, 0x0741, 0x073f, 0x073c, 0x0739, 0x0736, 0x0734, 0x0731, 0x072e, 0x072a,
    0x0727, 0x0724, 0x0720, 0x071d, 0x0719, 0x0716, 0x0712, 0x070e, 0x070a, 0x0706, 0x0702, 0x06fe,
    0x06fa, 0x06f5, 0x06f1, 0x06ec, 0x06e7, 0x06e3, 0x06de, 0x06d9, 0x06d4, 0x06cf, 0x06ca, 0x06c4,
    0x06bf, 0x06b9, 0x06b4, 0x06ae, 0x06a8, 0x06a3, 0x069d, 0x0697, 0x0691, 0x068a, 0x0684, 0x067e,
    0x0677, 0x0671, 0x066a, 0x0663, 0x065c, 0x0656, 0x064e, 0x0647, 0x0640, 0x0639, 0x0632, 0x062a,
    0x0623, 0x061b, 0x0613, 0x060b, 0x0603, 0x05fb, 0x05f3, 0x05eb, 0x05e3, 0x05da, 0x05d2, 0x05c9,
    0x05c1, 0x05b8, 0x05af, 0x05a6, 0x059d, 0x0594, 0x058b, 0x0582, 0x0579, 0x056f, 0x0566, 0x055c,
    0x0552, 0x0548, 0x053f, 0x0535, 0x052b, 0x0520, 0x0516, 0x050c, 0x0501, 0x04f7, 0x04ec, 0x04e2,
    0x04d7, 0x04cc, 0x04c1, 0x04b6, 0x04ab, 0x04a0, 0x0495, 0x0489, 0x047e, 0x0473, 0x0467, 0x045b,
    0x0450, 0x0444, 0x0438, 0x042c, 0x0420, 0x0413, 0x0407, 0x03fb, 0x03ee, 0x03e2, 0x03d5, 0x03c9,
    0x03bc, 0x03af, 0x03a2, 0x0395, 0x0388, 0x037b, 0x036e, 0x0361, 0x0353, 0x0346, 0x0338, 0x032b,
    0x031d, 0x030f, 0x0302, 0x02f4, 0x02e6, 0x02d8, 0x02ca, 0x02bb, 0x02ad, 0x029f, 0x0290, 0x0282,
    0x0273, 0x0265, 0x0256, 0x0247, 0x0238, 0x0229, 0x021a, 0x020b, 0x01fc, 0x01ed, 0x01de, 0x01cf,
    0x01bf, 0x01b0, 0x01a0, 0x0191, 0x0181, 0x0171, 0x0162, 0x0152, 0x0142, 0x0132, 0x0122, 0x0112,
    0x0102, 0x00f1, 0x00e1, 0x00d1, 0x00c0, 0x00b0, 0x00a0, 0x008f, 0x007e, 0x006e, 0x005d, 0x004c,
    0x003b, 0x002a, 0x001a, 0x0009, 0xfff7, 0xffe6, 0xffd5, 0xffc4, 0xffb3, 0xffa1, 0xff90, 0xff7f,
    0xff6d, 0xff5c, 0xff4a, 0xff39, 0xff27, 0xff15, 0xff04, 0xfef2, 0xfee0, 0xfece, 0xfebc, 0xfeaa,
    0xfe98, 0xfe86, 0xfe74, 0xfe62, 0xfe50, 0xfe3e, 0xfe2c, 0xfe19, 0xfe07, 0xfdf5, 0xfde2, 0xfdd0,
    0xfdbe, 0xfdab, 0xfd99, 0xfd86, 0xfd74, 0xfd61, 0xfd4f, 0xfd3c, 0xfd29, 0xfd17, 0xfd04, 0xfcf1,
    0xfcde, 0xfccc, 0xfcb9, 0xfca6, 0xfc93, 0xfc80, 0xfc6d, 0xfc5a, 0xfc47, 0xfc35, 0xfc22, 0xfc0f,
    0xfbfc, 0xfbe9, 0xfbd6, 0xfbc2, 0xfbaf, 0xfb9c, 0xfb89, 0xfb76, 0xfb63, 0xfb50, 0xfb3d, 0xfb2a,
    0xfb17, 0xfb03, 0xfaf0, 0xfadd, 0xfaca, 0xfab7, 0xfaa4, 0xfa91, 0xfa7d, 0xfa6a, 0xfa57, 0xfa44,
    0xfa31, 0xfa1e, 0xfa0a, 0xf9f7, 0xf9e4, 0xf9d1, 0xf9be, 0xf9ab, 0xf998, 0xf985, 0xf971, 0xf95e,
    0xf94b, 0xf938, 0xf925, 0xf912, 0xf8ff, 0xf8ec, 0xf8d9, 0xf8c6, 0xf8b3, 0xf8a0, 0xf88e, 0xf87b,
    0xf868, 0xf855, 0xf842, 0xf82f, 0xf81d, 0xf80a, 0xf7f7, 0xf7e5, 0xf7d2, 0xf7bf, 0xf7ad, 0xf79a,
    0xf788, 0xf775, 0xf763, 0xf750, 0xf73e, 0xf72c, 0xf719, 0xf707, 0xf6f5, 0xf6e3, 0xf6d0, 0xf6be,
    0xf6ac, 0xf69a, 0xf688, 0xf676, 0xf665, 0xf653, 0xf641, 0xf62f, 0xf61e, 0xf60c, 0xf5fa, 0xf5e9,
    0xf5d7, 0xf5c6, 0xf5b5, 0xf5a3, 0xf592, 0xf581, 0xf570, 0xf55f, 0xf54e, 0xf53d, 0xf52c, 0xf51b,
    0xf50a, 0xf4fa, 0xf4e9, 0xf4d8, 0xf4c8, 0xf4b7, 0xf4a7, 0xf497, 0xf487, 0xf476, 0xf466, 0xf456,
    0xf447, 0xf437, 0xf427, 0xf417, 0xf408, 0xf3f8, 0xf3e9, 0xf3d9, 0xf3ca, 0xf3bb, 0xf3ac, 0xf39d,
    0xf38e, 0xf37f, 0xf370, 0xf362, 0xf353, 0xf345, 0xf336, 0xf328, 0xf31a, 0xf30c, 0xf2fe, 0xf2f0,
    0xf2e2, 0xf2d4, 0xf2c6, 0xf2b9, 0xf2ab, 0xf29e, 0xf291, 0xf284, 0xf277, 0xf26a, 0xf25d, 0xf250,
    0xf244, 0xf237, 0xf22b, 0xf21f, 0xf212, 0xf206, 0xf1fa, 0xf1ef, 0xf1e3, 0xf1d7, 0xf1cc, 0xf1c0,
    0xf1b5, 0xf1aa, 0xf19f, 0xf194, 0xf189, 0xf17f, 0xf174, 0xf16a, 0xf160, 0xf156, 0xf14c, 0xf142,
    0xf138, 0xf12e, 0xf125, 0xf11b, 0xf112, 0xf109, 0xf100, 0xf0f7, 0xf0ef, 0xf0e6, 0xf0de, 0xf0d5,
    0xf0cd, 0xf0c5, 0xf0bd, 0xf0b5, 0xf0ae, 0xf0a6, 0xf09f, 0xf098, 0xf091, 0xf08a, 0xf083, 0xf07d,
    0xf076, 0xf070, 0xf06a, 0xf064, 0xf05e, 0xf058, 0xf053, 0xf04d, 0xf048, 0xf043, 0xf03e, 0xf039,
    0xf035, 0xf030, 0xf02c, 0xf028, 0xf024, 0xf020, 0xf01c, 0xf019, 0xf015, 0xf012, 0xf00f, 0xf00c,
    0xf00a, 0xf007, 0xf005, 0xf003, 0xf001, 0xefff, 0xeffd, 0xeffc, 0xeffa, 0xeff9, 0xeff8, 0xeff7,
    0xeff7, 0xeff6, 0xeff6, 0xeff6, 0xeff6, 0xeff6, 0xeff6, 0xeff7, 0xeff8, 0xeff9, 0xeffa, 0xeffb,
    0xeffd, 0xeffe, 0xf000, 0xf002, 0xf004, 0xf007, 0xf009, 0xf00c, 0xf00f, 0xf012, 0xf015, 0xf019,
    0xf01d, 0xf021, 0xf025, 0xf029, 0xf02d, 0xf032, 0xf037, 0xf03c, 0xf041, 0xf046, 0xf04c, 0xf052,
    0xf058, 0xf05e, 0xf064, 0xf06b, 0xf072, 0xf079, 0xf080, 0xf087, 0xf08f, 0xf096, 0xf09e, 0xf0a6,
    0xf0af, 0xf0b7, 0xf0c0, 0xf0c9, 0xf0d2, 0xf0db, 0xf0e5, 0xf0ef, 0xf0f9, 0xf103, 0xf10d, 0xf118,
    0xf122, 0xf12d, 0xf139, 0xf144, 0xf14f, 0xf15b, 0xf167, 0xf173, 0xf180, 0xf18c, 0xf199, 0xf1a6,
    0xf1b3, 0xf1c1, 0xf1cf, 0xf1dc, 0xf1ea, 0xf1f9, 0xf207, 0xf216, 0xf225, 0xf234, 0xf243, 0xf253,
    0xf262, 0xf272, 0xf283, 0xf293, 0xf2a3, 0xf2b4, 0xf2c5, 0xf2d7, 0xf2e8, 0xf2fa, 0xf30b, 0xf31d,
    0xf330, 0xf342, 0xf355, 0xf368, 0xf37b, 0xf38e, 0xf3a2, 0xf3b6, 0xf3ca, 0xf3de, 0xf3f2, 0xf407,
    0xf41c, 0xf431, 0xf446, 0xf45c, 0xf471, 0xf487, 0xf49d, 0xf4b4, 0xf4ca, 0xf4e1, 0xf4f8, 0xf50f,
    0xf527, 0xf53e, 0xf556, 0xf56e, 0xf586, 0xf59f, 0xf5b7, 0xf5d0, 0xf5ea, 0xf603, 0xf61c, 0xf636,
    0xf650, 0xf66a, 0xf685, 0xf69f, 0xf6ba, 0xf6d5, 0xf6f1, 0xf70c, 0xf728, 0xf744, 0xf760, 0xf77c,
    0xf799, 0xf7b5, 0xf7d2, 0xf7ef, 0xf80d, 0xf82a, 0xf848, 0xf866, 0xf884, 0xf8a3, 0xf8c2, 0xf8e0,
    0xf8ff, 0xf91f, 0xf93e, 0xf95e, 0xf97e, 0xf99e, 0xf9be, 0xf9df, 0xfa00, 0xfa20, 0xfa42, 0xfa63,
    0xfa85, 0xfaa6, 0xfac8, 0xfaea, 0xfb0d, 0xfb2f, 0xfb52, 0xfb75, 0xfb98, 0xfbbc, 0xfbdf, 0xfc03,
    0xfc27, 0xfc4b, 0xfc70, 0xfc94, 0xfcb9, 0xfcde, 0xfd03, 0xfd29, 0xfd4e, 0xfd74, 0xfd9a, 0xfdc0,
    0xfde7, 0xfe0d, 0xfe34, 0xfe5b, 0xfe82, 0xfeaa, 0xfed1, 0xfef9, 0xff21, 0xff49, 0xff71, 0xff9a,
    0xffc3, 0xffeb, 0x0015, 0x003e, 0x0067, 0x0091, 0x00bb, 0x00e5, 0x010f, 0x0139, 0x0164, 0x018f,
    0x01ba, 0x01e5, 0x0210, 0x023b, 0x0267, 0x0293, 0x02bf, 0x02eb, 0x0318, 0x0344, 0x0371, 0x039e,
    0x03cb, 0x03f8, 0x0425, 0x0453, 0x0481, 0x04af, 0x04dd, 0x050b, 0x0539, 0x0568, 0x0597, 0x05c6,
    0x05f5, 0x0624, 0x0654, 0x0683, 0x06b3, 0x06e3, 0x0713, 0x0743, 0x0773, 0x07a4, 0x07d5, 0x0806,
    0x0837, 0x0868, 0x0899, 0x08cb, 0x08fc, 0x092e, 0x0960, 0x0992, 0x09c4, 0x09f7, 0x0a29, 0x0a5c,
    0x0a8e, 0x0ac1, 0x0af4, 0x0b28, 0x0b5b, 0x0b8f, 0x0bc2, 0x0bf6, 0x0c2a, 0x0c5e, 0x0c92, 0x0cc6,
    0x0cfb, 0x0d2f, 0x0d64, 0x0d99, 0x0dce, 0x0e03, 0x0e38, 0x0e6e, 0x0ea3, 0x0ed9, 0x0f0e, 0x0f44,
    0x0f7a, 0x0fb0, 0x0fe6, 0x101d, 0x1053, 0x108a, 0x10c0, 0x10f7, 0x112e, 0x1165, 0x119c, 0x11d3,
    0x120a, 0x1242, 0x1279, 0x12b1, 0x12e9, 0x1320, 0x1358, 0x1390, 0x13c9, 0x1401, 0x1439, 0x1471,
    0x14aa, 0x14e2, 0x151b, 0x1554, 0x158d, 0x15c6, 0x15ff, 0x1638, 0x1671, 0x16aa, 0x16e4, 0x171d,
    0x1756, 0x1790, 0x17ca, 0x1803, 0x183d, 0x1877, 0x18b1, 0x18eb, 0x1925, 0x195f, 0x1999, 0x19d4,
    0x1a0e, 0x1a48, 0x1a83, 0x1abd, 0x1af8, 0x1b33, 0x1b6d, 0x1ba8, 0x1be3, 0x1c1e, 0x1c58, 0x1c93,
    0x1cce, 0x1d09, 0x1d45, 0x1d80, 0x1dbb, 0x1df6, 0x1e31, 0x1e6d, 0x1ea8, 0x1ee3, 0x1f1f, 0x1f5a,
    0x1f95, 0x1fd1, 0x200c, 0x2048, 0x2083, 0x20bf, 0x20fb, 0x2136, 0x2172, 0x21ae, 0x21e9, 0x2225,
    0x2261, 0x229c, 0x22d8, 0x2314, 0x2350, 0x238b, 0x23c7, 0x2403, 0x243f, 0x247b, 0x24b6, 0x24f2,
    0x252e, 0x256a, 0x25a5, 0x25e1, 0x261d, 0x2659, 0x2695, 0x26d0, 0x270c, 0x2748, 0x2783, 0x27bf,
    0x27fb, 0x2836, 0x2872, 0x28ae, 0x28e9, 0x2925, 0x2960, 0x299c, 0x29d7, 0x2a12, 0x2a4e, 0x2a89,
    0x2ac5, 0x2b00, 0x2b3b, 0x2b76, 0x2bb1, 0x2bed, 0x2c28, 0x2c63, 0x2c9e, 0x2cd9, 0x2d13, 0x2d4e,
    0x2d89, 0x2dc4, 0x2dfe, 0x2e39, 0x2e74, 0x2eae, 0x2ee9, 0x2f23, 0x2f5d, 0x2f97, 0x2fd2, 0x300c,
    0x3046, 0x3080, 0x30ba, 0x30f3, 0x312d, 0x3167, 0x31a0, 0x31da, 0x3213, 0x324d, 0x3286, 0x32bf,
    0x32f8, 0x3331, 0x336a, 0x33a3, 0x33dc, 0x3414, 0x344d, 0x3485, 0x34be, 0x34f6, 0x352e, 0x3566,
    0x359e, 0x35d6, 0x360d, 0x3645, 0x367c, 0x36b4, 0x36eb, 0x3722, 0x3759, 0x3790, 0x37c7, 0x37fe,
    0x3834, 0x386b, 0x38a1, 0x38d7, 0x390d, 0x3943, 0x3979, 0x39af, 0x39e5, 0x3a1a, 0x3a4f, 0x3a84,
    0x3ab9, 0x3aee, 0x3b23, 0x3b58, 0x3b8c, 0x3bc0, 0x3bf5, 0x3c29, 0x3c5d, 0x3c90, 0x3cc4, 0x3cf7,
    0x3d2b, 0x3d5e, 0x3d91, 0x3dc4, 0x3df6, 0x3e29, 0x3e5b, 0x3e8d, 0x3ebf, 0x3ef1, 0x3f23, 0x3f54,
    0x3f86, 0x3fb7, 0x3fe8, 0x4019, 0x4049, 0x407a, 0x40aa, 0x40da, 0x410a, 0x413a, 0x416a, 0x4199,
    0x41c8, 0x41f7, 0x4226, 0x4255, 0x4283, 0x42b2, 0x42e0, 0x430e, 0x433b, 0x4369, 0x4396, 0x43c3,
    0x43f0, 0x441d, 0x444a, 0x4476, 0x44a2, 0x44ce, 0x44fa, 0x4525, 0x4551, 0x457c, 0x45a7, 0x45d1,
    0x45fc, 0x4626, 0x4650, 0x467a, 0x46a4, 0x46cd, 0x46f6, 0x471f, 0x4748, 0x4771, 0x4799, 0x47c1,
    0x47e9, 0x4811, 0x4838, 0x485f, 0x4886, 0x48ad, 0x48d3, 0x48fa, 0x4920, 0x4946, 0x496b, 0x4990,
    0x49b6, 0x49da, 0x49ff, 0x4a23, 0x4a48, 0x4a6c, 0x4a8f, 0x4ab3, 0x4ad6, 0x4af9, 0x4b1b, 0x4b3e,
    0x4b60, 0x4b82, 0x4ba4, 0x4bc5, 0x4be6, 0x4c07, 0x4c28, 0x4c48, 0x4c69, 0x4c89, 0x4ca8, 0x4cc8,
    0x4ce7, 0x4d06, 0x4d24, 0x4d43, 0x4d61, 0x4d7f, 0x4d9c, 0x4dba, 0x4dd7, 0x4df4, 0x4e10, 0x4e2c,
    0x4e48, 0x4e64, 0x4e80, 0x4e9b, 0x4eb6, 0x4ed0, 0x4eeb, 0x4f05, 0x4f1f, 0x4f38, 0x4f52, 0x4f6b,
    0x4f83, 0x4f9c, 0x4fb4, 0x4fcc, 0x4fe4, 0x4ffb, 0x5012, 0x5029, 0x503f, 0x5056, 0x506c, 0x5081,
    0x5097, 0x50ac, 0x50c1, 0x50d5, 0x50e9, 0x50fd, 0x5111, 0x5124, 0x5138, 0x514a, 0x515d, 0x516f,
    0x5181, 0x5193, 0x51a4, 0x51b5, 0x51c6, 0x51d7, 0x51e7, 0x51f7, 0x5206, 0x5216, 0x5225, 0x5233,
    0x5242, 0x5250, 0x525e, 0x526b, 0x5279, 0x5286, 0x5292, 0x529f, 0x52ab, 0x52b6, 0x52c2, 0x52cd,
    0x52d8, 0x52e3, 0x52ed, 0x52f7, 0x5301, 0x530a, 0x5313, 0x531c, 0x5324, 0x532d, 0x5334, 0x533c,
    0x5343, 0x534a, 0x5351, 0x5357, 0x535d, 0x5363, 0x5369, 0x536e, 0x5373, 0x5377, 0x537b, 0x537f,
    0x5383, 0x5386, 0x5389, 0x538c, 0x538f, 0x5391, 0x5393, 0x5394, 0x5395, 0x5396, 0x5397, 0x5397,
};


// ----- 160 : 441 ---------------
// makeFIR(24, 160, 441, .9, 8);
static const int16_t fir_160_441[ 160 * FIR_TAP_24 / 2] = {
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xfffe, 0xfffe, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe,
    0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffe, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
    0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffc,
    0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc, 0xfffc,
    0xfffc, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb, 0xfffb,
    0xfffb, 0xfffb, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa, 0xfffa,
    0xfffa, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff9, 0xfff8,
    0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff8, 0xfff7, 0xfff7, 0xfff7,
    0xfff7, 0xfff7, 0xfff7, 0xfff7, 0xfff7, 0xfff7, 0xfff6, 0xfff6, 0xfff6, 0xfff6, 0xfff6, 0xfff6,
    0xfff6, 0xfff6, 0xfff6, 0xfff5, 0xfff5, 0xfff5, 0xfff5, 0xfff5, 0xfff5, 0xfff5, 0xfff5, 0xfff4,
    0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff4, 0xfff3, 0xfff3, 0xfff3, 0xfff3, 0xfff3,
    0xfff3, 0xfff3, 0xfff3, 0xfff2, 0xfff2, 0xfff2, 0xfff2, 0xfff2, 0xfff2, 0xfff2, 0xfff1, 0xfff1,
    0xfff1, 0xfff1, 0xfff1, 0xfff1, 0xfff1, 0xfff1, 0xfff0, 0xfff0, 0xfff0, 0xfff0, 0xfff0, 0xfff0,
    0xfff0, 0xffef, 0xffef, 0xffef, 0xffef, 0xffef, 0xffef, 0xffef, 0xffee, 0xffee, 0xffee, 0xffee,
    0xffee, 0xffee, 0xffee, 0xffed, 0xffed, 0xffed, 0xffed, 0xffed, 0xffed, 0xffed, 0xffed, 0xffec,
    0xffec, 0xffec, 0xffec, 0xffec, 0xffec, 0xffec, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb, 0xffeb,
    0xffeb, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffea, 0xffe9, 0xffe9, 0xffe9,
    0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8, 0xffe8,
    0xffe8, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe6, 0xffe6,
    0xffe6, 0xffe6, 0xffe6, 0xffe6, 0xffe6, 0xffe6, 0xffe6, 0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe5,
    0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4,
    0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe3, 0xffe3, 0xffe3, 0xffe3,
    0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3,
    0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3,
    0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3,
    0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3, 0xffe3,
    0xffe3, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4, 0xffe4,
    0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe5, 0xffe6, 0xffe6, 0xffe6, 0xffe6,
    0xffe6, 0xffe6, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe7, 0xffe8, 0xffe8, 0xffe8, 0xffe8,
    0xffe8, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffe9, 0xffea, 0xffea, 0xffea, 0xffea, 0xffeb, 0xffeb,
    0xffeb, 0xffeb, 0xffec, 0xffec, 0xffec, 0xffed, 0xffed, 0xffed, 0xffed, 0xffee, 0xffee, 0xffee,
    0xffef, 0xffef, 0xffef, 0xfff0, 0xfff0, 0xfff0, 0xfff1, 0xfff1, 0xfff1, 0xfff2, 0xfff2, 0xfff2,
    0xfff3, 0xfff3, 0xfff3, 0xfff4, 0xfff4, 0xfff5, 0xfff5, 0xfff5, 0xfff6, 0xfff6, 0xfff7, 0xfff7,
    0xfff8, 0xfff8, 0xfff8, 0xfff9, 0xfff9, 0xfffa, 0xfffa, 0xfffb, 0xfffb, 0xfffc, 0xfffc, 0xfffd,
    0xfffd, 0xfffe, 0xfffe, 0xffff, 0xffff, 0x0000, 0x0000, 0x0001, 0x0001, 0x0002, 0x0002, 0x0003,
    0x0004, 0x0004, 0x0005, 0x0005, 0x0006, 0x0006, 0x0007, 0x0008, 0x0008, 0x0009, 0x0009, 0x000a,
    0x000b, 0x000b, 0x000c, 0x000d, 0x000d, 0x000e, 0x000f, 0x000f, 0x0010, 0x0011, 0x0011, 0x0012,
    0x0013, 0x0013, 0x0014, 0x0015, 0x0015, 0x0016, 0x0017, 0x0018, 0x0018, 0x0019, 0x001a, 0x001b,
    0x001b, 0x001c, 0x001d, 0x001e, 0x001e, 0x001f, 0x0020, 0x0021, 0x0022, 0x0022, 0x0023, 0x0024,
    0x0025, 0x0026, 0x0027, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002d, 0x002e,
    0x002f, 0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0037, 0x0038, 0x0039,
    0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f, 0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045,
    0x0046, 0x0047, 0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, 0x0050, 0x0051,
    0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e,
    0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0069, 0x006a, 0x006b, 0x006c,
    0x006d, 0x006e, 0x006f, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0078, 0x0079, 0x007a,
    0x007b, 0x007c, 0x007e, 0x007f, 0x0080, 0x0081, 0x0082, 0x0083, 0x0085, 0x0086, 0x0087, 0x0088,
    0x0089, 0x008b, 0x008c, 0x008d, 0x008e, 0x0090, 0x0091, 0x0092, 0x0093, 0x0094, 0x0096, 0x0097,
    0x0098, 0x0099, 0x009b, 0x009c, 0x009d, 0x009e, 0x009f, 0x00a1, 0x00a2, 0x00a3, 0x00a4, 0x00a6,
    0x00a7, 0x00a8, 0x00a9, 0x00aa, 0x00ac, 0x00ad, 0x00ae, 0x00af, 0x00b1, 0x00b2, 0x00b3, 0x00b4,
    0x00b6, 0x00b7, 0x00b8, 0x00b9, 0x00ba, 0x00bc, 0x00bd, 0x00be, 0x00bf, 0x00c0, 0x00c2, 0x00c3,
    0x00c4, 0x00c5, 0x00c6, 0x00c8, 0x00c9, 0x00ca, 0x00cb, 0x00cc, 0x00ce, 0x00cf, 0x00d0, 0x00d1,
    0x00d2, 0x00d3, 0x00d5, 0x00d6, 0x00d7, 0x00d8, 0x00d9, 0x00da, 0x00db, 0x00dd, 0x00de, 0x00df,
    0x00e0, 0x00e1, 0x00e2, 0x00e3, 0x00e4, 0x00e5, 0x00e7, 0x00e8, 0x00e9, 0x00ea, 0x00eb, 0x00ec,
    0x00ed, 0x00ee, 0x00ef, 0x00f0, 0x00f1, 0x00f2, 0x00f3, 0x00f4, 0x00f5, 0x00f6, 0x00f7, 0x00f8,
    0x00f9, 0x00fa, 0x00fb, 0x00fc, 0x00fd, 0x00fe, 0x00ff, 0x0100, 0x0101, 0x0102, 0x0103, 0x0103,
    0x0104, 0x0105, 0x0106, 0x0107, 0x0108, 0x0109, 0x0109, 0x010a, 0x010b, 0x010c, 0x010d, 0x010d,
    0x010e, 0x010f, 0x0110, 0x0110, 0x0111, 0x0112, 0x0113, 0x0113, 0x0114, 0x0115, 0x0115, 0x0116,
    0x0117, 0x0117, 0x0118, 0x0118, 0x0119, 0x011a, 0x011a, 0x011b, 0x011b, 0x011c, 0x011c, 0x011d,
    0x011d, 0x011e, 0x011e, 0x011f, 0x011f, 0x0120, 0x0120, 0x0120, 0x0121, 0x0121, 0x0122, 0x0122,
    0x0122, 0x0123, 0x0123, 0x0123, 0x0123, 0x0124, 0x0124, 0x0124, 0x0124, 0x0124, 0x0125, 0x0125,
    0x0125, 0x0125, 0x0125, 0x0125, 0x0125, 0x0125, 0x0126, 0x0126, 0x0126, 0x0126, 0x0126, 0x0126,
    0x0125, 0x0125, 0x0125, 0x0125, 0x0125, 0x0125, 0x0125, 0x0125, 0x0124, 0x0124, 0x0124, 0x0124,
    0x0123, 0x0123, 0x0123, 0x0122, 0x0122, 0x0122, 0x0121, 0x0121, 0x0121, 0x0120, 0x0120, 0x011f,
    0x011f, 0x011e, 0x011e, 0x011d, 0x011c, 0x011c, 0x011b, 0x011b, 0x011a, 0x0119, 0x0119, 0x0118,
    0x0117, 0x0116, 0x0115, 0x0115, 0x0114, 0x0113, 0x0112, 0x0111, 0x0110, 0x010f, 0x010e, 0x010d,
    0x010c, 0x010b, 0x010a, 0x0109, 0x0108, 0x0107, 0x0106, 0x0105, 0x0103, 0x0102, 0x0101, 0x0100,
    0x00fe, 0x00fd, 0x00fc, 0x00fa, 0x00f9, 0x00f8, 0x00f6, 0x00f5, 0x00f3, 0x00f2, 0x00f0, 0x00ef,
    0x00ed, 0x00eb, 0x00ea, 0x00e8, 0x00e7, 0x00e5, 0x00e3, 0x00e1, 0x00e0, 0x00de, 0x00dc, 0x00da,
    0x00d8, 0x00d6, 0x00d4, 0x00d2, 0x00d1, 0x00cf, 0x00cc, 0x00ca, 0x00c8, 0x00c6, 0x00c4, 0x00c2,
    0x00c0, 0x00be, 0x00bb, 0x00b9, 0x00b7, 0x00b5, 0x00b2, 0x00b0, 0x00ad, 0x00ab, 0x00a9, 0x00a6,
    0x00a4, 0x00a1, 0x009f, 0x009c, 0x0099, 0x0097, 0x0094, 0x0092, 0x008f, 0x008c, 0x0089, 0x0087,
    0x0084, 0x0081, 0x007e, 0x007b, 0x0078, 0x0075, 0x0072, 0x006f, 0x006c, 0x0069, 0x0066, 0x0063,
    0x0060, 0x005d, 0x005a, 0x0057, 0x0053, 0x0050, 0x004d, 0x0049, 0x0046, 0x0043, 0x003f, 0x003c,
    0x0039, 0x0035, 0x0032, 0x002e, 0x002b, 0x0027, 0x0023, 0x0020, 0x001c, 0x0018, 0x0015, 0x0011,
    0x000d, 0x000a, 0x0006, 0x0002, 0xfffe, 0xfffa, 0xfff6, 0xfff2, 0xffee, 0xffea, 0xffe6, 0xffe2,
    0xffde, 0xffda, 0xffd6, 0xffd2, 0xffce, 0xffca, 0xffc6, 0xffc1, 0xffbd, 0xffb9, 0xffb5, 0xffb0,
    0xffac, 0xffa7, 0xffa3, 0xff9f, 0xff9a, 0xff96, 0xff91, 0xff8d, 0xff88, 0xff84, 0xff7f, 0xff7b,
    0xff76, 0xff71, 0xff6d, 0xff68, 0xff63, 0xff5e, 0xff5a, 0xff55, 0xff50, 0xff4b, 0xff46, 0xff42,
    0xff3d, 0xff38, 0xff33, 0xff2e, 0xff29, 0xff24, 0xff1f, 0xff1a, 0xff15, 0xff10, 0xff0b, 0xff06,
    0xff00, 0xfefb, 0xfef6, 0xfef1, 0xfeec, 0xfee6, 0xfee1, 0xfedc, 0xfed7, 0xfed1, 0xfecc, 0xfec7,
    0xfec1, 0xfebc, 0xfeb7, 0xfeb1, 0xfeac, 0xfea6, 0xfea1, 0xfe9b, 0xfe96, 0xfe90, 0xfe8b, 0xfe85,
    0xfe80, 0xfe7a, 0xfe75, 0xfe6f, 0xfe69, 0xfe64, 0xfe5e, 0xfe59, 0xfe53, 0xfe4d, 0xfe48, 0xfe42,
    0xfe3c, 0xfe36, 0xfe31, 0xfe2b, 0xfe25, 0xfe1f, 0xfe1a, 0xfe14, 0xfe0e, 0xfe08, 0xfe02, 0xfdfd,
    0xfdf7, 0xfdf1, 0xfdeb, 0xfde5, 0xfddf, 0xfdd9, 0xfdd4, 0xfdce, 0xfdc8, 0xfdc2, 0xfdbc, 0xfdb6,
    0xfdb0, 0xfdaa, 0xfda4, 0xfd9e, 0xfd98, 0xfd92, 0xfd8c, 0xfd86, 0xfd80, 0xfd7a, 0xfd74, 0xfd6e,
    0xfd68, 0xfd62, 0xfd5c, 0xfd56, 0xfd50, 0xfd4a, 0xfd44, 0xfd3e, 0xfd38, 0xfd32, 0xfd2c, 0xfd26,
    0xfd20, 0xfd1a, 0xfd14, 0xfd0e, 0xfd08, 0xfd03, 0xfcfd, 0xfcf7, 0xfcf1, 0xfceb, 0xfce5, 0xfcdf,
    0xfcd9, 0xfcd3, 0xfccd, 0xfcc7, 0xfcc1, 0xfcbb, 0xfcb5, 0xfcaf, 0xfca9, 0xfca3, 0xfc9d, 0xfc98,
    0xfc92, 0xfc8c, 0xfc86, 0xfc80, 0xfc7a, 0xfc75, 0xfc6f, 0xfc69, 0xfc63, 0xfc5d, 0xfc58, 0xfc52,
    0xfc4c, 0xfc46, 0xfc41, 0xfc3b, 0xfc35, 0xfc30, 0xfc2a, 0xfc24, 0xfc1f, 0xfc19, 0xfc14, 0xfc0e,
    0xfc09, 0xfc03, 0xfbfe, 0xfbf8, 0xfbf3, 0xfbed, 0xfbe8, 0xfbe2, 0xfbdd, 0xfbd8, 0xfbd2, 0xfbcd,
    0xfbc8, 0xfbc2, 0xfbbd, 0xfbb8, 0xfbb3, 0xfbae, 0xfba8, 0xfba3, 0xfb9e, 0xfb99, 0xfb94, 0xfb8f,
    0xfb8a, 0xfb85, 0xfb80, 0xfb7b, 0xfb77, 0xfb72, 0xfb6d, 0xfb68, 0xfb63, 0xfb5f, 0xfb5a, 0xfb55,
    0xfb51, 0xfb4c, 0xfb48, 0xfb43, 0xfb3f, 0xfb3a, 0xfb36, 0xfb31, 0xfb2d, 0xfb29, 0xfb25, 0xfb20,
    0xfb1c, 0xfb18, 0xfb14, 0xfb10, 0xfb0c, 0xfb08, 0xfb04, 0xfb00, 0xfafc, 0xfaf8, 0xfaf5, 0xfaf1,
    0xfaed, 0xfaea, 0xfae6, 0xfae2, 0xfadf, 0xfadb, 0xfad8, 0xfad5, 0xfad1, 0xface, 0xfacb, 0xfac8,
    0xfac5, 0xfac1, 0xfabe, 0xfabb, 0xfab8, 0xfab6, 0xfab3, 0xfab0, 0xfaad, 0xfaab, 0xfaa8, 0xfaa5,
    0xfaa3, 0xfaa0, 0xfa9e, 0xfa9c, 0xfa99, 0xfa97, 0xfa95, 0xfa93, 0xfa91, 0xfa8f, 0xfa8d, 0xfa8b,
    0xfa89, 0xfa87, 0xfa86, 0xfa84, 0xfa82, 0xfa81, 0xfa7f, 0xfa7e, 0xfa7d, 0xfa7b, 0xfa7a, 0xfa79,
    0xfa78, 0xfa77, 0xfa76, 0xfa75, 0xfa74, 0xfa74, 0xfa73, 0xfa72, 0xfa72, 0xfa71, 0xfa71, 0xfa70,
    0xfa70, 0xfa70, 0xfa70, 0xfa70, 0xfa70, 0xfa70, 0xfa70, 0xfa70, 0xfa70, 0xfa71, 0xfa71, 0xfa72,
    0xfa72, 0xfa73, 0xfa74, 0xfa75, 0xfa75, 0xfa76, 0xfa77, 0xfa79, 0xfa7a, 0xfa7b, 0xfa7c, 0xfa7e,
    0xfa7f, 0xfa81, 0xfa82, 0xfa84, 0xfa86, 0xfa88, 0xfa8a, 0xfa8c, 0xfa8e, 0xfa90, 0xfa93, 0xfa95,
    0xfa98, 0xfa9a, 0xfa9d, 0xfa9f, 0xfaa2, 0xfaa5, 0xfaa8, 0xfaab, 0xfaae, 0xfab2, 0xfab5, 0xfab8,
    0xfabc, 0xfabf, 0xfac3, 0xfac7, 0xfacb, 0xfacf, 0xfad3, 0xfad7, 0xfadb, 0xfadf, 0xfae4, 0xfae8,
    0xfaed, 0xfaf1, 0xfaf6, 0xfafb, 0xfb00, 0xfb05, 0xfb0a, 0xfb0f, 0xfb15, 0xfb1a, 0xfb20, 0xfb25,
    0xfb2b, 0xfb31, 0xfb36, 0xfb3c, 0xfb43, 0xfb49, 0xfb4f, 0xfb55, 0xfb5c, 0xfb62, 0xfb69, 0xfb70,
    0xfb77, 0xfb7d, 0xfb85, 0xfb8c, 0xfb93, 0xfb9a, 0xfba2, 0xfba9, 0xfbb1, 0xfbb9, 0xfbc0, 0xfbc8,
    0xfbd0, 0xfbd9, 0xfbe1, 0xfbe9, 0xfbf2, 0xfbfa, 0xfc03, 0xfc0b, 0xfc14, 0xfc1d, 0xfc26, 0xfc2f,
    0xfc39, 0xfc42, 0xfc4b, 0xfc55, 0xfc5f, 0xfc68, 0xfc72, 0xfc7c, 0xfc86, 0xfc91, 0xfc9b, 0xfca5,
    0xfcb0, 0xfcba, 0xfcc5, 0xfcd0, 0xfcdb, 0xfce6, 0xfcf1, 0xfcfc, 0xfd07, 0xfd13, 0xfd1e, 0xfd2a,
    0xfd36, 0xfd42, 0xfd4e, 0xfd5a, 0xfd66, 0xfd72, 0xfd7f, 0xfd8b, 0xfd98, 0xfda4, 0xfdb1, 0xfdbe,
    0xfdcb, 0xfdd8, 0xfde5, 0xfdf3, 0xfe00, 0xfe0e, 0xfe1b, 0xfe29, 0xfe37, 0xfe45, 0xfe53, 0xfe61,
    0xfe70, 0xfe7e, 0xfe8d, 0xfe9b, 0xfeaa, 0xfeb9, 0xfec8, 0xfed7, 0xfee6, 0xfef5, 0xff04, 0xff14,
    0xff23, 0xff33, 0xff43, 0xff53, 0xff63, 0xff73, 0xff83, 0xff93, 0xffa4, 0xffb4, 0xffc5, 0xffd6,
    0xffe7, 0xfff7, 0x0009, 0x001a, 0x002b, 0x003c, 0x004e, 0x005f, 0x0071, 0x0083, 0x0095, 0x00a7,
    0x00b9, 0x00cb, 0x00dd, 0x00f0, 0x0102, 0x0115, 0x0127, 0x013a, 0x014d, 0x0160, 0x0173, 0x0186,
    0x019a, 0x01ad, 0x01c1, 0x01d4, 0x01e8, 0x01fc, 0x0210, 0x0224, 0x0238, 0x024c, 0x0260, 0x0275,
    0x0289, 0x029e, 0x02b3, 0x02c7, 0x02dc, 0x02f1, 0x0306, 0x031b, 0x0331, 0x0346, 0x035c, 0x0371,
    0x0387, 0x039d, 0x03b2, 0x03c8, 0x03de, 0x03f5, 0x040b, 0x0421, 0x0437, 0x044e, 0x0465, 0x047b,
    0x0492, 0x04a9, 0x04c0, 0x04d7, 0x04ee, 0x0505, 0x051c, 0x0534, 0x054b, 0x0563, 0x057b, 0x0592,
    0x05aa, 0x05c2, 0x05da, 0x05f2, 0x060a, 0x0623, 0x063b, 0x0653, 0x066c, 0x0684, 0x069d, 0x06b6,
    0x06cf, 0x06e8, 0x0700, 0x071a, 0x0733, 0x074c, 0x0765, 0x077f, 0x0798, 0x07b2, 0x07cb, 0x07e5,
    0x07ff, 0x0818, 0x0832, 0x084c, 0x0866, 0x0880, 0x089b, 0x08b5, 0x08cf, 0x08ea, 0x0904, 0x091f,
    0x0939, 0x0954, 0x096f, 0x098a, 0x09a4, 0x09bf, 0x09da, 0x09f5, 0x0a11, 0x0a2c, 0x0a47, 0x0a62,
    0x0a7e, 0x0a99, 0x0ab5, 0x0ad0, 0x0aec, 0x0b08, 0x0b23, 0x0b3f, 0x0b5b, 0x0b77, 0x0b93, 0x0baf,
    0x0bcb, 0x0be7, 0x0c03, 0x0c20, 0x0c3c, 0x0c58, 0x0c75, 0x0c91, 0x0cad, 0x0cca, 0x0ce7, 0x0d03,
    0x0d20, 0x0d3d, 0x0d59, 0x0d76, 0x0d93, 0x0db0, 0x0dcd, 0x0dea, 0x0e07, 0x0e24, 0x0e41, 0x0e5e,
    0x0e7b, 0x0e99, 0x0eb6, 0x0ed3, 0x0ef0, 0x0f0e, 0x0f2b, 0x0f49, 0x0f66, 0x0f84, 0x0fa1, 0x0fbf,
    0x0fdc, 0x0ffa, 0x1017, 0x1035, 0x1053, 0x1070, 0x108e, 0x10ac, 0x10ca, 0x10e7, 0x1105, 0x1123,
    0x1141, 0x115f, 0x117d, 0x119b, 0x11b9, 0x11d7, 0x11f5, 0x1213, 0x1231, 0x124f, 0x126d, 0x128b,
    0x12a9, 0x12c7, 0x12e5, 0x1303, 0x1321, 0x133f, 0x135d, 0x137b, 0x1399, 0x13b7, 0x13d5, 0x13f4,
    0x1412, 0x1430, 0x144e, 0x146c, 0x148a, 0x14a8, 0x14c6, 0x14e4, 0x1502, 0x1521, 0x153f, 0x155d,
    0x157b, 0x1599, 0x15b7, 0x15d5, 0x15f3, 0x1611, 0x162f, 0x164d, 0x166b, 0x1689, 0x16a7, 0x16c4,
    0x16e2, 0x1700, 0x171e, 0x173c, 0x175a, 0x1777, 0x1795, 0x17b3, 0x17d1, 0x17ee, 0x180c, 0x182a,
    0x1847, 0x1865, 0x1882, 0x18a0, 0x18bd, 0x18db, 0x18f8, 0x1915, 0x1933, 0x1950, 0x196d, 0x198a,
    0x19a8, 0x19c5, 0x19e2, 0x19ff, 0x1a1c, 0x1a39, 0x1a56, 0x1a73, 0x1a8f, 0x1aac, 0x1ac9, 0x1ae6,
    0x1b02, 0x1b1f, 0x1b3b, 0x1b58, 0x1b74, 0x1b91, 0x1bad, 0x1bc9, 0x1be6, 0x1c02, 0x1c1e, 0x1c3a,
    0x1c56, 0x1c72, 0x1c8e, 0x1ca9, 0x1cc5, 0x1ce1, 0x1cfc, 0x1d18, 0x1d33, 0x1d4f, 0x1d6a, 0x1d85,
    0x1da1, 0x1dbc, 0x1dd7, 0x1df2, 0x1e0d, 0x1e28, 0x1e42, 0x1e5d, 0x1e78, 0x1e92, 0x1ead, 0x1ec7,
    0x1ee1, 0x1efc, 0x1f16, 0x1f30, 0x1f4a, 0x1f64, 0x1f7e, 0x1f97, 0x1fb1, 0x1fcb, 0x1fe4, 0x1ffd,
    0x2017, 0x2030, 0x2049, 0x2062, 0x207b, 0x2094, 0x20ad, 0x20c5, 0x20de, 0x20f6, 0x210f, 0x2127,
    0x213f, 0x2157, 0x216f, 0x2187, 0x219f, 0x21b6, 0x21ce, 0x21e6, 0x21fd, 0x2214, 0x222b, 0x2242,
    0x2259, 0x2270, 0x2287, 0x229d, 0x22b4, 0x22ca, 0x22e1, 0x22f7, 0x230d, 0x2323, 0x2339, 0x234e,
    0x2364, 0x237a, 0x238f, 0x23a4, 0x23b9, 0x23ce, 0x23e3, 0x23f8, 0x240d, 0x2421, 0x2436, 0x244a,
    0x245e, 0x2472, 0x2486, 0x249a, 0x24ae, 0x24c1, 0x24d5, 0x24e8, 0x24fb, 0x250e, 0x2521, 0x2534,
    0x2547, 0x2559, 0x256c, 0x257e, 0x2590, 0x25a2, 0x25b4, 0x25c6, 0x25d7, 0x25e9, 0x25fa, 0x260b,
    0x261c, 0x262d, 0x263e, 0x264e, 0x265f, 0x266f, 0x267f, 0x2690, 0x26a0, 0x26af, 0x26bf, 0x26ce,
    0x26de, 0x26ed, 0x26fc, 0x270b, 0x271a, 0x2728, 0x2737, 0x2745, 0x2753, 0x2762, 0x276f, 0x277d,
    0x278b, 0x2798, 0x27a6, 0x27b3, 0x27c0, 0x27cd, 0x27d9, 0x27e6, 0x27f2, 0x27fe, 0x280a, 0x2816,
    0x2822, 0x282e, 0x2839, 0x2845, 0x2850, 0x285b, 0x2866, 0x2870, 0x287b, 0x2885, 0x288f, 0x289a,
    0x28a3, 0x28ad, 0x28b7, 0x28c0, 0x28c9, 0x28d3, 0x28db, 0x28e4, 0x28ed, 0x28f5, 0x28fe, 0x2906,
    0x290e, 0x2916, 0x291d, 0x2925, 0x292c, 0x2933, 0x293a, 0x2941, 0x2948, 0x294e, 0x2955, 0x295b,
    0x2961, 0x2967, 0x296c, 0x2972, 0x2977, 0x297c, 0x2981, 0x2986, 0x298b, 0x298f, 0x2994, 0x2998,
    0x299c, 0x29a0, 0x29a4, 0x29a7, 0x29ab, 0x29ae, 0x29b1, 0x29b4, 0x29b6, 0x29b9, 0x29bb, 0x29bd,
    0x29bf, 0x29c1, 0x29c3, 0x29c4, 0x29c6, 0x29c7, 0x29c8, 0x29c9, 0x29c9, 0x29ca, 0x29ca, 0x29ca,
};

//---------Conversion table--------------------------------------------------------

typedef struct {
    int             inRate;
    int             outRate;
    int             upScale;
    int             dnScale;
    int             minInOutSamples;
    int             taps;
    const int16_t   *coefs;
} ConvTable;


// Note!! The MAX downscale ratio in the table below supported by the NEON
// assembly versions is 256 / taps when upscale=1.
// i.e. for 24 taps the max would be 10
static const ConvTable table[] = {
    { 8000,  48000,   6,   1,   6, FIR_TAP_24, fir_6_1 },
    { 11025, 48000,  48,  11,  48, FIR_TAP_24, fir_48_11 }, // should be 640_147?
    { 12000, 48000,   4,   1,   4, FIR_TAP_24, fir_4_1 },
    { 16000, 48000,   3,   1,   3, FIR_TAP_24, fir_3_1 },
    { 22050, 48000, 320, 147, 320, FIR_TAP_24, fir_320_147 },
    { 24000, 48000,   2,   1,   2, FIR_TAP_24, fir_2_1 },
    { 32000, 48000,   3,   2,   3, FIR_TAP_24, fir_3_1 },
    { 44100, 48000, 160, 147, 160, FIR_TAP_32, fir_160_147 },
    { 96000, 48000,   1,   2,   1, FIR_TAP_24, fir_1_2 },

    { 48000, 8000,    1,   6,   6, FIR_TAP_24, fir_1_6 },
    { 48000, 12000,   1,   4,   4, FIR_TAP_24, fir_1_4 },
    { 48000, 16000,   1,   3,   3, FIR_TAP_24, fir_1_3 },
    { 48000, 24000,   1,   2,   2, FIR_TAP_24, fir_1_2 },
    { 48000, 32000,   2,   3,   3, FIR_TAP_32, fir_2_3 },
    { 48000, 44100, 147, 160, 160, FIR_TAP_32, fir_147_160 },

    {  8000, 16000,   2,   1,   2, FIR_TAP_24, fir_2_1 },
    { 12000, 24000,   2,   1,   2, FIR_TAP_24, fir_2_1 },
    { 16000, 32000,   2,   1,   2, FIR_TAP_24, fir_2_1 },
    { 11025, 22050,   2,   1,   2, FIR_TAP_24, fir_2_1 },
    { 22050, 44100,   2,   1,   2, FIR_TAP_24, fir_2_1 },

    { 16000,  8000,   1,   2,   2, FIR_TAP_24, fir_1_2 },
    { 24000, 12000,   1,   2,   2, FIR_TAP_24, fir_1_2 },
    { 32000, 16000,   1,   2,   2, FIR_TAP_24, fir_1_2 },
    { 22050, 11025,   1,   2,   2, FIR_TAP_24, fir_1_2 },
    { 44100, 22050,   1,   2,   2, FIR_TAP_24, fir_1_2 },

    {  8000, 12000,   3,   2,   3, FIR_TAP_24, fir_3_1 },
    { 16000, 24000,   3,   2,   3, FIR_TAP_24, fir_3_1 },

    { 12000,  8000,   2,   3,   3, FIR_TAP_32, fir_2_3 },
    { 24000, 16000,   2,   3,   3, FIR_TAP_32, fir_2_3 },

    { 11025, 44100,   4,   1,   4, FIR_TAP_24, fir_4_1 },

    { 44100, 11025,   1,   4,   4, FIR_TAP_24, fir_1_4 },

    { 22050, 16000, 320, 441, 441, FIR_TAP_24, fir_320_441 },
    { 44100, 16000, 160, 441, 441, FIR_TAP_24, fir_160_441 },
};
#define CONV_TABLE_SIZE  (sizeof(table)/sizeof(table[0]))

// Helper functions used with arm reference code
#if defined(VARIANT_v7) && !defined(POLY_NEON_INTRINSICS) 

#define MULTIPLY_MONO \
    co = *coefW++; \
    dly0 = *dly++; \
    dly1 = *dly++; \
    __asm__ volatile (             \
        "smlabb %0, %1, %2, %0;"   \
        "smlatb %0, %1, %3, %0;"   \
        : "+r" (acc)               \
        : "r"(co), "r"(dly0), "r"(dly1) \
    );

static __inline int16_t __attribute__ ((__always_inline__)) dot_product_mono(const int16_t* coef, int16_t* dly, int taps, int channels)
{
    int32_t acc = 0;
    int32_t co, dly0, dly1;
    int32_t *coefW = (int32_t *)coef;
    int i;

    for (i = 0; i < taps/8; i++) {
        MULTIPLY_MONO;
        MULTIPLY_MONO;
        MULTIPLY_MONO;
        MULTIPLY_MONO;
    }
    acc = ((acc >> 14) + 1) >> 1;                // 2.30 -> 1.15
    __asm__ volatile (
        " ssat %0, #16, %0;"
        : "+r" (acc)
    );
    return (int16_t)acc;
}

// c0   - k0 k1
// dly0 - l0 r0
// dly  - l1 r1
// acc0 = k0 * l0
// acc1 = k0 * r0
// acc0 += k1 * l1
// acc1 += k1 * r1

#define MULTIPLY_STEREO \
    co = *coefW++;      \
    dly0 = *dlyW++;     \
    dly1 = *dlyW++;     \
    __asm__ volatile (                  \
        "smlabb %0, %2, %3, %0;"        \
        "smlabt %1, %2, %3, %1;"        \
        "smlatb %0, %2, %4, %0;"        \
        "smlatt %1, %2, %4, %1;"        \
        : "+r" (acc0), "+r"(acc1)       \
        : "r"(co), "r"(dly0), "r"(dly1) \
    );


static __inline int32_t __attribute__ ((__always_inline__)) dot_product_stereo(const int16_t* coef, int32_t* dly, int taps, int channels)
{
    int32_t acc0 = 0, acc1 = 0;
    int32_t acc;
    int32_t co, dly0, dly1;
    int32_t *coefW = (int32_t *)coef;
    int32_t *dlyW = (int32_t *)dly;
    int i;

    for (i = 0; i < taps/8 ; i++) {
        MULTIPLY_STEREO;
        MULTIPLY_STEREO;
        MULTIPLY_STEREO;
        MULTIPLY_STEREO;
    }
    acc0 = (((acc0 >> 14) + 1) >> 1);                // 2.30 -> 1.15
    __asm__ volatile (
        " ssat %0, #16, %0;"
        : "+r" (acc0)
    );
    acc1 = (((acc1 >> 14) + 1) >> 1);                // 2.30 -> 1.15
    __asm__ volatile (
        " ssat %0, #16, %0;"
        : "+r" (acc1)
    );
    acc = acc1 << 16 | (acc0 & 0xffff);
    return acc;
}

// Helper functions used with X86, target independent reference code
#elif !defined(VARIANT_v7)

static __inline int32_t saturate (int32_t acc)
{
    if (acc > 0x7fff) {
        return 0x7fff;
    } else if (acc < -0x8000) {
        return -0x8000;
    } else {
        return acc;
    }
}

//------------------------------------------------------------------------------
static __inline int16_t dot_product_mono(const int16_t* coef, int16_t* dly, int taps, int channels)
{
    int32_t acc = 0;
    int i;
    for (i = 0; i < taps/8; i++) {
        acc += (*coef++) * (*dly++);
        acc += (*coef++) * (*dly++);
        acc += (*coef++) * (*dly++);
        acc += (*coef++) * (*dly++);
        acc += (*coef++) * (*dly++);
        acc += (*coef++) * (*dly++);
        acc += (*coef++) * (*dly++);
        acc += (*coef++) * (*dly++);
    }
    acc = ((acc >> 14) + 1) >> 1;                // 2.30 -> 1.15
    acc = saturate(acc);
    return (int16_t)acc;
}

static __inline int32_t dot_product_stereo(const int16_t* coef, int32_t* dly, int taps, int channels)
{
    int32_t acc0 = 0, acc1 = 0;
    int32_t acc;
    int32_t mask = 0xffff;
    int i;

    for (i = 0; i < taps/8 ; i++) {
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
        acc0 += (*coef) * (((*dly) << 16) >> 16);
        acc1 += (*coef) * ((*dly) >> 16);
        dly ++; coef++;
    }

    acc0 = (((acc0 >> 14) + 1) >> 1);        // 2.30 -> 1.15
    acc0 = saturate(acc0);
    acc1 = (((acc1 >> 14) + 1) >> 1);        // 2.30 -> 1.15
    acc1 = saturate(acc1);
    acc = acc1 << 16 | (acc0 & mask);
    return acc;
}
#endif

#if defined(POLY_NEON_INTRINSICS)
static int PlaybackRateMixConvertMono_NEON_I(const int16_t* in, int outSamples, int16_t* out, int upScale, int dnScale, int channels,
                                             int *firPh, int *dlyUp, int16_t *dlyLine, int *dlyIdx, int16_t *poly, int taps)
{
    int i;
    int j;
    int firPhase = *firPh;
    int dlyUpdate = *dlyUp;
    int dlyI = *dlyIdx;
    int iter = taps>>2;
    int16x4_t dly;
    int32x4_t acc_q;
    int32x2_t acc_d;
    int16x4_t acc;
    int16_t *dly_base;
    int16_t *poly_base;

    if (outSamples == 0) {
        return 0;
    }
    // Update delay line when filter phase wraps
    while (dlyUpdate) {
        dlyI --;
        DLYLINE_CHECK_IDX(dlyI);
        dlyLine[dlyI] = *in;
        in += channels;
        dlyUpdate--;
    }
    for (i = 0; i < outSamples; i++) {
        dly = vld1_s16((dly_base = (dlyLine + dlyI)));
        DLYLINE_CHECK_PTR(dly_base);
        acc_q = vmull_s16(vld1_s16((poly_base = (poly + firPhase * taps))), dly);
        for (j=1; j < iter; j++) {
            DLYLINE_CHECK_PTR(dly_base + (j<<2));
            dly = vld1_s16(dly_base + (j<<2));
            acc_q = vmlal_s16(acc_q, vld1_s16(poly_base + (j<<2)), dly);
        }
        acc_d = vadd_s32(vget_low_s32(acc_q), vget_high_s32(acc_q));
        acc_d = vpadd_s32(acc_d, acc_d);
        acc_q = vcombine_s32(acc_d, acc_d);
        acc = vqrshrn_n_s32(acc_q, 15);
        vst1_lane_s16(out, acc, 0);
        out += channels;
        // Update filter phase and delay line  when filter phase wraps
        firPhase += dnScale;
        while (firPhase >= upScale && i + 1 < outSamples) {
            firPhase -= upScale;
            dlyI --;
            DLYLINE_CHECK_IDX(dlyI);
            dlyLine[dlyI] = *in;
            in += channels;
        }
    }
    // Update filter phase
    while (firPhase >= upScale) {
        firPhase -= upScale;
        dlyUpdate++;
    }
    i = *dlyIdx - dlyI;
    *firPh = firPhase;
    *dlyUp = dlyUpdate;
    *dlyIdx = dlyI;
    return i;
}

static int PlaybackRateMixConvertStereo_NEON_I(const int16_t* in, int outSamples, int16_t* out, int upScale, int dnScale, int channels,
                                               int *firPh, int *dlyUp, int32_t *dlyLine, int *dlyIdx, int16_t *poly, int taps)
{
    int i;
    int j;
    int32_t *inW = (int32_t *)in;
    int firPhase = *firPh;
    int dlyUpdate = *dlyUp;
    int dlyI = *dlyIdx;
    int iter = taps>>2;
    int16x4_t c;
    int16x4x2_t dly;
    int32x4_t acc_q0, acc_q1;
    int32x2_t acc_d0, acc_d1;
    int16x4_t acc;
    int32_t *dly_base;
    int16_t *poly_base;

    if (outSamples == 0) {
        return 0;
    }
    // Update delay line
    while (dlyUpdate) {
        dlyI --;
        DLYLINE_CHECK_IDX(dlyI);
        dlyLine[dlyI] = *inW;
        inW += channels/2;
        dlyUpdate--;
    }
    for (i = 0; i < outSamples; i++) {
        c = vld1_s16((poly_base = (poly + firPhase * taps)));
        dly = vld2_s16((int16_t*)(dly_base = (dlyLine + dlyI)));
        DLYLINE_CHECK_PTR(dly_base);
        acc_q0 = vmull_s16(c, dly.val[0]);
        acc_q1 = vmull_s16(c, dly.val[1]);
        for ( j = 1; j < iter; j++ ) {
            c = vld1_s16(poly_base + (j<<2));
            DLYLINE_CHECK_PTR(dly_base + (j<<2));
            dly = vld2_s16((int16_t*)(dly_base + (j<<2)));
            acc_q0 = vmlal_s16(acc_q0, c, dly.val[0]);
            acc_q1 = vmlal_s16(acc_q1, c, dly.val[1]);
        }
        acc_d0 = vadd_s32(vget_low_s32(acc_q0), vget_high_s32(acc_q0));
        acc_d1 = vadd_s32(vget_low_s32(acc_q1), vget_high_s32(acc_q1));
        acc_d0 = vpadd_s32(acc_d0, acc_d0);
        acc_d1 = vpadd_s32(acc_d1, acc_d1);

        acc_q0 = vcombine_s32(acc_d0, acc_d1);
        acc = vqrshrn_n_s32(acc_q0, 15);
        vst1_lane_s16(out, acc, 0);
        vst1_lane_s16(out+1, acc, 2);
        out += channels;

        // Update filter phase and delay line when filter phase wraps
        firPhase += dnScale;
        while (firPhase >= upScale && i + 1 < outSamples) {
            firPhase -= upScale;
            dlyI --;
            DLYLINE_CHECK_IDX(dlyI);
            dlyLine[dlyI] = *inW;
            inW += channels/2;
        }
    }
    // Update filter phase
    while (firPhase >= upScale) {
        firPhase -= upScale;
        dlyUpdate++;
    }
    i = *dlyIdx - dlyI;
    *firPh = firPhase;
    *dlyUp = dlyUpdate;
    *dlyIdx = dlyI;
    return i;
}

static int CaptureRateDownConvertMono_NEON_I(const int16_t* in, int inSamples, int16_t* out, int dnScale, int channels,
                                             int *dlyUp, int16_t *dlyLine, int16_t *polyDn, int taps)
{
    int i, j;
    int outSamples;
    int dlyUpdate = *dlyUp;;
    int iter = taps>>2;
    int16x4_t dly;
    int32x4_t acc_q;
    int32x2_t acc_d;
    int16x4_t acc;
    int16x4_t c[iter]; // this is hardcoded to c[64] in the assembly version
    int16_t *dlyI = dlyLine;

    if (inSamples == 0) {
        return 0;
    }
    while (dlyUpdate) {
        dlyI --;
        DLYLINE_CHECK_PTR(dlyI);
        *dlyI = *in;
        in += channels;
        dlyUpdate--;
        inSamples--;
    }
    outSamples = inSamples / dnScale + 1;
    for (j=0; j < iter; j++) {
        c[j] = vld1_s16(polyDn + (j<<2));
    }
    for (i = 0; i < outSamples; i++) {
        // Start filter kernel
        DLYLINE_CHECK_PTR(dlyI);
        dly = vld1_s16(dlyI);
        acc_q = vmull_s16(c[0], dly);
        for (j=1; j < iter; j++) {
            DLYLINE_CHECK_PTR(dlyI + (j<<2));
            dly = vld1_s16(dlyI + (j<<2));
            acc_q = vmlal_s16(acc_q, c[j], dly);
        }
        acc_d = vadd_s32(vget_low_s32(acc_q), vget_high_s32(acc_q));
        acc_d = vpadd_s32(acc_d, acc_d);
        acc_q = vcombine_s32(acc_d, acc_d);
        acc = vqrshrn_n_s32(acc_q, 15);
        vst1_lane_s16(out, acc, 0);
        out += channels;
        // Update filter delay line
        for (j = 0; j < dnScale && i + 1 < outSamples; j++) {
            dlyI --;
            DLYLINE_CHECK_PTR(dlyI);
            *dlyI = *in;
            in += channels;
            inSamples--;
        }
    }
    // Consume remaining input
    dlyUpdate = dnScale;
    while (inSamples > 0) {
        dlyI --;
        DLYLINE_CHECK_PTR(dlyI);
        *dlyI = *in;
        in += channels;
        dlyUpdate--;
        inSamples--;
    }

    *dlyUp = dlyUpdate;
    return outSamples;
}

static int CaptureRateDownConvertStereo_NEON_I(const int16_t* in, int inSamples, int16_t* out, int dnScale, int channels,
                                               int *dlyUp, int32_t *dlyLine, int16_t *polyDn, int taps)
{
    int i, j;
    int outSamples;
    int dlyUpdate = *dlyUp;
    int iter = taps>>2;
    int16x4x2_t dly;
    int32x4_t acc_q0, acc_q1;
    int32x2_t acc_d0, acc_d1;
    int16x4_t acc;
    int16x4_t c[iter]; // this is hardcoded to c[64] in the assembly version
    int32_t *dlyIW = dlyLine;
    int32_t *inW = (int32_t*)in;

    if (inSamples == 0) {
        return 0;
    }
    while (dlyUpdate) {
        dlyIW --;
        DLYLINE_CHECK_PTR(dlyIW);
        *dlyIW = *inW;
        inW += channels/2;
        dlyUpdate--;
        inSamples--;
    }
    outSamples = inSamples / dnScale + 1;
    for (j=0; j < iter; j++) {
        c[j] = vld1_s16(polyDn + (j<<2));
    }
    for (i = 0; i < outSamples; i++) {
        DLYLINE_CHECK_PTR(dlyIW);
        dly = vld2_s16((int16_t*)(dlyIW));
        acc_q0 = vmull_s16(c[0], dly.val[0]);
        acc_q1 = vmull_s16(c[0], dly.val[1]);
        for ( j = 1; j < iter; j++ ) {
            DLYLINE_CHECK_PTR(dlyIW + (j<<2));
            dly = vld2_s16((int16_t*)(dlyIW + (j<<2)));
            acc_q0 = vmlal_s16(acc_q0, c[j], dly.val[0]);
            acc_q1 = vmlal_s16(acc_q1, c[j], dly.val[1]);
        }
        acc_d0 = vadd_s32(vget_low_s32(acc_q0), vget_high_s32(acc_q0));
        acc_d1 = vadd_s32(vget_low_s32(acc_q1), vget_high_s32(acc_q1));
        acc_d0 = vpadd_s32(acc_d0, acc_d0);
        acc_d1 = vpadd_s32(acc_d1, acc_d1);

        acc_q0 = vcombine_s32(acc_d0, acc_d1);
        acc = vqrshrn_n_s32(acc_q0, 15);
        vst1_lane_s16(out, acc, 0);
        vst1_lane_s16(out+1, acc, 2);
        out += channels;
        // Update filter delay line
        for (j = 0; j < dnScale && i + 1 < outSamples; j++) {
            dlyIW --;
            DLYLINE_CHECK_PTR(dlyIW);
            *dlyIW = *inW;
            inW += channels/2;
            inSamples--;
        }
    }
    // Consume remaining input
    dlyUpdate = dnScale;
    while (inSamples > 0) {
        dlyIW --;
        DLYLINE_CHECK_PTR(dlyIW);
        *dlyIW = *inW;
        inW += channels/2;
        dlyUpdate--;
        inSamples--;
    }

    *dlyUp = dlyUpdate;
    return outSamples;
}


int CaptureRateMixConvertMono_NEON_I(const int16_t* in, int inSamples, int16_t* out, int upScale, int dnScale, int channels,
                                     int *firPh, int *dlyUp, int16_t *dlyLine, int16_t *poly, int taps)
{
    int i;
    int j;
    int firPhase = *firPh;
    int dlyUpdate = *dlyUp;
    int outSamples;
    int16_t *dlyI = dlyLine;
    int iter = taps>>2;
    int32x4_t acc_q;
    int32x2_t acc_d;
    int16x4_t acc;
    int16_t *poly_base;
    if (inSamples == 0) {
        return 0;
    }
    outSamples = inSamples * upScale / dnScale;
    for (i = 0; i < outSamples; i++) {
        // Update delay line when filter phase wraps
        while (dlyUpdate) {
            dlyI --;
            DLYLINE_CHECK_PTR(dlyI);
            *dlyI = *in;
            in += channels;
            dlyUpdate--;
            inSamples--;
        }
        // Start filter kernel
        DLYLINE_CHECK_PTR(dlyI);
        acc_q = vmull_s16(vld1_s16((poly_base = (poly + firPhase * taps))), vld1_s16(dlyI));
        for (j=1; j < iter; j++) {
            DLYLINE_CHECK_PTR(dlyI + (j<<2));
            acc_q = vmlal_s16(acc_q, vld1_s16(poly_base + (j<<2)), vld1_s16(dlyI + (j<<2)));
        }
        acc_d = vadd_s32(vget_low_s32(acc_q), vget_high_s32(acc_q));
        acc_d = vpadd_s32(acc_d, acc_d);
        acc_q = vcombine_s32(acc_d, acc_d);
        acc = vqrshrn_n_s32(acc_q, 15);
        vst1_lane_s16(out, acc, 0);
        out += channels;
        // Update filter phase
        firPhase += dnScale;
        while (firPhase >= upScale) {
            firPhase -= upScale;
            dlyUpdate++;
        }
    }

    // Consume remaining samples and generate remaining output
    if (inSamples > 0) {
        while (inSamples > 0 && dlyUpdate > 0) {
            dlyI --;
            DLYLINE_CHECK_PTR(dlyI);
            *dlyI = *in;
            in += channels;
            dlyUpdate--;
            inSamples--;
        }

        if (dlyUpdate == 0) {
            // Start filter kernel
            DLYLINE_CHECK_PTR(dlyI);
            acc_q = vmull_s16(vld1_s16((poly_base = (poly + firPhase * taps))), vld1_s16(dlyI));
            for (j=1; j < iter; j++) {
                DLYLINE_CHECK_PTR(dlyI + (j<<2));
                acc_q = vmlal_s16(acc_q, vld1_s16(poly_base + (j<<2)), vld1_s16(dlyI + (j<<2)));
            }
            acc_d = vadd_s32(vget_low_s32(acc_q), vget_high_s32(acc_q));
            acc_d = vpadd_s32(acc_d, acc_d);
            acc_q = vcombine_s32(acc_d, acc_d);
            acc = vqrshrn_n_s32(acc_q, 15);
            vst1_lane_s16(out, acc, 0);
            out += channels;
            outSamples++;

            // Update filter phase
            firPhase += dnScale;
            while (firPhase >= upScale) {
                firPhase -= upScale;
                dlyUpdate++;
            }
        }
    }
    *dlyUp = dlyUpdate;
    *firPh = firPhase;
    return outSamples;
}

int CaptureRateMixConvertStereo_NEON_I(const int16_t* in, int inSamples, int16_t* out, int upScale, int dnScale, int channels,
                                       int *firPh, int *dlyUp, int32_t *dlyLine, int16_t *poly, int taps)
{
    int i;
    int j;
    int firPhase = *firPh;
    int dlyUpdate = *dlyUp;
    int outSamples;
    int32_t *dlyIW = dlyLine;
    int32_t *inW = (int32_t*) in;
    int iter = taps>>2;
    int32x4_t acc_q0, acc_q1;
    int32x2_t acc_d0, acc_d1;
    int16x4_t acc;
    int16x4_t c;
    int16x4x2_t dly;
    int16_t *poly_base;

    if (inSamples == 0) {
        return 0;
    }
    outSamples = inSamples * upScale / dnScale;

    for (i = 0; i < outSamples; i++) {
        // Update delay line when filter phase wraps
        while (dlyUpdate) {
            dlyIW --;
            DLYLINE_CHECK_PTR(dlyIW);
            *dlyIW = *inW;
            inW += channels/2;
            dlyUpdate--;
            inSamples--;
        }
        // Start filter kernel
        c = vld1_s16((poly_base = (poly + firPhase * taps)));
        DLYLINE_CHECK_PTR(dlyIW);
        dly = vld2_s16((int16_t*)(dlyIW));
        acc_q0 = vmull_s16(c, dly.val[0]);
        acc_q1 = vmull_s16(c, dly.val[1]);
        for ( j = 1; j < iter; j++ ) {
            c = vld1_s16(poly_base + (j<<2));
            DLYLINE_CHECK_PTR(dlyIW + (j<<2));
            dly = vld2_s16((int16_t*)(dlyIW + (j<<2)));
            acc_q0 = vmlal_s16(acc_q0, c, dly.val[0]);
            acc_q1 = vmlal_s16(acc_q1, c, dly.val[1]);
        }
        acc_d0 = vadd_s32(vget_low_s32(acc_q0), vget_high_s32(acc_q0));
        acc_d1 = vadd_s32(vget_low_s32(acc_q1), vget_high_s32(acc_q1));
        acc_d0 = vpadd_s32(acc_d0, acc_d0);
        acc_d1 = vpadd_s32(acc_d1, acc_d1);

        acc_q0 = vcombine_s32(acc_d0, acc_d1);
        acc = vqrshrn_n_s32(acc_q0, 15);
        vst1_lane_s16(out, acc, 0);
        vst1_lane_s16(out+1, acc, 2);
        out += channels;

        // Update filter phase
        firPhase += dnScale;
        while (firPhase >= upScale) {
            firPhase -= upScale;
            dlyUpdate++;
        }
    }

    // Consume remaining samples and generate remaining output
    if (inSamples > 0) {
        while (inSamples > 0 && dlyUpdate > 0) {
            dlyIW --;
            DLYLINE_CHECK_PTR(dlyIW);
            *dlyIW = *inW;
            inW += channels/2;
            dlyUpdate--;
            inSamples--;
        }

        if (dlyUpdate == 0) {
            // Start filter kernel
            c = vld1_s16((poly_base = (poly + firPhase * taps)));
            DLYLINE_CHECK_PTR(dlyIW);
            dly = vld2_s16((int16_t*)(dlyIW));
            acc_q0 = vmull_s16(c, dly.val[0]);
            acc_q1 = vmull_s16(c, dly.val[1]);
            for ( j = 1; j < iter; j++ ) {
                c = vld1_s16(poly_base + (j<<2));
                DLYLINE_CHECK_PTR(dlyIW + (j<<2));
                dly = vld2_s16((int16_t*)(dlyIW + (j<<2)));
                acc_q0 = vmlal_s16(acc_q0, c, dly.val[0]);
                acc_q1 = vmlal_s16(acc_q1, c, dly.val[1]);
            }
            acc_d0 = vadd_s32(vget_low_s32(acc_q0), vget_high_s32(acc_q0));
            acc_d1 = vadd_s32(vget_low_s32(acc_q1), vget_high_s32(acc_q1));
            acc_d0 = vpadd_s32(acc_d0, acc_d0);
            acc_d1 = vpadd_s32(acc_d1, acc_d1);

            acc_q0 = vcombine_s32(acc_d0, acc_d1);
            acc = vqrshrn_n_s32(acc_q0, 15);
            vst1_lane_s16(out, acc, 0);
            vst1_lane_s16(out+1, acc, 2);
            out += channels;
            outSamples++;

            // Update filter phase
            firPhase += dnScale;
            while (firPhase >= upScale) {
                firPhase -= upScale;
                dlyUpdate++;
            }
        }
    }
    *dlyUp = dlyUpdate;
    *firPh = firPhase;
    return outSamples;
}

int PlaybackRateMixConvertMono_24t_NEON_I(const int16_t* in, int outSamples, int16_t* out, int upScale, int dnScale, int channels,
                                          int *firPh, int *dlyUp, int16_t *dlyLine, int *dlyIdx, int16_t *poly, int taps)
{
    int i;
    int firPhase = *firPh;
    int dlyUpdate = *dlyUp;
    int dlyI = *dlyIdx;
    int32x4_t acc_q;
    int32x2_t acc_d;
    int16x4_t acc;
    int16x4_t c[6];
    int16x4_t dly[6];
    int16_t *dly_base;
    int16_t *poly_base;
    int dlyChanged;

    if (outSamples == 0) {
        return 0;
    }
    // Update delay line when filter phase wraps
    while (dlyUpdate) {
        dlyI --;
        DLYLINE_CHECK_IDX(dlyI);
        dlyLine[dlyI] = *in;
        in += channels;
        dlyUpdate--;
    }
    dly[0] = vld1_s16((dly_base = (dlyLine + dlyI)));
    dly[1] = vld1_s16(dly_base + 4);
    dly[2] = vld1_s16(dly_base + 8);
    dly[3] = vld1_s16(dly_base + 12);
    dly[4] = vld1_s16(dly_base + 16);
    dly[5] = vld1_s16(dly_base + 20);
    for (i = 0; i < outSamples; i++) {
        DLYLINE_CHECK_PTR(dly_base);
        DLYLINE_CHECK_PTR(dly_base + 23);
        c[0] = vld1_s16((poly_base = (poly + firPhase * 24)));
        c[1] = vld1_s16(poly_base + 4);
        c[2] = vld1_s16(poly_base + 8);
        c[3] = vld1_s16(poly_base + 12);
        c[4] = vld1_s16(poly_base + 16);
        c[5] = vld1_s16(poly_base + 20);
        acc_q = vmull_s16(c[0], dly[0]);
        acc_q = vmlal_s16(acc_q, c[1], dly[1]);
        acc_q = vmlal_s16(acc_q, c[2], dly[2]);
        acc_q = vmlal_s16(acc_q, c[3], dly[3]);
        acc_q = vmlal_s16(acc_q, c[4], dly[4]);
        acc_q = vmlal_s16(acc_q, c[5], dly[5]);
        acc_d = vadd_s32(vget_low_s32(acc_q), vget_high_s32(acc_q));
        acc_d = vpadd_s32(acc_d, acc_d);
        acc_q = vcombine_s32(acc_d, acc_d);
        acc = vqrshrn_n_s32(acc_q, 15);
        vst1_lane_s16(out, acc, 0);
        out += channels;

        // Update filter phase and delay line  when filter phase wraps
        firPhase += dnScale;
        dlyChanged = dlyI;
        while (firPhase >= upScale && i + 1 < outSamples) {
            firPhase -= upScale;
            dlyI --;
            DLYLINE_CHECK_IDX(dlyI);
            dlyLine[dlyI] = *in;
            in += channels;
        }
        if ( dlyI != dlyChanged ) {
            dly[0] = vld1_s16((dly_base = (dlyLine + dlyI)));
            dly[1] = vld1_s16(dly_base + 4);
            dly[2] = vld1_s16(dly_base + 8);
            dly[3] = vld1_s16(dly_base + 12);
            dly[4] = vld1_s16(dly_base + 16);
            dly[5] = vld1_s16(dly_base + 20);
        }
    }
    // Update filter phase
    while (firPhase >= upScale) {
        firPhase -= upScale;
        dlyUpdate++;
    }
    i = *dlyIdx - dlyI;
    *firPh = firPhase;
    *dlyUp = dlyUpdate;
    *dlyIdx = dlyI;
    return i;
}

int PlaybackRateMixConvertStereo_24t_NEON_I(const int16_t* in, int outSamples, int16_t* out, int upScale, int dnScale, int channels,
                                            int *firPh, int *dlyUp, int32_t *dlyLine, int *dlyIdx, int16_t *poly, int taps)
{
    int i;
    int32_t *inW = (int32_t *)in;
    int firPhase = *firPh;
    int dlyUpdate = *dlyUp;
    int dlyI = *dlyIdx;
    int16x4_t c[6];
    int16x4x2_t dly[6];
    int32x4_t acc_q0, acc_q1;
    int32x2_t acc_d0, acc_d1;
    int16x4_t acc;
    int32_t *dly_base;
    int16_t *poly_base;
    int dlyChanged;

    if (outSamples == 0) {
        return 0;
    }
    // Update delay line
    while (dlyUpdate) {
        dlyI --;
        DLYLINE_CHECK_IDX(dlyI);
        dlyLine[dlyI] = *inW;
        inW += channels/2;
        dlyUpdate--;
    }
    dly[0] = vld2_s16((int16_t*)(dly_base = (dlyLine + dlyI)));
    dly[1] = vld2_s16((int16_t*)(dly_base + 4));
    dly[2] = vld2_s16((int16_t*)(dly_base + 8));
    dly[3] = vld2_s16((int16_t*)(dly_base + 12));
    dly[4] = vld2_s16((int16_t*)(dly_base + 16));
    dly[5] = vld2_s16((int16_t*)(dly_base + 20));
    for (i = 0; i < outSamples; i++) {
        DLYLINE_CHECK_PTR(dly_base);
        DLYLINE_CHECK_PTR(dly_base + 23);
        c[0] = vld1_s16((poly_base = (poly + firPhase * 24)));
        c[1] = vld1_s16(poly_base + 4);
        c[2] = vld1_s16(poly_base + 8);
        c[3] = vld1_s16(poly_base + 12);
        c[4] = vld1_s16(poly_base + 16);
        c[5] = vld1_s16(poly_base + 20);
        acc_q0 = vmull_s16(c[0], dly[0].val[0]);
        acc_q1 = vmull_s16(c[0], dly[0].val[1]);
        acc_q0 = vmlal_s16(acc_q0, c[1], dly[1].val[0]);
        acc_q1 = vmlal_s16(acc_q1, c[1], dly[1].val[1]);
        acc_q0 = vmlal_s16(acc_q0, c[2], dly[2].val[0]);
        acc_q1 = vmlal_s16(acc_q1, c[2], dly[2].val[1]);
        acc_q0 = vmlal_s16(acc_q0, c[3], dly[3].val[0]);
        acc_q1 = vmlal_s16(acc_q1, c[3], dly[3].val[1]);
        acc_q0 = vmlal_s16(acc_q0, c[4], dly[4].val[0]);
        acc_q1 = vmlal_s16(acc_q1, c[4], dly[4].val[1]);
        acc_q0 = vmlal_s16(acc_q0, c[5], dly[5].val[0]);
        acc_q1 = vmlal_s16(acc_q1, c[5], dly[5].val[1]);
        acc_d0 = vadd_s32(vget_low_s32(acc_q0), vget_high_s32(acc_q0));
        acc_d1 = vadd_s32(vget_low_s32(acc_q1), vget_high_s32(acc_q1));
        acc_d0 = vpadd_s32(acc_d0, acc_d0);
        acc_d1 = vpadd_s32(acc_d1, acc_d1);

        acc_q0 = vcombine_s32(acc_d0, acc_d1);
        acc = vqrshrn_n_s32(acc_q0, 15);
        vst1_lane_s16(out, acc, 0);
        vst1_lane_s16(out+1, acc, 2);
        out += channels;

        // Update filter phase and delay line when filter phase wraps
        firPhase += dnScale;
        dlyChanged = dlyI;
        while (firPhase >= upScale && i + 1 < outSamples) {
            firPhase -= upScale;
            dlyI --;
            DLYLINE_CHECK_IDX(dlyI);
            dlyLine[dlyI] = *inW;
            inW += channels/2;
        }
        if ( dlyChanged != dlyI ) {
            dly[0] = vld2_s16((int16_t*)(dly_base = (dlyLine + dlyI)));
            dly[1] = vld2_s16((int16_t*)(dly_base + 4));
            dly[2] = vld2_s16((int16_t*)(dly_base + 8));
            dly[3] = vld2_s16((int16_t*)(dly_base + 12));
            dly[4] = vld2_s16((int16_t*)(dly_base + 16));
            dly[5] = vld2_s16((int16_t*)(dly_base + 20));
        }
    }
    // Update filter phase
    while (firPhase >= upScale) {
        firPhase -= upScale;
        dlyUpdate++;
    }
    i = *dlyIdx - dlyI;
    *firPh = firPhase;
    *dlyUp = dlyUpdate;
    *dlyIdx = dlyI;
    return i;
}

#else // X86, Target independent

static int PlaybackRateMixConvertMono(const int16_t* in, int outSamples, int16_t* out, int upScale, int dnScale, int channels,
                                      int *firPh, int *dlyUp, int16_t *dlyLine, int *dlyIdx, int16_t *poly, int taps)
{
    int i;
    int consumed;
    int firPhase;
    int dlyUpdate;
    unsigned int dlyI;

    if (outSamples == 0) {
        return 0;
    }
    firPhase = *firPh;
    dlyUpdate = *dlyUp;
    dlyI = *dlyIdx;

    // Update delay line when filter phase wraps
    while (dlyUpdate) {
        dlyI --;
        DLYLINE_CHECK_IDX(dlyI);
        dlyLine[dlyI] = *in;
        in += channels;
        dlyUpdate--;
    }

    for (i = 0; i < outSamples - 1; i++) {
        // Start filter kernel
        DLYLINE_CHECK_IDX(dlyI);
        DLYLINE_CHECK_IDX(dlyI + taps - 1);
        *out = dot_product_mono(poly + (firPhase*taps), &dlyLine[dlyI], taps, channels);
        out += channels;
        // Update filter phase and delay line  when filter phase wraps
        firPhase += dnScale;
        while (firPhase >= upScale) {
            firPhase -= upScale;
            dlyI --;
            DLYLINE_CHECK_IDX(dlyI);
            dlyLine[dlyI] = *in;
            in += channels;
        }
    }

    // Final sample
    DLYLINE_CHECK_IDX(dlyI);
    DLYLINE_CHECK_IDX(dlyI + taps - 1);
    *out = dot_product_mono(poly + (firPhase*taps), &dlyLine[dlyI], taps, channels);
    out += channels;

    // Update filter phase
    firPhase += dnScale;
    while (firPhase >= upScale) {
        firPhase -= upScale;
        dlyUpdate++;
    }

    consumed = *dlyIdx - dlyI;
    *firPh = firPhase;
    *dlyUp = dlyUpdate;
    *dlyIdx = dlyI;
    return consumed;
}

static int PlaybackRateMixConvertStereo(const int16_t* in, int outSamples, int16_t* out, int upScale, int dnScale, int channels,
                                        int *firPh, int *dlyUp, int32_t *dlyLine, int *dlyIdx, int16_t* poly, int taps)
{
    int i;
    int consumed;
    int32_t *inW;
    int32_t *outW;
    int firPhase;
    int dlyUpdate;
    unsigned int dlyI;

    if (outSamples == 0) {
        return 0;
    }
    firPhase = *firPh;
    dlyUpdate = *dlyUp;
    dlyI = *dlyIdx;
    inW = (int32_t *)in;
    outW = (int32_t *)out;

    // Update delay line
    while (dlyUpdate) {
        dlyI --;
        DLYLINE_CHECK_IDX(dlyI);
        dlyLine[dlyI] = *inW;
        inW += channels/2;
        dlyUpdate--;
    }

    for (i = 0; i < outSamples - 1; i++) {
        // Start filter kernel
        DLYLINE_CHECK_IDX(dlyI);
        DLYLINE_CHECK_IDX(dlyI + (taps-1)*channels/2);
        *outW = dot_product_stereo(poly + (firPhase*taps), &dlyLine[dlyI], taps, channels);
        outW += channels/2;
        // Update filter phase and delay line  when filter phase wraps
        firPhase += dnScale;
        while (firPhase >= upScale) {
            firPhase -= upScale;
            dlyI --;
            DLYLINE_CHECK_IDX(dlyI);
            dlyLine[dlyI] = *inW;
            inW += channels/2;
        }
    }

    // Final sample
    DLYLINE_CHECK_IDX(dlyI);
    DLYLINE_CHECK_IDX(dlyI + taps - 1);
    *outW = dot_product_stereo(poly + (firPhase*taps), &dlyLine[dlyI], taps, channels);
    outW += channels/2;

    // Update filter phase
    firPhase += dnScale;
    while (firPhase >= upScale) {
        firPhase -= upScale;
        dlyUpdate++;
    }

    consumed = *dlyIdx - dlyI;
    *firPh = firPhase;
    *dlyUp = dlyUpdate;
    *dlyIdx = dlyI;
    return consumed;
}

//------------------------------------------------------------------------------
static int CaptureRateMixConvertMono(const int16_t* in, int inSamples, int16_t* out, int upScale, int dnScale, int channels,
                                         int *firPh, int *dlyUp, int16_t *dlyLine, int16_t *poly, int taps)
{
    int i;
    int firPhase;
    int dlyUpdate;
    int outSamples;
    int16_t *dly;

    if (inSamples == 0) {
        return 0;
    }
    dly = dlyLine;
    firPhase = *firPh;
    dlyUpdate = *dlyUp;

    outSamples = inSamples * upScale / dnScale;

    for (i = 0; i < outSamples; i++) {
        // Update delay line when filter phase wraps
        while (dlyUpdate) {
            dly --;
            DLYLINE_CHECK_PTR(dly);
            *dly = *in;
            in += channels;
            dlyUpdate--;
            inSamples--;
        }

        // Start filter kernel
        DLYLINE_CHECK_PTR(dly);
        DLYLINE_CHECK_PTR(dly + taps - 1);
        *out = dot_product_mono(poly + firPhase * taps, dly, taps, channels);
        out += channels;

        // Update filter phase
        firPhase += dnScale;
        while (firPhase >= upScale) {
            firPhase -= upScale;
            dlyUpdate++;
        }
    }

    // Consume remaining samples and generate remaining output
    if (inSamples > 0) {
        while (inSamples > 0 && dlyUpdate > 0) {
            dly --;
            DLYLINE_CHECK_PTR(dly);
            *dly = *in;
            in += channels;
            dlyUpdate--;
            inSamples--;
        }

        if (dlyUpdate == 0) {
            // Start filter kernel
            DLYLINE_CHECK_PTR(dly);
            DLYLINE_CHECK_PTR(dly + taps - 1);
            *out = dot_product_mono(poly + firPhase * taps, dly, taps, channels);
            out += channels;
            outSamples++;

            // Update filter phase
            firPhase += dnScale;
            while (firPhase >= upScale) {
                firPhase -= upScale;
                dlyUpdate++;
            }
        }
    }
    *dlyUp = dlyUpdate;
    *firPh = firPhase;
    return outSamples;
}

static int CaptureRateMixConvertStereo(const int16_t* in, int inSamples, int16_t* out, int upScale, int dnScale, int channels,
                                           int *firPh, int *dlyUp, int32_t *dlyLine, int16_t *poly, int taps)
{
    int i;
    int firPhase;
    int dlyUpdate;
    int outSamples;
    int32_t *inW;
    int32_t *outW;
    int32_t *dlyW;

    if (inSamples == 0) {
        return 0;
    }
    inW = (int32_t *)in;
    outW = (int32_t *)out;
    dlyW = (int32_t *)dlyLine;
    firPhase = *firPh;
    dlyUpdate = *dlyUp;

    outSamples = inSamples * upScale / dnScale;

    for (i = 0; i < outSamples; i++) {
        // Update delay line when filter phase wraps
        while (dlyUpdate) {
            dlyW --;
            DLYLINE_CHECK_PTR(dlyW);
            *dlyW = *inW;
            inW += channels/2;
            dlyUpdate--;
            inSamples--;
        }

        // Start filter kernel
        DLYLINE_CHECK_PTR(dlyW);
        DLYLINE_CHECK_PTR(dlyW + taps - 1);
        *outW = dot_product_stereo(poly + firPhase * taps, dlyW, taps, channels);
        outW += channels/2;

        // Update filter phase
        firPhase += dnScale;
        while (firPhase >= upScale) {
            firPhase -= upScale;
            dlyUpdate++;
        }
    }

    // Consume remaining samples and generate remaining output
    if (inSamples > 0) {
        while (inSamples > 0 && dlyUpdate > 0) {
            dlyW --;
            DLYLINE_CHECK_PTR(dlyW);
            *dlyW = *inW;
            inW += channels/2;
            dlyUpdate--;
            inSamples--;
        }

        if (dlyUpdate == 0) {
            // Start filter kernel
            DLYLINE_CHECK_PTR(dlyW);
            DLYLINE_CHECK_PTR(dlyW + taps - 1);
            *outW = dot_product_stereo(poly + firPhase * taps, dlyW, taps, channels);
            outW += channels/2;
            outSamples++;

            // Update filter phase
            firPhase += dnScale;
            while (firPhase >= upScale) {
                firPhase -= upScale;
                dlyUpdate++;
            }
        }
    }
    *dlyUp = dlyUpdate;
    *firPh = firPhase;
    return outSamples;
}

//------------------------------------------------------------------------------
static int CaptureRateDownConvertMono(const int16_t* in, int inSamples, int16_t* out, int dnScale, int channels,
                                      int *dlyUp, int16_t *dlyLine, int16_t *polyDn, int taps)
{
    int i, j;
    int outSamples;
    int16_t *dly;
    int dlyUpdate;

    if (inSamples == 0) {
        return 0;
    }
    dly = dlyLine;

    dlyUpdate = *dlyUp;
    while (dlyUpdate) {
        dly --;
        DLYLINE_CHECK_PTR(dly);
        *dly = *in;
        in += channels;
        dlyUpdate--;
        inSamples--;
    }

    outSamples = inSamples / dnScale;

    for (i = 0; i < outSamples; i++) {
        // Start filter kernel
        DLYLINE_CHECK_PTR(dly);
        DLYLINE_CHECK_PTR(dly + taps - 1);
        *out = dot_product_mono(polyDn, dly, taps, channels);
        out += channels;

        // Update filter delay line
        for (j = 0; j < dnScale; j++) {
            dly --;
            DLYLINE_CHECK_PTR(dly);
            *dly = *in;
            in += channels;
            inSamples--;
        }
    }

    // Last sample
    DLYLINE_CHECK_PTR(dly);
    DLYLINE_CHECK_PTR(dly + taps - 1);
    *out = dot_product_mono(polyDn, dly, taps, channels);
    out += channels;
    outSamples++;

    // Consume remaining input
    dlyUpdate = dnScale;
    while (inSamples > 0) {
        dly -= channels;
        DLYLINE_CHECK_PTR(dly);
        *dly = *in;
        in += channels;
        dlyUpdate--;
        inSamples--;
    }

    *dlyUp = dlyUpdate;
    return outSamples;
}

static int CaptureRateDownConvertStereo(const int16_t* in, int inSamples, int16_t* out, int dnScale, int channels,
                          int *dlyUp, int32_t *dlyLine, int16_t *polyDn, int taps)
{
    int i, j;
    int outSamples;
    int32_t *inW;
    int32_t *outW;
    int32_t *dlyW;
    int dlyUpdate;

    if (inSamples == 0) {
        return 0;
    }
    inW = (int32_t *)in;
    outW = (int32_t *)out;
    dlyW = (int32_t *)dlyLine;

    dlyUpdate = *dlyUp;
    while (dlyUpdate) {
        dlyW --;
        DLYLINE_CHECK_PTR(dlyW);
        *dlyW = *inW;
        inW += channels/2;
        dlyUpdate--;
        inSamples--;
    }

    outSamples = inSamples / dnScale;

    for (i = 0; i < outSamples; i++) {
        // Start filter kernel
        DLYLINE_CHECK_PTR(dlyW);
        DLYLINE_CHECK_PTR(dlyW + taps - 1);
        *outW = dot_product_stereo(polyDn, dlyW, taps, channels);
        outW += channels/2;

        // Update filter delay line
        for (j = 0; j < dnScale; j++) {
            dlyW --;
            DLYLINE_CHECK_PTR(dlyW);
            *dlyW = *inW;
            inW += channels/2;
            inSamples--;
        }
    }

    // Last sample
    DLYLINE_CHECK_PTR(dlyW);
    DLYLINE_CHECK_PTR(dlyW + taps - 1);
    *outW = dot_product_stereo(polyDn, dlyW, taps, channels);
    outW += channels/2;
    outSamples++;

    // Consume remaining input
    dlyUpdate = dnScale;
    while (inSamples > 0) {
        dlyW --;
        DLYLINE_CHECK_PTR(dlyW);
        *dlyW = *inW;
        inW += channels/2;
        dlyUpdate--;
        inSamples--;
    }

    *dlyUp = dlyUpdate;
    return outSamples;
}
#endif // POLY_NEON_INTRINSICS

//------------------------------------------------------------------------------
int rate_poly_process( void *state, const int16_t* in, uint32_t inSizeBytes, int16_t* out, uint32_t outSizeBytes)
{
    ResampleState_t *s;
    uint32_t inSamples;
    uint32_t outSamples;
#if defined(POLY_ENABLE_PROFILING) && defined(VARIANT_v7)
    uint64_t cycles = ClockCycles();
#endif
    int i;
    int dlyIdx;
    int firPhase;
    int dlyUpdate;

    if (state == NULL) {
        return -EINVAL;
    }

    s = (ResampleState_t *)state;

    if (s->dlyIdx < 0) {
        return -EINVAL;
    }

    inSamples = inSizeBytes / (2 * s->channels);
    outSamples = outSizeBytes / (2 * s->channels);
    s->lastSrcSize = 0;
    s->lastDstSize = 0;

    // Convert
    if (s->playback) {
        if (s->channelOptimization == RATE_POLY_MONO) {
            for (i = 0; i < s->channels; i++ ) {
                dlyIdx = s->dlyIdx;
                firPhase = s->firPhase;
                dlyUpdate = s->dlyUpdate;
                inSamples = s->funcs.playbackRateMixConvertMono( in + i, outSamples, out + i,
                                                                 s->upScale, s->dnScale, s->channels,
                                                                 &firPhase, &dlyUpdate,
                                                                 s->dlyLineM[i], &dlyIdx,
                                                                 s->poly, s->taps );
            }
        } else {
            for (i = 0; i < s->channels/2; i++ ) {
                dlyIdx = s->dlyIdx;
                firPhase = s->firPhase;
                dlyUpdate = s->dlyUpdate;
                inSamples = s->funcs.playbackRateMixConvertStereo( in + i + i, outSamples, out + i + i,
                                                                   s->upScale, s->dnScale, s->channels,
                                                                   &firPhase, &dlyUpdate,
                                                                   s->dlyLineS[i], &dlyIdx,
                                                                   s->poly, s->taps );
            }
        }
        s->dlyIdx = dlyIdx;
        s->firPhase = firPhase;
        s->dlyUpdate = dlyUpdate;
    } else {
        if (s->upScale == 1) {
            if (s->channelOptimization == RATE_POLY_MONO) {
                for (i = 0; i < s->channels; i++ ) {
                    dlyUpdate = s->dlyUpdate;
                    outSamples = s->funcs.captureRateDownConvertMono ( in + i, inSamples, out + i,
                                                                       s->dnScale, s->channels,
                                                                       &dlyUpdate,
                                                                       s->dlyLineM[i] + s->dlyIdx,
                                                                       s->poly, s->taps );
                }
            } else {
                for (i = 0; i < s->channels/2; i++ ) {
                    dlyUpdate = s->dlyUpdate;
                    outSamples = s->funcs.captureRateDownConvertStereo( in + i + i, inSamples, out + i + i,
                                                                        s->dnScale, s->channels,
                                                                        &dlyUpdate,
                                                                        s->dlyLineS[i] + s->dlyIdx,
                                                                        s->poly, s->taps );
                }
            }
            s->dlyUpdate = dlyUpdate;
        } else {
            if (s->channelOptimization == RATE_POLY_MONO) {
                for (i = 0; i < s->channels; i++ ) {
                    dlyUpdate = s->dlyUpdate;
                    firPhase = s->firPhase;
                    outSamples = s->funcs.captureRateMixConvertMono ( in + i, inSamples, out + i,
                                                                      s->upScale, s->dnScale, s->channels,
                                                                      &s->firPhase, &dlyUpdate,
                                                                      s->dlyLineM[i] + s->dlyIdx,
                                                                      s->poly, s->taps );
                }
            } else {
                for (i = 0; i < s->channels/2; i++ ) {
                    dlyUpdate = s->dlyUpdate;
                    firPhase = s->firPhase;
                    outSamples = s->funcs.captureRateMixConvertStereo( in + i + i, inSamples, out + i + i,
                                                                       s->upScale, s->dnScale, s->channels,
                                                                       &s->firPhase, &dlyUpdate,
                                                                       s->dlyLineS[i] + s->dlyIdx,
                                                                       s->poly, s->taps );
                }
            }
            s->dlyUpdate = dlyUpdate;
            s->firPhase = firPhase;
        }
        s->dlyIdx -= inSamples; // we always consume all input samples in the recording case
    }

    // Update delay line
    DLYLINE_CHECK_IDX(s->dlyIdx);
    if (s->channelOptimization == RATE_POLY_MONO) {
        for( i = 0; i < s->channels; i++ ) {
            memmove(s->dlyLineM[i] + s->dlySize - s->taps, s->dlyLineM[i] + s->dlyIdx, 2 * s->taps);
        }
    } else {
        for( i = 0; i < s->channels/2; i++ ) {
            memmove(s->dlyLineS[i] + s->dlySize - s->taps, s->dlyLineS[i] + s->dlyIdx, 4 * s->taps);
        }
    }
    s->dlyIdx = s->dlySize - s->taps;

    s->icount += inSamples;
    s->ocount += outSamples;
    s->lastSrcSize = inSamples * 2 * s->channels;
    s->lastDstSize = outSamples * 2 * s->channels;
#if defined(POLY_ENABLE_PROFILING) && defined(VARIANT_v7)
    cycles = ClockCycles() - cycles;
    s->cycles += cycles;
    s->iter++;
#endif
    return 0;
}
//------------------------------------------------------------------------------
int rate_poly_free(void *state)
{
#if defined(POLY_ENABLE_PROFILING) && defined(VARIANT_v7)
    uint64_t cps, mcs;
#endif
    ResampleState_t *s;
    if (state == NULL) {
        return -EINVAL;
    }
    s = (ResampleState_t *)state;
    if (s->dlyLine) {
        free(s->dlyLine);
        s->dlyLine = NULL;
    }
    if (s->poly) {
        free(s->poly);
        s->poly = NULL;
    }
    s->magic = 0;
#if defined(POLY_ENABLE_PROFILING) && defined(VARIANT_v7)
    cps = SYSPAGE_ENTRY(qtime)->cycles_per_sec;
    mcs = (s->cycles * 10) / (cps / 100000);
    slogf (_SLOGC_AUDIO, _SLOG_INFO, "RATE_POLY Stat: Mcs: %d. Iter: %d. %d -> %d\n",
        (uint32_t)mcs, s->iter, s->icount, s->ocount);
#endif
    return 0;
}

//------------------------------------------------------------------------------
int rate_poly_mem_required(void)
{
    return sizeof(ResampleState_t);
}

//------------------------------------------------------------------------------
int rate_poly_support_idx(uint32_t inRate, uint32_t channels, uint32_t outRate, uint32_t hwFragSize, uint32_t fixedTransferSize)
{
    int i;
    int samples;
    int found = -1;

    for (i = 0; i < CONV_TABLE_SIZE; i++) {
        if (inRate == table[i].inRate && outRate == table[i].outRate) {
            if (!fixedTransferSize) {
                found = i;
                break;
            }
            samples = hwFragSize / (2 * channels);
            if ((samples % table[i].minInOutSamples) == 0) {
                found = i;
                break;
            }
        }
    }
    return found;
}

//------------------------------------------------------------------------------
int rate_poly_reset(void *state, uint32_t hwFragSize)
{
    int n, i;
    int old_src_size;
    ResampleState_t *s;

    s = (ResampleState_t *)state;
    s->firPhase = 0;
    s->dlyUpdate = s->dlyUpdate0;
    s->lastSrcSize = s->lastDstSize = 0;
    old_src_size = s->src_size;

    // Delay line indexes
    if (s->playback) {
        s->dst_size = hwFragSize;
        s->src_size = (int32_t)( (int64_t)s->dst_size * s->dnScale / s->upScale);
        // playback - constant dst_size, variable src_size
        while (s->src_size * s->upScale < s->dst_size * s->dnScale) {
            s->src_size++;
        }
        while (s->src_size % (2 * s->channels) != 0) {
            s->src_size++;
        }
    }
    else {
        // record - constant src_size, variable dst_size
        s->src_size = hwFragSize;
        s->dst_size = (int32_t)( (int64_t)s->src_size * s->upScale / s->dnScale);
        while (s->dst_size * s->dnScale < s->src_size * s->upScale) {
            s->dst_size++;
        }
        while (s->dst_size % (2 * s->channels) != 0) {
            s->dst_size++;
        }
    }

    s->dlySize = s->src_size/(2*s->channels);
    s->dlySize += s->taps;
    s->dlyIdx = s->dlySize - s->taps;
    n = s->dlySize * 2 * s->channels;

    if (s->dlyLine != NULL && old_src_size != s->src_size ) {
        free(s->dlyLine);
        s->dlyLine = NULL;
    }
    if (s->dlyLine == NULL) {
        s->dlyLine = malloc(n);
        if (s->dlyLine == NULL) {
            return -ENOMEM;
        }
    }
    memset(s->dlyLine, 0, n);
    memset(s->dlyLineM, 0, sizeof(s->dlyLineM));
    memset(s->dlyLineS, 0, sizeof(s->dlyLineS));

    if( s->channelOptimization == RATE_POLY_MONO ) {
        s->dlyLineM[0] = (int16_t*) s->dlyLine;
        for( i = 1; i < s->channels; i++ ) {
            s->dlyLineM[i] = s->dlyLineM[i-1] + s->dlySize;
        }
    } else {
        s->dlyLineS[0] = (int32_t*) s->dlyLine;
        for( i = 1; i < s->channels/2; i++ ) {
            s->dlyLineS[i] = s->dlyLineS[i-1] + s->dlySize;
        }
    }

    DLYLINE_SET_INFO( s->dlyLine, n/2, s->channels );

    return 0;
}

//------------------------------------------------------------------------------
int rate_poly_flush(void *state)
{
    int n;
    ResampleState_t *s;
    if (state == NULL) {
        return -EINVAL;
    }
    s = (ResampleState_t *)state;
    s->firPhase = 0;
    s->dlyUpdate = s->dlyUpdate0;
    s->lastSrcSize = s->lastDstSize = 0;
    s->dlyIdx = s->dlySize - s->taps;
    n =  s->dlySize * 2 * s->channels;
    memset(s->dlyLine, 0, n);

    return 0;
}

//------------------------------------------------------------------------------
int rate_poly_create(void *state, uint32_t inRate, uint32_t channels, uint32_t outRate, uint32_t hwFragSize, uint32_t fixedSizes, int playback)
{
    int i, j, n, n0, off, nmax;
    ResampleState_t *s;

    if (state == NULL) {
        return -EINVAL;
    }
    if (channels == 0 || channels > 8) {
        return -EINVAL;
    }
    if (inRate == outRate) {
        return -EINVAL;
    }
    i = rate_poly_support_idx(inRate, channels, outRate, hwFragSize, fixedSizes);
    if ( i < 0 ) {
        return -EINVAL;
    }

    s = (ResampleState_t *)state;
    memset(s, 0, sizeof(ResampleState_t));

    s->magic = MAGIC_POLY;
    s->playback = playback;
    s->inRate = inRate;
    s->outRate = outRate;
    s->upScale = table[i].upScale;
    s->dnScale = table[i].dnScale;
    s->taps = table[i].taps;
    s->coefs = table[i].coefs;
    s->channels = channels;
    s->firPhase = 0;
    if (s->upScale > s->dnScale) {
        s->dlyUpdate0 = 1;
    } else {
        s->dlyUpdate0 = (s->dnScale + s->upScale - 1) / s->upScale;
    }
    s->dlyUpdate = s->dlyUpdate0;
    s->dlySize = 0;
    s->dlyLine = NULL;
    s->src_size = s->dst_size = 0;
    s->lastSrcSize = s->lastDstSize = 0;

    // Allocate filter bank
    s->poly = NULL;

    // Prepare filter banks
    if (s->upScale == 1) {
        // pure downconversion
        s->taps *= s->dnScale;
        s->poly = malloc(s->taps * sizeof(int16_t));
        if (s->poly == NULL) {
            return -ENOMEM;
        }
        memset(s->poly, 0, s->taps * sizeof(int16_t));
        nmax = s->taps;
        for (j = 0; j < s->taps; j++) {
            n0 = (j < nmax/2) ? j : (nmax - j - 1);
            s->poly[j] = s->coefs[n0];
        }
    }
    else {
        // mix conversion
        s->poly = malloc(s->upScale * s->taps * sizeof(int16_t));
        if (s->poly == NULL) {
            return -ENOMEM;
        }
        memset(s->poly, 0, s->upScale * s->taps * sizeof(int16_t));
        nmax = s->upScale * s->taps;
        for (i = 0; i < s->upScale; i++) {
            n = i;
            off = i * s->taps;
            for (j = 0; j < s->taps; j++) {
                n0 = (n < nmax/2) ? n : (nmax - n - 1);
                s->poly[off + j] = s->coefs[n0];
                n += s->upScale;
            }
        }
    }

#if defined(POLY_NEON_INTRINSICS)
    if ( s->taps == 24 ) {
       s->funcs.playbackRateMixConvertMono = PlaybackRateMixConvertMono_24t_NEON_I;
       s->funcs.playbackRateMixConvertStereo = PlaybackRateMixConvertStereo_24t_NEON_I;
       s->conversionFunctionVersion = RATE_POLY_NEON_I_24T_CONV_FUNC;
    } else {
       s->funcs.playbackRateMixConvertMono = PlaybackRateMixConvertMono_NEON_I;
       s->funcs.playbackRateMixConvertStereo = PlaybackRateMixConvertStereo_NEON_I;
       s->conversionFunctionVersion = RATE_POLY_NEON_I_CONV_FUNC;
    }
    s->funcs.captureRateMixConvertMono = CaptureRateMixConvertMono_NEON_I;
    s->funcs.captureRateMixConvertStereo = CaptureRateMixConvertStereo_NEON_I;
    s->funcs.captureRateDownConvertMono = CaptureRateDownConvertMono_NEON_I;
    s->funcs.captureRateDownConvertStereo = CaptureRateDownConvertStereo_NEON_I;
#else
    s->funcs.playbackRateMixConvertMono = PlaybackRateMixConvertMono;
    s->funcs.playbackRateMixConvertStereo = PlaybackRateMixConvertStereo;
    s->funcs.captureRateMixConvertMono = CaptureRateMixConvertMono;
    s->funcs.captureRateMixConvertStereo = CaptureRateMixConvertStereo;
    s->funcs.captureRateDownConvertMono = CaptureRateDownConvertMono;
    s->funcs.captureRateDownConvertStereo = CaptureRateDownConvertStereo;
    s->conversionFunctionVersion = RATE_POLY_GENERIC_CONV_FUNC;
#endif // POLY_NEON_INTRINSICS

    s->channelOptimization = RATE_POLY_MONO;
#ifdef POLY_ENABLE_STEREO_OPTIMIZATION
    if((s->channels % 2) == 0) {
        s->channelOptimization = RATE_POLY_STEREO;
    }
#endif

#if defined(POLY_ENABLE_PROFILING)
    s->cycles = 0;
    s->iter = 0;
#endif

    return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/conversion/rate_poly_core.c $ $Rev: 804396 $")
#endif
