/*
 *  Mixer Interface - main file
 *  Copyright (c) 1998 by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include "asoundlib.h"
#include <devctl.h>
#include <pthread.h>
#include "pcm_local.h"
#include "mixer_local.h"



#define __USE_GNU
#include <search.h>

struct snd_mixer
{
	int	 fd;
};

int
snd_mixer_open_name (snd_mixer_t **handle, const char *filename)
{
	int	 fd, ver;
	snd_mixer_t *mixer;

	*handle = NULL;

	if (!filename || !filename[0])
		return -EINVAL;

	if ((fd = open (filename, O_RDWR | O_CLOEXEC)) < 0)
	{
		if ((fd = open (filename, O_RDWR | O_CLOEXEC)) < 0)
			return -errno;
	}
	if (ioctl (fd, SND_MIXER_IOCTL_PVERSION, &ver) < 0)
	{
		close (fd);
		return -errno;
	}
	if (SND_PROTOCOL_INCOMPATIBLE (ver, SND_MIXER_VERSION))
	{
		close (fd);
		return -SND_ERROR_INCOMPATIBLE_VERSION;
	}
	mixer = (snd_mixer_t *) calloc (1, sizeof (snd_mixer_t));
	if (mixer == NULL)
	{
		close (fd);
		return -ENOMEM;
	}
	mixer->fd = fd;

	*handle = mixer;
	return 0;
}

int
snd_mixer_open (snd_mixer_t ** handle, int card, int device)
{
	char	filename[32];

	if (card < 0 || card >= INT_MAX)
		return -EINVAL;
	snprintf (filename, sizeof(filename), SND_FILE_MIXER, card, device);
	return (snd_mixer_open_name (handle, filename));
}

int
snd_mixer_open_pcm (snd_mixer_t ** handle,  snd_pcm_t *pcm)
{
	int rtn;
	char path[_POSIX_PATH_MAX];
	snd_pcm_channel_setup_t setup;
	snd_mixer_gid_t gid;

	memset(&setup, 0, sizeof(setup));
	memset(&gid, 0, sizeof(gid));
	setup.channel = (pcm->mode & SND_PCM_OPEN_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE;
	setup.mixer_gid = &gid;
	if( (rtn = snd_pcm_channel_setup(pcm, &setup)) != EOK ) {
		return rtn;
	}

	snprintf (path, sizeof(path), SND_FILE_MIXER, setup.mixer_card, setup.mixer_device);
	return (snd_mixer_open_name (handle, path));
}

int
snd_mixer_close (snd_mixer_t * handle)
{
	snd_mixer_t *mixer;
	int	 res;

	mixer = handle;
	if (!mixer)
		return -EINVAL;

	res = close (mixer->fd) < 0 ? -errno : EOK;
	free (mixer);
	return res;
}

int
snd_mixer_file_descriptor (snd_mixer_t * mixer)
{
	if (!mixer)
		return -EINVAL;
	return mixer->fd;
}

int
snd_mixer_info (snd_mixer_t * mixer, snd_mixer_info_t * info)
{
	if (!mixer || !info)
		return -EINVAL;
	if (ioctl (mixer->fd, SND_MIXER_IOCTL_INFO, info) < 0)
		return -errno;
	return EOK;
}

int
snd_mixer_elements (snd_mixer_t * mixer, snd_mixer_elements_t * elements)
{
	iov_t   iov[2];

	if (!mixer || !elements)
		return -EINVAL;
	SETIOV (&iov[0], elements, sizeof (snd_mixer_elements_t));
	SETIOV (&iov[1], elements->pelements, elements->elements_size * sizeof (snd_mixer_eid_t));
	return (-devctlv (mixer->fd, SND_MIXER_IOCTL_ELEMENTS, 2, 2, iov, iov, NULL));
}

int
snd_mixer_routes (snd_mixer_t * mixer, snd_mixer_routes_t * routes)
{
	iov_t   iov[2];

	if (!mixer || !routes)
		return -EINVAL;
	SETIOV (&iov[0], routes, sizeof (snd_mixer_routes_t));
	SETIOV (&iov[1], routes->proutes, routes->routes_size * sizeof (snd_mixer_eid_t));
	return (-devctlv (mixer->fd, SND_MIXER_IOCTL_ROUTES, 2, 2, iov, iov, NULL));
}

int
snd_mixer_groups (snd_mixer_t * mixer, snd_mixer_groups_t * groups)
{
	iov_t   iov[2];

	if (!mixer || !groups)
		return -EINVAL;
	SETIOV (&iov[0], groups, sizeof (snd_mixer_groups_t));
	SETIOV (&iov[1], groups->pgroups, groups->groups_size * sizeof (snd_mixer_gid_t));
	return (-devctlv (mixer->fd, SND_MIXER_IOCTL_GROUPS, 2, 2, iov, iov, NULL));
}

int
snd_mixer_group_read (snd_mixer_t * mixer, snd_mixer_group_t * group)
{
	iov_t   iov[2];

	if (!mixer || !group)
		return -EINVAL;
	SETIOV (&iov[0], group, sizeof (snd_mixer_group_t));
	SETIOV (&iov[1], group->pelements, group->elements_size * sizeof (snd_mixer_eid_t));
	return (-devctlv (mixer->fd, SND_MIXER_IOCTL_GROUP_READ, 2, 2, iov, iov, NULL));
}

int
snd_mixer_group_write (snd_mixer_t * mixer, snd_mixer_group_t * group)
{
	if (!mixer || !group)
		return -EINVAL;

	if (ioctl (mixer->fd, SND_MIXER_IOCTL_GROUP_WRITE, group) < 0)
		return -errno;
	return EOK;
}

int
snd_mixer_element_info (snd_mixer_t * mixer, snd_mixer_element_info_t * info)
{
	iov_t   iov[3];

	if (!mixer || !info)
		return -EINVAL;
	SETIOV (&iov[0], info, sizeof (snd_mixer_element_info_t));
	SETIOV (&iov[1], NULL, 0);
	SETIOV (&iov[2], NULL, 0);
	switch (info->eid.type)
	{
	case SND_MIXER_ETYPE_INPUT:
	case SND_MIXER_ETYPE_OUTPUT:
		SETIOV (&iov[1], info->data.io.pvoices,
			info->data.io.voices_size * sizeof (snd_mixer_voice_t));
		break;
	case SND_MIXER_ETYPE_CAPTURE1:
	case SND_MIXER_ETYPE_PLAYBACK1:
		SETIOV (&iov[1], info->data.pcm1.pdevices, info->data.pcm1.devices_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_CAPTURE2:
	case SND_MIXER_ETYPE_PLAYBACK2:
		break;
	case SND_MIXER_ETYPE_ADC:
	case SND_MIXER_ETYPE_DAC:
		break;
	case SND_MIXER_ETYPE_SWITCH1:
	case SND_MIXER_ETYPE_SWITCH2:
		break;
	case SND_MIXER_ETYPE_SWITCH3:
		SETIOV (&iov[1], info->data.switch3.pvoices,
			info->data.switch3.voices_size * sizeof (snd_mixer_voice_t));
		break;
	case SND_MIXER_ETYPE_VOLUME1:
		SETIOV (&iov[1], info->data.volume1.prange,
			info->data.volume1.range_size * sizeof (struct snd_mixer_element_volume1_range));
		break;
	case SND_MIXER_ETYPE_VOLUME2:
		SETIOV (&iov[1], info->data.volume2.psvoices,
			info->data.volume2.svoices_size * sizeof (snd_mixer_voice_t));
		SETIOV (&iov[2], info->data.volume2.prange,
			info->data.volume2.range_size * sizeof (struct snd_mixer_element_volume2_range));
		break;
	case SND_MIXER_ETYPE_ACCU1:
	case SND_MIXER_ETYPE_ACCU2:
		break;
	case SND_MIXER_ETYPE_ACCU3:
		SETIOV (&iov[1], info->data.accu3.prange,
			info->data.accu3.range_size * sizeof (struct snd_mixer_element_accu3_range));
		break;
	case SND_MIXER_ETYPE_MUX1:
	case SND_MIXER_ETYPE_MUX2:
		break;
	case SND_MIXER_ETYPE_TONE_CONTROL1:
	case SND_MIXER_ETYPE_EQUALIZER1:
		break;
	case SND_MIXER_ETYPE_PAN_CONTROL1:
		SETIOV (&iov[1], info->data.pc1.prange,
			info->data.pc1.range_size * sizeof (struct snd_mixer_element_pan_control1_range));
		break;
	case SND_MIXER_ETYPE_3D_EFFECT1:
		break;
	case SND_MIXER_ETYPE_PRE_EFFECT1:
		SETIOV (&iov[1], info->data.peffect1.pitems,
			info->data.peffect1.items_size *
			sizeof (struct snd_mixer_element_pre_effect1_info_item));
		SETIOV (&iov[2], info->data.peffect1.pparameters,
			info->data.peffect1.parameters_size *
			sizeof (struct snd_mixer_element_pre_effect1_info_parameter));
		break;
	default:
		return -EINVAL;
	}
	return (-devctlv (mixer->fd, SND_MIXER_IOCTL_ELEMENT_INFO, 3, 3, iov, iov, NULL));
}

int
snd_mixer_element_read (snd_mixer_t * mixer, snd_mixer_element_t * element)
{
	iov_t   iov[2];

	if (!mixer || !element)
		return -EINVAL;
	SETIOV (&iov[0], element, sizeof (snd_mixer_element_t));
	SETIOV (&iov[1], NULL, 0);
	switch (element->eid.type)
	{
	case SND_MIXER_ETYPE_INPUT:
	case SND_MIXER_ETYPE_OUTPUT:
		break;
	case SND_MIXER_ETYPE_CAPTURE1:
	case SND_MIXER_ETYPE_PLAYBACK1:
		break;
	case SND_MIXER_ETYPE_CAPTURE2:
	case SND_MIXER_ETYPE_PLAYBACK2:
		break;
	case SND_MIXER_ETYPE_ADC:
	case SND_MIXER_ETYPE_DAC:
		break;
	case SND_MIXER_ETYPE_SWITCH1:
		SETIOV (&iov[1], element->data.switch1.psw,
			element->data.switch1.sw_size * sizeof (unsigned int));
		break;
	case SND_MIXER_ETYPE_SWITCH2:
		break;
	case SND_MIXER_ETYPE_SWITCH3:
		SETIOV (&iov[1], element->data.switch3.prsw,
			element->data.switch3.rsw_size * sizeof (unsigned int));
		break;
	case SND_MIXER_ETYPE_VOLUME1:
		SETIOV (&iov[1], element->data.volume1.pvoices,
			element->data.volume1.voices_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_VOLUME2:
		SETIOV (&iov[1], element->data.volume2.pavoices,
			element->data.volume2.avoices_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_ACCU1:
	case SND_MIXER_ETYPE_ACCU2:
		break;
	case SND_MIXER_ETYPE_ACCU3:
		SETIOV (&iov[1], element->data.accu3.pvoices,
			element->data.accu3.voices_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_MUX1:
		SETIOV (&iov[1], element->data.mux1.poutput,
			element->data.mux1.output_size * sizeof (snd_mixer_eid_t));
		break;
	case SND_MIXER_ETYPE_MUX2:
		break;
	case SND_MIXER_ETYPE_TONE_CONTROL1:
	case SND_MIXER_ETYPE_EQUALIZER1:
		break;
	case SND_MIXER_ETYPE_PAN_CONTROL1:
		SETIOV (&iov[1], element->data.pc1.ppan, element->data.pc1.pan_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_3D_EFFECT1:
		break;
	case SND_MIXER_ETYPE_PRE_EFFECT1:
		SETIOV (&iov[1], element->data.peffect1.pparameters,
			element->data.peffect1.parameters_size * sizeof (int));
		break;
	default:
		return -EINVAL;
	}
	return (-devctlv (mixer->fd, SND_MIXER_IOCTL_ELEMENT_READ, 2, 2, iov, iov, NULL));
}

int
snd_mixer_element_write (snd_mixer_t * mixer, snd_mixer_element_t * element)
{
	iov_t   iov[2];

	if (!mixer || !element)
		return -EINVAL;
	SETIOV (&iov[0], element, sizeof (snd_mixer_element_t));
	SETIOV (&iov[1], NULL, 0);
	switch (element->eid.type)
	{
	case SND_MIXER_ETYPE_INPUT:
	case SND_MIXER_ETYPE_OUTPUT:
		break;
	case SND_MIXER_ETYPE_CAPTURE1:
	case SND_MIXER_ETYPE_PLAYBACK1:
		break;
	case SND_MIXER_ETYPE_CAPTURE2:
	case SND_MIXER_ETYPE_PLAYBACK2:
		break;
	case SND_MIXER_ETYPE_ADC:
	case SND_MIXER_ETYPE_DAC:
		break;
	case SND_MIXER_ETYPE_SWITCH1:
		SETIOV (&iov[1], element->data.switch1.psw,
			element->data.switch1.sw_size * sizeof (unsigned int));
		break;
	case SND_MIXER_ETYPE_SWITCH2:
		break;
	case SND_MIXER_ETYPE_SWITCH3:
		SETIOV (&iov[1], element->data.switch3.prsw,
			element->data.switch3.rsw_size * sizeof (unsigned int));
		break;
	case SND_MIXER_ETYPE_VOLUME1:
		SETIOV (&iov[1], element->data.volume1.pvoices,
			element->data.volume1.voices_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_VOLUME2:
		SETIOV (&iov[1], element->data.volume2.pavoices,
			element->data.volume2.avoices_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_ACCU1:
	case SND_MIXER_ETYPE_ACCU2:
		break;
	case SND_MIXER_ETYPE_ACCU3:
		SETIOV (&iov[1], element->data.accu3.pvoices,
			element->data.accu3.voices_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_MUX1:
		SETIOV (&iov[1], element->data.mux1.poutput,
			element->data.mux1.output_size * sizeof (snd_mixer_eid_t));
		break;
	case SND_MIXER_ETYPE_MUX2:
		break;
	case SND_MIXER_ETYPE_TONE_CONTROL1:
	case SND_MIXER_ETYPE_EQUALIZER1:
		break;
	case SND_MIXER_ETYPE_PAN_CONTROL1:
		SETIOV (&iov[1], element->data.pc1.ppan, element->data.pc1.pan_size * sizeof (int));
		break;
	case SND_MIXER_ETYPE_3D_EFFECT1:
		break;
	case SND_MIXER_ETYPE_PRE_EFFECT1:
		SETIOV (&iov[1], element->data.peffect1.pparameters,
			element->data.peffect1.parameters_size * sizeof (int));
		break;
	default:
		return -EINVAL;
	}
	return (-devctlv (mixer->fd, SND_MIXER_IOCTL_ELEMENT_WRITE, 2, 2, iov, iov, NULL));
}

int
snd_mixer_get_filter (snd_mixer_t * mixer, snd_mixer_filter_t * filter)
{
	if (!mixer || !filter)
		return -EINVAL;
	if (ioctl (mixer->fd, SND_MIXER_IOCTL_GET_FILTER, filter) < 0)
		return -errno;
	return EOK;
}

int
snd_mixer_set_filter (snd_mixer_t * mixer, snd_mixer_filter_t * filter)
{
	if (!mixer || !filter)
		return -EINVAL;
	if (ioctl (mixer->fd, SND_MIXER_IOCTL_SET_FILTER, filter) < 0)
		return -errno;
	return EOK;
}

int
snd_mixer_read (snd_mixer_t * mixer, snd_mixer_callbacks_t * callbacks)
{
	int	 result, count;
	snd_mixer_read_t r;

	if (!mixer)
		return -EINVAL;
	count = 0;
	while ((result = read (mixer->fd, &r, sizeof (r))) > 0)
	{
		if (result != sizeof (r))
			return -EIO;
		if (!callbacks)
			continue;
		switch (r.cmd)
		{
		case SND_MIXER_READ_REBUILD:
			if (callbacks->rebuild)
				callbacks->rebuild (callbacks->private_data);
			break;
		case SND_MIXER_READ_ELEMENT_VALUE:
		case SND_MIXER_READ_ELEMENT_CHANGE:
		case SND_MIXER_READ_ELEMENT_ROUTE:
		case SND_MIXER_READ_ELEMENT_ADD:
		case SND_MIXER_READ_ELEMENT_REMOVE:
			if (callbacks->element)
				callbacks->element (callbacks->private_data, r.cmd, &r.data.eid);
			break;
		case SND_MIXER_READ_GROUP_VALUE:
		case SND_MIXER_READ_GROUP_CHANGE:
		case SND_MIXER_READ_GROUP_ADD:
		case SND_MIXER_READ_GROUP_REMOVE:
			if (callbacks->group)
				callbacks->group (callbacks->private_data, r.cmd, &r.data.gid);
			break;
		}
		count++;
	}
	return result >= 0 ? count : -errno;
}

void
snd_mixer_set_bit (unsigned int *bitmap, int bit, int val)
{
	if (val)
	{
		bitmap[bit >> 5] |= 1 << (bit & 31);
	}
	else
	{
		bitmap[bit >> 5] &= ~(1 << (bit & 31));
	}
}

int
snd_mixer_get_bit (unsigned int *bitmap, int bit)
{
	return (bitmap[bit >> 5] & (1 << (bit & 31))) ? 1 : 0;
}

const char *
snd_mixer_channel_name (int channel)
{
	static char *array[6] = {
		"Front-Left",
		"Front-Right",
		"Front-Center",
		"Rear-Left",
		"Rear-Right",
		"Woofer"
	};

	if (channel < 0 || channel > 5)
		return "?";
	return array[channel];
}




static int
snd_mixer_compare_gid_table (const void *p1, const void *p2)
{
	const snd_mixer_gid_t *gid1 = (snd_mixer_gid_t *) p1;
	const snd_mixer_gid_t *gid2 = (snd_mixer_gid_t *) p2;
	int	 r;

	r = gid1->weight - gid2->weight;
	if (r != 0)
		return r;
	r = strcmp (gid1->name, gid2->name);
	if (r != 0)
		return r;
	return gid1->index - gid2->index;
}

static void
snd_mixer_sort_gid (snd_mixer_gid_t * list, int count, snd_mixer_weight_entry_t * table)
{
	int	 k, t;

	for (k = 0; k < count; ++k)
	{
		for (list[k].weight = 0, t = 0; table != NULL && table[t].name != NULL; t++)
			if (strcmp (list[k].name, table[t].name) == 0)
				list[k].weight = table[t].weight;
	}

	qsort (list, count, sizeof (snd_mixer_gid_t), snd_mixer_compare_gid_table);
}

void
snd_mixer_sort_gid_name_index (snd_mixer_gid_t * list, int count)
{
	snd_mixer_sort_gid (list, count, NULL);
}

void
snd_mixer_sort_gid_table (snd_mixer_gid_t * list, int count, snd_mixer_weight_entry_t * table)
{
	snd_mixer_sort_gid (list, count, table);
}



static int
snd_mixer_compare_eid_table (const void *p1, const void *p2)
{
	const snd_mixer_eid_t *eid1 = (snd_mixer_eid_t *) p1;
	const snd_mixer_eid_t *eid2 = (snd_mixer_eid_t *) p2;
	int	 r;

	r = eid1->weight - eid2->weight;
	if (r != 0)
		return r;
	r = strcmp (eid1->name, eid2->name);
	if (r != 0)
		return r;
	return eid1->index - eid2->index;
}

static void
snd_mixer_sort_eid (snd_mixer_eid_t * list, int count, snd_mixer_weight_entry_t * table)
{
	int	 k, t;

	for (k = 0; k < count; ++k)
	{
		for (list[k].weight = 0, t = 0; table != NULL && table[t].name != NULL; t++)
			if (strcmp (list[k].name, table[t].name) == 0)
				list[k].weight = table[t].weight;
	}

	qsort (list, count, sizeof (snd_mixer_eid_t), snd_mixer_compare_eid_table);
}

void
snd_mixer_sort_eid_name_index (snd_mixer_eid_t * list, int count)
{
	snd_mixer_sort_eid (list, count, NULL);
}

void
snd_mixer_sort_eid_table (snd_mixer_eid_t * list, int count, snd_mixer_weight_entry_t * table)
{
	snd_mixer_sort_eid (list, count, table);
}



static snd_mixer_weight_entry_t _snd_mixer_default_weights[] = {

	{SND_MIXER_MASTER_OUT, -5070},
	{SND_MIXER_MASTER_DIGITAL_OUT, -5060},
	{SND_MIXER_MASTER_MONO_OUT, -5050},
	{SND_MIXER_HEADPHONE_OUT, -5040},
	{SND_MIXER_GRP_OGAIN, -5030},
	{SND_MIXER_GRP_IGAIN, -5020},
	{SND_MIXER_GRP_EFFECT_3D, -5010},
	{SND_MIXER_GRP_BASS, -5000},
	{SND_MIXER_GRP_TREBLE, -4090},
	{SND_MIXER_GRP_EQUALIZER, -4080},
	{SND_MIXER_GRP_FADER, -4070},
	{SND_MIXER_CENTER_OUT, -4060},
	{SND_MIXER_FRONT_OUT, -4050},
	{SND_MIXER_REAR_OUT, -4040},
	{SND_MIXER_SURROUND_OUT, -4030},
	{SND_MIXER_WOOFER_OUT, -4020},
	{SND_MIXER_GRP_EFFECT, -4010},
	{SND_MIXER_DSP_OUT, -4000},
	{SND_MIXER_DSP_IN, -3090},
	{SND_MIXER_PCM_OUT, -3080},
	{SND_MIXER_PCM_IN, -3070},
	{SND_MIXER_DAC_OUT, -3060},
	{SND_MIXER_ADC_IN, -3050},
	{SND_MIXER_LINE_OUT, -3040},
	{SND_MIXER_LINE_IN, -3030},
	{SND_MIXER_SPDIF_OUT, -3020},
	{SND_MIXER_SPDIF_IN, -3010},
	{SND_MIXER_GRP_MIC_GAIN, -3000},
	{SND_MIXER_MIC_OUT, -2090},
	{SND_MIXER_MIC_IN, -2080},
	{SND_MIXER_CD_OUT, -2070},
	{SND_MIXER_CD_IN, -2060},
	{SND_MIXER_VIDEO_OUT, -2050},
	{SND_MIXER_VIDEO_IN, -2040},
	{SND_MIXER_RADIO_OUT, -2030},
	{SND_MIXER_RADIO_IN, -2020},
	{SND_MIXER_PHONE_OUT, -2010},
	{SND_MIXER_PHONE_IN, -2000},
	{SND_MIXER_AUX_OUT, -1090},
	{SND_MIXER_AUX_IN, -1080},
	{SND_MIXER_SYNTHESIZER_OUT, -1070},
	{SND_MIXER_SYNTHESIZER_IN, -1060},
	{SND_MIXER_FM_OUT, -1050},
	{SND_MIXER_FM_IN, -1040},
	{SND_MIXER_GRP_ANALOG_LOOPBACK, -1030},
	{SND_MIXER_GRP_DIGITAL_LOOPBACK, -1020},
	{SND_MIXER_SPEAKER_OUT, -1010},
	{SND_MIXER_MONO_OUT, -1000},

	{SND_MIXER_PCM_OUT_SUBCHN, -20},
	{SND_MIXER_PCM_IN_SUBCHN, -10},
	{NULL, 0}
};

snd_mixer_weight_entry_t *snd_mixer_default_weights = _snd_mixer_default_weights;

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/mixer/mixer.c $ $Rev: 812869 $")
#endif
