/*
 *  PCM Interface - main file
 *  Copyright (c) 1998 by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/pathmgr.h>
#include <sys/poll.h>
#include <pthread.h>
#include <libgen.h>
#include <sys/utsname.h>
#include "pcm_local.h"
#include <stdbool.h>
#include <devctl.h>
#include <sys/slog2.h>
#include <gulliver.h>

// Define to log internally for cases where the app doesn't register a slog buffer
//#define INTERNAL_SLOG_BUFFER

static pthread_mutex_t _link_mutex = PTHREAD_MUTEX_INITIALIZER;

#ifdef INTERNAL_SLOG_BUFFER
static bool slog_initialized = false;
static slog2_buffer_t slog_handle;
static slog2_buffer_set_config_t slog_config;
#define TRY_START_SLOG                                          \
{                                                               \
    if(!slog_initialized) {                                     \
        slog_initialized = true;                                \
        memset(&slog_config, 0, sizeof(slog_config));           \
        slog_config.num_buffers = 1;                            \
        slog_config.verbosity_level = SLOG2_INFO;               \
        slog_config.buffer_set_name = "libasound";              \
        slog_config.buffer_config[0].buffer_name="libasound";   \
        slog_config.buffer_config[0].num_pages = 5;             \
        slog2_register(&slog_config, &slog_handle, 0);          \
        slog2_set_verbosity(slog_handle, SLOG2_DEBUG2);         \
        slog2_set_default_buffer(slog_handle);                  \
    }                                                           \
}
#else
#define TRY_START_SLOG
#endif

int
open_pcm_log (snd_pcm_t *pcm, snd_pcm_channel_params_t *params)
{
    char *str;
    char filename_buf[_POSIX_PATH_MAX];
    int mSampleChannels = 0;
    int mSampleRate = 0;
    int mSampleBits = 0;
    bool bigEndian = false;
    struct
    {
        char     riff_id[4];
        uint32_t wave_len;
        struct
        {
            char     fmt_id[8];
            uint32_t fmt_len;
            struct
            {
                uint16_t format_tag;
                uint16_t voices;
                uint32_t rate;
                uint32_t char_per_sec;
                uint16_t block_align;
                uint16_t bits_per_sample;
            }
            fmt;
            struct
            {
                char     data_id[4];
                uint32_t data_len;
            }
            data;
        }
        wave;
    }
    riff_hdr =
    {
        {'R', 'I', 'F', 'F' },
        sizeof (riff_hdr.wave),
        {
            {'W', 'A', 'V', 'E', 'f', 'm', 't', ' ' },
            sizeof (riff_hdr.wave.fmt),
            {
                0x0001,
                0x0000,
                0x00000000,
                0x00000000,
                0x0000,
                0x0000
            },
            {
                {'d', 'a', 't', 'a' },
                0x00000000
            }
        }
    };
/* *INDENT-ON* */
    pcm->log_flags = 0;
    if (params->channel == SND_PCM_CHANNEL_PLAYBACK)
    {
        str = getenv("PLAYBACK_LOGGING");
        if (str)
        {
            time_t t = time(NULL);
            struct tm *current_time = gmtime ( &t );
            if (strcasecmp(str, "processed") == 0)
            {
                pcm->log_flags = LOG_POSTPROCESSED_DATA;
                snprintf(filename_buf, _POSIX_PATH_MAX,  "/dev/shmem/playback_%d_%d_%d_%d_processed_%04d%02d%02d-%02d%02d%02dUTC.wav",
                                getpid(),
                                pcm->card,
                                pcm->device,
                                pcm->log_id,
                                1900 + current_time->tm_year,
                                current_time->tm_mon + 1,
                                current_time->tm_mday,
                                current_time->tm_hour,
                                current_time->tm_min,
                                current_time->tm_sec);
                bigEndian = snd_pcm_format_big_endian(pcm->setup[params->channel].format.format);
                mSampleChannels = pcm->setup[params->channel].format.voices;
                mSampleRate = pcm->setup[params->channel].format.rate;
                mSampleBits = snd_pcm_format_width(pcm->setup[params->channel].format.format);
            }
            else if (strcasecmp(str, "unprocessed") == 0)
            {
                pcm->log_flags = LOG_PREPROCESSED_DATA;
                snprintf(filename_buf, _POSIX_PATH_MAX, "/dev/shmem/playback_%d_%d_%d_%d_unprocessed_%04d%02d%02d-%02d%02d%02dUTC.wav",
                                getpid(),
                                pcm->card,
                                pcm->device,
                                pcm->log_id,
                                1900 + current_time->tm_year,
                                current_time->tm_mon + 1,
                                current_time->tm_mday,
                                current_time->tm_hour,
                                current_time->tm_min,
                                current_time->tm_sec);
                bigEndian = snd_pcm_format_big_endian(params->format.format);
                mSampleChannels = params->format.voices;
                mSampleRate = params->format.rate;
                mSampleBits = snd_pcm_format_width(params->format.format);
            }
        }
    }
    else if (params->channel == SND_PCM_CHANNEL_CAPTURE)
    {
        str = getenv("CAPTURE_LOGGING");
        if (str)
        {
            time_t t = time(NULL);
            struct tm *current_time = gmtime ( &t );
            if (strcasecmp(str, "processed") == 0)
            {
                pcm->log_flags = LOG_POSTPROCESSED_DATA;
                snprintf(filename_buf, _POSIX_PATH_MAX, "/dev/shmem/capture_%d_%d_%d_%d_processed_%04d%02d%02d-%02d%02d%02dUTC.wav",
                                getpid(),
                                pcm->card,
                                pcm->device,
                                pcm->log_id,
                                1900 + current_time->tm_year,
                                current_time->tm_mon + 1,
                                current_time->tm_mday,
                                current_time->tm_hour,
                                current_time->tm_min,
                                current_time->tm_sec);
                bigEndian = snd_pcm_format_big_endian(params->format.format);
                mSampleChannels = params->format.voices;
                mSampleRate = params->format.rate;
                mSampleBits = snd_pcm_format_width(params->format.format);
            }
            else if (strcasecmp(str, "unprocessed") == 0)
            {
                pcm->log_flags = LOG_PREPROCESSED_DATA;
                snprintf(filename_buf, _POSIX_PATH_MAX, "/dev/shmem/capture_%d_%d_%d_%d_unprocessed_%04d%02d%02d-%02d%02d%02dUTC.wav",
                                getpid(),
                                pcm->card,
                                pcm->device,
                                pcm->log_id,
                                1900 + current_time->tm_year,
                                current_time->tm_mon + 1,
                                current_time->tm_mday,
                                current_time->tm_hour,
                                current_time->tm_min,
                                current_time->tm_sec);
                bigEndian = snd_pcm_format_big_endian(pcm->setup[params->channel].format.format);
                mSampleChannels = pcm->setup[params->channel].format.voices;
                mSampleRate = pcm->setup[params->channel].format.rate;
                mSampleBits = snd_pcm_format_width(pcm->setup[params->channel].format.format);
            }
        }
    }

    if (pcm->log_flags)
    {
        close_pcm_log (pcm);
        if ((pcm->log_fd = open ( filename_buf, O_RDWR | O_CREAT, 0400)) == -1)
        {
            slog2fa (NULL, 0, SLOG2_CRITICAL, "Failed to create PCM log file (%s) - %s", SLOG2_FA_STRING(filename_buf), SLOG2_FA_STRING(strerror(errno)), SLOG2_FA_END);
            fprintf(stderr, "Failed to create PCM log file (%s) - %s\n", filename_buf, strerror(errno));
            pcm->log_flags = 0;
            return (-errno);
        }
        pcm->log_id++;
        if( bigEndian ) {
            riff_hdr.riff_id[3] = 'X';
            riff_hdr.wave.fmt_len = ENDIAN_BE32(sizeof(riff_hdr.wave.fmt));
            riff_hdr.wave.fmt.format_tag = ENDIAN_BE16(1);
            riff_hdr.wave.fmt.voices = ENDIAN_BE16 (mSampleChannels);
            riff_hdr.wave.fmt.rate = ENDIAN_BE32 (mSampleRate);
            riff_hdr.wave.fmt.char_per_sec = ENDIAN_BE32 (mSampleRate * mSampleChannels * mSampleBits / 8);
            riff_hdr.wave.fmt.block_align = ENDIAN_BE16 (mSampleChannels * mSampleBits / 8);
            riff_hdr.wave.fmt.bits_per_sample = ENDIAN_BE16 (mSampleBits);
        } else {
            riff_hdr.wave.fmt_len = ENDIAN_LE32(sizeof(riff_hdr.wave.fmt));
            riff_hdr.wave.fmt.format_tag = ENDIAN_LE16(1);
            riff_hdr.wave.fmt.voices = ENDIAN_LE16 (mSampleChannels);
            riff_hdr.wave.fmt.rate = ENDIAN_LE32 (mSampleRate);
            riff_hdr.wave.fmt.char_per_sec = ENDIAN_LE32 (mSampleRate * mSampleChannels * mSampleBits / 8);
            riff_hdr.wave.fmt.block_align = ENDIAN_LE16 (mSampleChannels * mSampleBits / 8);
            riff_hdr.wave.fmt.bits_per_sample = ENDIAN_LE16 (mSampleBits);
        }
        write (pcm->log_fd, &riff_hdr, sizeof(riff_hdr));
    }
    return (EOK);
}

int
close_pcm_log (snd_pcm_t * pcm)
{
    if (pcm->log_fd != -1 && pcm->log_flags)
    {
        int size = 0;
        struct stat statbuf;

        fsync( pcm->log_fd );
        if( fstat( pcm->log_fd, &statbuf ) == 0 )
        {
            char endian;
            /* Get file size minus RIFF header for wave length*/
            lseek( pcm->log_fd, 3, SEEK_SET );
            read( pcm->log_fd, &endian, 1 );
            if( endian == 'X' ) {
                size = ENDIAN_BE32(statbuf.st_size - 8);
            } else {
                size = ENDIAN_LE32(statbuf.st_size - 8);
            }
            if( lseek( pcm->log_fd, 4, SEEK_SET ) == 4) /* Seek 4 bytes to wave length position */
                write( pcm->log_fd, &size, 4 );
            /* Get file size minus RIFF and WAVE header for data length */
            if( endian == 'X' ) {
                size = ENDIAN_BE32(statbuf.st_size - 44);
            } else {
                size = ENDIAN_LE32(statbuf.st_size - 44);
            }
            if( lseek( pcm->log_fd, 40, SEEK_SET ) == 40)   /* Seek 40 bytes to data length position */
                write( pcm->log_fd, &size, 4 );
            fsync( pcm->log_fd );
        }
        close(pcm->log_fd);
        pcm->log_fd = -1;
    }

    return (EOK);
}

int
snd_pcm_find (unsigned int format, int *number, int *cards, int *devices, int mode)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(format), SLOG2_FA_UNSIGNED(mode), SLOG2_FA_END);

    char   *filefmt;
    int     index;
    int     max_cards, card;
    snd_ctl_t *handle;
    int     max_devs, dev;
    char    filename[32];
    int     fd;
    snd_ctl_hw_info_t hw_info;
    snd_pcm_channel_info_t info;

    for (index = 0; index < *number; index++)
    {
        cards[index] = -1;
        devices[index] = -1;
    }

    switch (mode)
    {
    case SND_PCM_CHANNEL_PLAYBACK:
        filefmt = SND_FILE_PCM_PLAYBACK;
        break;
    case SND_PCM_CHANNEL_CAPTURE:
        filefmt = SND_FILE_PCM_CAPTURE;
        break;
    default:
        return -EINVAL;
    }

    index = 0;
    max_cards = snd_cards ();
    for (card = 0; card < max_cards; card++)
    {
        if (snd_ctl_open (&handle, card) < 0)
            continue;

        if (snd_ctl_hw_info (handle, &hw_info) < 0) {
            snd_ctl_close(handle);
            continue;
        }

        max_devs = hw_info.pcmdevs;
        for (dev = 0; dev < max_devs; dev++)
        {
            memset (&info, 0, sizeof (info));

            if (snd_ctl_driver_version (handle) >= 3)
            {
                if (snd_ctl_pcm_channel_info (handle, dev, mode, 0, &info) < 0)
                    continue;
            }
            else
            {
                snprintf (filename, sizeof(filename), filefmt, card, dev);
                if ((fd = open (filename, O_RDWR | O_NONBLOCK | O_CLOEXEC)) == -1)
                {
                    continue;
                }
                if (ioctl (fd, SND_PCM_IOCTL_CHANNEL_INFO, &info) < 0)
                {
                    close (fd);
                    continue;
                }
                close (fd);
            }

            if ((info.formats & format))
            {
                cards[index] = card;
                devices[index] = dev;
                index++;
            }
            if (index == *number)
                break;
        }

        snd_ctl_close (handle);
    }

    return (*number = index);
}

int
snd_pcm_open_internal (snd_pcm_t **pcm, const char *name, int *rcard, int *rdevice, int mode)
{
    int  fd;
    int  ver;
    snd_pcm_info_t pinfo;

    if (((*pcm) = (snd_pcm_t *) calloc (1, sizeof (snd_pcm_t))) == NULL)
    {
        return -errno;
    }
    (*pcm)->fd[SND_PCM_CHANNEL_PLAYBACK] = -1;
    (*pcm)->fd[SND_PCM_CHANNEL_CAPTURE] = -1;
    (*pcm)->log_fd = -1;
    (*pcm)->log_id = 0;

    if ((fd = open (name, O_RDWR | O_CLOEXEC |
                ((mode & SND_PCM_OPEN_NONBLOCK) ? O_NONBLOCK : 0))) != -1)
    {
        if (ioctl (fd, SND_PCM_IOCTL_PVERSION, &ver) < 0)
        {
            close (fd);
            free (*pcm);
            *pcm = NULL;
            return -errno;
        }
        if (SND_PROTOCOL_INCOMPATIBLE (ver, SND_PCM_VERSION))
        {
            close (fd);
            free (*pcm);
            *pcm = NULL;
            return -SND_ERROR_INCOMPATIBLE_VERSION;
        }
        if (ioctl (fd, SND_PCM_IOCTL_INFO, &pinfo) < 0)
        {
            close (fd);
            free (*pcm);
            *pcm = NULL;
            return -errno;
        }
        (*pcm)->card = pinfo.card;
        (*pcm)->device = pinfo.device;

        slog2fa(NULL, 0, SLOG2_DEBUG2, "%s:%d - %p open pcm %s by, current fd %d/%d",
                SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(__LINE__), SLOG2_FA_UNSIGNED(pcm),
                SLOG2_FA_STRING(name?name:"NA"), SLOG2_FA_SIGNED((*pcm)->fd[SND_PCM_CHANNEL_PLAYBACK]),
                SLOG2_FA_SIGNED((*pcm)->fd[SND_PCM_CHANNEL_CAPTURE]), SLOG2_FA_END);

        if (mode & SND_PCM_OPEN_PLAYBACK) {
            (*pcm)->fd[SND_PCM_CHANNEL_PLAYBACK] = fd;
        } else {
            (*pcm)->fd[SND_PCM_CHANNEL_CAPTURE] = fd;
        }
        (*pcm)->mode = mode;
        (*pcm)->ver = ver;

        /* List of plugins which are disabled by default
         * Note: The voice converter is used for both voice conversion and voice re-mapping
         *       so there is a discreet PLUGIN_VOICE rather than only being bundled in with PLUGIN_CONVERSION
         *       Disable voice plugin by default so the voice converter will only be instantiated for voice conversions
         */
        (*pcm)->plugin_disable_mask = PLUGIN_VOICE | PLUGIN_MMAP;
    }
    else
    {
        free (*pcm);
        *pcm = NULL;
        return -ENOENT;
    }

    if (rcard != NULL)
        *rcard = (*pcm)->card;
    if (rdevice != NULL)
        *rdevice = (*pcm)->device;
    return EOK;
}

int
snd_pcm_open (snd_pcm_t ** handle, int card, int device, int mode)
{
    TRY_START_SLOG
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d %d %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(card), SLOG2_FA_UNSIGNED(device), SLOG2_FA_UNSIGNED(mode), SLOG2_FA_END);

    char name[32];
    int  rtn;

    *handle = NULL;
    if ((mode & (SND_PCM_OPEN_PLAYBACK | SND_PCM_OPEN_CAPTURE)) == 0)
        return -EINVAL;
    if (mode & SND_PCM_OPEN_PLAYBACK)
    {
        snprintf (name, sizeof(name), SND_FILE_PCM_PLAYBACK, card, device);
        if ((rtn = snd_pcm_open_internal (handle, name, NULL, NULL,
                    (mode & ~SND_PCM_OPEN_CAPTURE))) != 0)
            return rtn;
    }
    if (mode & SND_PCM_OPEN_CAPTURE)
    {
        snprintf (name, sizeof(name), SND_FILE_PCM_CAPTURE, card, device);
        if ((rtn = snd_pcm_open_internal (handle, name, NULL, NULL,
                    (mode & ~SND_PCM_OPEN_PLAYBACK))) != 0)
            return rtn;
    }
    return 0;
}

int
snd_pcm_open_name (snd_pcm_t ** handle, const char *name, int mode)
{
    char    path[_POSIX_PATH_MAX];

    TRY_START_SLOG
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %s %d", SLOG2_FA_STRING(__func__), SLOG2_FA_STRING(name), SLOG2_FA_UNSIGNED(mode), SLOG2_FA_END);

    *handle = NULL;
    if ((mode & (SND_PCM_OPEN_PLAYBACK | SND_PCM_OPEN_CAPTURE)) == 0)
        return -EINVAL;
    if (mode & SND_PCM_OPEN_CAPTURE && mode & SND_PCM_OPEN_PLAYBACK)
        return -EINVAL;

    if (name[0] != '/')
    {
        strlcpy (path, SND_FILE_PCM_PATH, sizeof(path));
        strlcat (path, name, sizeof(path));
        name = path;
    }
    return (snd_pcm_open_internal (handle, name, NULL, NULL, mode));
}

int
snd_pcm_open_preferred (snd_pcm_t ** handle, int *rcard, int *rdevice, int mode)
{
    char   *name;

    TRY_START_SLOG
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(mode), SLOG2_FA_END);

    *handle = NULL;

    if (mode & SND_PCM_OPEN_CAPTURE && mode & SND_PCM_OPEN_PLAYBACK)
        return -EINVAL;
    else if (mode & SND_PCM_OPEN_PLAYBACK)
        name = SND_FILE_PCM_PREFERRED_PLAYBACK;
    else if (mode & SND_PCM_OPEN_CAPTURE)
        name = SND_FILE_PCM_PREFERRED_CAPTURE;
    else
        return -EINVAL;
    return (snd_pcm_open_internal (handle, name, rcard, rdevice, mode));
}

int
snd_pcm_close (snd_pcm_t * pcm)
{
    int res = EOK;
    int channel;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;

    snd_pcm_unlink(pcm);

    for (channel = 0; channel < 2; ++channel)
    {
        snd_pcm_munmap (pcm, channel);
        snd_pcm_plugin_clear (pcm, channel);
        if (pcm->fd[channel] >= 0) {
            if (close (pcm->fd[channel])) {
                res = -errno;
            }
            pcm->fd[channel] = -1;
        }
    }

    close_pcm_log (pcm);
    free (pcm);
    pcm = NULL;
    return res;
}

int
snd_pcm_file_descriptor (snd_pcm_t * pcm, int channel)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if (channel < 0 || channel >= SND_PCM_CHANNEL_MAX)
        return -EINVAL;

    return pcm->fd[channel];
}

int
snd_pcm_nonblock_mode (snd_pcm_t * pcm, int nonblock)
{
    long    flags;
    int     fd, channel;
    int     rtn = EOK;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(nonblock), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;

    for (channel = 0; channel < SND_PCM_CHANNEL_MAX; ++channel)
    {
        fd = pcm->fd[channel];
        if (fd < 0)
            continue;
        if ((flags = fcntl (fd, F_GETFL)) < 0)
            return -errno;
        if (nonblock)
            flags |= O_NONBLOCK;
        else
            flags &= ~O_NONBLOCK;
        if (fcntl (fd, F_SETFL, flags) < 0)
            return -errno;
        if (nonblock)
            pcm->mode |= SND_PCM_OPEN_NONBLOCK;
        else
            pcm->mode &= ~SND_PCM_OPEN_NONBLOCK;
    }
    return rtn;

}

int
snd_pcm_get_filter (snd_pcm_t * pcm, int channel, snd_pcm_filter_t * filter)
{
    if (!pcm || !filter)
        return -EINVAL;
    if (ioctl (pcm->fd[channel], SND_PCM_IOCTL_GET_FILTER, filter) < 0)
        return -errno;
    return EOK;
}

int
snd_pcm_set_filter (snd_pcm_t * pcm, int channel, snd_pcm_filter_t * filter)
{
    if (!pcm || !filter)
        return -EINVAL;
    if (ioctl (pcm->fd[channel], SND_PCM_IOCTL_SET_FILTER, filter) < 0)
        return -errno;
    return EOK;
}

int
snd_pcm_info (snd_pcm_t * pcm, snd_pcm_info_t * info)
{
    int     fd, channel;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm || !info)
        return -EINVAL;
    for (channel = 0; channel < 2; ++channel)
    {
        fd = pcm->fd[channel];
        if (fd >= 0)
            break;
    }
    if (ioctl (fd, SND_PCM_IOCTL_INFO, info) < 0)
        return -errno;
    return 0;
}

int
snd_pcm_channel_info (snd_pcm_t * pcm, snd_pcm_channel_info_t * info)
{
    int     fd;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm || !info)
        return -EINVAL;
    if (info->channel < 0 || info->channel >= SND_PCM_CHANNEL_MAX)
        return -EINVAL;
    fd = pcm->fd[info->channel];
    if (fd < 0)
        return -EINVAL;
    if (ioctl (fd, SND_PCM_IOCTL_CHANNEL_INFO, info) < 0)
        return -errno;
    return 0;
}

int
snd_pcm_channel_setup (snd_pcm_t * pcm, snd_pcm_channel_setup_t * setup)
{
    int     fd;
    iov_t   iov[3];
    int     rtn;
    snd_mixer_eid_t *eid_ptr;
    snd_mixer_gid_t *gid_ptr;

    if (!pcm || !setup)
        return -EINVAL;
    if (setup->channel < 0 || setup->channel >= SND_PCM_CHANNEL_MAX)
        return -EINVAL;
    fd = pcm->fd[setup->channel];
    if (fd < 0)
        return -EINVAL;
    if (pcm->setup_is_valid[setup->channel] == 0)
    {
        SETIOV (&iov[0], &pcm->setup[setup->channel], sizeof (snd_pcm_channel_setup_t));
        SETIOV (&iov[1], &pcm->setup_eid[setup->channel], sizeof (snd_mixer_eid_t));
        SETIOV (&iov[2], &pcm->setup_gid[setup->channel], sizeof (snd_mixer_gid_t));
        if ((rtn = devctlv (fd, SND_PCM_IOCTL_CHANNEL_SETUP, 3, 3, iov, iov, NULL)) != EOK)
            return -rtn;
        pcm->setup_is_valid[setup->channel] = 1;
    }

    // If we are doing a setup internal to params, we don't have to copy data over to
    // the caller
    if( setup != &pcm->setup[setup->channel] ) {
        eid_ptr = setup->mixer_eid;
        gid_ptr = setup->mixer_gid;
        memcpy (setup, &pcm->setup[setup->channel], sizeof (*setup));
        if ((setup->mixer_eid = eid_ptr))
            memcpy (setup->mixer_eid, &pcm->setup_eid[setup->channel], sizeof (snd_mixer_eid_t));
        if ((setup->mixer_gid = gid_ptr))
            memcpy (setup->mixer_gid, &pcm->setup_gid[setup->channel], sizeof (snd_mixer_gid_t));
    }

    return 0;
}

int
snd_pcm_channel_params (snd_pcm_t * pcm, snd_pcm_channel_params_t * params)
{
    int  sample_size;
    snd_pcm_channel_params_t hwparams;
    snd_pcm_plugin_t *plugin;
    int  err;
    char *new_buffer;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if ( !params || params->channel < 0 || params->channel >= SND_PCM_CHANNEL_MAX ||
        pcm->fd[params->channel] < 0)
        return -EINVAL;

    params->why_failed = 0;

    /* make fragsize even number of samples*channels for plugins */
    if (!(pcm->plugin_disable_mask & PLUGIN_CONVERSION)) {
        if (snd_pcm_format_width (params->format.format) >= 0) {
            sample_size = snd_pcm_format_width (params->format.format) / _BITS_BYTE * params->format.voices;
            if( sample_size != 0 ) {
                if ((params->mode & SND_PCM_MODE_MASK) == SND_PCM_MODE_STREAM)
                    params->buf.stream.queue_size = (params->buf.stream.queue_size / sample_size) * sample_size;
                else if ((params->mode & SND_PCM_MODE_MASK) == SND_PCM_MODE_BLOCK)
                    params->buf.block.frag_size = (params->buf.block.frag_size / sample_size) * sample_size;
                else {
                    params->why_failed = SND_PCM_PARAMS_BAD_MODE;
                    return -EINVAL;
                }
            }
        }
    }

    /*
     *  try to decide, if a conversion is required
     */
    pcm->pre_plugin_format[params->channel] = params->format;
    pcm->start_mode[params->channel] = params->start_mode;
    memcpy (&hwparams, params, sizeof (hwparams));
    pdprintf ("params requested params: format = %i, rate = %i, voices = %i\n",
        hwparams.format.format, hwparams.format.rate, hwparams.format.voices);
    if ((pcm->plugin_disable_mask & PLUGIN_CONVERSION)) {
        if (ioctl (pcm->fd[params->channel], SND_PCM_IOCTL_CHANNEL_PARAMS, &hwparams) < 0)
        {
            params->why_failed = hwparams.why_failed;
            return -errno;
        }
    } else {
        if (ioctl (pcm->fd[params->channel], SND_PCM_IOCTL_CHANNEL_PARAM_FIT, &hwparams) < 0)
        {
            params->why_failed = hwparams.why_failed;
            return -errno;
        }
    }

    pcm->setup_is_valid[params->channel] = 0;
    memset (&pcm->setup[params->channel], 0, sizeof (snd_pcm_channel_setup_t));
    pcm->setup[params->channel].channel = params->channel;
    if ((err = snd_pcm_channel_setup (pcm, &pcm->setup[params->channel])) < 0)
        return err;
    pcm->setup_is_valid[params->channel] = 1;

    snd_pcm_plugin_clear (pcm, params->channel);

    /* add necessary plugins */
    if (!(pcm->plugin_disable_mask & PLUGIN_CONVERSION)) {
        if ((err = snd_pcm_plugin_format (pcm, params, &hwparams)) < 0) {
            return err;
        }
    }

    /*
     *  I/O plugins
     */

    if ((params->mode & SND_PCM_MODE_MASK) == SND_PCM_MODE_STREAM)
    {
        pdprintf ("params stream plugin\n");
        plugin = NULL;
        err = snd_pcm_plugin_build_stream (pcm, params->channel, &plugin);
    }
    else if ((params->mode & SND_PCM_MODE_MASK) == SND_PCM_MODE_BLOCK)
    {
        /* Cache params so we can do a pre-param from within the plugins */
        memmove( &pcm->params[params->channel], params, sizeof(*params) );

        if (!(pcm->plugin_disable_mask & PLUGIN_MMAP) &&
            pcm->setup[params->channel].mmap_valid)
        {
            plugin = NULL;
            err = snd_pcm_plugin_build_mmap (pcm, params->channel, &plugin);
        }
        else
        {
            plugin = NULL;
            err = snd_pcm_plugin_build_block (pcm, params->channel, &plugin);
        }
    }
    else
    {
        err = -EINVAL;
    }
    if (err < 0) {
        snd_pcm_plugin_clear (pcm, params->channel);
        return err;
    }

    if (params->channel == SND_PCM_CHANNEL_PLAYBACK)
    {
        err = snd_pcm_plugin_append (pcm, params->channel, plugin);
    }
    else
    {
        err = snd_pcm_plugin_insert (pcm, params->channel, plugin);
    }
    if (err < 0)
    {
        snd_pcm_plugin_free (plugin);
        snd_pcm_plugin_clear (pcm, params->channel);
        return err;
    }

    if ((params->mode & SND_PCM_MODE_MASK) == SND_PCM_MODE_BLOCK)
    {
        params->buf.block.frag_size = snd_pcm_plugin_transfer_size (pcm, params->channel,
            pcm->setup[params->channel].buf.block.frag_size);
        pcm->plugin_partial_block_size[params->channel] = params->buf.block.frag_size;
        if (!(pcm->plugin_disable_mask & PLUGIN_BUFFER_PARTIAL_BLOCKS))
        {
            new_buffer = realloc (pcm->plugin_partial_block_buffer[params->channel],
                        pcm->plugin_partial_block_size[params->channel]);
            if( new_buffer == NULL ) {
                snd_pcm_plugin_clear (pcm, params->channel);
                err = -ENOMEM;
                return err;
            }
            pcm->plugin_partial_block_buffer[params->channel] = new_buffer;
            pcm->plugin_partial_block_len[params->channel] = 0;
        }
        pcm->plugin_ratio[params->channel] = 1.0 * params->buf.block.frag_size /
            pcm->setup[params->channel].buf.block.frag_size;
        pcm->plugin_cratio[params->channel] = 1.0 * pcm->pre_plugin_format[params->channel].rate /
            pcm->setup[params->channel].format.rate;
    }
    else
    {
        pcm->plugin_partial_block_size[params->channel] = 0;
        pcm->plugin_partial_block_len[params->channel] = 0;
        params->buf.stream.queue_size = snd_pcm_plugin_transfer_size (pcm, params->channel,
            pcm->setup[params->channel].buf.stream.queue_size);
        params->buf.stream.max_fill = params->buf.stream.queue_size;
        pcm->plugin_ratio[params->channel] = 1.0 * params->buf.stream.queue_size /
            pcm->setup[params->channel].buf.stream.queue_size;
        pcm->plugin_cratio[params->channel] = 1.0 * pcm->pre_plugin_format[params->channel].rate /
            pcm->setup[params->channel].format.rate;
    }

    err = snd_pcm_plugin_action (pcm, hwparams.channel, INIT);
    if (err < 0) {
        snd_pcm_plugin_clear (pcm, params->channel);
        return err;
    }

    open_pcm_log(pcm, params);

    return EOK;
}

int
snd_pcm_channel_status (snd_pcm_t * pcm, snd_pcm_channel_status_t * status)
{
    int  fd;
    int  channel;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm || !status || status->channel < 0 || status->channel >= SND_PCM_CHANNEL_MAX)
        return -EINVAL;

    channel = status->channel;
    fd = pcm->fd[channel];
    if (fd < 0)
        return -EINVAL;

    if (ioctl (fd, SND_PCM_IOCTL_CHANNEL_STATUS, status) < 0)
        return -errno;

    if (pcm->plugin_disable_mask & PLUGIN_BUFFER_PARTIAL_BLOCKS) {
        status->subbuffered = 0;
    } else {
        status->subbuffered = pcm->plugin_partial_block_len[channel];
    }

    if (!(pcm->plugin_disable_mask & PLUGIN_CONVERSION)) {
        if (pcm->plugin_src_mode == SND_SRC_MODE_NORMAL)
        {
            status->scount *= pcm->plugin_ratio[channel];
            status->count *= pcm->plugin_ratio[channel];
            status->free *= pcm->plugin_ratio[channel];
        }
        else /* average rate */
        {
            status->scount *= pcm->plugin_cratio[channel];
            status->count *= pcm->plugin_cratio[channel];
            status->free *= pcm->plugin_cratio[channel];
        }
    }
    return EOK;
}

static int snd_pcm_prepare(snd_pcm_t * pcm, int channel )
{
    int  err;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x, %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_SIGNED(channel), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if (pcm->fd[channel] < 0)
        return -EINVAL;

    if ((err = snd_pcm_plugin_action (pcm, channel, PREPARE)) < 0) {
        return err;
    }

    if (ioctl (pcm->fd[channel], SND_PCM_IOCTL_CHANNEL_PREPARE) < 0) {
        return -errno;
    }

    return EOK;
}

int
snd_pcm_playback_prepare (snd_pcm_t * pcm)
{
    return snd_pcm_prepare( pcm, SND_PCM_CHANNEL_PLAYBACK );
}

int
snd_pcm_capture_prepare (snd_pcm_t * pcm)
{
    return snd_pcm_prepare( pcm, SND_PCM_CHANNEL_CAPTURE );
}

int
snd_pcm_channel_prepare (snd_pcm_t * pcm, int channel)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

    switch (channel)
    {
    case SND_PCM_CHANNEL_PLAYBACK:
        return snd_pcm_playback_prepare (pcm);
    case SND_PCM_CHANNEL_CAPTURE:
        return snd_pcm_capture_prepare (pcm);
    default:
        return -EIO;
    }
}

int
snd_pcm_playback_go (snd_pcm_t * pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if (pcm->fd[SND_PCM_CHANNEL_PLAYBACK] < 0)
        return -EINVAL;
    if (ioctl (pcm->fd[SND_PCM_CHANNEL_PLAYBACK], SND_PCM_IOCTL_CHANNEL_GO) < 0)
        return -errno;

    return EOK;
}

int
snd_pcm_capture_go (snd_pcm_t * pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if (pcm->fd[SND_PCM_CHANNEL_CAPTURE] < 0)
        return -EINVAL;
    if (ioctl (pcm->fd[SND_PCM_CHANNEL_CAPTURE], SND_PCM_IOCTL_CHANNEL_GO) < 0)
        return -errno;

    return EOK;
}

int
snd_pcm_channel_go (snd_pcm_t * pcm, int channel)
{
    switch (channel)
    {
    case SND_PCM_CHANNEL_PLAYBACK:
        return snd_pcm_playback_go (pcm);
    case SND_PCM_CHANNEL_CAPTURE:
        return snd_pcm_capture_go (pcm);
    default:
        return -EIO;
    }
}

int
snd_pcm_playback_pause (snd_pcm_t * pcm)
{
    int  on = 1;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if (pcm->fd[SND_PCM_CHANNEL_PLAYBACK] < 0)
        return -EINVAL;
    if (ioctl (pcm->fd[SND_PCM_CHANNEL_PLAYBACK], SND_PCM_IOCTL_CHANNEL_PAUSE, &on) < 0)
        return -errno;
    return 0;
}

int
snd_pcm_capture_pause (snd_pcm_t * pcm)
{
    int  on = 1;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if (pcm->fd[SND_PCM_CHANNEL_CAPTURE] < 0)
        return -EINVAL;
    if (ioctl (pcm->fd[SND_PCM_CHANNEL_CAPTURE], SND_PCM_IOCTL_CHANNEL_PAUSE, &on) < 0)
        return -errno;
    return 0;
}

int
snd_pcm_channel_pause (snd_pcm_t * pcm, int channel)
{
    switch (channel)
    {
    case SND_PCM_CHANNEL_PLAYBACK:
        return snd_pcm_playback_pause (pcm);
    case SND_PCM_CHANNEL_CAPTURE:
        return snd_pcm_capture_pause (pcm);
    default:
        return -EIO;
    }
}

int
snd_pcm_playback_resume (snd_pcm_t * pcm)
{
    int  off = 0;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if (pcm->fd[SND_PCM_CHANNEL_PLAYBACK] < 0)
        return -EINVAL;
    if (ioctl (pcm->fd[SND_PCM_CHANNEL_PLAYBACK], SND_PCM_IOCTL_CHANNEL_PAUSE, &off) < 0)
        return -errno;
    return 0;
}

int
snd_pcm_capture_resume (snd_pcm_t * pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    int  off = 0;

    if (!pcm)
        return -EINVAL;
    if (pcm->fd[SND_PCM_CHANNEL_CAPTURE] < 0)
        return -EINVAL;
    if (ioctl (pcm->fd[SND_PCM_CHANNEL_CAPTURE], SND_PCM_IOCTL_CHANNEL_PAUSE, &off) < 0)
        return -errno;
    return 0;
}

int
snd_pcm_channel_resume (snd_pcm_t * pcm, int channel)
{
    switch (channel)
    {
    case SND_PCM_CHANNEL_PLAYBACK:
        return snd_pcm_playback_resume (pcm);
    case SND_PCM_CHANNEL_CAPTURE:
        return snd_pcm_capture_resume (pcm);
    default:
        return -EIO;
    }
}

int
snd_pcm_playback_drain (snd_pcm_t * pcm)
{
    int  err;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;
    if ((err = snd_pcm_plugin_action (pcm, SND_PCM_CHANNEL_PLAYBACK, DRAIN)) < 0)
        return err;
    if (pcm->fd[SND_PCM_CHANNEL_PLAYBACK] < 0)
        return -EINVAL;
    pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] = 0;
    if ((err = snd_pcm_plugin_action (pcm, SND_PCM_CHANNEL_PLAYBACK, DRAIN)) < 0)
        return err;
    if (ioctl (pcm->fd[SND_PCM_CHANNEL_PLAYBACK], SND_PCM_IOCTL_CHANNEL_DRAIN) < 0)
        return -errno;
    if ((err = snd_pcm_plugin_action (pcm, SND_PCM_CHANNEL_PLAYBACK, POST_DRAIN)) < 0)
        return err;
    return 0;
}

int
snd_pcm_playback_flush (snd_pcm_t * pcm)
{
    int err, n = 0;
    unsigned char silence;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;

    if (pcm->fd[SND_PCM_CHANNEL_PLAYBACK] < 0)
        return -EINVAL;

    if (pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] != 0)
    {
        if (pcm->setup_is_valid[SND_PCM_CHANNEL_PLAYBACK])
            silence = snd_pcm_plugin_silence (&pcm->pre_plugin_format[SND_PCM_CHANNEL_PLAYBACK]);
        else
            silence = 0x0;
        n = pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK] -
            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK];
        memset (pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_PLAYBACK] +
            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK], silence, n);
        snd_pcm_plugin_write1 (pcm, pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_PLAYBACK],
            pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK]);
    }
    pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] = 0;

    if ((err = snd_pcm_plugin_action (pcm, SND_PCM_CHANNEL_PLAYBACK, FLUSH)) < 0) {
        return err;
    }

    if (ioctl (pcm->fd[SND_PCM_CHANNEL_PLAYBACK], SND_PCM_IOCTL_CHANNEL_FLUSH) < 0) {
        return -errno;
    }

    return 0;
}

int
snd_pcm_capture_flush (snd_pcm_t * pcm)
{
    int err;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;

     if (pcm->fd[SND_PCM_CHANNEL_CAPTURE] < 0)
        return -EINVAL;

    pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE] = 0;

    if ((err = snd_pcm_plugin_action (pcm, SND_PCM_CHANNEL_CAPTURE, FLUSH)) < 0)
        return err;

    if (ioctl (pcm->fd[SND_PCM_CHANNEL_CAPTURE], SND_PCM_IOCTL_CHANNEL_FLUSH) < 0)
        return -errno;

    return 0;
}

int
snd_pcm_channel_flush (snd_pcm_t * pcm, int channel)
{
    int  err;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

    if( channel < 0 || channel >= SND_PCM_CHANNEL_MAX ) {
        return -EINVAL;
    }

    switch (channel)
    {
    case SND_PCM_CHANNEL_PLAYBACK:
        err = snd_pcm_playback_flush (pcm);
        break;
    case SND_PCM_CHANNEL_CAPTURE:
        err = snd_pcm_capture_flush (pcm);
        break;
    default:
        return -EIO;
    }

    return err;
}

int
snd_pcm_link(snd_pcm_t * pcm1, snd_pcm_t *pcm2)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm1), SLOG2_FA_UNSIGNED(pcm2), SLOG2_FA_END);

    int channel1, channel2;
    int ret = EINVAL;
    struct snd_pcm_link_group *group, *prev = NULL;

    if( pcm1 == pcm2 ) {
        return EOK;
    }

    channel1 = (pcm1->mode & SND_PCM_OPEN_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE;
    channel2 = (pcm2->mode & SND_PCM_OPEN_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE;
    pthread_mutex_lock( &_link_mutex );
    // Try to link the streams in libasound
    if( pcm1->group ) {
        if( pcm2->group ) {
            // Are they already in the same link group?
            for(group = pcm1->group; group != NULL; group = group->next)
            {
                if( group->handle == pcm2 ) {
                    break;
                }
                prev = group;
            }
            if( group == NULL ) {
                // Not in the same group
                prev->next = pcm2->group;
                pcm2->group = pcm1->group;
            }
        } else {
            // Add pcm2 to pcm1's group
            for(group = pcm1->group; group->next != NULL; group = group->next);
            group->next = calloc(1, sizeof(struct snd_pcm_link_group));
            if( group->next ) {
                group->next->handle = pcm2;
                pcm2->group = pcm1->group;
            }
        }
    } else {
        if( pcm2->group ) {
            // Add pcm1 to pcm2's group
            for(group = pcm2->group; group->next != NULL; group = group->next);
            group->next = calloc(1, sizeof(struct snd_pcm_link_group));
            if( group->next ) {
                group->next->handle = pcm1;
                pcm1->group = pcm2->group;
            }
        } else {
            pcm1->group = calloc(1, sizeof(struct snd_pcm_link_group));
            if( pcm1->group ) {
                pcm1->group->handle = pcm1;
                pcm2->group = calloc(1, sizeof(struct snd_pcm_link_group));
                if( pcm2->group ) {
                    pcm2->group->handle = pcm2;
                    pcm1->group->next = pcm2->group;
                    pcm2->group = pcm1->group;
                } else {
                    // Give up
                    free( pcm1->group );
                    pcm1->group = NULL;
                }
            }
        }
    }

/* TODO: On failure we need to cleanup the asound group lists */

    // Try to link the streams in io-audio
    if( (ret = ioctl (pcm1->fd[channel1], SND_PCM_IOCTL_NEW_LINK_GROUP)) != EOK) {
        slog2fa (NULL, 0, SLOG2_CRITICAL, "io-audio failed to create a link group: 0x%s", SLOG2_FA_STRING(snd_strerror(ret)), SLOG2_FA_END);
        pthread_mutex_unlock( &_link_mutex );
        return ret;
    }

    ret = ioctl (pcm2->fd[channel2], SND_PCM_IOCTL_ADD_LINK_GROUP);
    if( ret != EOK ) {
        slog2fa (NULL, 0, SLOG2_CRITICAL, "io-audio failed to add to PCM link group: 0x%s", SLOG2_FA_STRING(snd_strerror(ret)), SLOG2_FA_END);
        ioctl (pcm1->fd[channel1], SND_PCM_IOCTL_END_LINK_GROUP);
        ioctl (pcm1->fd[channel1], SND_PCM_IOCTL_REMOVE_LINK_GROUP);
        pthread_mutex_unlock( &_link_mutex );
        return ret;
    }

    ioctl (pcm1->fd[channel1], SND_PCM_IOCTL_END_LINK_GROUP);
    pthread_mutex_unlock( &_link_mutex );

    return ret;
}

int
snd_pcm_unlink(snd_pcm_t * pcm)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    int rtn = EINVAL;

    pthread_mutex_lock( &_link_mutex );
    if( pcm->group ) {
        struct snd_pcm_link_group *group, *newparent;
        if( pcm->group->handle == pcm ) {
            newparent = pcm->group->next;
            // On the first group, we have to update all descendents
            for(group = pcm->group->next; group != NULL; group = group->next)
            {
                group->handle->group = newparent;
            }
            free( pcm->group );
        } else {
            for(group = pcm->group; group->next->handle != pcm; group = group->next);
            newparent = group->next;
            group->next = group->next->next;
            free( group->next );
        }
        pcm->group = NULL;
        rtn = EOK;
    }
    pthread_mutex_unlock( &_link_mutex );

    if ( rtn == EOK ) {
        rtn = ioctl (pcm->fd[(pcm->mode & SND_PCM_OPEN_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE],
            SND_PCM_IOCTL_REMOVE_LINK_GROUP);
    }
    return rtn;
}

int snd_pcm_query_channel_map(snd_pcm_t * pcm, snd_pcm_chmap_t *chmap)
{
	int status;
	int channel;
	int chmap_size;

	if (!pcm || !chmap)
		return (-EINVAL);

	channel = (pcm->mode & SND_PCM_OPEN_PLAYBACK) ? SND_PCM_CHANNEL_PLAYBACK : SND_PCM_CHANNEL_CAPTURE;
	chmap_size = sizeof(snd_pcm_chmap_t) + (chmap->channels * sizeof(int));

	status = devctl(pcm->fd[channel], SND_PCM_IOCTL_GET_CHANNEL_MAP, chmap, chmap_size, NULL);
	return -status;
}

snd_pcm_chmap_query_t ** snd_pcm_query_chmaps(snd_pcm_t * pcm)
{
	int status = EINVAL, chmap_size,
		channel, msg_size;
	unsigned int *msgbuf;
	snd_pcm_channel_info_t info;
	snd_pcm_chmap_query_t **mappings = NULL;

	if (!pcm)
	{
		errno = EINVAL;
		return (NULL);
	}

	channel = (pcm->mode & SND_PCM_OPEN_PLAYBACK) ? SND_PCM_CHANNEL_PLAYBACK : SND_PCM_CHANNEL_CAPTURE;

	/* Size the mapping entry based on the max_voices the device supports */
	if (!pcm->setup_is_valid[channel])	/* If called before snd_pcm_plugin_params() */
	{
		info.channel = channel;
		if ((status = snd_pcm_plugin_info (pcm, &info)) < 0)
		{
			errno = -status;	/* snd_pcm_plugin_info() returns a negative errno, flip it */
			return NULL;
		}
		chmap_size = sizeof(snd_pcm_chmap_query_t) + (info.max_voices * sizeof(int));
		if ((msgbuf = calloc(1 , chmap_size)) == NULL)
			return NULL;
	}
	else	/* If called after snd_pcm_plugin_params() */
	{
		chmap_size = sizeof(snd_pcm_chmap_query_t) + (pcm->setup[channel].format.voices * sizeof(int));
		if ((msgbuf = calloc(1 , chmap_size)) == NULL)
			return NULL;
	}

	/* We don't know how many mapping entries the device supports, so we start with a message buffer sized for
	 * 1 entry. On ENOMEM failure, the total size is returned to us in the first integer of the msgbuf, so we can
	 * re-allocated msgbuf to the correct size and re-issue the request.
	 */
	msg_size = chmap_size;
	while(1)
	{
		status = devctl(pcm->fd[channel], SND_PCM_IOCTL_QUERY_CHANNEL_MAP, msgbuf, msg_size, NULL);
		if (status == ENOMEM)
		{
			/* Re-allocate the msgbuf to the required size */
			unsigned int *newdata;
			msg_size = msgbuf[0];
			if ((newdata = realloc( msgbuf, msg_size )) == NULL)
			{
				status = errno;
				free( msgbuf );
				break;
			}
			msgbuf = newdata;
		}
		else if (status == EOK)
		{
			void *rptr = msgbuf;
			int mapcount, byte_cnt, i;
			snd_pcm_chmap_query_t *chmap;

			/* Spin through and count the entries */
			for (mapcount = 0, byte_cnt = msg_size; byte_cnt > 0; byte_cnt -= chmap_size, mapcount++, rptr += chmap_size)
			{
				chmap = rptr;
				chmap_size = sizeof(snd_pcm_chmap_query_t) + (chmap->map.channels * sizeof(int));
			}

			/* Allocate the mappings array, +1 for the NULL terminating entry */
			if ((mappings = calloc(mapcount + 1, sizeof(snd_pcm_chmap_query_t *))) == NULL)
			{
				status = errno;
				free( msgbuf );
				break;
			}

			/* Fill in the mapping array */
			for (i = 0, rptr = msgbuf; i < mapcount; i++, rptr += chmap_size)
			{
				mappings[i] = chmap = rptr;
				chmap_size = sizeof(snd_pcm_chmap_query_t) + (chmap->map.channels * sizeof(int));
			}
			break;
		}
		else
		{
			free(msgbuf);
			break;
		}
	}
	errno = status;
	return (mappings);
}

/* Note: Client application must free the returned chmap pointer when finished with it */
snd_pcm_chmap_t * snd_pcm_get_chmap(snd_pcm_t * pcm)
{
	int i, j;
	int status, channel, chmap_size;
	snd_pcm_chmap_t *hw_map, *map;
	snd_pcm_channel_info_t info;
	snd_pcm_voice_conversion_t voice_conversion;

	if (!pcm)
	{
		errno = EINVAL;
		return (NULL);
	}

	channel = (pcm->mode & SND_PCM_OPEN_PLAYBACK) ? SND_PCM_CHANNEL_PLAYBACK : SND_PCM_CHANNEL_CAPTURE;

	if (!pcm->setup_is_valid[channel])
	{
		info.channel = channel;
		if ((status = snd_pcm_plugin_info (pcm, &info)) < 0)
		{
			errno = -status;	/* snd_pcm_plugin_info() returns a negative errno, flip it */
			return NULL;
		}
		chmap_size = sizeof(snd_pcm_chmap_t) + (info.max_voices * sizeof(int));
		if ((hw_map = calloc(1, chmap_size)) == NULL)
			return NULL;
		hw_map->channels = info.max_voices;
	}
	else
	{
		chmap_size = sizeof(snd_pcm_chmap_t) + (pcm->setup[channel].format.voices * sizeof(int));
		if ((hw_map = calloc(1, chmap_size)) == NULL)
			return NULL;
		hw_map->channels = pcm->setup[channel].format.voices;
	}

	status = devctl(pcm->fd[channel], SND_PCM_IOCTL_GET_CHANNEL_MAP, hw_map, chmap_size, NULL);
	if (status != EOK)
	{
		free(hw_map);
		errno = status;
		return (NULL);
	}

	/* If capture interface, then account for voice conversion */
	if((pcm->mode & SND_PCM_OPEN_CAPTURE) && (status = snd_pcm_plugin_get_voice_conversion (pcm, channel, &voice_conversion)) == EOK)
	{
		/* Allocate map buffer to hold the post conversion mapping */
		chmap_size = sizeof(snd_pcm_chmap_t) + (voice_conversion.app_voices * sizeof(int));
		if ((map = calloc(1, chmap_size)) == NULL)
		{
			status = errno;
			free(hw_map);
			errno = status;
			return NULL;
		}

		/* Map hardware channels according to the voice map */
		for( i = 0; i < voice_conversion.app_voices; i++ )
		{
			bool found = false;
			for( j = 0; j < voice_conversion.hw_voices; j++ )
			{
				if( voice_conversion.matrix[i] & (1<<j) )
				{
					/* We must continue to look through all the hardware channels
					 * even after we find a match, as the voice convertor may be mixing
					 * multiple hw channels into a single position in which case
					 * we must mark the channel as UNKNOWN.
					 */
					if( !found )
					{
						map->pos[i] = hw_map->pos[j];
						found = true;
					}
					else
					{
						/* Just in case the hardware contains multiples of
						 * the same channel, a down mix of those channels would
						 * not change anything so we can still give it a
						 * known label rather then marking it as UNKNOWN.
						 */
						if (map->pos[i] != hw_map->pos[j])
						{
							/* The channel is a combination of multiple hardware channels */
							map->pos[i] = SND_CHMAP_UNKNOWN;
						}
						break;
					}
				}
			}
		}
		free(hw_map);
		map->channels = voice_conversion.app_voices;
	}
	else
		map = hw_map;

	return map;
}

int snd_pcm_set_chmap(snd_pcm_t *pcm, const snd_pcm_chmap_t *map)
{
	int channel, status;
	int chmap_size;

	if (!pcm || !map)
		return (-EINVAL);

	chmap_size = sizeof(snd_pcm_chmap_t) + (map->channels * sizeof(unsigned int));

	channel = (pcm->mode & SND_PCM_OPEN_PLAYBACK) ? SND_PCM_CHANNEL_PLAYBACK : SND_PCM_CHANNEL_CAPTURE;
	status = devctl(pcm->fd[channel], SND_PCM_IOCTL_SET_CHANNEL_MAP, (void*)map, chmap_size, NULL);
	return -status;
}

ssize_t
snd_pcm_transfer_size (snd_pcm_t * pcm, int channel)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);
    if (!pcm || channel < 0 || channel >= SND_PCM_CHANNEL_MAX)
        return -EINVAL;
    if (!pcm->setup_is_valid[channel])
        return -EBADFD;
    if (pcm->setup[channel].mode != SND_PCM_MODE_BLOCK)
        return -EBADFD;
    return pcm->setup[channel].buf.block.frag_size;
}

ssize_t
snd_pcm_write_plugin_internal(snd_pcm_t * pcm, const void*buffer, size_t count)
{
    char   *ptr;
    size_t  n;
    ssize_t total, rtn;

    /* BLOCK_MODE */
    if (pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK] != 0 && pcm->plugin_src_mode == SND_SRC_MODE_NORMAL)
    {
        if ((pcm->plugin_disable_mask & PLUGIN_BUFFER_PARTIAL_BLOCKS) &&
            pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK] != count) {
                return -EINVAL;
        }
        else
        {
            for (ptr = (char *) buffer, total = 0; count > 0;)
            {                                              /* do we have a partial block subbuffered | less then a full block */
                if (pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] > 0 ||
                    count < pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK])
                {                                          /* if we can't make a full block just add to subbuffer */
                    if (count <
                        (pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK] -
                            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK]))
                    {
                        memcpy (pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_PLAYBACK] +
                            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK], ptr, count);
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] += count;
                        total += count;
                        count = 0;
                    }
                    else
                    {                                      /* make a full block and write it out */
                        if (pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] <
                            pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK])
                        {
                            n = pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK] -
                                pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK];
                            memcpy (pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_PLAYBACK] +
                                pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK], ptr, n);
                        } else {
                            n = 0;
                        }
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] =
                            pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK];
                        if ((rtn =
                                snd_pcm_plugin_write1 (pcm,
                                    pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_PLAYBACK],
                                    pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK])) < 0)
                        {
                            /* Write failed so role back changes to the partial block buffer */
                            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] -= n;
                            errno = -rtn;
                            break;                         /* error */
                        }
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] -= rtn;
                        if (pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK] > 0)
                        {
                            memmove(pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_PLAYBACK],
                            pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_PLAYBACK] + rtn,
                            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_PLAYBACK]);
                        }
                        // increase by n as that's how much is copied/consumed
                        // from the source instead of how much is being written
                        // to the deva
                        total += n;
                        count -= n;
                        ptr += n;
                    }
                }
                else
                {                                          /* write a full block directly */
                    if ((rtn =
                            snd_pcm_plugin_write1 (pcm, ptr,
                                pcm->plugin_partial_block_size[SND_PCM_CHANNEL_PLAYBACK])) < 0)
                    {
                        errno = -rtn;
                        break;                             /* error */
                    }
                    total += rtn;
                    count -= rtn;
                    ptr += rtn;
                }
            }
        }
    }
    else
        total = snd_pcm_plugin_write1 (pcm, buffer, count);
    return (total);

}

ssize_t
snd_pcm_write (snd_pcm_t * pcm, const void *buffer, size_t count)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(buffer), SLOG2_FA_UNSIGNED(count), SLOG2_FA_END);
    return snd_pcm_write_plugin_internal(pcm, buffer, count);
}

ssize_t
snd_pcm_write_internal (snd_pcm_t * pcm, const void *buffer, size_t count)
{
    int result;

    result = write (pcm->fd[SND_PCM_CHANNEL_PLAYBACK], buffer, count);
    if( result < 0 ) {
        return -errno;
    }
    return result;
}

ssize_t
snd_pcm_read (snd_pcm_t * pcm, void *buffer, size_t count)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(buffer), SLOG2_FA_UNSIGNED(count), SLOG2_FA_END);

    char   *ptr = NULL;
    ssize_t total = 0, rtn = 0;

    /* BLOCK_MODE */
    if (pcm->plugin_partial_block_size[SND_PCM_CHANNEL_CAPTURE] != 0)
    {
        if ((pcm->plugin_disable_mask & PLUGIN_BUFFER_PARTIAL_BLOCKS) &&
            pcm->plugin_partial_block_size[SND_PCM_CHANNEL_CAPTURE] != count)
            return -EINVAL;
        else
        {
            for (ptr = (char *) buffer, total = 0; count > 0;)
            {                                              /* do we have a partial block remaining from a previous read? */
                if (pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE])
                {                                          /* do we have enough already in the partial_block_buffer? */
                    if (count <= pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE])
                    {
                        memcpy (ptr, pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE], count);
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE] -= count;
                        memmove (pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE],
                            pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE] + count,
                            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE]);
                        ptr += count;
                        total += count;
                        count = 0;
                    }
                    else
                    {
                        memcpy (ptr, pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE],
                            pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE]);
                        ptr += pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE];
                        total += pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE];
                        count -= pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE];
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE] = 0;
                    }

                    memset (pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE] +
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE], 0,
                        pcm->plugin_partial_block_size[SND_PCM_CHANNEL_CAPTURE] -
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE]);
                }
                /* do we want less than one real block? */
                else if (count < pcm->plugin_partial_block_size[SND_PCM_CHANNEL_CAPTURE])
                {                                          /* read in a full block */
                    if ((rtn =
                            snd_pcm_plugin_read1 (pcm,
                                pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE],
                                pcm->plugin_partial_block_size[SND_PCM_CHANNEL_CAPTURE])) < 0)
                    {
                        errno = -rtn;
                        pdprintf
                            ("snd_pcm_plugin_read error: unable to read partial block - %s (%i)\n",
                            snd_strerror (rtn), rtn);
                        break;                             /* error */
                    }
                    rtn = (rtn > count) ? rtn : count;
                    memcpy (ptr, pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE], count);
                    pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE] = rtn - count;
                    memmove (pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE],
                        pcm->plugin_partial_block_buffer[SND_PCM_CHANNEL_CAPTURE] + count,
                        pcm->plugin_partial_block_len[SND_PCM_CHANNEL_CAPTURE]);
                    ptr += count;
                    total += count;
                    count = 0;
                }
                else
                {                                          /* read a full block directly */
                    if ((rtn =
                            snd_pcm_plugin_read1 (pcm, ptr,
                                pcm->plugin_partial_block_size[SND_PCM_CHANNEL_CAPTURE])) < 0)
                    {
                        errno = -rtn;
                        pdprintf
                            ("snd_pcm_plugin_read error: unable to read full block - %s (%i)\n",
                            snd_strerror (rtn), rtn);
                        break;                             /* error */
                    }
                    ptr += rtn;
                    total += rtn;
                    count -= rtn;
                }
            }
        }
    }
    else
        total = snd_pcm_plugin_read1 (pcm, buffer, count);

    if (total)
        return (total);
    else
        return -errno;
}

ssize_t
snd_pcm_read_internal (snd_pcm_t * pcm, void *buffer, size_t count)
{
    int result;

    result = read (pcm->fd[SND_PCM_CHANNEL_CAPTURE], buffer, count);
    if( result < 0 ) {
        return -errno;
    }

    return result;
}

int
snd_pcm_mmap (snd_pcm_t * pcm, int channel, snd_pcm_mmap_control_t ** control, void **buffer)
{
    snd_pcm_mmap_info_t info;
    int  fd, err;
    void   *caddr, *daddr;
    int flags = 0;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (control)
        *control = NULL;
    if (buffer)
        *buffer = NULL;
    if (!pcm || channel < 0 || channel >= SND_PCM_CHANNEL_MAX || !control || !buffer)
        return -EINVAL;
    fd = pcm->fd[channel];
    if (fd < 0)
        return -EINVAL;
    if (ioctl (fd, SND_PCM_IOCTL_MMAP_INFO, &info) < 0) {
        return -errno;
    }

    if ((fd = shm_open (info.dmactl_name, O_RDWR | O_CLOEXEC, 0)) == -1)
        return -errno;
    if ((caddr =
            mmap (NULL, info.ctl_size, PROT_READ | PROT_WRITE | PROT_NOCACHE, MAP_SHARED, fd,
                0)) == MAP_FAILED)
    {
        close (fd);
        return -errno;
    }
    close (fd);
    if ((fd = shm_open (info.dmabuf_name, O_RDWR | O_CLOEXEC, 0)) == -1)
        return -errno;

    if (!(info.driver_flags & ADO_BUF_CACHE))
    {
        flags = PROT_NOCACHE;
    }

    if ((daddr =
            mmap (NULL, info.size, PROT_READ | PROT_WRITE | flags, MAP_SHARED, fd,
                0)) == MAP_FAILED)
    {
        close (fd);
        err = -errno;
        munmap (caddr, info.ctl_size);
        return err;
    }
    close (fd);
    *control = pcm->mmap_caddr[channel] = caddr;
    *buffer = pcm->mmap_daddr[channel] = daddr;
    pcm->mmap_size[channel] = info.size;
    return 0;
}

int
snd_pcm_munmap (snd_pcm_t * pcm, int channel)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

    if (!pcm || channel < 0 || channel >= SND_PCM_CHANNEL_MAX)
        return -EINVAL;

    if (pcm->mmap_caddr[channel])
    {
        int  frags = pcm->mmap_caddr[channel]->status.frags;

        munmap (pcm->mmap_caddr[channel],
            sizeof (snd_pcm_mmap_control_t) + (sizeof (snd_pcm_mmap_fragment_t) * frags));
        pcm->mmap_caddr[channel] = NULL;
    }
    if (pcm->mmap_daddr[channel])
    {
        munmap (pcm->mmap_daddr[channel], pcm->mmap_size[channel]);
        pcm->mmap_daddr[channel] = NULL;
        pcm->mmap_size[channel] = 0;
    }
    return 0;
}

void snd_pcm_free_chmaps(snd_pcm_chmap_query_t **maps)
{
	free(maps[0]);
	free(maps);
}

int snd_pcm_channel_audio_ducking(snd_pcm_t *pcm, int channel, uint32_t enable)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x, %d, %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_SIGNED(channel), SLOG2_FA_UNSIGNED(enable), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;

    if (pcm->fd[channel] < 0)
        return -EINVAL;

    if (ioctl (pcm->fd[channel], SND_PCM_IOCTL_CHANNEL_SET_FORCE_DUCKING, &enable) != EOK) {
        return -errno;
    }

    return EOK;
}

int
snd_pcm_channel_read_event (snd_pcm_t * pcm, int channel, snd_pcm_event_t * event)
{
    int     fd;

    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

    if (!pcm || !event)
        return -EINVAL;
    if (channel < 0 || channel >= SND_PCM_CHANNEL_MAX)
        return -EINVAL;
    fd = pcm->fd[channel];
    if (fd < 0)
        return -EINVAL;
    if (ioctl (fd, SND_PCM_IOCTL_CHANNEL_READ_EVENT, event) < 0)
        return -errno;

    return 0;
}

int snd_pcm_set_output_class(snd_pcm_t *pcm, uint32_t output_class)
{
    slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x, %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(output_class), SLOG2_FA_END);

    if (!pcm)
        return -EINVAL;

    if (pcm->fd[SND_PCM_CHANNEL_PLAYBACK] < 0)
        return -EINVAL;

    if (ioctl (pcm->fd[SND_PCM_CHANNEL_PLAYBACK], SND_PCM_IOCTL_SET_OUTPUT_CLASS, &output_class) != EOK) {
        return -errno;
    }

    return EOK;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/pcm/pcm.c $ $Rev: 819590 $")
#endif
