/*
 *  PCM Plug-In Interface
 *  Copyright (c) 1999 by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <sys/ioctl.h>
#include <sys/slog2.h>
#include <sys/poll.h>
#include "pcm_local.h"

unsigned char
snd_pcm_plugin_silence (snd_pcm_format_t * format)
{
	switch (format->format)
	{
	case SND_PCM_SFMT_IMA_ADPCM:						   /* special case */
	case SND_PCM_SFMT_MPEG:
	case SND_PCM_SFMT_GSM:
	case SND_PCM_SFMT_MU_LAW:
	case SND_PCM_SFMT_A_LAW:
		return 0;
	case SND_PCM_SFMT_U8:
	case SND_PCM_SFMT_U16_LE:
	case SND_PCM_SFMT_U16_BE:
	case SND_PCM_SFMT_U24_LE:
	case SND_PCM_SFMT_U24_BE:
	case SND_PCM_SFMT_U32_LE:
	case SND_PCM_SFMT_U32_BE:
		return 0x80;
	case SND_PCM_SFMT_S8:
	case SND_PCM_SFMT_S16_LE:
	case SND_PCM_SFMT_S16_BE:
	case SND_PCM_SFMT_S24_LE:
	case SND_PCM_SFMT_S24_BE:
	case SND_PCM_SFMT_S32_LE:
	case SND_PCM_SFMT_S32_BE:
		return 0;
	case SND_PCM_SFMT_FLOAT:
	case SND_PCM_SFMT_FLOAT64:
	case SND_PCM_SFMT_IEC958_SUBFRAME_LE:
	case SND_PCM_SFMT_IEC958_SUBFRAME_BE:
		return 0;
	}
	return 0;
}

snd_pcm_plugin_t *
snd_pcm_plugin_build (const char *name, int extra)
{
	snd_pcm_plugin_t *plugin;

	if (extra < 0)
		return NULL;
	plugin = (snd_pcm_plugin_t *) calloc (1, sizeof (*plugin) + extra);
	if (plugin == NULL)
		return NULL;
	plugin->name = name ? strdup (name) : NULL;
	return plugin;
}

int
snd_pcm_plugin_free (snd_pcm_plugin_t * plugin)
{
	if (plugin)
	{
		if (plugin->private_free)
			plugin->private_free (plugin, plugin->private_data);
		if (plugin->name)
			free (plugin->name);
		free (plugin);
	}
	return 0;
}

int
snd_pcm_plugin_clear (snd_pcm_t * pcm, int channel)
{
	snd_pcm_plugin_t *plugin, *plugin_next;
	int	 idx;

	if (!pcm)
		return -EINVAL;
	plugin = pcm->plugin_first[channel];
	pcm->plugin_first[channel] = NULL;
	pcm->plugin_last[channel] = NULL;
	while (plugin)
	{
		plugin_next = plugin->next;
		snd_pcm_plugin_free (plugin);
		plugin = plugin_next;
	}
	for (idx = 0; idx < 4; idx++)
	{
		if (pcm->plugin_alloc_ptr[idx])
			free (pcm->plugin_alloc_ptr[idx]);
		pcm->plugin_alloc_ptr[idx] = 0;
		pcm->plugin_alloc_size[idx] = 0;
		pcm->plugin_alloc_lock[idx] = 0;
	}
	if (pcm->plugin_alloc_xptr[channel])
		free (pcm->plugin_alloc_xptr[channel]);
	pcm->plugin_alloc_xptr[channel] = NULL;
	pcm->plugin_alloc_xsize[channel] = 0;

	pcm->plugin_partial_block_size[channel] = 0;
	if (pcm->plugin_partial_block_buffer[channel])
		free (pcm->plugin_partial_block_buffer[channel]);
	pcm->plugin_partial_block_buffer[channel] = NULL;
	pcm->plugin_partial_block_len[channel] = 0;
	return 0;
}

int
snd_pcm_plugin_insert (snd_pcm_t * pcm, int channel, snd_pcm_plugin_t * plugin)
{
	if (!pcm || channel < 0 || channel > 1 || !plugin)
		return -EINVAL;
	plugin->next = pcm->plugin_first[channel];
	plugin->prev = NULL;
	if (pcm->plugin_first[channel])
	{
		pcm->plugin_first[channel]->prev = plugin;
		pcm->plugin_first[channel] = plugin;
	}
	else
	{
		pcm->plugin_last[channel] = pcm->plugin_first[channel] = plugin;
	}
	return 0;
}

int
snd_pcm_plugin_append (snd_pcm_t * pcm, int channel, snd_pcm_plugin_t * plugin)
{
	if (!pcm || channel < 0 || channel > 1 || !plugin)
		return -EINVAL;
	plugin->next = NULL;
	plugin->prev = pcm->plugin_last[channel];
	if (pcm->plugin_last[channel])
	{
		pcm->plugin_last[channel]->next = plugin;
		pcm->plugin_last[channel] = plugin;
	}
	else
	{
		pcm->plugin_last[channel] = pcm->plugin_first[channel] = plugin;
	}
	return 0;
}

int
snd_pcm_plugin_remove_to (snd_pcm_t * pcm, int channel, snd_pcm_plugin_t * plugin)
{
	snd_pcm_plugin_t *plugin1, *plugin1_prev;

	if (!pcm || channel < 0 || channel > 1 || !plugin || !plugin->prev)
		return -EINVAL;
	plugin1 = plugin;
	while (plugin1->prev)
		plugin1 = plugin1->prev;
	if (pcm->plugin_first[channel] != plugin1)
		return -EINVAL;
	pcm->plugin_first[channel] = plugin;
	plugin1 = plugin->prev;
	plugin->prev = NULL;
	while (plugin1)
	{
		plugin1_prev = plugin1->prev;
		snd_pcm_plugin_free (plugin1);
		plugin1 = plugin1_prev;
	}
	return 0;
}

snd_pcm_plugin_t *
snd_pcm_plugin_first (snd_pcm_t * pcm, int channel)
{
	if (!pcm || channel < 0 || channel > 1)
		return NULL;
	return pcm->plugin_first[channel];
}

snd_pcm_plugin_t *
snd_pcm_plugin_last (snd_pcm_t * pcm, int channel)
{
	if (!pcm || channel < 0 || channel > 1)
		return NULL;
	return pcm->plugin_last[channel];
}

snd_pcm_plugin_t *
snd_pcm_plugin_find (snd_pcm_t * pcm, int channel, char * plugin_name)
{
	snd_pcm_plugin_t *plugin;

	plugin = pcm->plugin_first[channel];
	while (plugin)
	{
		if (strcmp(plugin_name, plugin->name) == 0)
		{
			return(plugin);
		}
		plugin = plugin->next;
	}
	return NULL;
}

int
snd_pcm_plugin_info (snd_pcm_t * pcm, snd_pcm_channel_info_t * info)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

	int rtn;

	rtn = snd_pcm_channel_info (pcm, info);
	if (!(pcm->plugin_disable_mask & PLUGIN_CONVERSION)) {
		info->formats = snd_pcm_plugin_formats (info->formats);
		info->min_rate = 4000;
		info->max_rate = 192000;
		info->rates = SND_PCM_RATE_8000_48000;
	}

	return rtn;
}

int
snd_pcm_plugin_action (snd_pcm_t * pcm, int channel, int action)
{
	snd_pcm_plugin_t *plugin;
	int	 err;

	if(!pcm || channel < 0 || channel >= SND_PCM_CHANNEL_MAX) {
		return -EINVAL;
	}

	plugin = pcm->plugin_first[channel];
	while (plugin)
	{
		if (plugin->action)
		{
			if ((err = plugin->action (plugin, action)) < 0) {
				return err;
			}
		}
		plugin = plugin->next;
	}
	return 0;
}

unsigned int
snd_pcm_plugin_set_disable (snd_pcm_t * pcm, unsigned int mask)
{
	int rtn;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(mask), SLOG2_FA_END);

	rtn = (pcm->plugin_disable_mask |= mask);

	return rtn;
}

unsigned int
snd_pcm_plugin_get_inactive (snd_pcm_t *pcm)
{
	return pcm->plugin_inactive_mask;
}

unsigned int
snd_pcm_plugin_set_enable (snd_pcm_t * pcm, unsigned int mask)
{
	int rtn;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(mask), SLOG2_FA_END);

	rtn = (pcm->plugin_disable_mask &= ~mask);

	return rtn;
}

int
snd_pcm_plugin_params (snd_pcm_t * pcm, snd_pcm_channel_params_t * params)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);
	return snd_pcm_channel_params(pcm, params);
}

int
snd_pcm_plugin_setup (snd_pcm_t * pcm, snd_pcm_channel_setup_t * setup)
{
	int err;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

	if (!pcm || !setup || setup->channel < 0 || setup->channel > 1)
		return -EINVAL;

	if ((err = snd_pcm_channel_setup (pcm, setup)) < 0) {
		return err;
	}

	if (setup->mode == SND_PCM_MODE_STREAM)
	{
		pdprintf ("params setup: queue_size = %i\n", setup->buf.stream.queue_size);
		setup->buf.stream.queue_size =
			snd_pcm_plugin_transfer_size (pcm, setup->channel, setup->buf.stream.queue_size);
		pdprintf ("params setup: queue_size = %i\n", setup->buf.stream.queue_size);
	}
	else if (setup->mode == SND_PCM_MODE_BLOCK)
	{
		pdprintf ("params setup: frag_size = %i\n", setup->buf.block.frag_size);
		setup->buf.block.frag_size =
			snd_pcm_plugin_transfer_size (pcm, setup->channel, setup->buf.block.frag_size);
		pdprintf ("params setup: frag_size = %i\n", setup->buf.block.frag_size);
	}
	else
	{
		return -EINVAL;
	}

	if (setup->buf.block.max_frag_size == 0)
	{
		if (pcm->plugin_src_mode > SND_SRC_MODE_NORMAL)
			setup->buf.block.max_frag_size = snd_pcm_plugin_src_max_frag(pcm, setup->channel, setup->buf.block.frag_size);
		else
			setup->buf.block.max_frag_size = setup->buf.block.frag_size;
	}

	setup->format = pcm->pre_plugin_format[setup->channel];

	return err;
}

int
snd_pcm_plugin_status (snd_pcm_t * pcm, snd_pcm_channel_status_t * status)
{
	int ret;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

	if( pcm == NULL ) {
		return -EINVAL;
	}

	ret = snd_pcm_channel_status(pcm, status);
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s = %d (%d)", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(ret), SLOG2_FA_UNSIGNED(status->status), SLOG2_FA_END);
	return ret;
}

int
snd_pcm_plugin_prepare (snd_pcm_t * pcm, int channel)
{
	int ret;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

	ret = snd_pcm_channel_prepare (pcm, channel);
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s = %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(ret), SLOG2_FA_END);
	return ret;
}

int
snd_pcm_plugin_playback_drain (snd_pcm_t * pcm)
{
	int ret;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

	ret = snd_pcm_playback_drain (pcm);
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s = %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(ret), SLOG2_FA_END);
	return ret;
}

int
snd_pcm_plugin_flush (snd_pcm_t * pcm, int channel)
{
	int ret;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_END);

	ret = snd_pcm_channel_flush (pcm, channel);
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s = %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(ret), SLOG2_FA_END);
	return ret;
}

int
snd_pcm_plugin_pointer (snd_pcm_t * pcm, int channel, void **ptr, size_t * size)
{
	snd_pcm_plugin_t *plugin;
	int err;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

	if (!ptr || !size)
		return -EINVAL;
	*ptr = NULL;
	if (!pcm || channel < 0 || channel > 1)
		return -EINVAL;
	plugin = pcm->plugin_first[channel];
	if (!plugin)
		return -EINVAL;
	if (plugin->transfer_src_ptr)
	{
		err = plugin->transfer_src_ptr (plugin, ptr, size);
		if (err >= 0)
			return 0;
	}
	if (pcm->plugin_alloc_xptr[channel])
	{
		if (pcm->plugin_alloc_xsize[channel] >= *size)
		{
			*ptr = (char *) pcm->plugin_alloc_xptr[channel];
			return 0;
		}
		*ptr = (char *) realloc (pcm->plugin_alloc_xptr[channel], *size);
	}
	else
	{
		*ptr = (char *) malloc (*size);
		if (*ptr != NULL)
			pcm->plugin_alloc_xsize[channel] = *size;
	}
	if (*ptr == NULL)
		return -ENOMEM;
	pcm->plugin_alloc_xptr[channel] = *ptr;
	return 0;
}

static void *
snd_pcm_plugin_malloc (snd_pcm_t * pcm, ssize_t size)
{
	int	 idx;
	void   *ptr;

	for (idx = 0; idx < 4; idx++)
	{
		if (pcm->plugin_alloc_lock[idx])
			continue;
		if (pcm->plugin_alloc_ptr[idx] == NULL)
			continue;
		if (pcm->plugin_alloc_size[idx] >= size)
		{
			pcm->plugin_alloc_lock[idx] = 1;
			return pcm->plugin_alloc_ptr[idx];
		}
	}
	for (idx = 0; idx < 4; idx++)
	{
		if (pcm->plugin_alloc_lock[idx])
			continue;
		if (pcm->plugin_alloc_ptr[idx] == NULL)
			continue;
		ptr = realloc (pcm->plugin_alloc_ptr[idx], size);
		if (ptr == NULL)
			continue;
		pcm->plugin_alloc_size[idx] = size;
		pcm->plugin_alloc_lock[idx] = 1;
		return pcm->plugin_alloc_ptr[idx] = ptr;
	}
	for (idx = 0; idx < 4; idx++)
	{
		if (pcm->plugin_alloc_ptr[idx] != NULL)
			continue;
		ptr = malloc (size);
		if (ptr == NULL)
			continue;
		pcm->plugin_alloc_size[idx] = size;
		pcm->plugin_alloc_lock[idx] = 1;
		return pcm->plugin_alloc_ptr[idx] = ptr;
	}
	return NULL;
}

static int
snd_pcm_plugin_alloc_unlock (snd_pcm_t * pcm, void *ptr)
{
	int	 idx;

	for (idx = 0; idx < 4; idx++)
	{
		if (pcm->plugin_alloc_ptr[idx] == ptr)
		{
			pcm->plugin_alloc_lock[idx] = 0;
			return 0;
		}
	}
	return -ENOENT;
}

ssize_t
snd_pcm_plugin_write1 (snd_pcm_t * pcm, const void *buffer, size_t count)
{
	snd_pcm_plugin_t *plugin, *next;
	void   *dst_ptr, *dst_ptr1 = NULL, *src_ptr, *src_ptr1 = NULL;
	ssize_t  dst_size, src_size;
	ssize_t size = 0, result = 0;
	int	 err;
	snd_pcm_channel_setup_t setup;

	memset (&setup, 0, sizeof (setup));
	setup.channel = SND_PCM_CHANNEL_PLAYBACK;
	if ((err = snd_pcm_plugin_setup (pcm, &setup)) < 0)
		return (err);
	if (setup.mode == SND_PCM_MODE_BLOCK && pcm->mode & SND_PCM_OPEN_NONBLOCK)
	{
		struct pollfd fds;
		fds.fd = pcm->fd[SND_PCM_CHANNEL_PLAYBACK];
		fds.events = POLLWRNORM;
		fds.revents = 0;
		switch ((result = poll(&fds, 1, 0)))
		{
		case -1:
			return -EIO;
		case 0:
			return -EWOULDBLOCK;
		}
	}

	/* Log Pre-processed PCM Data */
	if (pcm->log_fd != -1 && pcm->log_flags & LOG_PREPROCESSED_DATA)
		write(pcm->log_fd, buffer, count);

	if ((plugin = snd_pcm_plugin_first (pcm, SND_PCM_CHANNEL_PLAYBACK)) == NULL) {
		return snd_pcm_write_internal (pcm, buffer, count);
	}
	src_ptr = (char *) buffer;
	dst_size = src_size = count;

	while (plugin)
	{
		next = plugin->next;
		if (plugin->dst_size)
		{
			dst_size = plugin->dst_size (plugin, dst_size);
			if (dst_size < 0)
			{
				result = dst_size;
				goto __free;
			}
		}
		if (!plugin->transfer && !plugin->pass_through)
		{
			plugin = next;
			continue;
		}
		if (next != NULL)
		{
			if (next->transfer_src_ptr)
			{
				size_t new_dst_size = (size_t)dst_size;
				if ((err = next->transfer_src_ptr (next, &dst_ptr, &new_dst_size)) < 0)
				{
					if (dst_ptr == NULL)
						goto __alloc;
					result = err;
					goto __free;
				}
				dst_size = (ssize_t) new_dst_size;
				if ( dst_size < 0 ) {
					result = -EINVAL;
					goto __free;
				}
			}
			else if (!plugin->pass_through)
			{
			  __alloc:
				dst_ptr = dst_ptr1 = (char *) snd_pcm_plugin_malloc (pcm, dst_size);
				if (dst_ptr == NULL)
				{
					result = -ENOMEM;
					goto __free;
				}
			} else {
				dst_ptr = src_ptr;
				dst_size = src_size;
			}
		}
		else
		{
			dst_ptr = src_ptr;
			dst_size = src_size;
		}
		pdprintf ("write plugin: %s, %i, %i\n", plugin->name, src_size, dst_size);
		if (plugin->pass_through) {
			if ((size = plugin->pass_through (plugin, src_ptr, src_size)) < 0)
			{
				result = size;
				goto __free;
			}
			// we could pass through, but in this case, the next plugin
			// required the source to be filled so we have no choice but doing
			// a memcpy
			if (src_ptr != dst_ptr) {
				memcpy(dst_ptr, src_ptr, min(src_size, dst_size));
			}
		} else if ((size = plugin->transfer (plugin, src_ptr, src_size, dst_ptr, dst_size)) <= 0)
		{
			result = size;
			goto __free;
		}
		if (src_ptr1)
			snd_pcm_plugin_alloc_unlock (pcm, src_ptr1);
		src_ptr = dst_ptr;
		src_ptr1 = dst_ptr1;
		dst_ptr1 = NULL;
		src_size = dst_size = size;
		plugin = next;
	}
	result = snd_pcm_plugin_transfer_size (pcm, SND_PCM_CHANNEL_PLAYBACK, size);
	pdprintf ("size = %i, result = %i, count = %i\n", size, result, count);
	/* Log Post-processed PCM Data */
	if (pcm->log_fd != -1 && pcm->log_flags & LOG_POSTPROCESSED_DATA)
		write(pcm->log_fd, dst_ptr, dst_size);

__free:
	if (dst_ptr1)
		snd_pcm_plugin_alloc_unlock (pcm, dst_ptr1);
	if (src_ptr1)
		snd_pcm_plugin_alloc_unlock (pcm, src_ptr1);
	return result;
}

ssize_t
snd_pcm_plugin_write (snd_pcm_t * pcm, const void *buffer, size_t count)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(buffer), SLOG2_FA_UNSIGNED(count), SLOG2_FA_END);
	return snd_pcm_write(pcm, buffer, count);
}

ssize_t
snd_pcm_plugin_read1 (snd_pcm_t * pcm, void *buffer, size_t count)
{
	snd_pcm_plugin_t *plugin, *next;
	void   *dst_ptr, *dst_ptr1 = NULL, *src_ptr, *src_ptr1 = NULL;
	ssize_t  dst_size, src_size;
	ssize_t size = 0, result = 0;
	int	 err;

	if ((plugin = snd_pcm_plugin_first (pcm, SND_PCM_CHANNEL_CAPTURE)) == NULL) {
		count = snd_pcm_read_internal (pcm, buffer, count);
		/* Log Pre-processed data
		 * Note: If plugins are enabled the PCM data will be written to the log
		 *       from the MMAP or Block plugin (after it is read from io-audio)
		 */
		if (count > 0 && pcm->log_fd != -1 && pcm->log_flags & LOG_PREPROCESSED_DATA)
			write(pcm->log_fd, buffer, count);
		return count;
	}
	src_ptr = NULL;
	src_size = 0;
	dst_size = snd_pcm_plugin_hardware_size (pcm, SND_PCM_CHANNEL_CAPTURE, count);

	if (dst_size < 0)
		return dst_size;
	while (plugin)
	{
		next = plugin->next;
		if (plugin->dst_size)
		{
			dst_size = plugin->dst_size (plugin, dst_size);
			if (dst_size < 0)
			{
				result = dst_size;
				goto __free;
			}
		}
		if (!plugin->transfer && !plugin->pass_through)
		{
			plugin = next;
			if( plugin == NULL ) {
				// If this was the last plugin, then we must transfer the data
				// back to the client's buffer
				memcpy(buffer, src_ptr, src_size);
			}
			continue;
		}
		if (next != NULL)
		{
			if (next->transfer_src_ptr)
			{
				size_t new_dst_size = (size_t)dst_size;
				if ((err = next->transfer_src_ptr (next, &dst_ptr, &new_dst_size)) < 0)
				{
					if (dst_ptr == NULL)
						goto __alloc;
					result = err;
					goto __free;
				}
				dst_size = (ssize_t)new_dst_size;
				if ( dst_size < 0 ) {
					result = -EINVAL;
					goto __free;
				}
			}
			else if (!plugin->pass_through)
			{
			  __alloc:
				dst_ptr = dst_ptr1 = (char *) snd_pcm_plugin_malloc (pcm, dst_size);
				if (dst_ptr == NULL)
				{
					result = -ENOMEM;
					goto __free;
				}
			} else {
				dst_ptr = buffer;
			}
		}
		else
		{
			dst_ptr = buffer;
		}
		pdprintf ("read plugin: %s, %i, %i\n", plugin->name, src_size, dst_size);
		if (plugin->pass_through) {
			if ((size = plugin->pass_through (plugin, src_ptr, src_size)) < 0)
			{
				result = size;
				goto __free;
			}
		} else if ((size = plugin->transfer (plugin, src_ptr, src_size, dst_ptr, dst_size)) <= 0)
		{
			result = size;
			goto __free;
		}
		if (src_ptr1)
			snd_pcm_plugin_alloc_unlock (pcm, src_ptr1);
		src_ptr = dst_ptr;
		src_ptr1 = dst_ptr1;
		dst_ptr1 = NULL;
		plugin = plugin->next;
		src_size = dst_size = size;
	}
	result = size;
	/* Log Post-processed PCM Data */
	if (pcm->log_fd != -1 && pcm->log_flags & LOG_POSTPROCESSED_DATA)
		write(pcm->log_fd, dst_ptr, size);

__free:
	if (dst_ptr1)
		snd_pcm_plugin_alloc_unlock (pcm, dst_ptr1);
	if (src_ptr1)
		snd_pcm_plugin_alloc_unlock (pcm, src_ptr1);
	return result;
}

ssize_t
snd_pcm_plugin_read (snd_pcm_t * pcm, void *buffer, size_t count)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(buffer), SLOG2_FA_UNSIGNED(count), SLOG2_FA_END);
	return snd_pcm_read(pcm, buffer, count);
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/pcm/pcm_plugin.c $ $Rev: 813565 $")
#endif
