/*
 *  PCM Plug-In shared (kernel/library) code
 *  Copyright (c) 1999 by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <sys/slog2.h>
#include "pcm_local.h"

ssize_t
snd_pcm_plugin_transfer_size (snd_pcm_t *pcm, int channel, size_t drv_size)
{
	snd_pcm_plugin_t *plugin, *plugin_prev, *plugin_next;

	if (pcm == NULL || (channel != SND_PCM_CHANNEL_PLAYBACK && channel != SND_PCM_CHANNEL_CAPTURE))
		return -EINVAL;
	if (drv_size == 0)
		return 0;
	if (drv_size < 0)
		return -EINVAL;
	if (channel == SND_PCM_CHANNEL_PLAYBACK)
	{
		plugin = snd_pcm_plugin_last (pcm, channel);
		while (plugin)
		{
			plugin_prev = plugin->prev;
			if (plugin->src_size)
				drv_size = plugin->src_size (plugin, drv_size);
			plugin = plugin_prev;
		}
	}
	else if (channel == SND_PCM_CHANNEL_CAPTURE)
	{
		plugin = snd_pcm_plugin_first (pcm, channel);
		while (plugin)
		{
			plugin_next = plugin->next;
			if (plugin->dst_size)
				drv_size = plugin->dst_size (plugin, drv_size);
			plugin = plugin_next;
		}
	}
	return drv_size;
}

ssize_t
snd_pcm_plugin_hardware_size (snd_pcm_t * pcm, int channel, size_t trf_size)
{
	snd_pcm_plugin_t *plugin, *plugin_prev, *plugin_next;

	if (pcm == NULL || (channel != SND_PCM_CHANNEL_PLAYBACK && channel != SND_PCM_CHANNEL_CAPTURE))
		return -EINVAL;
	if (trf_size == 0)
		return 0;
	if (trf_size < 0)
		return -EINVAL;
	if (channel == SND_PCM_CHANNEL_PLAYBACK)
	{
		plugin = snd_pcm_plugin_first (pcm, channel);
		while (plugin)
		{
			plugin_next = plugin->next;
			if (plugin->dst_size)
				trf_size = plugin->dst_size (plugin, trf_size);
			plugin = plugin_next;
		}
	}
	else if (channel == SND_PCM_CHANNEL_CAPTURE)
	{
		plugin = snd_pcm_plugin_last (pcm, channel);
		while (plugin)
		{
			plugin_prev = plugin->prev;
			if (plugin->src_size)
				trf_size = plugin->src_size (plugin, trf_size);
			plugin = plugin_prev;
		}
	}
	return trf_size;
}

unsigned int
snd_pcm_plugin_formats (unsigned int formats)
{
	int	 linfmts = (SND_PCM_FMT_U8 | SND_PCM_FMT_S8 |
		SND_PCM_FMT_U16_LE | SND_PCM_FMT_S16_LE |
		SND_PCM_FMT_U16_BE | SND_PCM_FMT_S16_BE |
		SND_PCM_FMT_U24_LE | SND_PCM_FMT_S24_LE |
		SND_PCM_FMT_U24_BE | SND_PCM_FMT_S24_BE |
		SND_PCM_FMT_U32_LE | SND_PCM_FMT_S32_LE | SND_PCM_FMT_U32_BE | SND_PCM_FMT_S32_BE);
	formats |= SND_PCM_FMT_MU_LAW;
#ifndef __KERNEL__
	formats |= SND_PCM_FMT_A_LAW | SND_PCM_FMT_IMA_ADPCM;
#endif

	if (formats & linfmts)
		formats |= linfmts;
	return formats;
}


int
snd_pcm_plugin_format (snd_pcm_t * pcm,
	snd_pcm_channel_params_t * params, snd_pcm_channel_params_t * hwparams)
{
	snd_pcm_channel_params_t tmpparams;
	snd_pcm_channel_params_t srcparams;
	snd_pcm_channel_params_t dstparams;
	snd_pcm_plugin_t *plugin;
	int	 err;
	int	 fixed_sizes = 0;  
	int	 playback;

	switch (params->channel)
	{
	case SND_PCM_CHANNEL_PLAYBACK:
		memcpy (&srcparams, params, sizeof (*params));
		memcpy (&dstparams, hwparams, sizeof (*hwparams));
		playback = 1;
		break;
	case SND_PCM_CHANNEL_CAPTURE:
		memcpy (&srcparams, hwparams, sizeof (*hwparams));
		memcpy (&dstparams, params, sizeof (*params));
		playback = 0;
		break;
	default:
		return -EINVAL;
	}
	memcpy (&tmpparams, &srcparams, sizeof (srcparams));

	pdprintf ("srcparams: interleave=%i, format=%i, rate=%i, voices=%i\n",
		srcparams.format.interleave, srcparams.format.format,
		srcparams.format.rate, srcparams.format.voices);
	pdprintf ("dstparams: interleave=%i, format=%i, rate=%i, voices=%i\n",
		dstparams.format.interleave,
		dstparams.format.format, dstparams.format.rate, dstparams.format.voices);

	if (srcparams.format.format != dstparams.format.format ||
		(dstparams.format.voices > 1 && srcparams.format.interleave != dstparams.format.interleave) )
		fixed_sizes = 1;

	/* Convert to interleaved format if needed */
	if (!srcparams.format.interleave &&
		(srcparams.format.voices != dstparams.format.voices ||
			(srcparams.format.rate != dstparams.format.rate && srcparams.format.voices > 1)))
	{
		tmpparams.format.interleave = 1;
		err = snd_pcm_plugin_build_interleave (&srcparams.format, &tmpparams.format, &plugin);
		pdprintf ("params interleave change: src=%i, dst=%i returns %i\n",
			srcparams.format.interleave, tmpparams.format.interleave, err);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.interleave = 1;
		fixed_sizes = 1;
	}

	/* voices reduction */
	if (srcparams.format.voices > dstparams.format.voices || !(pcm->plugin_disable_mask & PLUGIN_VOICE))
	{
		tmpparams.format.voices = dstparams.format.voices;
		err =
			snd_pcm_plugin_build_voices (pcm, &srcparams.format, &tmpparams.format, params->channel,
			&plugin);
		pdprintf ("params voices reduction: src=%i, dst=%i returns %i\n", srcparams.format.voices,
			tmpparams.format.voices, err);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.voices = tmpparams.format.voices;
	}

	/* rate down resampling */
	if (srcparams.format.rate > dstparams.format.rate &&
		snd_pcm_format_linear (srcparams.format.format) &&
		snd_pcm_format_width (srcparams.format.format) <= 16 )
	{
		tmpparams.format.rate = dstparams.format.rate;
		err = snd_pcm_plugin_build_rate_poly (&srcparams.format, &tmpparams.format,
			&plugin, pcm->plugin_src_mode, params->mode & SND_PCM_MODE_MASK, fixed_sizes,
			pcm->setup[params->channel].buf.block.frag_size, playback);
		if (err < 0) 
		{
			snd_pcm_plugin_free (plugin);
			err = snd_pcm_plugin_build_rate (&srcparams.format, &tmpparams.format,
				pcm->plugin_src_method, &plugin, pcm->plugin_src_mode, pcm->plugin_src_target,
				pcm->setup[params->channel].buf.block.frag_size);
		}
		pdprintf ("params rate down resampling: src=%i, dst=%i returns %i\n", srcparams.format.rate,
			tmpparams.format.rate, err);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.rate = tmpparams.format.rate;
		fixed_sizes = 1;
	}

	/* format change (linearization) */
	if (srcparams.format.format != dstparams.format.format &&
		!snd_pcm_format_linear (srcparams.format.format) &&
		!snd_pcm_format_linear (dstparams.format.format))
	{
		tmpparams.format.format = SND_PCM_SFMT_S16_LE;
		switch (srcparams.format.format)
		{
		case SND_PCM_SFMT_MU_LAW:
			err = snd_pcm_plugin_build_mulaw (&srcparams.format, &tmpparams.format, &plugin);
			break;
#ifndef __KERNEL__
		case SND_PCM_SFMT_A_LAW:
			err = snd_pcm_plugin_build_alaw (&srcparams.format, &tmpparams.format, &plugin);
			break;
		case SND_PCM_SFMT_IMA_ADPCM:
			err = snd_pcm_plugin_build_adpcm (&srcparams.format, &tmpparams.format, &plugin);
			break;
#endif
		default:
			return -EINVAL;
		}
		pdprintf ("params format change: src=%i, dst=%i returns %i\n", srcparams.format.format,
			tmpparams.format.format, err);
		if (err < 0)
			return err;
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.format = tmpparams.format.format;
		fixed_sizes = 1;
	}

	/* format change */
	if (srcparams.format.format != dstparams.format.format)
	{
		tmpparams.format.format = dstparams.format.format;
		if (srcparams.format.format == SND_PCM_SFMT_MU_LAW ||
			tmpparams.format.format == SND_PCM_SFMT_MU_LAW)
		{
			err = snd_pcm_plugin_build_mulaw (&srcparams.format, &tmpparams.format, &plugin);
		}
#ifndef __KERNEL__
		else if (srcparams.format.format == SND_PCM_SFMT_A_LAW ||
			tmpparams.format.format == SND_PCM_SFMT_A_LAW)
		{
			err = snd_pcm_plugin_build_alaw (&srcparams.format, &tmpparams.format, &plugin);
		}
		else if (srcparams.format.format == SND_PCM_SFMT_IMA_ADPCM ||
			tmpparams.format.format == SND_PCM_SFMT_IMA_ADPCM)
		{
			err = snd_pcm_plugin_build_adpcm (&srcparams.format, &tmpparams.format, &plugin);
		}
#endif
		else if (snd_pcm_format_linear (srcparams.format.format) &&
			snd_pcm_format_linear (tmpparams.format.format))
		{
			err = snd_pcm_plugin_build_linear (&srcparams.format, &tmpparams.format, &plugin);
		}
		else
			return -EINVAL;
		pdprintf ("params format change: src=%i, dst=%i returns %i\n", srcparams.format.format,
			tmpparams.format.format, err);
		if (err < 0)
			return err;
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.format = tmpparams.format.format;
		fixed_sizes = 1;
	}

	/* rate resampling */
	if (srcparams.format.rate != dstparams.format.rate || pcm->plugin_src_mode > SND_SRC_MODE_NORMAL)
	{
		tmpparams.format.rate = dstparams.format.rate;
		err = snd_pcm_plugin_build_rate_poly (&srcparams.format, &tmpparams.format,
			&plugin, pcm->plugin_src_mode, params->mode & SND_PCM_MODE_MASK, fixed_sizes,
			pcm->setup[params->channel].buf.block.frag_size, playback);
		if (err < 0) 
		{
			snd_pcm_plugin_free (plugin);
			err = snd_pcm_plugin_build_rate (&srcparams.format, &tmpparams.format,
				pcm->plugin_src_method, &plugin, pcm->plugin_src_mode, pcm->plugin_src_target,
				pcm->setup[params->channel].buf.block.frag_size);
		}
		pdprintf ("params rate resampling: src=%i, dst=%i return %i\n", srcparams.format.rate,
			tmpparams.format.rate, err);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.rate = tmpparams.format.rate;
	}

	/* voices extension  */
	if (srcparams.format.voices != dstparams.format.voices)
	{
		tmpparams.format.voices = dstparams.format.voices;
		err =
			snd_pcm_plugin_build_voices (pcm, &srcparams.format, &tmpparams.format, params->channel,
			&plugin);
		pdprintf ("params voices extension: src=%i, dst=%i returns %i\n", srcparams.format.voices,
			tmpparams.format.voices, err);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.voices = tmpparams.format.voices;
	}

	/* interleave change */
	if (dstparams.format.voices > 1 &&
		dstparams.mode == SND_PCM_MODE_BLOCK &&
		srcparams.format.interleave != dstparams.format.interleave)
	{
		tmpparams.format.interleave = dstparams.format.interleave;
		err = snd_pcm_plugin_build_interleave (&srcparams.format, &tmpparams.format, &plugin);
		pdprintf ("params interleave change: src=%i, dst=%i return %i\n",
			srcparams.format.interleave, tmpparams.format.interleave, err);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		err = snd_pcm_plugin_append (pcm, params->channel, plugin);
		if (err < 0)
		{
			snd_pcm_plugin_free (plugin);
			return err;
		}
		srcparams.format.interleave = tmpparams.format.interleave;
	}
	pdprintf ("newparams: interleave=%i, format=%i, rate=%i, voices=%i\n",
		srcparams.format.interleave,
		srcparams.format.format, srcparams.format.rate, srcparams.format.voices);
	return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/pcm/pcm_plugin_build.c $ $Rev: 812869 $")
#endif
