/*
 *  PCM MMAP Plug-In Interface
 *  Copyright (c) 1999 by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <poll.h>
#include "../pcm_local.h"

/* Use the fragment period as a basis for the poll period, Make the poll period 2x the fragment period to account for jitter and preemptions */
#define POLL_TIME(p, i) (2 * ((p)->setup[(i)].buf.block.frag_size / ((p)->setup[(i)].format.rate * (p)->setup[(i)].format.voices * (snd_pcm_format_width((p)->setup[(i)].format.format) / _BITS_BYTE) / 1000)))

/*
 *  Basic mmap plugin
 */

struct mmap_private_data
{
	snd_pcm_t *pcm;
	snd_pcm_mmap_control_t *control;
	int  channel;
	int  frag;
	char *buffer;
};

static  ssize_t mmap_size (snd_pcm_plugin_t * plugin, size_t dst_size)
{
	struct mmap_private_data *data;

	if (plugin == NULL || dst_size <= 0) {
		return -EINVAL;
	}
	data = (struct mmap_private_data *) snd_pcm_plugin_extra_data (plugin);
	if ( data->pcm->setup_is_valid[data->channel] ) {
		int frag_size = data->pcm->setup[data->channel].buf.block.frag_size;
		dst_size = (dst_size / frag_size) * frag_size;
		if ( dst_size == 0 ) {
			dst_size = frag_size;
		}
		return dst_size;
	} else {
		return -EINVAL;
	}
}

static int
poll_playback (snd_pcm_t * pcm)
{
	struct pollfd fds;
	int err;

	fds.fd = pcm->fd[SND_PCM_CHANNEL_PLAYBACK];
	fds.events = POLLWRNORM;
	fds.revents = 0;
	err = poll(&fds, 1, (pcm->mode & SND_PCM_OPEN_NONBLOCK) ? 0 : POLL_TIME(pcm, SND_PCM_CHANNEL_PLAYBACK));
	if ( err > 0 ) {
		err = EOK;
	} else if (err == 0) {

		err = -EWOULDBLOCK;
	} else {
		err = -EIO;
	}
	return err;
}

static int
query_playback (struct mmap_private_data *data, snd_pcm_mmap_control_t * control, int not_use_poll)
{
	int	 err = EOK;

	switch (control->status.status)
	{
	case SND_PCM_STATUS_PAUSED:
		if (!not_use_poll) {
			struct pollfd fds;
			fds.fd = data->pcm->fd[SND_PCM_CHANNEL_PLAYBACK];
			fds.events = POLLWRNORM;
			err = poll(&fds, 1, 0);
			if ( err > 0 ) {
				if ( control->fragments[data->frag].data ) {
					err = -EWOULDBLOCK;
				} else {
					err = EOK;
				}
			} else {

				err = -EIO;
			}
		}
		break;
	case SND_PCM_STATUS_PREPARED:
	case SND_PCM_STATUS_RUNNING:
		if (!not_use_poll)
		{
			if (control->fragments[data->frag].data)
			{
				err = poll_playback (data->pcm);
				if (err < 0)
					return err;
			}
		}
		break;
	default:
		return -EIO;
	}
	return err;
}

static int
poll_capture (snd_pcm_t * pcm)
{
	struct pollfd fds;
	int err;

	fds.fd = pcm->fd[SND_PCM_CHANNEL_CAPTURE];
	fds.events = POLLRDNORM;
	fds.revents = 0;
	err = poll(&fds, 1, (pcm->mode & SND_PCM_OPEN_NONBLOCK) ? 0 : POLL_TIME(pcm, SND_PCM_CHANNEL_CAPTURE));
	if ( err > 0 ) {
		err = EOK;
	} else if (err == 0) {

		err = -EWOULDBLOCK;
	} else {
		err = -EIO;
	}
	return err;

}

static int
query_capture (struct mmap_private_data *data, snd_pcm_mmap_control_t * control, int not_use_poll)
{
	int	 err;

	switch (control->status.status)
	{
	case SND_PCM_STATUS_PREPARED:
		err = snd_pcm_channel_go (data->pcm, data->channel);
		if (err < 0)
			return err;
		break;
	case SND_PCM_STATUS_RUNNING:
		if (!not_use_poll)
		{
			if (!control->fragments[data->frag].data)
			{
				err = poll_capture (data->pcm);
				if (err < 0)
				{
					return err;
				}
			}
		}
		break;
	default:
		return -EIO;
	}
	return 0;
}

static int
mmap_transfer_src_ptr (snd_pcm_plugin_t * plugin, void **buffer, size_t * size)
{
	struct mmap_private_data *data;
	snd_pcm_mmap_control_t *control;
	int	 interleave, err;

	if (plugin == NULL || buffer == NULL || size == NULL)
		return -EINVAL;
	data = (struct mmap_private_data *) snd_pcm_plugin_extra_data (plugin);
	if (data == NULL)
		return -EINVAL;
	control = data->control;
	if (control == NULL)
		return -EINVAL;
	interleave = control->status.voices < 0;
	if (interleave)
	{
		*buffer = data->buffer + control->fragments[data->frag].addr;
		if (data->channel == SND_PCM_CHANNEL_PLAYBACK)
		{
			/* wait until the block is free */
			do
			{
				err = query_playback (data, control, 0);
				if (err < 0)
					return err;
			} while (control->fragments[data->frag].data);
		}
		*size = control->status.frag_size;
	}
	else
	{
		*buffer = NULL;									   /* use another buffer */
		return -EINVAL;
	}
	return 0;
}

static  ssize_t
mmap_transfer (snd_pcm_plugin_t * plugin,
	void *src_ptr, size_t src_size, void *dst_ptr, size_t dst_size)
{
	struct mmap_private_data *data;
	snd_pcm_mmap_control_t *control;
	int	 interleave, voice, err;
	char   *addr;


	if (plugin == NULL || dst_ptr == NULL || dst_size <= 0)
		return -EINVAL;
	data = (struct mmap_private_data *) snd_pcm_plugin_extra_data (plugin);
	if (data == NULL)
		return -EINVAL;
	control = data->control;
	if (control == NULL)
		return -EINVAL;
	interleave = control->status.voices < 0;
	if (interleave)
	{
		if (dst_size != control->status.frag_size)
			return -EINVAL;
	}
	else
	{
		if (dst_size != control->status.frag_size)
			return -EINVAL;
	}
	if (data->channel == SND_PCM_CHANNEL_PLAYBACK)
	{
		do
		{
			err = query_playback (data, control, 0);
			if (err < 0)
				return err;
		}while (control->fragments[data->frag].data);
		/* detect underrun before we fill buffer */
		if (control->status.status != SND_PCM_STATUS_PREPARED &&
			control->status.status != SND_PCM_STATUS_RUNNING &&
			control->status.status != SND_PCM_STATUS_PAUSED)
		{
			return -EIO;
		}
		if (interleave)
		{
			addr = data->buffer + control->fragments[data->frag].addr;
			if (dst_ptr != addr)
			{
				memcpy (addr, dst_ptr, dst_size);
			}
			/* Call into io-audio to complete fragment processing */
			if ((err = snd_pcm_write_internal (data->pcm, &data->frag, sizeof(int))) <= 0)
			{
				return(err);
			}
			data->frag++;
			data->frag %= control->status.frags;
		}
		else
		{
			for (voice = 0; voice < control->status.voices; voice++)
			{
				addr = data->buffer + control->fragments[data->frag].addr;
				if (dst_ptr != addr)
				{
					memcpy (addr, dst_ptr, control->status.frag_size / control->status.voices);
				}
				/* Call into io-audio to complete fragment processing */
				if ((err = snd_pcm_write_internal (data->pcm, &data->frag, sizeof(int))) <= 0)
				{
					return(err);
				}
				data->frag++;
				data->frag %= control->status.frags * control->status.voices;
				dst_ptr += control->status.frag_size / control->status.voices;
			}
		}
		if (control->status.status == SND_PCM_STATUS_PREPARED)
		{
			if (data->pcm->start_mode[data->channel] == SND_PCM_START_DATA)
				err = snd_pcm_channel_go (data->pcm, data->channel);
			else if (data->pcm->start_mode[data->channel] == SND_PCM_START_FULL)
			{
				if (control->status.block == control->status.frags)
					err = snd_pcm_channel_go (data->pcm, data->channel);
			}
		}
		return dst_size;
	}
	else if (data->channel == SND_PCM_CHANNEL_CAPTURE)
	{
		do
		{
			err = query_capture (data, control, 0);
			if (err < 0)
				return err;
		}while (!control->fragments[data->frag].data);

		if (interleave)
		{
			addr = data->buffer + control->fragments[data->frag].addr;
			if (dst_ptr != addr)
				memcpy (dst_ptr, addr, dst_size);
			/* Call into io-audio to complete fragment processing */
			if ((err = snd_pcm_read_internal (data->pcm, &data->frag, sizeof(int))) != EOK)
			{
				return(err);
			}
			control->fragments[data->frag++].data = 0;
			data->frag %= control->status.frags;
			/* Log Pre-processed PCM data */
			if (data->pcm->log_fd != -1 && data->pcm->log_flags & LOG_PREPROCESSED_DATA)
				write(data->pcm->log_fd, addr, dst_size);
		}
		else
		{
			for (voice = 0; voice < control->status.voices; voice++)
			{
				addr = data->buffer + control->fragments[data->frag].addr;
				if (dst_ptr != addr)
					memcpy (dst_ptr, addr, control->status.frag_size / control->status.voices);
				/* Call into io-audio to complete fragment processing */
				if ((err = snd_pcm_read_internal (data->pcm, &data->frag, sizeof(int))) != EOK)
				{
					return(err);
				}
				control->fragments[data->frag++].data = 0;
				data->frag %= control->status.frags * control->status.voices;
				dst_ptr += control->status.frag_size / control->status.voices;
				/* Log Pre-processed PCM data */
				if (data->pcm->log_fd != -1 && data->pcm->log_flags & LOG_PREPROCESSED_DATA)
					write(data->pcm->log_fd, addr, control->status.frag_size / control->status.voices);
			}
		}
		return dst_size;
	}
	else
	{
		return -EINVAL;
	}
}

static int
mmap_action (snd_pcm_plugin_t * plugin, snd_pcm_plugin_action_t action)
{
	struct mmap_private_data *data;

	if (plugin == NULL)
		return -EINVAL;
	data = (struct mmap_private_data *) snd_pcm_plugin_extra_data (plugin);
	if (action == INIT)
	{
		if (data->control)
			snd_pcm_munmap (data->pcm, data->channel);
		return snd_pcm_mmap (data->pcm, data->channel, &data->control, (void **) &data->buffer);
	}
	else if (action == PREPARE)
	{
		data->frag = 0;
	}
	else if (action == DRAIN && data->channel == SND_PCM_CHANNEL_PLAYBACK)
	{
		data->frag = 0;
	}
	else if (action == POST_DRAIN && data->channel == SND_PCM_CHANNEL_PLAYBACK)
	{
		if (data->control)
			snd_pcm_munmap (data->pcm, data->channel);
		return snd_pcm_mmap (data->pcm, data->channel, &data->control, (void **) &data->buffer);
	}
	else if (action == FLUSH)
	{
		data->frag = 0;
	}
	return 0;											   /* silenty ignore other actions */
}

static void
mmap_free (snd_pcm_plugin_t * plugin, void *private_data)
{
	struct mmap_private_data *data;

	if (plugin == NULL)
		return;
	data = (struct mmap_private_data *) snd_pcm_plugin_extra_data (plugin);
	if (data->control)
		snd_pcm_munmap (data->pcm, data->channel);
}

int
snd_pcm_plugin_build_mmap (snd_pcm_t * pcm, int channel, snd_pcm_plugin_t ** r_plugin)
{
	struct mmap_private_data *data;
	snd_pcm_plugin_t *plugin;

	if (r_plugin == NULL)
		return -EINVAL;
	*r_plugin = NULL;
	if (!pcm || channel < 0 || channel > 1)
		return -EINVAL;
	plugin = snd_pcm_plugin_build (channel == SND_PCM_CHANNEL_PLAYBACK ?
		"I/O mmap playback" : "I/O mmap capture", sizeof (struct mmap_private_data));
	if (plugin == NULL)
		return -ENOMEM;
	data = (struct mmap_private_data *) snd_pcm_plugin_extra_data (plugin);
	data->pcm = pcm;
	data->channel = channel;
	plugin->transfer_src_ptr = mmap_transfer_src_ptr;
	plugin->transfer = mmap_transfer;
	plugin->action = mmap_action;
	plugin->private_free = mmap_free;
	plugin->src_size = mmap_size;
	plugin->dst_size = mmap_size;
	*r_plugin = plugin;
	return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/pcm/plugin/mmap.c $ $Rev: 813565 $")
#endif
