/*
 *  Rate conversion Plug-In
 *  Copyright (c) 1999 by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <asound_endian.h>
#include <asound_byteswap.h>
#include "../pcm_local.h"
#include <sys/slog.h>
#include <sys/slogcodes.h>
#include <sys/slog2.h>



#define SHIFT	30
#define BITS	(1<<SHIFT)
#define MASK	(BITS-1)
#define MAX_VOICES 8
#define MAX_SUPPORTED_SRC_METHODS   4     /* 0 = default,1 = 7-pt kaiser windowed, 2 = 20-pt remez, 3 = linear interpolation */
#define CNTRESET 1000000.0f
#define MAXPHASE asrc_maxphase
#define MINPHASE asrc_minphase
#define ASRCTC   asrc_tc
#define ASRCDEBUG asrc_debug

float asrc_maxphase = 1.0025f; //1.02f //1.01f
float asrc_minphase = 0.9975f; //0.98f //0.99f
float asrc_tc = 0.9f; //0.6f //0.999
int asrc_debug = 0;

static int asrc_conf_done = 0;


/*
 *  Basic rate conversion plugin
 */

struct rate_private_data;
typedef void (*resample_f) (struct rate_private_data * data, int voices, char *src_ptr,
	int src_size, char *dst_ptr, int dst_size);
typedef signed short (*take_sample_f) (void *ptr);
typedef void (*put_sample_f) (void *ptr, signed int val);

struct rate_private_data
{
	snd_pcm_format_t format;
	int     src_voices;
	int     dst_voices;
	int     src_rate;
	int     dst_rate;
	int     sample_size;
	resample_f method;
	take_sample_f take;
	put_sample_f put;
	int64_t pitch;
	ssize_t old_src_size, old_dst_size;

	unsigned int src_mode;
	int	icount;
	int ocount;
	int newfragsize;
	double src_ratio;
	double fractfrag;
	int target;
	double filttarget;
	int maxfragsize;
	int minfragsize;
	int dstfragsize;
	int initialized;
	float error;
	float old_error;
	/* used for the linear converter */
	int16_t S[MAX_VOICES];

	/* used for the band limited converter */
	uint16_t Nmult;						/* Filter length for up-conversions */
	int16_t *Imp;						/* Filter coefficients */
	int16_t *ImpD;						/* ImpD[n] = Imp[n+1]-Imp[n] */
	uint16_t LpScl;						/* Unity-gain scale factor */
	uint16_t Nwing;						/* Filter table size */
	int16_t *X[MAX_VOICES];
};


#define RATE_TAKE_SAMPLE(name, type, val) \
static signed short rate_take_sample_##name(void *ptr) \
{ \
	signed int smp = *(type *)ptr; \
	return val; \
}

#define RATE_PUT_SAMPLE(name, type, val) \
static void rate_put_sample_##name(void *ptr, signed int smp) \
{ \
	*(type *)ptr = val; \
}

#if __BYTE_ORDER == __LITTLE_ENDIAN
#define my_little_swap16(x) (x)
#define my_little_swap32(x) (x)
#define my_big_swap16(x) bswap_16(x)
#define my_big_swap32(x) bswap_32(x)
#else
#define my_little_swap16(x) bswap_16(x)
#define my_little_swap32(x) bswap_32(x)
#define my_big_swap16(x) (x)
#define my_big_swap32(x) (x)
#endif

RATE_TAKE_SAMPLE (s8, int8_t, smp << 8)
RATE_TAKE_SAMPLE (u8, int8_t, (smp << 8) ^ 0x8000)
RATE_TAKE_SAMPLE (s16_le, int16_t, my_little_swap16 (smp))
RATE_TAKE_SAMPLE (s16_be, int16_t, my_big_swap16 (smp))
RATE_TAKE_SAMPLE (u16_le, int16_t, my_little_swap16 (smp) ^ 0x8000)
RATE_TAKE_SAMPLE (u16_be, int16_t, my_big_swap16 (smp) ^ 0x8000)
RATE_TAKE_SAMPLE (s24_le, int32_t, my_little_swap32 (smp) >> 8)
RATE_TAKE_SAMPLE (s24_be, int32_t, my_big_swap32 (smp) >> 8)
RATE_TAKE_SAMPLE (u24_le, int32_t, (my_little_swap32 (smp) >> 8) ^ 0x8000)
RATE_TAKE_SAMPLE (u24_be, int32_t, (my_big_swap32 (smp) >> 8) ^ 0x8000)
RATE_TAKE_SAMPLE (s32_le, int32_t, my_little_swap32 (smp) >> 16)
RATE_TAKE_SAMPLE (s32_be, int32_t, my_big_swap32 (smp) >> 16)
RATE_TAKE_SAMPLE (u32_le, int32_t, (my_little_swap32 (smp) >> 16) ^ 0x8000)
RATE_TAKE_SAMPLE (u32_be, int32_t, (my_big_swap32 (smp) >> 16) ^ 0x8000)
	 static take_sample_f rate_take_sample[] = {
		 [SND_PCM_SFMT_S8] rate_take_sample_s8,
		 [SND_PCM_SFMT_U8] rate_take_sample_u8,
		 [SND_PCM_SFMT_S16_LE] rate_take_sample_s16_le,
		 [SND_PCM_SFMT_S16_BE] rate_take_sample_s16_be,
		 [SND_PCM_SFMT_U16_LE] rate_take_sample_u16_le,
		 [SND_PCM_SFMT_U16_BE] rate_take_sample_u16_be,
		 [SND_PCM_SFMT_S24_LE] rate_take_sample_s24_le,
		 [SND_PCM_SFMT_S24_BE] rate_take_sample_s24_be,
		 [SND_PCM_SFMT_U24_LE] rate_take_sample_u24_le,
		 [SND_PCM_SFMT_U24_BE] rate_take_sample_u24_be,
		 [SND_PCM_SFMT_S32_LE] rate_take_sample_s32_le,
		 [SND_PCM_SFMT_S32_BE] rate_take_sample_s32_be,
		 [SND_PCM_SFMT_U32_LE] rate_take_sample_u32_le,
		 [SND_PCM_SFMT_U32_BE] rate_take_sample_u32_be
	 };

RATE_PUT_SAMPLE (s8, int8_t, smp >> 8)
RATE_PUT_SAMPLE (u8, int8_t, (smp >> 8) ^ 0x80)
RATE_PUT_SAMPLE (s16_le, int16_t, my_little_swap16 (smp))
RATE_PUT_SAMPLE (s16_be, int16_t, my_big_swap16 (smp))
RATE_PUT_SAMPLE (u16_le, int16_t, my_little_swap16 (smp ^ 0x8000))
RATE_PUT_SAMPLE (u16_be, int16_t, my_big_swap16 (smp ^ 0x8000))
RATE_PUT_SAMPLE (s24_le, int32_t, my_little_swap32 (smp << 8))
RATE_PUT_SAMPLE (s24_be, int32_t, my_big_swap32 (smp << 8))
RATE_PUT_SAMPLE (u24_le, int32_t, my_little_swap32 ((smp ^ 0x8000) << 8))
RATE_PUT_SAMPLE (u24_be, int32_t, my_big_swap32 ((smp ^ 0x8000) << 8))
RATE_PUT_SAMPLE (s32_le, int32_t, my_little_swap32 (smp << 16))
RATE_PUT_SAMPLE (s32_be, int32_t, my_big_swap32 (smp << 16))
RATE_PUT_SAMPLE (u32_le, int32_t, my_little_swap32 ((smp ^ 0x8000) << 16))
RATE_PUT_SAMPLE (u32_be, int32_t, my_big_swap32 ((smp ^ 0x8000) << 16))
	 static put_sample_f rate_put_sample[] = {
		 [SND_PCM_SFMT_S8] rate_put_sample_s8,
		 [SND_PCM_SFMT_U8] rate_put_sample_u8,
		 [SND_PCM_SFMT_S16_LE] rate_put_sample_s16_le,
		 [SND_PCM_SFMT_S16_BE] rate_put_sample_s16_be,
		 [SND_PCM_SFMT_U16_LE] rate_put_sample_u16_le,
		 [SND_PCM_SFMT_U16_BE] rate_put_sample_u16_be,
		 [SND_PCM_SFMT_S24_LE] rate_put_sample_s24_le,
		 [SND_PCM_SFMT_S24_BE] rate_put_sample_s24_be,
		 [SND_PCM_SFMT_U24_LE] rate_put_sample_u24_le,
		 [SND_PCM_SFMT_U24_BE] rate_put_sample_u24_be,
		 [SND_PCM_SFMT_S32_LE] rate_put_sample_s32_le,
		 [SND_PCM_SFMT_S32_BE] rate_put_sample_s32_be,
		 [SND_PCM_SFMT_U32_LE] rate_put_sample_u32_le,
		 [SND_PCM_SFMT_U32_BE] rate_put_sample_u32_be
	 };

/******************************************************************************/
/* BSD License:                                                               */
/*                                                                            */
/* Copyright (c) 2003.  Julius O. Smith III . All Rights Reserved.            */
/*                                                                            */
/* Redistribution and use in source and binary forms, with or without         */
/* modifications, are permitted provided that the following conditions are    */
/* met:                                                                       */
/* i.          Redistribution of source code must retain the above copyright  */
/* notice, this list of conditions and the following disclaimer.              */
/* ii.         Redistribution in binary form must reproduce the above         */
/* copyright notice, this list of conditions and the following disclaimer in  */
/* the documentation and/or other materials provided with the distribution.   */
/* iii.        Neither the name(s) of Julius O. Smith III  nor the names of   */
/* its/their contributors may be used to endorse or promote products derived  */
/* from this software without specific prior written permission.              */
/*                                                                            */
/* THIS SOFTWARE IS PROVIDED BY Julius O. Smith III  AND CONTRIBUTORS         */
/* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING , BUT NOT LIMITED */
/* TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR */
/* PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE [NAME OF LICENSOR(S)] OR    */
/* CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,      */
/* EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,        */
/* PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR         */
/* PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF     */
/* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING       */
/* NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS         */
/* SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.               */
/******************************************************************************/
/* Conversion constants */
#define Nhc       8
#define Na        7
#define Np       (Nhc+Na)
#define Npc      (1<<Nhc)
#define Amask    ((1<<Na)-1)
#define Pmask    ((1<<Np)-1)
#define Nh       16
#define Nb       16
#define Nhxn     14
#define Nhg      (Nh-Nhxn)
#define NLpScl   13

#define	USE_INTERP		1


/* *INDENT-OFF* */
#define A_FILTER_NMULT 7
#define A_FILTER_SCALE 13175
#define A_FILTER_NWING 768
static int16_t A_FILTER_IMP[] = {
 32767, 32766, 32764, 32760, 32755, 32748, 32740, 32730, 32718, 32705, 32691, 32675, 32657, 32638,
 32618, 32596, 32572, 32547, 32521, 32493, 32463, 32432, 32400, 32366, 32330, 32293, 32255, 32215,
 32173, 32131, 32086, 32041, 31993, 31945, 31895, 31843, 31790, 31736, 31680, 31623, 31565, 31505,
 31444, 31381, 31317, 31251, 31185, 31117, 31047, 30976, 30904, 30831, 30756, 30680, 30602, 30524,
 30444, 30363, 30280, 30197, 30112, 30025, 29938, 29849, 29760, 29669, 29576, 29483, 29388, 29293,
 29196, 29098, 28999, 28899, 28797, 28695, 28591, 28487, 28381, 28275, 28167, 28058, 27948, 27838,
 27726, 27613, 27499, 27385, 27269, 27153, 27035, 26917, 26798, 26678, 26557, 26435, 26312, 26189,
 26064, 25939, 25813, 25686, 25559, 25431, 25302, 25172, 25042, 24910, 24779, 24646, 24513, 24379,
 24245, 24110, 23974, 23838, 23701, 23564, 23426, 23287, 23148, 23008, 22868, 22728, 22587, 22445,
 22303, 22161, 22018, 21875, 21732, 21588, 21443, 21299, 21154, 21008, 20863, 20717, 20571, 20424,
 20278, 20131, 19983, 19836, 19688, 19541, 19393, 19245, 19096, 18948, 18799, 18651, 18502, 18353,
 18204, 18055, 17907, 17758, 17609, 17460, 17311, 17162, 17013, 16864, 16715, 16566, 16418, 16269,
 16121, 15972, 15824, 15676, 15528, 15381, 15233, 15086, 14939, 14792, 14645, 14499, 14353, 14207,
 14061, 13916, 13771, 13626, 13482, 13338, 13194, 13050, 12907, 12765, 12623, 12481, 12339, 12198,
 12058, 11918, 11778, 11639, 11500, 11362, 11224, 11086, 10950, 10813, 10677, 10542, 10407, 10273,
 10140, 10007,  9874,  9742,  9611,  9480,  9350,  9221,  9092,  8963,  8836,  8709,  8582,  8457,
  8332,  8207,  8084,  7961,  7838,  7717,  7596,  7476,  7356,  7237,  7119,  7002,  6886,  6770,
  6655,  6540,  6427,  6314,  6202,  6090,  5980,  5870,  5761,  5653,  5546,  5439,  5333,  5228,
  5124,  5021,  4918,  4816,  4715,  4615,  4516,  4417,  4320,  4223,  4127,  4032,  3938,  3844,
  3751,  3660,  3569,  3479,  3389,  3301,  3213,  3127,  3041,  2956,  2872,  2788,  2706,  2624,
  2543,  2463,  2384,  2306,  2229,  2152,  2077,  2002,  1928,  1855,  1782,  1711,  1640,  1571,
  1502,  1434,  1367,  1300,  1235,  1170,  1106,  1043,   981,   919,   859,   799,   740,   682,
   625,   568,   513,   458,   404,   351,   298,   246,   196,   146,    96,    48,     0,   -46,
   -92,  -138,  -182,  -226,  -269,  -312,  -353,  -394,  -434,  -474,  -513,  -551,  -588,  -624,
  -660,  -695,  -730,  -763,  -797,  -829,  -861,  -891,  -922,  -951,  -980, -1009, -1036, -1063,
 -1090, -1115, -1141, -1165, -1189, -1212, -1235, -1257, -1278, -1299, -1319, -1339, -1358, -1376,
 -1394, -1412, -1428, -1445, -1460, -1475, -1490, -1504, -1518, -1531, -1543, -1555, -1567, -1578,
 -1588, -1599, -1608, -1617, -1626, -1634, -1642, -1649, -1656, -1662, -1668, -1674, -1679, -1684,
 -1688, -1692, -1696, -1699, -1701, -1704, -1706, -1707, -1709, -1710, -1710, -1710, -1710, -1710,
 -1709, -1708, -1706, -1705, -1703, -1700, -1697, -1695, -1691, -1688, -1684, -1680, -1676, -1671,
 -1666, -1661, -1656, -1650, -1645, -1638, -1632, -1626, -1619, -1612, -1605, -1598, -1590, -1583,
 -1575, -1567, -1559, -1550, -1542, -1533, -1524, -1515, -1506, -1497, -1488, -1478, -1468, -1458,
 -1449, -1439, -1428, -1418, -1408, -1397, -1387, -1376, -1365, -1355, -1344, -1333, -1322, -1311,
 -1299, -1288, -1277, -1265, -1254, -1243, -1231, -1220, -1208, -1196, -1185, -1173, -1161, -1150,
 -1138, -1126, -1114, -1103, -1091, -1079, -1067, -1055, -1044, -1032, -1020, -1008,  -997,  -985,
  -973,  -961,  -950,  -938,  -926,  -915,  -903,  -892,  -880,  -869,  -857,  -846,  -835,  -823,
  -812,  -801,  -790,  -779,  -768,  -757,  -746,  -735,  -724,  -714,  -703,  -692,  -682,  -671,
  -661,  -650,  -640,  -630,  -620,  -610,  -600,  -590,  -580,  -570,  -561,  -551,  -542,  -532,
  -523,  -514,  -504,  -495,  -486,  -477,  -469,  -460,  -451,  -442,  -434,  -426,  -417,  -409,
  -401,  -393,  -385,  -377,  -369,  -361,  -354,  -346,  -339,  -332,  -324,  -317,  -310,  -303,
  -296,  -289,  -283,  -276,  -269,  -263,  -257,  -250,  -244,  -238,  -232,  -226,  -220,  -215,
  -209,  -203,  -198,  -192,  -187,  -182,  -177,  -172,  -167,  -162,  -157,  -152,  -147,  -143,
  -138,  -134,  -130,  -125,  -121,  -117,  -113,  -109,  -105,  -101,   -98,   -94,   -91,   -87,
   -84,   -80,   -77,   -74,   -71,   -67,   -64,   -61,   -59,   -56,   -53,   -50,   -48,   -45,
   -43,   -40,   -38,   -35,   -33,   -31,   -29,   -27,   -25,   -23,   -21,   -19,   -17,   -15,
   -14,   -12,   -10,    -9,    -7,    -6,    -4,    -3,    -2,     0,     0,     1,     2,     4,
     5,     6,     7,     8,     9,     9,    10,    11,    12,    13,    13,    14,    15,    15,
    16,    16,    17,    18,    18,    18,    19,    19,    20,    20,    20,    21,    21,    21,
    21,    21,    22,    22,    22,    22,    22,    22,    22,    22,    22,    22,    22,    22,
    22,    22,    22,    22,    22,    22,    22,    22,    22,    22,    21,    21,    21,    21,
    21,    21,    20,    20,    20,    20,    20,    19,    19,    19,    19,    18,    18,    18,
    18,    17,    17,    17,    16,    16,    16,    16,    15,    15,    15,    14,    14,    14,
    14,    13,    13,    13,    12,    12,    12,    12,    11,    11,    11,    10,    10,    10,
    10,     9,     9,     9,     9,     8,     8,     8,     8,     7,     7,     7,     7,     6,
     6,     6,     6,     5,     5,     5,     5,     5,     4,     4,     4,     4,};
static int16_t A_FILTER_IMPD[] = {
  -1,  -2,  -4,  -5,  -7,  -8, -10, -12, -13, -14, -16, -18, -19, -20, -22, -24, -25, -26, -28, -30,
 -31, -32, -34, -36, -37, -38, -40, -42, -42, -45, -45, -48, -48, -50, -52, -53, -54, -56, -57, -58,
 -60, -61, -63, -64, -66, -66, -68, -70, -71, -72, -73, -75, -76, -78, -78, -80, -81, -83, -83, -85,
 -87, -87, -89, -89, -91, -93, -93, -95, -95, -97, -98, -99,-100,-102,-102,-104,-104,-106,-106,-108,
-109,-110,-110,-112,-113,-114,-114,-116,-116,-118,-118,-119,-120,-121,-122,-123,-123,-125,-125,-126,
-127,-127,-128,-129,-130,-130,-132,-131,-133,-133,-134,-134,-135,-136,-136,-137,-137,-138,-139,-139,
-140,-140,-140,-141,-142,-142,-142,-143,-143,-143,-144,-145,-144,-145,-146,-145,-146,-146,-147,-146,
-147,-148,-147,-148,-147,-148,-148,-149,-148,-149,-148,-149,-149,-149,-149,-148,-149,-149,-149,-149,
-149,-149,-149,-149,-149,-148,-149,-148,-149,-148,-148,-148,-147,-148,-147,-147,-147,-147,-146,-146,
-146,-146,-145,-145,-145,-144,-144,-144,-144,-143,-142,-142,-142,-142,-141,-140,-140,-140,-139,-139,
-138,-138,-138,-136,-137,-136,-135,-135,-134,-133,-133,-133,-132,-131,-131,-130,-129,-129,-129,-127,
-127,-127,-125,-125,-125,-123,-123,-123,-121,-121,-120,-120,-119,-118,-117,-116,-116,-115,-115,-113,
-113,-112,-112,-110,-110,-109,-108,-107,-107,-106,-105,-104,-103,-103,-102,-101,-100, -99, -99, -97,
 -97, -96, -95, -94, -94, -93, -91, -91, -90, -90, -88, -88, -86, -86, -85, -84, -84, -82, -82, -81,
 -80, -79, -78, -77, -77, -75, -75, -74, -73, -73, -71, -71, -69, -69, -68, -67, -67, -65, -65, -64,
 -63, -62, -62, -60, -60, -59, -58, -57, -57, -55, -55, -54, -53, -53, -52, -50, -50, -50, -48, -48,
 -46, -46, -46, -44, -44, -43, -43, -41, -41, -40, -40, -39, -38, -37, -36, -36, -35, -35, -33, -34,
 -32, -32, -30, -31, -29, -29, -29, -27, -27, -27, -25, -26, -24, -24, -23, -23, -22, -21, -21, -20,
 -20, -19, -18, -18, -18, -16, -17, -15, -15, -15, -14, -14, -13, -12, -12, -12, -11, -10, -11,  -9,
  -9,  -9,  -8,  -8,  -7,  -7,  -6,  -6,  -6,  -5,  -5,  -4,  -4,  -4,  -3,  -2,  -3,  -2,  -1,  -2,
  -1,   0,   0,   0,   0,   1,   1,   2,   1,   2,   3,   3,   2,   4,   3,   4,   4,   4,   5,   5,
   5,   5,   6,   5,   7,   6,   6,   7,   7,   7,   7,   8,   7,   8,   8,   8,   9,   8,   9,   9,
   9,   9,   9,   9,  10,  10,  10,   9,  10,  11,  10,  10,  11,  10,  11,  11,  10,  11,  11,  11,
  11,  12,  11,  11,  12,  11,  11,  12,  11,  12,  12,  11,  12,  12,  11,  12,  12,  12,  11,  12,
  12,  12,  12,  11,  12,  12,  12,  11,  12,  12,  12,  11,  12,  12,  11,  12,  11,  12,  11,  12,
  11,  11,  12,  11,  11,  11,  11,  11,  11,  11,  11,  11,  10,  11,  11,  10,  11,  10,  11,  10,
  10,  10,  10,  10,  10,  10,  10,   9,  10,   9,  10,   9,   9,  10,   9,   9,   9,   8,   9,   9,
   9,   8,   8,   9,   8,   8,   8,   8,   8,   8,   8,   7,   8,   7,   7,   8,   7,   7,   7,   7,
   7,   6,   7,   7,   6,   6,   7,   6,   6,   6,   6,   6,   5,   6,   6,   5,   6,   5,   5,   5,
   5,   5,   5,   5,   5,   5,   4,   5,   4,   4,   5,   4,   4,   4,   4,   4,   4,   3,   4,   3,
   4,   3,   4,   3,   3,   3,   4,   3,   3,   2,   3,   3,   3,   2,   3,   2,   3,   2,   3,   2,
   2,   2,   2,   2,   2,   2,   2,   2,   2,   1,   2,   2,   1,   2,   1,   2,   1,   1,   2,   0,
   1,   1,   2,   1,   1,   1,   1,   1,   0,   1,   1,   1,   1,   0,   1,   1,   0,   1,   0,   1,
   1,   0,   0,   1,   0,   1,   0,   0,   1,   0,   0,   0,   0,   1,   0,   0,   0,   0,   0,   0,
   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  -1,   0,   0,   0,   0,
   0,  -1,   0,   0,   0,   0,  -1,   0,   0,   0,  -1,   0,   0,   0,  -1,   0,   0,  -1,   0,   0,
   0,  -1,   0,   0,  -1,   0,   0,   0,  -1,   0,   0,  -1,   0,   0,   0,  -1,   0,   0,  -1,   0,
   0,   0,  -1,   0,   0,   0,  -1,   0,   0,   0,  -1,   0,   0,   0,  -1,   0,   0,   0,  -1,   0,
   0,   0,   0,  -1,   0,   0,   0,  -4,};


#define B_FILTER_NMULT 20
#define B_FILTER_SCALE 16059
#define B_FILTER_NWING 2432
static int16_t B_FILTER_IMP[] = {
32767, 32766, 32764, 32760, 32754, 32747, 32738, 32728, 32715, 32702,
32686, 32669, 32651, 32630, 32608, 32585, 32560, 32534, 32505, 32475,
32444, 32411, 32376, 32341, 32303, 32263, 32222, 32180, 32136, 32090,
32043, 31995, 31944, 31893, 31839, 31784, 31728, 31670, 31610, 31550,
31487, 31423, 31357, 31290, 31222, 31152, 31080, 31008, 30933, 30858,
30780, 30702, 30622, 30540, 30457, 30373, 30287, 30200, 30111, 30021,
29930, 29837, 29743, 29648, 29551, 29453, 29353, 29253, 29151, 29048,
28943, 28838, 28730, 28622, 28512, 28401, 28289, 28176, 28061, 27946,
27829, 27711, 27591, 27471, 27350, 27227, 27103, 26978, 26852, 26725,
26597, 26468, 26337, 26206, 26073, 25940, 25806, 25670, 25534, 25397,
25258, 25119, 24978, 24838, 24695, 24553, 24408, 24264, 24118, 23972,
23824, 23676, 23527, 23378, 23227, 23076, 22924, 22771, 22617, 22463,
22308, 22152, 21996, 21839, 21681, 21523, 21364, 21205, 21044, 20884,
20722, 20560, 20398, 20235, 20071, 19907, 19743, 19578, 19412, 19246,
19080, 18913, 18746, 18579, 18411, 18243, 18074, 17905, 17736, 17566,
17396, 17226, 17056, 16885, 16714, 16543, 16371, 16200, 16028, 15856,
15684, 15512, 15340, 15167, 14995, 14822, 14649, 14477, 14304, 14131,
13958, 13785, 13613, 13440, 13267, 13095, 12922, 12749, 12577, 12405,
12232, 12060, 11888, 11717, 11545, 11374, 11203, 11032, 10861, 10691,
10520, 10350, 10181, 10011,  9842,  9674,  9505,  9337,  9169,  9002,
 8835,  8669,  8503,  8337,  8172,  8007,  7843,  7679,  7515,  7352,
 7190,  7028,  6867,  6706,  6546,  6386,  6227,  6069,  5911,  5753,
 5597,  5441,  5285,  5131,  4977,  4823,  4671,  4519,  4367,  4217,
 4067,  3918,  3770,  3622,  3475,  3329,  3184,  3039,  2896,  2753,
 2611,  2470,  2330,  2190,  2052,  1914,  1777,  1641,  1506,  1372,
 1238,  1106,   975,   844,   715,   586,   459,   332,   206,    81,
  -41,  -164,  -286,  -406,  -526,  -645,  -762,  -879,  -994, -1109,
-1222, -1335, -1446, -1556, -1665, -1774, -1881, -1986, -2091, -2195,
-2298, -2399, -2500, -2599, -2697, -2794, -2890, -2985, -3079, -3171,
-3262, -3353, -3442, -3530, -3617, -3702, -3787, -3870, -3952, -4033,
-4113, -4192, -4269, -4346, -4421, -4495, -4567, -4639, -4710, -4779,
-4847, -4914, -4980, -5044, -5107, -5170, -5231, -5290, -5349, -5407,
-5463, -5518, -5572, -5625, -5676, -5726, -5776, -5824, -5871, -5916,
-5961, -6004, -6046, -6087, -6127, -6166, -6203, -6240, -6275, -6309,
-6342, -6374, -6404, -6434, -6462, -6489, -6515, -6540, -6564, -6587,
-6608, -6629, -6648, -6667, -6684, -6700, -6715, -6729, -6742, -6754,
-6764, -6774, -6783, -6790, -6797, -6802, -6806, -6810, -6812, -6813,
-6814, -6813, -6811, -6809, -6805, -6800, -6795, -6788, -6780, -6772,
-6762, -6752, -6741, -6728, -6715, -6701, -6686, -6670, -6653, -6635,
-6617, -6597, -6577, -6556, -6533, -6511, -6487, -6462, -6437, -6411,
-6384, -6356, -6327, -6298, -6268, -6237, -6205, -6173, -6140, -6106,
-6071, -6036, -6000, -5963, -5926, -5888, -5849, -5810, -5769, -5729,
-5687, -5646, -5603, -5560, -5516, -5472, -5427, -5381, -5335, -5288,
-5241, -5194, -5145, -5097, -5047, -4998, -4947, -4897, -4846, -4794,
-4742, -4689, -4636, -4583, -4529, -4475, -4421, -4366, -4311, -4255,
-4199, -4143, -4086, -4029, -3972, -3914, -3856, -3798, -3740, -3681,
-3622, -3563, -3503, -3444, -3384, -3324, -3263, -3203, -3142, -3081,
-3020, -2959, -2898, -2837, -2775, -2713, -2652, -2590, -2528, -2466,
-2404, -2342, -2279, -2217, -2155, -2093, -2030, -1968, -1906, -1843,
-1781, -1719, -1657, -1594, -1532, -1470, -1408, -1346, -1285, -1223,
-1161, -1100, -1038,  -977,  -916,  -855,  -794,  -734,  -673,  -613,
 -553,  -493,  -433,  -374,  -314,  -255,  -197,  -138,   -80,   -21,
   35,    93,   150,   207,   264,   320,   377,   432,   488,   543,
  598,   653,   707,   761,   814,   867,   920,   973,  1025,  1076,
 1128,  1179,  1229,  1279,  1329,  1378,  1427,  1475,  1523,  1571,
 1618,  1665,  1711,  1757,  1802,  1847,  1891,  1935,  1979,  2022,
 2064,  2106,  2148,  2189,  2229,  2269,  2309,  2348,  2386,  2424,
 2462,  2499,  2535,  2571,  2606,  2641,  2675,  2709,  2742,  2775,
 2807,  2838,  2869,  2900,  2929,  2959,  2988,  3016,  3043,  3070,
 3097,  3123,  3148,  3173,  3197,  3221,  3244,  3266,  3288,  3310,
 3330,  3351,  3370,  3389,  3408,  3425,  3443,  3459,  3476,  3491,
 3506,  3520,  3534,  3547,  3560,  3572,  3584,  3595,  3605,  3615,
 3624,  3632,  3640,  3648,  3655,  3661,  3667,  3672,  3677,  3681,
 3684,  3687,  3690,  3691,  3693,  3693,  3694,  3693,  3692,  3691,
 3689,  3686,  3683,  3680,  3676,  3671,  3666,  3660,  3654,  3647,
 3640,  3632,  3624,  3615,  3606,  3596,  3586,  3575,  3564,  3552,
 3540,  3527,  3514,  3500,  3486,  3472,  3457,  3441,  3425,  3409,
 3392,  3375,  3357,  3339,  3320,  3302,  3282,  3262,  3242,  3222,
 3201,  3179,  3157,  3135,  3113,  3090,  3067,  3043,  3019,  2995,
 2970,  2945,  2919,  2894,  2868,  2841,  2815,  2788,  2760,  2733,
 2705,  2676,  2648,  2619,  2590,  2561,  2531,  2501,  2471,  2440,
 2410,  2379,  2348,  2316,  2285,  2253,  2221,  2189,  2156,  2124,
 2091,  2058,  2025,  1991,  1958,  1924,  1890,  1856,  1822,  1788,
 1753,  1719,  1684,  1649,  1614,  1579,  1544,  1509,  1473,  1438,
 1402,  1367,  1331,  1295,  1259,  1224,  1188,  1152,  1116,  1080,
 1044,  1008,   971,   935,   899,   863,   827,   791,   755,   719,
  683,   647,   611,   575,   539,   503,   467,   432,   396,   360,
  325,   289,   254,   219,   184,   148,   113,    79,    44,     9,
  -24,   -59,   -93,  -127,  -161,  -195,  -228,  -262,  -295,  -328,
 -361,  -394,  -427,  -459,  -492,  -524,  -556,  -587,  -619,  -650,
 -681,  -712,  -743,  -773,  -804,  -834,  -864,  -893,  -922,  -952,
 -980, -1009, -1037, -1065, -1093, -1121, -1148, -1175, -1202, -1228,
-1255, -1281, -1306, -1332, -1357, -1382, -1406, -1430, -1454, -1478,
-1501, -1524, -1547, -1569, -1591, -1613, -1635, -1656, -1677, -1697,
-1717, -1737, -1757, -1776, -1795, -1813, -1832, -1849, -1867, -1884,
-1901, -1918, -1934, -1950, -1965, -1980, -1995, -2009, -2023, -2037,
-2051, -2064, -2076, -2089, -2101, -2112, -2124, -2135, -2145, -2155,
-2165, -2175, -2184, -2193, -2201, -2209, -2217, -2224, -2231, -2238,
-2244, -2250, -2256, -2261, -2266, -2271, -2275, -2279, -2282, -2286,
-2288, -2291, -2293, -2295, -2296, -2297, -2298, -2298, -2298, -2298,
-2297, -2296, -2295, -2293, -2291, -2289, -2286, -2284, -2280, -2277,
-2273, -2268, -2264, -2259, -2254, -2248, -2242, -2236, -2230, -2223,
-2216, -2208, -2201, -2193, -2184, -2176, -2167, -2158, -2148, -2139,
-2129, -2118, -2108, -2097, -2086, -2074, -2063, -2051, -2039, -2026,
-2013, -2000, -1987, -1974, -1960, -1946, -1932, -1917, -1903, -1888,
-1873, -1857, -1842, -1826, -1810, -1793, -1777, -1760, -1743, -1726,
-1709, -1692, -1674, -1656, -1638, -1620, -1601, -1583, -1564, -1545,
-1526, -1507, -1487, -1468, -1448, -1428, -1408, -1388, -1368, -1348,
-1327, -1306, -1286, -1265, -1244, -1222, -1201, -1180, -1158, -1137,
-1115, -1093, -1072, -1050, -1028, -1006,  -983,  -961,  -939,  -916,
 -894,  -872,  -849,  -826,  -804,  -781,  -758,  -736,  -713,  -690,
 -667,  -644,  -622,  -599,  -576,  -553,  -530,  -507,  -484,  -461,
 -438,  -415,  -392,  -370,  -347,  -324,  -301,  -278,  -256,  -233,
 -210,  -188,  -165,  -142,  -120,   -98,   -75,   -53,   -31,    -9,
   12,    34,    56,    78,   100,   122,   143,   165,   186,   207,
  229,   250,   271,   291,   312,   333,   353,   374,   394,   414,
  434,   454,   474,   493,   513,   532,   551,   570,   589,   608,
  626,   645,   663,   681,   699,   717,   734,   752,   769,   786,
  803,   820,   836,   852,   869,   885,   900,   916,   931,   947,
  962,   977,   991,  1006,  1020,  1034,  1048,  1062,  1075,  1089,
 1102,  1115,  1127,  1140,  1152,  1164,  1176,  1187,  1199,  1210,
 1221,  1232,  1242,  1253,  1263,  1273,  1282,  1292,  1301,  1310,
 1319,  1327,  1336,  1344,  1352,  1359,  1367,  1374,  1381,  1388,
 1394,  1400,  1406,  1412,  1418,  1423,  1428,  1433,  1438,  1442,
 1447,  1451,  1454,  1458,  1461,  1464,  1467,  1470,  1472,  1475,
 1477,  1478,  1480,  1481,  1482,  1483,  1484,  1484,  1484,  1484,
 1484,  1483,  1483,  1482,  1481,  1479,  1478,  1476,  1474,  1472,
 1469,  1467,  1464,  1461,  1458,  1454,  1451,  1447,  1443,  1438,
 1434,  1429,  1424,  1419,  1414,  1409,  1403,  1397,  1391,  1385,
 1379,  1372,  1365,  1359,  1351,  1344,  1337,  1329,  1321,  1313,
 1305,  1297,  1289,  1280,  1271,  1262,  1253,  1244,  1235,  1225,
 1215,  1205,  1195,  1185,  1175,  1165,  1154,  1143,  1133,  1122,
 1111,  1099,  1088,  1077,  1065,  1053,  1042,  1030,  1018,  1006,
  993,   981,   969,   956,   943,   931,   918,   905,   892,   879,
  866,   852,   839,   825,   812,   798,   785,   771,   757,   743,
  729,   715,   701,   687,   673,   659,   644,   630,   616,   601,
  587,   573,   558,   543,   529,   514,   500,   485,   470,   456,
  441,   426,   411,   397,   382,   367,   352,   338,   323,   308,
  293,   278,   264,   249,   234,   220,   205,   190,   175,   161,
  146,   132,   117,   103,    88,    74,    59,    45,    31,    16,
    2,   -11,   -25,   -39,   -53,   -67,   -81,   -95,  -109,  -123,
 -136,  -150,  -164,  -177,  -190,  -204,  -217,  -230,  -243,  -256,
 -269,  -282,  -295,  -307,  -320,  -332,  -345,  -357,  -369,  -381,
 -393,  -405,  -417,  -429,  -440,  -452,  -463,  -475,  -486,  -497,
 -508,  -519,  -529,  -540,  -550,  -561,  -571,  -581,  -591,  -601,
 -611,  -620,  -630,  -639,  -649,  -658,  -667,  -676,  -684,  -693,
 -702,  -710,  -718,  -726,  -734,  -742,  -750,  -757,  -765,  -772,
 -779,  -786,  -793,  -799,  -806,  -812,  -819,  -825,  -831,  -837,
 -842,  -848,  -853,  -859,  -864,  -869,  -874,  -878,  -883,  -887,
 -891,  -895,  -899,  -903,  -907,  -910,  -914,  -917,  -920,  -923,
 -926,  -928,  -931,  -933,  -935,  -937,  -939,  -941,  -943,  -944,
 -945,  -947,  -948,  -949,  -949,  -950,  -950,  -951,  -951,  -951,
 -951,  -950,  -950,  -950,  -949,  -948,  -947,  -946,  -945,  -943,
 -942,  -940,  -939,  -937,  -935,  -933,  -930,  -928,  -925,  -923,
 -920,  -917,  -914,  -911,  -907,  -904,  -900,  -897,  -893,  -889,
 -885,  -881,  -877,  -872,  -868,  -863,  -859,  -854,  -849,  -844,
 -839,  -833,  -828,  -823,  -817,  -811,  -806,  -800,  -794,  -788,
 -782,  -775,  -769,  -763,  -756,  -749,  -743,  -736,  -729,  -722,
 -715,  -708,  -701,  -694,  -686,  -679,  -671,  -664,  -656,  -648,
 -641,  -633,  -625,  -617,  -609,  -601,  -593,  -585,  -576,  -568,
 -560,  -551,  -543,  -534,  -526,  -517,  -508,  -500,  -491,  -482,
 -473,  -464,  -455,  -446,  -437,  -428,  -419,  -410,  -401,  -392,
 -383,  -374,  -365,  -355,  -346,  -337,  -328,  -318,  -309,  -300,
 -290,  -281,  -272,  -262,  -253,  -244,  -234,  -225,  -216,  -206,
 -197,  -188,  -178,  -169,  -160,  -150,  -141,  -132,  -122,  -113,
 -104,   -95,   -85,   -76,   -67,   -58,   -49,   -40,   -31,   -22,
  -13,    -4,     4,    13,    22,    31,    40,    48,    57,    66,
   74,    83,    92,   100,   108,   117,   125,   134,   142,   150,
  158,   166,   174,   182,   190,   198,   206,   214,   222,   229,
  237,   244,   252,   259,   266,   274,   281,   288,   295,   302,
  309,   316,   322,   329,   336,   342,   349,   355,   361,   368,
  374,   380,   386,   392,   397,   403,   409,   414,   420,   425,
  431,   436,   441,   446,   451,   456,   461,   466,   470,   475,
  479,   484,   488,   492,   496,   500,   504,   508,   512,   516,
  519,   523,   526,   529,   533,   536,   539,   542,   545,   547,
  550,   553,   555,   558,   560,   562,   564,   566,   568,   570,
  572,   574,   575,   577,   578,   579,   581,   582,   583,   584,
  584,   585,   586,   587,   587,   587,   588,   588,   588,   588,
  588,   588,   588,   588,   587,   587,   586,   585,   585,   584,
  583,   582,   581,   580,   579,   577,   576,   575,   573,   572,
  570,   568,   566,   564,   562,   560,   558,   556,   554,   551,
  549,   547,   544,   541,   539,   536,   533,   530,   528,   525,
  521,   518,   515,   512,   509,   505,   502,   499,   495,   491,
  488,   484,   480,   477,   473,   469,   465,   461,   457,   453,
  448,   444,   440,   436,   431,   427,   422,   418,   413,   408,
  404,   399,   394,   389,   384,   379,   374,   369,   364,   359,
  354,   349,   343,   338,   333,   328,   322,   317,   312,   306,
  301,   295,   290,   285,   279,   274,   268,   263,   257,   251,
  246,   240,   235,   229,   223,   218,   212,   207,   201,   195,
  190,   184,   178,   173,   167,   161,   156,   150,   144,   138,
  133,   127,   121,   116,   110,   105,    99,    93,    88,    82,
   76,    71,    65,    60,    54,    49,    43,    38,    32,    27,
   21,    16,    10,     5,     0,    -5,   -10,   -15,   -20,   -26,
  -31,   -36,   -41,   -46,   -51,   -56,   -61,   -66,   -71,   -76,
  -81,   -86,   -91,   -96,  -101,  -105,  -110,  -115,  -119,  -124,
 -129,  -133,  -138,  -142,  -146,  -151,  -155,  -159,  -164,  -168,
 -172,  -176,  -180,  -184,  -188,  -192,  -196,  -200,  -204,  -207,
 -211,  -215,  -218,  -222,  -226,  -229,  -232,  -236,  -239,  -242,
 -246,  -249,  -252,  -255,  -258,  -261,  -264,  -267,  -270,  -272,
 -275,  -278,  -280,  -283,  -285,  -288,  -290,  -293,  -295,  -297,
 -299,  -301,  -304,  -306,  -307,  -309,  -311,  -313,  -315,  -316,
 -318,  -320,  -321,  -323,  -324,  -325,  -327,  -328,  -329,  -330,
 -331,  -333,  -334,  -334,  -335,  -336,  -337,  -338,  -338,  -339,
 -339,  -340,  -340,  -341,  -341,  -341,  -342,  -342,  -342,  -342,
 -342,  -342,  -342,  -342,  -342,  -342,  -341,  -341,  -341,  -340,
 -340,  -339,  -339,  -338,  -337,  -337,  -336,  -335,  -334,  -334,
 -333,  -332,  -331,  -330,  -328,  -327,  -326,  -325,  -324,  -322,
 -321,  -320,  -318,  -317,  -315,  -314,  -312,  -310,  -309,  -307,
 -305,  -303,  -301,  -300,  -298,  -296,  -294,  -292,  -290,  -288,
 -285,  -283,  -281,  -279,  -277,  -274,  -272,  -270,  -267,  -265,
 -263,  -260,  -258,  -255,  -253,  -250,  -247,  -245,  -242,  -239,
 -237,  -234,  -231,  -229,  -226,  -223,  -220,  -217,  -214,  -212,
 -209,  -206,  -203,  -200,  -197,  -194,  -191,  -188,  -185,  -182,
 -179,  -176,  -172,  -169,  -166,  -163,  -160,  -157,  -154,  -150,
 -147,  -144,  -141,  -138,  -135,  -131,  -128,  -125,  -122,  -118,
 -115,  -112,  -109,  -105,  -102,   -99,   -96,   -92,   -89,   -86,
  -83,   -79,   -76,   -73,   -70,   -66,   -63,   -60,   -57,   -53,
  -50,   -47,   -44,   -40,   -37,   -34,   -31,   -28,   -25,   -21,
  -18,   -15,   -12,    -9,    -6,    -3,     0,     3,     6,     9,
   12,    15,    18,    21,    24,    27,    30,    33,    36,    38,
   41,    44,    47,    50,    53,    56,    58,    61,    64,    67,
   69,    72,    75,    77,    80,    83,    85,    88,    90,    93,
   95,    98,   100,   103,   105,   107,   110,   112,   115,   117,
  119,   121,   124,   126,   128,   130,   132,   134,   136,   138,
  141,   143,   144,   146,   148,   150,   152,   154,   156,   158,
  159,   161,   163,   165,   166,   168,   170,   171,   173,   174,
  176,   177,   179,   180,   181,   183,   184,   185,   187,   188,
  189,   190,   192,   193,   194,   195,   196,   197,   198,   199,
  200,   201,   202,   203,   203,   204,   205,   206,   207,   207,
  208,   209,   209,   210,   210,   211,   212,   212,   212,   213,
  213,   214,   214,   214,   215,   215,   215,   216,   216,   216,
  216,   216,   216,   216,   217,   217,   217,   217,   217,   217,
  216,   216,   216,   216,   216,   216,   216,   215,   215,   215,
  215,   214,   214,   214,   213,   213,   212,   212,   212,   211,
  211,   210,   210,   209,   209,   208,   207,   207,   206,   206,
  205,   204,   204,   203,   202,   201,   201,   200,   199,   198,
  198,   197,   196,   195,   194,   193,   192,   192,   191,   190,
  189,   188,   187,   186,   185,   184,   183,   182,   181,   180,
  179,   178,   177,   176,   175,   174,   172,   171,   170,   169,
  168,   167,   166,   165,   163,   162,   161,   160,   159,   158,
  156,   155,   154,   153,   152,   150,   149,   148,   147,   146,
  144,   143,   142,   141,   140,   138,   137,   136,   135,   134,
  132,   131,   130,   129,   128,   126,   125,   124,   123,   122,
  120,   119,   118,   117,   116,   115,   113,   112,   111,   110,
  109,   108,   107,   105,   104,   103,   102,   101,   100,    98,
   97,    96,    95,    94,    93,    92,    90,    89,    88,    87,
   86,    84,    83,    82,    81,    79,    78,    77,    75,    74,
   73,    71,    70,    68,    67,    65,    63,    62,    60,    58,
   57,    55,    53,    51,    49,    47,    45,    43,    41,    39,
   37,    35,    33,    31,    28,    26,    24,    22,    20,    18,
   16,    14,    12,    10,     8,     6,     5,     3,     2,     0,
    0,    -1,    -2,    -2,    -3,    -3,    -4,    -4,    -4,    -4,
   -4,    -4,    -3,    -3,    -3,    -2,    -2,    -1,    -1,     0,
    0,     0 };

static int16_t B_FILTER_IMPD[] = {
   -1,    -2,    -4,    -6,    -7,    -9,   -10,   -13,   -13,   -16,
  -17,   -18,   -21,   -22,   -23,   -25,   -26,   -29,   -30,   -31,
  -33,   -35,   -35,   -38,   -40,   -41,   -42,   -44,   -46,   -47,
  -48,   -51,   -51,   -54,   -55,   -56,   -58,   -60,   -60,   -63,
  -64,   -66,   -67,   -68,   -70,   -72,   -72,   -75,   -75,   -78,
  -78,   -80,   -82,   -83,   -84,   -86,   -87,   -89,   -90,   -91,
  -93,   -94,   -95,   -97,   -98,  -100,  -100,  -102,  -103,  -105,
 -105,  -108,  -108,  -110,  -111,  -112,  -113,  -115,  -115,  -117,
 -118,  -120,  -120,  -121,  -123,  -124,  -125,  -126,  -127,  -128,
 -129,  -131,  -131,  -133,  -133,  -134,  -136,  -136,  -137,  -139,
 -139,  -141,  -140,  -143,  -142,  -145,  -144,  -146,  -146,  -148,
 -148,  -149,  -149,  -151,  -151,  -152,  -153,  -154,  -154,  -155,
 -156,  -156,  -157,  -158,  -158,  -159,  -159,  -161,  -160,  -162,
 -162,  -162,  -163,  -164,  -164,  -164,  -165,  -166,  -166,  -166,
 -167,  -167,  -167,  -168,  -168,  -169,  -169,  -169,  -170,  -170,
 -170,  -170,  -171,  -171,  -171,  -172,  -171,  -172,  -172,  -172,
 -172,  -172,  -173,  -172,  -173,  -173,  -172,  -173,  -173,  -173,
 -173,  -172,  -173,  -173,  -172,  -173,  -173,  -172,  -172,  -173,
 -172,  -172,  -171,  -172,  -171,  -171,  -171,  -171,  -170,  -171,
 -170,  -169,  -170,  -169,  -168,  -169,  -168,  -168,  -167,  -167,
 -166,  -166,  -166,  -165,  -165,  -164,  -164,  -164,  -163,  -162,
 -162,  -161,  -161,  -160,  -160,  -159,  -158,  -158,  -158,  -156,
 -156,  -156,  -154,  -154,  -154,  -152,  -152,  -152,  -150,  -150,
 -149,  -148,  -148,  -147,  -146,  -145,  -145,  -143,  -143,  -142,
 -141,  -140,  -140,  -138,  -138,  -137,  -136,  -135,  -134,  -134,
 -132,  -131,  -131,  -129,  -129,  -127,  -127,  -126,  -125,  -122,
 -123,  -122,  -120,  -120,  -119,  -117,  -117,  -115,  -115,  -113,
 -113,  -111,  -110,  -109,  -109,  -107,  -105,  -105,  -104,  -103,
 -101,  -101,   -99,   -98,   -97,   -96,   -95,   -94,   -92,   -91,
  -91,   -89,   -88,   -87,   -85,   -85,   -83,   -82,   -81,   -80,
  -79,   -77,   -77,   -75,   -74,   -72,   -72,   -71,   -69,   -68,
  -67,   -66,   -64,   -63,   -63,   -61,   -59,   -59,   -58,   -56,
  -55,   -54,   -53,   -51,   -50,   -50,   -48,   -47,   -45,   -45,
  -43,   -42,   -41,   -40,   -39,   -37,   -37,   -35,   -34,   -33,
  -32,   -30,   -30,   -28,   -27,   -26,   -25,   -24,   -23,   -21,
  -21,   -19,   -19,   -17,   -16,   -15,   -14,   -13,   -12,   -10,
  -10,    -9,    -7,    -7,    -5,    -4,    -4,    -2,    -1,    -1,
    1,     2,     2,     4,     5,     5,     7,     8,     8,    10,
   10,    11,    13,    13,    14,    15,    16,    17,    18,    18,
   20,    20,    21,    23,    22,    24,    25,    25,    26,    27,
   28,    29,    29,    30,    31,    32,    32,    33,    34,    35,
   35,    36,    37,    37,    38,    39,    39,    41,    40,    42,
   41,    43,    43,    44,    44,    45,    46,    46,    47,    47,
   47,    49,    48,    50,    49,    51,    50,    51,    52,    52,
   53,    53,    53,    54,    54,    54,    55,    55,    56,    56,
   56,    57,    57,    57,    58,    58,    58,    58,    59,    59,
   59,    60,    59,    60,    60,    61,    60,    61,    61,    61,
   61,    61,    61,    62,    62,    61,    62,    62,    62,    62,
   62,    63,    62,    62,    62,    63,    62,    62,    63,    62,
   62,    62,    63,    62,    62,    62,    62,    61,    62,    62,
   61,    62,    61,    61,    61,    61,    60,    61,    60,    60,
   60,    60,    59,    60,    59,    58,    59,    58,    59,    56,
   58,    57,    57,    57,    56,    57,    55,    56,    55,    55,
   55,    54,    54,    53,    53,    53,    53,    52,    51,    52,
   51,    50,    50,    50,    49,    49,    48,    48,    48,    47,
   47,    46,    46,    45,    45,    44,    44,    44,    43,    42,
   42,    42,    41,    40,    40,    40,    39,    38,    38,    38,
   37,    36,    36,    35,    35,    34,    34,    33,    33,    32,
   31,    31,    31,    29,    30,    29,    28,    27,    27,    27,
   26,    25,    25,    24,    24,    23,    22,    22,    22,    20,
   21,    19,    19,    19,    17,    18,    16,    17,    15,    15,
   14,    14,    13,    13,    12,    12,    11,    10,    10,     9,
    8,     8,     8,     7,     6,     6,     5,     5,     4,     3,
    3,     3,     1,     2,     0,     1,    -1,    -1,    -1,    -2,
   -3,    -3,    -3,    -4,    -5,    -5,    -6,    -6,    -7,    -7,
   -8,    -8,    -9,    -9,   -10,   -10,   -11,   -11,   -12,   -12,
  -13,   -13,   -14,   -14,   -14,   -15,   -16,   -16,   -16,   -17,
  -17,   -18,   -18,   -19,   -18,   -20,   -20,   -20,   -20,   -21,
  -22,   -22,   -22,   -22,   -23,   -23,   -24,   -24,   -24,   -25,
  -25,   -26,   -25,   -26,   -27,   -26,   -27,   -28,   -27,   -28,
  -29,   -28,   -29,   -29,   -29,   -30,   -30,   -30,   -31,   -30,
  -31,   -31,   -32,   -31,   -32,   -32,   -32,   -33,   -32,   -33,
  -33,   -33,   -34,   -33,   -34,   -34,   -34,   -34,   -34,   -35,
  -34,   -35,   -35,   -35,   -35,   -35,   -35,   -36,   -35,   -36,
  -35,   -36,   -36,   -36,   -35,   -36,   -36,   -36,   -36,   -36,
  -36,   -37,   -36,   -36,   -36,   -36,   -36,   -36,   -36,   -36,
  -36,   -36,   -36,   -36,   -36,   -36,   -35,   -36,   -36,   -35,
  -36,   -35,   -35,   -35,   -36,   -35,   -34,   -35,   -35,   -33,
  -35,   -34,   -34,   -34,   -34,   -33,   -34,   -33,   -33,   -33,
  -33,   -33,   -32,   -33,   -32,   -32,   -31,   -32,   -31,   -31,
  -31,   -31,   -30,   -31,   -30,   -30,   -29,   -29,   -30,   -28,
  -29,   -28,   -28,   -28,   -28,   -27,   -27,   -27,   -26,   -27,
  -26,   -25,   -26,   -25,   -25,   -24,   -24,   -24,   -24,   -23,
  -23,   -23,   -22,   -22,   -22,   -22,   -21,   -21,   -20,   -20,
  -20,   -20,   -19,   -19,   -18,   -19,   -17,   -18,   -17,   -17,
  -17,   -16,   -16,   -15,   -15,   -15,   -14,   -14,   -14,   -14,
  -13,   -12,   -13,   -12,   -11,   -12,   -11,   -10,   -10,   -10,
  -10,    -9,    -9,    -8,    -8,    -8,    -7,    -7,    -7,    -6,
   -6,    -6,    -5,    -5,    -5,    -4,    -4,    -3,    -4,    -2,
   -3,    -2,    -2,    -1,    -1,    -1,     0,     0,     0,     1,
    1,     1,     2,     2,     2,     3,     2,     4,     3,     4,
    5,     4,     5,     5,     6,     6,     6,     6,     7,     7,
    8,     7,     8,     9,     8,     9,     9,    10,     9,    10,
   11,    10,    11,    11,    12,    11,    12,    12,    13,    13,
   13,    13,    13,    14,    14,    14,    15,    14,    15,    15,
   16,    15,    16,    16,    17,    16,    17,    17,    17,    17,
   17,    18,    18,    18,    18,    19,    18,    19,    19,    19,
   19,    20,    19,    20,    20,    20,    20,    20,    20,    21,
   21,    20,    21,    21,    22,    21,    21,    22,    21,    22,
   22,    21,    22,    22,    22,    23,    22,    22,    23,    22,
   22,    23,    23,    22,    23,    23,    22,    23,    23,    23,
   23,    22,    23,    23,    23,    23,    23,    23,    23,    23,
   23,    23,    22,    23,    23,    23,    23,    22,    23,    23,
   22,    23,    23,    22,    22,    23,    22,    22,    22,    21,
   22,    22,    22,    22,    22,    21,    22,    21,    21,    22,
   21,    21,    20,    21,    21,    20,    21,    20,    20,    20,
   20,    20,    19,    20,    19,    19,    19,    19,    19,    18,
   19,    18,    18,    18,    18,    17,    18,    17,    17,    17,
   17,    16,    16,    17,    16,    15,    16,    15,    16,    15,
   15,    14,    15,    14,    14,    14,    14,    13,    14,    13,
   13,    12,    13,    12,    12,    12,    11,    12,    11,    11,
   11,    10,    11,    10,    10,     9,    10,     9,     9,     9,
    8,     9,     8,     8,     7,     8,     7,     7,     7,     6,
    6,     6,     6,     6,     5,     5,     5,     5,     4,     5,
    4,     3,     4,     3,     3,     3,     3,     2,     3,     2,
    1,     2,     1,     1,     1,     1,     0,     0,     0,     0,
   -1,     0,    -1,    -1,    -2,    -1,    -2,    -2,    -2,    -3,
   -2,    -3,    -3,    -3,    -4,    -3,    -4,    -4,    -5,    -4,
   -5,    -5,    -5,    -5,    -5,    -6,    -6,    -6,    -6,    -6,
   -7,    -7,    -6,    -8,    -7,    -7,    -8,    -8,    -8,    -8,
   -8,    -8,    -9,    -9,    -9,    -9,    -9,    -9,   -10,   -10,
  -10,   -10,   -10,   -10,   -10,   -11,   -11,   -10,   -11,   -11,
  -12,   -11,   -11,   -12,   -12,   -11,   -12,   -12,   -12,   -13,
  -12,   -12,   -13,   -13,   -12,   -13,   -13,   -13,   -13,   -13,
  -14,   -13,   -14,   -13,   -14,   -13,   -14,   -14,   -14,   -14,
  -14,   -14,   -14,   -14,   -14,   -15,   -14,   -14,   -15,   -14,
  -14,   -15,   -15,   -14,   -15,   -14,   -15,   -15,   -14,   -15,
  -15,   -15,   -14,   -15,   -15,   -15,   -14,   -15,   -15,   -15,
  -15,   -14,   -15,   -15,   -14,   -15,   -15,   -15,   -14,   -15,
  -14,   -15,   -14,   -15,   -14,   -15,   -14,   -14,   -15,   -14,
  -13,   -14,   -14,   -14,   -14,   -14,   -14,   -14,   -14,   -13,
  -14,   -14,   -13,   -13,   -14,   -13,   -13,   -13,   -13,   -13,
  -13,   -13,   -12,   -13,   -12,   -13,   -12,   -12,   -12,   -12,
  -12,   -12,   -12,   -11,   -12,   -11,   -12,   -11,   -11,   -11,
  -11,   -10,   -11,   -10,   -11,   -10,   -10,   -10,   -10,   -10,
   -9,   -10,    -9,   -10,    -9,    -9,    -9,    -8,    -9,    -9,
   -8,    -8,    -8,    -8,    -8,    -8,    -7,    -8,    -7,    -7,
   -7,    -7,    -6,    -7,    -6,    -7,    -6,    -6,    -6,    -5,
   -6,    -5,    -6,    -5,    -5,    -5,    -4,    -5,    -4,    -4,
   -4,    -4,    -4,    -4,    -3,    -4,    -3,    -3,    -3,    -3,
   -2,    -3,    -2,    -2,    -2,    -2,    -2,    -2,    -1,    -1,
   -2,    -1,    -1,     0,    -1,     0,    -1,     0,     0,     0,
    1,     0,     0,     1,     1,     1,     1,     1,     2,     1,
    2,     1,     2,     2,     2,     3,     2,     3,     2,     3,
    3,     3,     3,     4,     3,     4,     3,     4,     4,     4,
    4,     4,     5,     4,     5,     4,     5,     5,     5,     5,
    6,     5,     5,     6,     6,     5,     6,     6,     6,     6,
    7,     6,     6,     7,     7,     6,     7,     7,     7,     7,
    7,     7,     7,     8,     7,     8,     7,     8,     8,     7,
    8,     8,     8,     8,     8,     8,     8,     9,     8,     8,
    9,     8,     9,     8,     9,     9,     8,     9,     9,     9,
    9,     9,     9,     9,     9,     9,     9,     9,     9,     9,
    9,     9,    10,     9,     9,     9,    10,     9,     9,    10,
    9,     9,    10,     9,     9,    10,     9,     9,    10,     9,
    9,    10,     9,     9,    10,     9,     9,    10,     9,     9,
    9,    10,     9,     9,     9,     9,     9,     9,     9,     9,
    9,     8,     9,     9,     9,     9,     8,     9,     9,     8,
    9,     9,     8,     8,     9,     8,     9,     8,     8,     8,
    8,     8,     8,     8,     8,     8,     8,     8,     7,     8,
    7,     8,     7,     7,     8,     7,     7,     7,     7,     7,
    7,     6,     7,     7,     6,     7,     6,     6,     7,     6,
    6,     6,     6,     5,     6,     6,     5,     6,     5,     6,
    5,     5,     5,     5,     5,     5,     5,     4,     5,     4,
    5,     4,     4,     4,     4,     4,     4,     4,     4,     3,
    4,     3,     3,     4,     3,     3,     3,     3,     2,     3,
    3,     2,     3,     2,     2,     2,     2,     2,     2,     2,
    2,     1,     2,     1,     1,     2,     1,     1,     1,     0,
    1,     1,     1,     0,     0,     1,     0,     0,     0,     0,
    0,     0,     0,    -1,     0,    -1,    -1,     0,    -1,    -1,
   -1,    -1,    -1,    -1,    -2,    -1,    -1,    -2,    -1,    -2,
   -2,    -2,    -2,    -2,    -2,    -2,    -2,    -2,    -3,    -2,
   -2,    -3,    -3,    -2,    -3,    -3,    -3,    -2,    -3,    -4,
   -3,    -3,    -3,    -3,    -4,    -3,    -3,    -4,    -4,    -3,
   -4,    -4,    -3,    -4,    -4,    -4,    -4,    -4,    -4,    -5,
   -4,    -4,    -4,    -5,    -4,    -5,    -4,    -5,    -5,    -4,
   -5,    -5,    -5,    -5,    -5,    -5,    -5,    -5,    -5,    -5,
   -5,    -6,    -5,    -5,    -5,    -6,    -5,    -5,    -6,    -5,
   -6,    -5,    -5,    -6,    -5,    -6,    -5,    -6,    -6,    -5,
   -6,    -5,    -6,    -6,    -5,    -6,    -5,    -6,    -6,    -5,
   -6,    -6,    -5,    -6,    -6,    -5,    -6,    -6,    -6,    -5,
   -6,    -6,    -5,    -6,    -5,    -6,    -6,    -5,    -6,    -6,
   -5,    -6,    -5,    -6,    -5,    -6,    -5,    -6,    -5,    -6,
   -5,    -6,    -5,    -5,    -5,    -5,    -5,    -5,    -6,    -5,
   -5,    -5,    -5,    -5,    -5,    -5,    -5,    -5,    -5,    -5,
   -5,    -5,    -5,    -5,    -4,    -5,    -5,    -4,    -5,    -5,
   -4,    -5,    -4,    -4,    -5,    -4,    -4,    -5,    -4,    -4,
   -4,    -4,    -4,    -4,    -4,    -4,    -4,    -4,    -3,    -4,
   -4,    -3,    -4,    -4,    -3,    -3,    -4,    -3,    -3,    -4,
   -3,    -3,    -3,    -3,    -3,    -3,    -3,    -3,    -2,    -3,
   -3,    -2,    -3,    -2,    -3,    -2,    -3,    -2,    -2,    -2,
   -2,    -3,    -2,    -1,    -2,    -2,    -2,    -2,    -1,    -2,
   -2,    -1,    -2,    -1,    -1,    -2,    -1,    -1,    -1,    -1,
   -2,    -1,     0,    -1,    -1,    -1,    -1,     0,    -1,     0,
   -1,     0,    -1,     0,     0,    -1,     0,     0,     0,     0,
    0,     0,     0,     0,     0,     1,     0,     0,     1,     0,
    1,     0,     1,     1,     0,     1,     1,     1,     0,     1,
    1,     1,     1,     2,     1,     1,     1,     1,     2,     1,
    1,     2,     1,     2,     1,     2,     2,     1,     2,     2,
    2,     2,     1,     2,     2,     2,     2,     2,     2,     3,
    2,     2,     2,     2,     3,     2,     2,     3,     2,     2,
    3,     2,     3,     2,     3,     3,     2,     3,     3,     2,
    3,     3,     2,     3,     3,     3,     3,     3,     2,     3,
    3,     3,     3,     3,     3,     3,     3,     3,     3,     3,
    3,     4,     3,     3,     3,     3,     3,     3,     4,     3,
    3,     3,     3,     3,     4,     3,     3,     3,     4,     3,
    3,     3,     4,     3,     3,     3,     4,     3,     3,     3,
    4,     3,     3,     3,     4,     3,     3,     3,     4,     3,
    3,     3,     4,     3,     3,     3,     3,     3,     4,     3,
    3,     3,     3,     3,     3,     3,     3,     3,     3,     3,
    3,     3,     3,     3,     3,     3,     3,     3,     2,     3,
    3,     3,     3,     3,     3,     2,     3,     3,     3,     2,
    3,     3,     2,     3,     3,     2,     3,     2,     3,     2,
    3,     2,     3,     2,     2,     3,     2,     3,     2,     2,
    2,     3,     2,     2,     2,     2,     2,     2,     2,     3,
    2,     1,     2,     2,     2,     2,     2,     2,     2,     1,
    2,     2,     2,     1,     2,     2,     1,     2,     1,     2,
    1,     2,     1,     1,     2,     1,     1,     2,     1,     1,
    1,     2,     1,     1,     1,     1,     1,     1,     1,     1,
    1,     1,     1,     0,     1,     1,     1,     1,     0,     1,
    1,     0,     1,     0,     1,     1,     0,     0,     1,     0,
    1,     0,     0,     1,     0,     0,     1,     0,     0,     0,
    0,     0,     0,     1,     0,     0,     0,     0,     0,    -1,
    0,     0,     0,     0,     0,     0,    -1,     0,     0,     0,
   -1,     0,     0,    -1,     0,    -1,     0,     0,    -1,     0,
   -1,     0,    -1,     0,    -1,    -1,     0,    -1,     0,    -1,
   -1,     0,    -1,    -1,    -1,     0,    -1,    -1,    -1,     0,
   -1,    -1,    -1,    -1,    -1,    -1,     0,    -1,    -1,    -1,
   -1,    -1,    -1,    -1,    -1,    -1,    -1,    -1,    -1,    -1,
   -1,    -1,    -1,    -1,    -1,    -2,    -1,    -1,    -1,    -1,
   -1,    -1,    -1,    -2,    -1,    -1,    -1,    -1,    -1,    -2,
   -1,    -1,    -1,    -1,    -2,    -1,    -1,    -1,    -1,    -2,
   -1,    -1,    -1,    -1,    -2,    -1,    -1,    -1,    -1,    -2,
   -1,    -1,    -1,    -1,    -2,    -1,    -1,    -1,    -1,    -2,
   -1,    -1,    -1,    -1,    -1,    -2,    -1,    -1,    -1,    -1,
   -1,    -1,    -2,    -1,    -1,    -1,    -1,    -1,    -2,    -1,
   -1,    -1,    -1,    -1,    -1,    -2,    -1,    -1,    -1,    -1,
   -2,    -1,    -1,    -1,    -2,    -1,    -1,    -2,    -1,    -1,
   -2,    -1,    -2,    -1,    -2,    -2,    -1,    -2,    -2,    -1,
   -2,    -2,    -2,    -2,    -2,    -2,    -2,    -2,    -2,    -2,
   -2,    -2,    -2,    -3,    -2,    -2,    -2,    -2,    -2,    -2,
   -2,    -2,    -2,    -2,    -2,    -1,    -2,    -1,    -2,     0,
   -1,    -1,     0,    -1,     0,    -1,     0,     0,     0,     0,
    0,     1,     0,     0,     1,     0,     1,     0,     1,     0,
    0,     0 };
/* *INDENT-ON* */

static inline int32_t
FilterUp (int16_t Imp[], int16_t ImpD[], uint16_t Nwing, char Interp,
	int16_t * Xp, int16_t Ph, int16_t Inc)
{
	int16_t *Hp, *Hdp = NULL, *End;
	int16_t a = 0;
	int32_t v, t;

	v = 0;
	Hp = &Imp[Ph >> Na];
	End = &Imp[Nwing];
	if (Interp)
	{
		Hdp = &ImpD[Ph >> Na];
		a = Ph & Amask;
	}
	if (Inc == 1)										   /* If doing right wing...              */
	{													   /* ...drop extra coeff, so when Ph is  */
		End--;											   /*    0.5, we don't do too many mult's */
		if (Ph == 0)									   /* If the phase is zero...           */
		{												   /* ...then we've already skipped the */
			Hp += Npc;									   /*    first sample, so we must also  */
			Hdp += Npc;									   /*    skip ahead in Imp[] and ImpD[] */
		}
	}
	if (Interp)
		while (Hp < End)
		{
			t = *Hp;									   /* Get filter coeff */
			t += (((int32_t) * Hdp) * a) >> Na;			   /* t is now interp'd filter coeff */
			Hdp += Npc;									   /* Filter coeff differences step */
			t *= *Xp;									   /* Mult coeff by input sample */
			if (t & (1 << (Nhxn - 1)))					   /* Round, if needed */
				t += (1 << (Nhxn - 1));
			t >>= Nhxn;									   /* Leave some guard bits, but come back some */
			v += t;										   /* The filter output */
			Hp += Npc;									   /* Filter coeff step */
			Xp += Inc;									   /* Input signal step. NO CHECK ON BOUNDS */
		}
	else
		while (Hp < End)
		{
			t = *Hp;									   /* Get filter coeff */
			t *= *Xp;									   /* Mult coeff by input sample */
			if (t & (1 << (Nhxn - 1)))					   /* Round, if needed */
				t += (1 << (Nhxn - 1));
			t >>= Nhxn;									   /* Leave some guard bits, but come back some */
			v += t;										   /* The filter output */
			Hp += Npc;									   /* Filter coeff step */
			Xp += Inc;									   /* Input signal step. NO CHECK ON BOUNDS */
		}
	return (v);
}


static inline int32_t
FilterUD (int16_t Imp[], int16_t ImpD[], uint16_t Nwing, char Interp,
	int16_t * Xp, int16_t Ph, int16_t Inc, uint16_t dhb)
{
	int16_t a;
	int16_t *Hp, *Hdp, *End;
	int32_t v, t;
	uint32_t Ho;

	v = 0;
	Ho = (Ph * (uint32_t) dhb) >> Np;
	End = &Imp[Nwing];
	if (Inc == 1)										   /* If doing right wing...              */
	{													   /* ...drop extra coeff, so when Ph is  */
		End--;											   /*    0.5, we don't do too many mult's */
		if (Ph == 0)									   /* If the phase is zero...           */
			Ho += dhb;									   /* ...then we've already skipped the */
	}													   /*    first sample, so we must also  */
	/*    skip ahead in Imp[] and ImpD[] */
	if (Interp)
		while ((Hp = &Imp[Ho >> Na]) < End)
		{
			t = *Hp;									   /* Get IR sample */
			Hdp = &ImpD[Ho >> Na];						   /* get interp (lower Na) bits from diff table */
			a = Ho & Amask;								   /* a is logically between 0 and 1 */
			t += (((int32_t) * Hdp) * a) >> Na;			   /* t is now interp'd filter coeff */
			t *= *Xp;									   /* Mult coeff by input sample */
			if (t & 1 << (Nhxn - 1))					   /* Round, if needed */
				t += 1 << (Nhxn - 1);
			t >>= Nhxn;									   /* Leave some guard bits, but come back some */
			v += t;										   /* The filter output */
			Ho += dhb;									   /* IR step */
			Xp += Inc;									   /* Input signal step. NO CHECK ON BOUNDS */
		}
	else
		while ((Hp = &Imp[Ho >> Na]) < End)
		{
			t = *Hp;									   /* Get IR sample */
			t *= *Xp;									   /* Mult coeff by input sample */
			if (t & 1 << (Nhxn - 1))					   /* Round, if needed */
				t += 1 << (Nhxn - 1);
			t >>= Nhxn;									   /* Leave some guard bits, but come back some */
			v += t;										   /* The filter output */
			Ho += dhb;									   /* IR step */
			Xp += Inc;									   /* Input signal step. NO CHECK ON BOUNDS */
		}
	return (v);
}


static inline int16_t
WordToHword (int32_t v, int scl)
{
	int16_t out;
	int32_t llsb = (1 << (scl - 1));

	v += llsb;											   /* round */
	v >>= scl;
	if (v > SHRT_MAX)
		v = SHRT_MAX;
	else if (v < SHRT_MIN)
		v = SHRT_MIN;
	out = (int16_t) v;
	return out;
}


static int
SrcUp (struct rate_private_data *data, int voice, int16_t X[], char *dst_ptr,
	uint64_t * Time, uint32_t Nx, uint32_t Ny, uint16_t Nwing, uint16_t LpScl,
	int16_t Imp[], int16_t ImpD[], char Interp)
{
	int16_t *Xp;
	int32_t v;
	int32_t ocount;

	uint32_t dtb;						/* Fixed-point version of Dt */
	uint64_t endTime;					/* When Time reaches EndTime, return to user */

#ifdef USE_FPU
	double  factor;						/* factor = outSampleRate/inSampleRate */

	factor = 1.0 * BITS / data->pitch;
	double  dt;							/* Step through input signal */

	dt = 1.0 / factor;									   /* Output sampling period */
	dtb = dt * (1 << Np) + 0.5;							   /* Fixed-point representation */
#else
	dtb = (data->pitch * (1 << Np) + BITS / 2) / BITS;
#endif

	ocount = 0;
	endTime = *Time + (1 << Np) * (uint64_t) Nx;
	while (*Time < endTime)
	{
		Xp = &X[*Time >> Np];							   /* Ptr to current input sample */
		/* Perform left-wing inner product */
		v = FilterUp (Imp, ImpD, Nwing, Interp, Xp, (int16_t) (*Time & Pmask), -1);
		/* Perform right-wing inner product */
		v += FilterUp (Imp, ImpD, Nwing, Interp, Xp + 1,
			/* previous (triggers warning): (int16_t)((-*Time)&Pmask),1); */
			(int16_t) ((((*Time) ^ Pmask) + 1) & Pmask), 1);
		v >>= Nhg;										   /* Make guard bits */
		v *= LpScl;										   /* Normalize for unity filter gain */
		data->put (dst_ptr + ocount++ * data->src_voices * data->sample_size +
			voice * data->sample_size, WordToHword (v, NLpScl));
		if (ocount >= Ny)
			break;
		*Time += dtb;									   /* Move to next sample by time increment */
	}
	return (ocount);									   /* Return the number of output samples */
}


static int
SrcUD (struct rate_private_data *data, int voice,
	int16_t X[], char *dst_ptr, uint64_t * Time, uint32_t Nx, uint32_t Ny,
	uint16_t Nwing, uint16_t LpScl, int16_t Imp[], int16_t ImpD[], char Interp)
{
	int16_t *Xp;
	int32_t v;
	int32_t ocount;

	uint64_t endTime;					/* When Time reaches EndTime, return to user */
	uint32_t dhb, dtb;					/* Fixed-point versions of Dh,Dt */

#ifdef USE_FPU
	double  factor;						/* factor = outSampleRate/inSampleRate */
	double  dh;							/* Step through filter impulse response */
	double  dt;							/* Step through input signal */

	factor = 1.0 * BITS / data->pitch;
	dt = 1.0 / factor;									   /* Output sampling period */
	dtb = dt * (1 << Np) + 0.5;							   /* Fixed-point representation */
	dh = min (Npc, factor * Npc);						   /* Filter sampling period */
	dhb = dh * (1 << Na) + 0.5;							   /* Fixed-point representation */
#else
	dtb = (data->pitch * (1 << Np) + BITS / 2) / BITS;
	dhb = min (Npc * (1 << Na),
		(((uint64_t) 1 << (2 * SHIFT)) / data->pitch * Npc * (1 << Na) + BITS / 2) / BITS);
#endif


	ocount = 0;
	endTime = *Time + (1 << Np) * (uint64_t) Nx;
	while (*Time < endTime)
	{
		Xp = &X[*Time >> Np];							   /* Ptr to current input sample */
		v = FilterUD (Imp, ImpD, Nwing, Interp, Xp, (int16_t) (*Time & Pmask), -1, dhb);	/* Perform left-wing inner product */
		v += FilterUD (Imp, ImpD, Nwing, Interp, Xp + 1,
			/* previous (triggers warning): (int16_t)((-*Time)&Pmask), */
			(int16_t) ((((*Time) ^ Pmask) + 1) & Pmask), 1, dhb);	/* Perform right-wing inner product */
		v >>= Nhg;										   /* Make guard bits */
		v *= LpScl;										   /* Normalize for unity filter gain */
		data->put (dst_ptr + ocount++ * data->src_voices * data->sample_size +
			voice * data->sample_size, WordToHword (v, NLpScl));
		if (ocount >= Ny)
			break;
		*Time += dtb;									   /* Move to next sample by time increment */
	}
	return (ocount);									   /* Return the number of output samples */
}

/************************************************************************/
/* END BSD License                                                      */
/************************************************************************/


static void
resample_band_limited (struct rate_private_data *data, int voices,
	char *src_ptr, int src_size, char *dst_ptr, int dst_size)
{
	uint16_t Nmult;						/* Filter length for up-conversions */
	int16_t *Imp = 0;					/* Filter coefficients */
	int16_t *ImpD = 0;					/* ImpD[n] = Imp[n+1]-Imp[n] */
	uint16_t LpScl;						/* Unity-gain scale factor */
	uint16_t Nwing;						/* Filter table size */

	uint64_t Time;						/* Current time/pos in input sample */

	uint32_t Xoff;
	uint32_t Nx;
	int     voice;
	int     count;
	char   *src;
	int     src_inc;

#ifdef USE_FPU
	double  factor;						/* factor = outSampleRate/inSampleRate */
#endif

	Nmult = data->Nmult;
	Imp = data->Imp;
	ImpD = data->ImpD;
	LpScl = data->LpScl;
	Nwing = data->Nwing;

#ifdef USE_FPU
	factor = 1.0 * BITS / data->pitch;
	Xoff = ((Nmult + 1) / 2.0) * max (1.0, 1.0 / factor) + 10;
#else
	Xoff = (Nmult + 1) * max ((1 << SHIFT), data->pitch) / BITS / 2 + 10;
#endif

#ifdef USE_FPU
	if (factor < 1)
		LpScl = LpScl * factor + 0.5;
#else
	if (data->pitch >= (1 << SHIFT))
		LpScl = ((uint64_t) LpScl * BITS + data->pitch / 2) / data->pitch;
#endif

	for (voice = 0; voice < voices; ++voice)
	{
		src = src_ptr + voice * data->sample_size;
		src_inc = voices * data->sample_size;
		for (count = 0; count < src_size; count++)
		{
			data->X[voice][2 * Xoff + count] = data->take (src);
			src += src_inc;
		}

		Nx = src_size;
		Time = 0;
		Time = (Xoff << Np);							   /* Current-time pointer for converter */

#ifdef USE_FPU
		if (factor >= 1)
#else
		if (data->pitch < (1 << SHIFT))
#endif
		{												   /* SrcUp() is faster if we can use it */
			SrcUp (data, voice, data->X[voice], dst_ptr, &Time, Nx, dst_size,
				Nwing, LpScl, Imp, ImpD, USE_INTERP);
		}
		else
		{
			SrcUD (data, voice, data->X[voice], dst_ptr, &Time, Nx, dst_size,
				Nwing, LpScl, Imp, ImpD, USE_INTERP);
		}

		for (count = 0; count < 2 * Xoff; count++)
		{
			data->X[voice][count] = data->X[voice][count + Nx];
		}
	}
}


static void
resample_linear (struct rate_private_data *data, int voices,
	char *src_ptr, int src_size, char *dst_ptr, int dst_size)
{
	int64_t pos;
	int     voice;
	int     count;
	signed int a;
	char   *src;
	signed short S1, S2;
	signed int val;
	char   *dst;

	for (voice = 0; voice < voices; ++voice)
	{
		pos = 0;
		for (count = 0; count < dst_size; count++)
		{
			a = (pos) >> SHIFT;
			if ((a - 1) >= 0)
			{
				src = src_ptr + (a - 1) * voices * data->sample_size + voice * data->sample_size;
				S1 = data->take (src);
			}
			else
				S1 = data->S[voice];
			src = src_ptr + (a + 0) * voices * data->sample_size + voice * data->sample_size;
			S2 = data->take (src);

			val = S1 + ((S2 - S1) * (pos & MASK) / BITS);

			if (val < SHRT_MIN)
				val = SHRT_MIN;
			else if (val > SHRT_MAX)
				val = SHRT_MAX;

			dst = dst_ptr + count * voices * data->sample_size + voice * data->sample_size;
			data->put (dst, val);

			pos += data->pitch;
		}
		src = src_ptr + (src_size - 1) * voices * data->sample_size + voice * data->sample_size;
		data->S[voice] = data->take (src);
	}
}


static  ssize_t
rate_src_size (snd_pcm_plugin_t * plugin, size_t size)
{
	struct rate_private_data *data;
	int64_t dst_fs, src_fs;

	if (plugin == NULL || size <= 0)
		return -EINVAL;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);

	if (data->src_mode == SND_SRC_MODE_NORMAL)
	{
		if (data->old_dst_size == size)
			return (data->old_src_size);

		dst_fs = size;
		src_fs = ((((int64_t) size * data->pitch) + (BITS / 2)) >> SHIFT);
		src_fs = src_fs / (data->src_voices * data->sample_size) *
			(data->src_voices * data->sample_size);
		data->pitch = ((int64_t) src_fs << SHIFT) / dst_fs;

		data->old_dst_size = dst_fs;
		data->old_src_size = src_fs;
	}
	else /* variable src frag size */
	{
		dst_fs = data->dstfragsize = size;
		if (data->newfragsize == 0)
		{
			src_fs = ((((int64_t) size * data->pitch) + (BITS / 2)) >> SHIFT);
			src_fs = src_fs / (data->src_voices * data->sample_size) *
				(data->src_voices * data->sample_size);
		}
		else
			src_fs = data->newfragsize;
		data->pitch = ((int64_t) src_fs << SHIFT) / dst_fs;

		data->old_dst_size = dst_fs;
		data->old_src_size = src_fs;
	}

	/* Initialize minfragsize and maxfragsize rate converter variables */
	if (!data->initialized)
	{
		int extrafrag = data->src_voices * data->sample_size;
		if (data->src_mode == SND_SRC_MODE_ACTUAL)
		{
			data->maxfragsize = src_fs + extrafrag;
			data->minfragsize = src_fs;
		}
		else if (data->src_mode == SND_SRC_MODE_ASYNC)
		{
			data->maxfragsize = (src_fs + extrafrag) * MAXPHASE;
			data->minfragsize = (src_fs - extrafrag) * MINPHASE;
		}
		else
		{
			data->maxfragsize = data->minfragsize = src_fs;
		}
		data->initialized = 1;
	}

	return (src_fs);
}

static  ssize_t
rate_dst_size (snd_pcm_plugin_t * plugin, size_t size)
{
	struct rate_private_data *data;
	int64_t dst_fs, src_fs;

	if (plugin == NULL || size <= 0)
		return -EINVAL;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);

	if (data->old_src_size == size)
		return (data->old_dst_size);

	/* adjust the pitch for integer fragsizes at src & dst */
	src_fs = size;
	if (data->src_mode == SND_SRC_MODE_NORMAL)
	{

		dst_fs = ((((int64_t) src_fs << SHIFT) + (data->pitch / 2)) / data->pitch);
		dst_fs = dst_fs / (data->dst_voices * data->sample_size)
			* (data->dst_voices * data->sample_size);
	}
	else
		dst_fs = data->dstfragsize;
	data->pitch = ((int64_t) src_fs << SHIFT) / dst_fs;

	data->old_dst_size = dst_fs;
	data->old_src_size = src_fs;

	/* Initialize minfragsize and maxfragsize rate converter variables */
	if (!data->initialized)
	{
		data->maxfragsize = data->minfragsize = src_fs;
		data->initialized = 1;
	}
	return (dst_fs);
}

static int
rate_init (snd_pcm_plugin_t * plugin)
{
	struct rate_private_data *data;
	int     i;
	uint16_t extra;

	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);
	if ((data->src_mode == SND_SRC_MODE_ASYNC) && !asrc_conf_done) 
	{
		char *str;
		asrc_conf_done = 1;

		str = getenv("MAXPHASE");
		if (str) {
			asrc_maxphase = strtof(str, NULL);
		}

		str = getenv("MINPHASE");
		if (str) {
			asrc_minphase = strtof(str, NULL);
		}

		str = getenv("ASRCTC");
		if (str) {
			asrc_tc = strtof(str, NULL);
		}

		str = getenv("ASRCDEBUG");
		if (str) {
			asrc_debug = strtol(str, NULL, 10);
		}

		if (asrc_debug)
		{
			slogf (_SLOGC_AUDIO, _SLOG_ERROR, "ASRC Using values: MAXPHASE: %f, MINPHASE: %f, ASRCTC: %f, ASRCDEBUG: %d",
				asrc_maxphase, asrc_minphase, asrc_tc, asrc_debug);
		}
	}

	if (data->X[0] == NULL && data->method == resample_band_limited)
	{
		extra = ((data->Nmult + 1) * 8 / 2) + 10;
		for (i = 0; i < data->src_voices; i++)
		{
			if ((data->X[i] = malloc (data->maxfragsize * sizeof (int16_t)
						+ extra * 2 + 10)) == NULL)
				return (-ENOMEM);
		}
	}
	return (0);
}

static void
rate_prepare (snd_pcm_plugin_t * plugin)
{
	struct rate_private_data *data;
	unsigned char silence;
	int     i;
	uint16_t extra;

	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);
	silence = snd_pcm_plugin_silence (&data->format);
	if (data->method == resample_band_limited)
	{
		for (i = 0; i < data->src_voices; i++)
		{
			extra = ((data->Nmult + 1) * 8 / 2) + 10;
			if (data->X[i])
				memset (data->X[i], silence, data->maxfragsize *
					sizeof (int16_t) + extra * 2 + 10);
		}
	}
	else
	{
		for (i = 0; i < data->src_voices; i++)
		{
			memset (&data->S[i], silence, sizeof (data->S[i]));
		}
	}
}

static  ssize_t
rate_transfer (snd_pcm_plugin_t * plugin,
	void *src_ptr, size_t src_size, void *dst_ptr, size_t dst_size)
{
	struct rate_private_data *data;

	if (plugin == NULL || src_ptr == NULL || src_size < 0 || dst_ptr == NULL || dst_size < 0)
		return -EINVAL;
	if (src_size == 0)
		return 0;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);
	if (data == NULL)
		return -EINVAL;

	if (src_size != data->old_src_size || dst_size != data->old_dst_size)
	{
		data->pitch = ((int64_t) src_size << SHIFT) / dst_size;
		data->old_src_size = src_size;
		data->old_dst_size = dst_size;
		rate_prepare (plugin);
	}

	data->method (data, data->src_voices,
		src_ptr, src_size / (data->src_voices * data->sample_size),
		dst_ptr, dst_size / (data->dst_voices * data->sample_size));

	data->icount += src_size;
	data->ocount += dst_size;

	return rate_dst_size (plugin, src_size);
}



static int
rate_action (snd_pcm_plugin_t * plugin, snd_pcm_plugin_action_t action)
{
	int     rtn = 0;
	struct rate_private_data *data;

	if (plugin == NULL)
		return -EINVAL;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);

	switch (action)
	{
	case INIT:
		slogf (_SLOGC_AUDIO, _SLOG_INFO, "SRC Init: Size %zd -> %zd", data->old_src_size, data->old_dst_size);
		rtn = rate_init (plugin);
		break;
	case PREPARE:
		rate_prepare (plugin);
		break;
	case DRAIN:
		break;
	case POST_DRAIN:
		break;
	case FLUSH:
		break;
	}
	return rtn;
}

static void
rate_free (snd_pcm_plugin_t * plugin, void *private_data)
{
	struct rate_private_data *data;
	int     i;

	if (plugin == NULL)
		return;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);

	for (i = 0; i < data->src_voices; i++)
	{
		if (data->X[i])
		{
			free (data->X[i]);
			data->X[i] = NULL;
		}
	}
	slogf (_SLOGC_AUDIO, _SLOG_INFO, "SRC Free: Converted %d -> %d", data->icount, data->ocount);
}

int
snd_pcm_plugin_build_rate (snd_pcm_format_t * src_format,
	snd_pcm_format_t * dst_format, unsigned int method,
	snd_pcm_plugin_t ** r_plugin, unsigned int src_mode,
	int target, int32_t out_frag_size)
{
	struct rate_private_data *data;
	snd_pcm_plugin_t *plugin;

	if (r_plugin == NULL)
		return -EINVAL;
	*r_plugin = NULL;

	if (src_format->interleave != dst_format->interleave && src_format->voices > 1)
		return -EINVAL;
	if (src_format->format != dst_format->format)
		return -EINVAL;
	if (!dst_format->interleave && src_format->voices > 1)
		return -EINVAL;
	if (src_format->voices != dst_format->voices)
		return -EINVAL;
	if (dst_format->voices < 1 || dst_format->voices > MAX_VOICES)
		return -EINVAL;

	if (src_format->format > SND_PCM_SFMT_S32_BE)
		return -EINVAL;
	if (src_format->rate == dst_format->rate && src_mode == SND_SRC_MODE_NORMAL)
		return -EINVAL;
	plugin = snd_pcm_plugin_build ("rate conversion", sizeof (struct rate_private_data));
	if (plugin == NULL)
		return -ENOMEM;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);
	data->format = *src_format;
	data->src_voices = src_format->voices;
	data->dst_voices = dst_format->voices;
	data->src_rate = src_format->rate;
	data->dst_rate = dst_format->rate;
	data->sample_size = snd_pcm_format_width (src_format->format) / 8;

	switch (method)
	{
		case 3:
			/* linear */
			slogf (_SLOGC_AUDIO, _SLOG_INFO, "SRC Create: Linear %d -> %d. Ch: %d. FragSize: %d",
					src_format->rate, dst_format->rate, src_format->voices, out_frag_size);
			data->method = resample_linear;
			break;
		case 0:
			/* Default */
		case 1:
			/* 7-pt */
			slogf (_SLOGC_AUDIO, _SLOG_INFO, "SRC Create: 7-pt Band limited %d -> %d. Ch: %d. FragSize: %d",
					src_format->rate, dst_format->rate, src_format->voices, out_frag_size);
			data->method = resample_band_limited;
			data->Nmult = A_FILTER_NMULT;
			data->Imp = A_FILTER_IMP;			  /* Impulse response */
			data->ImpD = A_FILTER_IMPD;			/* Impulse response deltas */
			data->LpScl = A_FILTER_SCALE;		  /* Unity-gain scale factor */
			data->Nwing = A_FILTER_NWING;		  /* Filter table length */
			break;
		case 2:
		default:
			/* 20-pt */
			slogf (_SLOGC_AUDIO, _SLOG_INFO, "SRC Create: 20-pt Band limited %d -> %d. Ch: %d. FragSize: %d",
					src_format->rate, dst_format->rate, src_format->voices, out_frag_size);
			data->method = resample_band_limited;
			data->Nmult = B_FILTER_NMULT;
			data->Imp = B_FILTER_IMP;			/* Impulse response */
			data->ImpD = B_FILTER_IMPD;			/* Impulse response deltas */
			data->LpScl = B_FILTER_SCALE;		  /* Unity-gain scale factor */
			data->Nwing = B_FILTER_NWING;			/* Filter table length */
			break;
	}

	data->take = rate_take_sample[src_format->format];
	data->put = rate_put_sample[dst_format->format];
	data->pitch = (((uint64_t) src_format->rate << SHIFT) +
		(dst_format->rate >> 1)) / dst_format->rate;
	data->old_src_size = data->old_dst_size = 0;
	data->src_mode = SND_SRC_MODE_NORMAL;
	if (src_mode > SND_SRC_MODE_NORMAL)
	{
		data->src_mode = src_mode;
		data->src_ratio = (double)data->src_rate / data->dst_rate;
		data->target = target;
		data->filttarget = data->target;
		data->icount = 0;
		data->ocount = 0;
		data->newfragsize = 0;
		data->fractfrag = 0;
		data->dstfragsize = out_frag_size;
		data->initialized = 0;
	}

	plugin->transfer = rate_transfer;
	plugin->src_size = rate_src_size;
	plugin->dst_size = rate_dst_size;
	plugin->action = rate_action;
	plugin->private_free = rate_free;
	*r_plugin = plugin;
	return 0;
}


unsigned int
snd_pcm_plugin_set_src_method (snd_pcm_t * pcm, unsigned int method)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(method), SLOG2_FA_END);

	if (method < MAX_SUPPORTED_SRC_METHODS)
		pcm->plugin_src_method = method;
	return (pcm->plugin_src_method);
}

unsigned int
snd_pcm_plugin_set_src_mode (snd_pcm_t * pcm, unsigned int src_mode, int target)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(target), SLOG2_FA_UNSIGNED(target), SLOG2_FA_END);

	if (src_mode <= SND_SRC_MODE_PITCH)
	{
		pcm->plugin_src_mode = src_mode;
		pcm->plugin_src_target = target;
	}
	return (pcm->plugin_src_mode);
}

/* To properly generate the fragsize as seen by the client we must run it through all the
 * plugins in the chain (relative to the rate conversion plugin). To do this we need the
 * channel argument to know what direction we need to walk the plugin chain relative to
 * the rate converison plugin.
 */
static int
get_client_fragsize (snd_pcm_t * pcm, int channel, unsigned int fragsize)
{
	snd_pcm_plugin_t *plugin ;
	int client_fragsize = fragsize;

	plugin = snd_pcm_plugin_find (pcm, channel, "rate conversion");
	if (plugin == NULL)
		return 0;

	if (channel == SND_PCM_CHANNEL_PLAYBACK)
	{
		/* Walk back through the plugin chain starting from the rate conversion
		 * plugin computing the new fragsize (as seen by the client app).
		 */
		while (plugin->prev)
		{
			plugin = plugin->prev;
			if (plugin->src_size)
				client_fragsize = plugin->src_size (plugin, client_fragsize);
		}
	}
	else if (channel == SND_PCM_CHANNEL_CAPTURE)
	{
		/* Walk forward through the plugin chain starting from the rate conversion
		 * plugin computing the new fragsize (as seen by the client app)
		 */
		while (plugin->next)
		{
			plugin = plugin->next;
			if (plugin->dst_size)
				client_fragsize = plugin->dst_size (plugin, client_fragsize);
		}
	}

	return (client_fragsize);
}

/*
 * Note: For compatibiliy the original publicly defined snd_pcm_plugin_src_max_frag()
 *		  is maintained and will call the internal version of the function with a
 *		  channel direction of SND_PCM_CHANNEL_PLAYBACK.
 */
int
snd_pcm_plugin_src_max_frag (snd_pcm_t * pcm, int channel, unsigned int fragsize)
{
	struct rate_private_data *data;
	snd_pcm_plugin_t *plugin;
	int client_fragsize;

	plugin = snd_pcm_plugin_find (pcm, channel, "rate conversion");
	if (plugin == NULL)
		return 0;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);

	/* In normal mode the maxfragsize is the same as the fragsize, so
	 * we can just return the size passed in. For ASRC we must call
	 * get_client_fragsize() to account for the plugins in the chain.
	 */
	if (data->src_mode > SND_SRC_MODE_NORMAL)
		client_fragsize = get_client_fragsize (pcm, channel, data->maxfragsize);
	else
		client_fragsize = data->maxfragsize;

	return (client_fragsize);
}

int
snd_pcm_plugin_update_src_internal (snd_pcm_t * pcm, snd_pcm_channel_setup_t * setup, int currlevel)
{
	struct rate_private_data *data;
	snd_pcm_plugin_t *plugin;
	double truecount;
	int extrafrag;

	plugin = snd_pcm_plugin_find (pcm, 0, "rate conversion");
	if (plugin == NULL)
		return -EINVAL;
	data = (struct rate_private_data *) snd_pcm_plugin_extra_data (plugin);

	/* In NORMAL mode return without changing the src size */
	if (data->src_mode == SND_SRC_MODE_NORMAL)
		return data->old_src_size;

	/* Let the currlevel change the block size and cause pitch artifacts */
	if (data->src_mode == SND_SRC_MODE_PITCH)
	{
		/* Adjust pitch based on currlevel */
		int pitchshift;
		pitchshift = data->maxfragsize * currlevel * 0.001;

		extrafrag = data->src_voices * data->sample_size;

		data->newfragsize = data->maxfragsize + (int)(pitchshift / extrafrag) * extrafrag;

		if (data->newfragsize <= 0)
			data->newfragsize = data->old_src_size;

		if (data->newfragsize != data->old_src_size)
			snd_pcm_channel_setup (pcm, setup);

		return data->newfragsize;
	}

	/* ASRC compensation - adjusts src ratio */
	if (data->src_mode == SND_SRC_MODE_ASYNC)
	{
		/* Calculate the exponential moving average of the currlevel */
		data->filttarget = ASRCTC * data->filttarget + (1 - ASRCTC) * currlevel;
		
		/* The target and currlevel are intergers so round the filttarget up when calculating the error
		 * NOTE: This does 2 things for us: 
		 *       1) Since ASRCTC is < 1, the above averaging will result in an error on the first call when
		 *          currlevel == filttarget == target, rounding up ensures we don't treat this as an error.
		 *       2) This will slow the rate of change, preventing us from being hyper reactive.
		 */
		data->error = ((int)(data->filttarget + 0.999)) - data->target;

		if (asrc_debug) {
			slogf (_SLOGC_AUDIO, _SLOG_ERROR, "ASRC: target=%i, filttarget=%f, src_ratio=%f, error=%f, lvl = %d",
				data->target, data->filttarget, data->src_ratio, data->error, currlevel);
		}
	}

	/* True rate - adjusts frag source size */
	truecount = data->ocount * data->src_ratio + data->fractfrag;
	extrafrag = data->src_voices * data->sample_size;
	data->newfragsize = data->old_src_size;

	/* We only adjust the fragment size if the error is increasing, this allows us to stablize and reduces
	 * the chances of oscillating around the target level
	 */
	if ((data->error >= 0.0 && data->error > data->old_error) && ((data->old_src_size + extrafrag) <= data->maxfragsize))
	{
		data->newfragsize = data->old_src_size + extrafrag;
		data->fractfrag = truecount - data->icount - extrafrag;
		data->icount = 0;
		data->ocount = 0;
		if (asrc_debug)
			slogf (_SLOGC_AUDIO, _SLOG_ERROR, "ASRC: adding a sample, frag_size = %d, error = %f, old_error = %f",
				data->newfragsize, data->error, data->old_error);
	}
	else if ((data->error <= 0.0 && data->error < data->old_error) && ((data->old_src_size - extrafrag) >= data->minfragsize))
	{
		data->newfragsize = data->old_src_size - extrafrag;
		data->fractfrag = truecount - data->icount + extrafrag;
		data->icount = 0;
		data->ocount = 0;
		if (asrc_debug)
			slogf (_SLOGC_AUDIO, _SLOG_ERROR, "ASRC: dropping a sample, frag_size = %d, error = %f, old_error = %f",
				data->newfragsize, data->error, data->old_error);
	}
	data->old_error = data->error;

	/* Prevent the counters from wrapping */
	if (data->src_mode == SND_SRC_MODE_ASYNC)
	{
		if ((data->old_src_size + extrafrag >= data->maxfragsize || data->old_src_size - extrafrag <= data->minfragsize) &&
				truecount > CNTRESET )
		{
			data->icount = 0;
			data->ocount = 0;
			data->fractfrag = 0;
		}
	}
	else if (data->icount == data->ocount)
	{
		data->icount = 0;
		data->ocount = 0;
	}

	if (asrc_debug) {
		slogf (_SLOGC_AUDIO, _SLOG_ERROR, "ASRC: truecount=%f, icount=%i, ocount=%i, ratio=%f, newfragsize=%i, old_src_size = %zd, lvl=%d",
			truecount, data->icount, data->ocount, data->src_ratio, data->newfragsize, data->old_src_size, currlevel);
	}

	if (data->newfragsize != data->old_src_size)
		snd_pcm_plugin_setup(pcm, setup);

	return(get_client_fragsize (pcm, setup->channel, data->newfragsize));
}

int
snd_pcm_plugin_update_src (snd_pcm_t * pcm, snd_pcm_channel_setup_t * setup, int currlevel)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(currlevel), SLOG2_FA_END);
	return snd_pcm_plugin_update_src_internal(pcm, setup, currlevel);
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/pcm/plugin/rate.c $ $Rev: 812869 $")
#endif
