/*
 * $QNXLicenseC:
 * Copyright 2015, QNX Software Systems. All Rights Reserved.
 * 
 * You must obtain a written license from and pay applicable license fees to QNX 
 * Software Systems before you may reproduce, modify or distribute this software, 
 * or any work that includes all or part of this software.   Free development 
 * licenses are available for evaluation and non-commercial purposes.  For more 
 * information visit http://licensing.qnx.com or email licensing@qnx.com.
 *  
 * This file may contain contributions from others.  Please review this entire 
 * file for other proprietary rights or license notices, as well as the QNX 
 * Development Suite License Guide at http://licensing.qnx.com/license-guide/ 
 * for other information.
 * $
 */

#include "../pcm_local.h"
#include <rate_poly.h>
#include <errno.h>
#include <sys/slog.h>
#include <sys/slogcodes.h>

static  ssize_t rate_poly_src_size (snd_pcm_plugin_t * plugin, size_t dst_size)
{
    ResampleState_t *s;

    if (plugin == NULL || dst_size <= 0) {
        return -EINVAL;
    }
    s = (ResampleState_t *) snd_pcm_plugin_extra_data (plugin);

    if (s->playback) {
        if (s->dst_size == dst_size) {
            if (s->lastSrcSize == 0) {
                return s->src_size;
            } else {
                return s->lastSrcSize;
            }
        }
        s->dst_size = dst_size;
        s->src_size = (int32_t)( (int64_t)dst_size * s->dnScale / s->upScale);
        while (s->src_size * s->upScale < s->dst_size * s->dnScale) {
            s->src_size++;
        }
        while (s->src_size % (2 * s->channels) != 0) {
            s->src_size++;
        }
    }
    else {
        if (!(dst_size == s->dst_size || dst_size == s->lastDstSize)) {
            s->dst_size = dst_size;
            s->src_size = (int32_t)( (int64_t)dst_size * s->dnScale / s->upScale);
            while (s->src_size * s->upScale < s->dst_size * s->dnScale) {
                s->src_size++;
            }
            while (s->src_size % (2 * s->channels) != 0) {
                s->src_size++;
            }
        }
    }
    return s->src_size;
}

static  ssize_t rate_poly_dst_size (snd_pcm_plugin_t * plugin, size_t src_size)
{
    ResampleState_t *s;

    if (plugin == NULL || src_size <= 0)
        return -EINVAL;
    s = (ResampleState_t *) snd_pcm_plugin_extra_data (plugin);

    if (s->playback) {
        if (!(src_size == s->src_size || src_size == s->lastSrcSize)) {
            s->src_size = src_size;
            s->dst_size = (int32_t)( (int64_t)src_size * s->upScale / s->dnScale);
            while (s->dst_size * s->dnScale < s->src_size * s->upScale) {
                s->dst_size++;
            }
            while (s->dst_size % (2 * s->channels) != 0) {
                s->dst_size++;
            }
        }
    }
    else {
        if (s->src_size == src_size) {
            if (s->lastDstSize == 0) {
                return s->dst_size;
            } else {
                return s->lastDstSize;
            }
        }
        s->src_size = src_size;
        s->dst_size = (int32_t)( (int64_t)src_size * s->upScale / s->dnScale);
        while (s->dst_size * s->dnScale < s->src_size * s->upScale) {
            s->dst_size++;
        }
        while (s->dst_size % (2 * s->channels) != 0) {
            s->dst_size++;
        }
    }
    return s->dst_size;
}


static  ssize_t rate_poly_transfer (snd_pcm_plugin_t * plugin,
    void *src_ptr, size_t src_size, void *dst_ptr, size_t dst_size)
{
    ResampleState_t *s;
    int     rtn;

    if (plugin == NULL || src_ptr == NULL || src_size < 0 || dst_ptr == NULL || dst_size < 0)
        return -EINVAL;
    if (src_size == 0)
        return 0;
    s = (ResampleState_t *) snd_pcm_plugin_extra_data (plugin);
    if (s == NULL)
        return -EINVAL;

    if (s->playback && dst_size != s->dst_size) {
        if ((rtn = rate_poly_reset (s, dst_size)) != 0)
            return (rtn);
    }
    else if (!s->playback && src_size != s->src_size) {
        if ((rtn = rate_poly_reset (s, src_size)) != 0)
            return (rtn);
    }

    rtn = rate_poly_process(s, (int16_t *)src_ptr, s->src_size, (int16_t *)dst_ptr, s->dst_size);
    if (rtn < 0) 
        return rtn;

    return s->lastDstSize;
}

static int rate_poly_action (snd_pcm_plugin_t * plugin, snd_pcm_plugin_action_t action)
{
    ResampleState_t *s;
    int     rtn = 0;
    int     hwSize;

    if (plugin == NULL)
        return -EINVAL;
    s = (ResampleState_t *) snd_pcm_plugin_extra_data (plugin);
    if (s->playback) {
        hwSize = s->dst_size;
    } else {
        hwSize = s->src_size;
    }
    switch (action)
    {
        case INIT:
            slogf (_SLOGC_AUDIO, _SLOG_INFO, "RATE_POLY Init. Size %d -> %d", s->src_size, s->dst_size);
            rtn = rate_poly_reset(s, hwSize);
            break;
        case PREPARE:
            slogf (_SLOGC_AUDIO, _SLOG_INFO, "RATE_POLY Reset. Size %d -> %d", s->src_size, s->dst_size);
            rtn = rate_poly_reset(s, hwSize);
            break;
        case DRAIN:
            break;
        case POST_DRAIN:
            break;
        case FLUSH:
            rtn = rate_poly_flush(s);
            break;
    }
    return rtn;
}

static void rate_poly_private_free (snd_pcm_plugin_t * plugin, void *private_data)
{
    ResampleState_t *s;

    if (plugin == NULL)
        return;
    s = (ResampleState_t *) snd_pcm_plugin_extra_data (plugin);
    slogf (_SLOGC_AUDIO, _SLOG_INFO, "RATE_POLY Free. Converted %d -> %d", s->icount, s->ocount);
    rate_poly_free(s);
}


int snd_pcm_plugin_build_rate_poly (snd_pcm_format_t * src_format,
    snd_pcm_format_t * dst_format, snd_pcm_plugin_t ** r_plugin, 
    uint32_t src_mode, int32_t play_mode, uint32_t fixed_transfer_size,
    int32_t hw_frag_size, int32_t playback )
{
    ResampleState_t *s;
    snd_pcm_plugin_t *plugin;

    if (r_plugin == NULL)
        return -EINVAL;
    *r_plugin = NULL;

    slogf (_SLOGC_AUDIO, _SLOG_INFO, "RATE_POLY Create: %d -> %d. Ch: %d. Pb: %d. FragSize: %d. FixedTransferSize: %d",
        src_format->rate, dst_format->rate, src_format->voices, playback, hw_frag_size, fixed_transfer_size);
    if (src_mode != SND_SRC_MODE_NORMAL) 
        return -EINVAL;
    if (play_mode != SND_PCM_MODE_BLOCK)
        return -EINVAL;
    if (src_format->format != SND_PCM_SFMT_S16_LE)
        return -EINVAL;
    if (src_format->format != dst_format->format)
        return -EINVAL;
    if (src_format->interleave != dst_format->interleave && src_format->voices > 1)
        return -EINVAL;
    if (!dst_format->interleave && src_format->voices > 1)
        return -EINVAL;
    if (src_format->voices != dst_format->voices)
        return -EINVAL;
    if (dst_format->voices < 1 || dst_format->voices > 8)
        return -EINVAL;
    if (src_format->rate == dst_format->rate)
        return -EINVAL;
    if (rate_poly_support_idx(src_format->rate, src_format->voices, dst_format->rate, hw_frag_size, fixed_transfer_size) < 0)
        return -EINVAL;

    plugin = snd_pcm_plugin_build ("rate_poly conversion", sizeof (ResampleState_t));
    if (plugin == NULL)
        return -ENOMEM;

    s = (ResampleState_t *) snd_pcm_plugin_extra_data (plugin);
    if (rate_poly_create(s, src_format->rate, src_format->voices, dst_format->rate, hw_frag_size, fixed_transfer_size, playback) < 0) {
        snd_pcm_plugin_free( plugin );
        return -EINVAL;
    }

    plugin->transfer = rate_poly_transfer;
    plugin->src_size = rate_poly_src_size;
    plugin->dst_size = rate_poly_dst_size;
    plugin->action = rate_poly_action;
    plugin->private_free = rate_poly_private_free;

    *r_plugin = plugin;
    return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/pcm/plugin/rate_poly.c $ $Rev: 812869 $")
#endif
