/*
 *  Voices conversion Plug-In
 *  Copyright (c) 1999 by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdbool.h>
#include <sys/ioctl.h>
#include <devctl.h>
#include <asound_endian.h>
#include <asound_byteswap.h>
#include <sys/slog2.h>
#include "../pcm_local.h"

/*
 *  Basic voices conversion plugin
 */

struct voices_private_data
{
	snd_pcm_t * pcm;
	int	 src_voices;
	int	 dst_voices;
	int	 width;						/* in bytes */
	int	 flg_signed:1;
	int	 flg_endian:1;				/* true indicates big endian */
	int	 channel;
	uint32_t voice_matrix[32];
};

// To make the endian macros consistent, because there are no little and big-endian 8-bit macros
#define ENDIAN_8(x) x

#define CONVERT_VOICES_FUNC(name, sign, type, endian, acctype) \
void convert_voices_##name##sign##endian(int channel, void *void_src_ptr, void *void_dst_ptr, size_t src_buf_size, int src_voices,	\
									 int dst_voices, uint32_t voice_matrix[])			\
{																						\
	type *src_ptr = (type *)void_src_ptr;												\
	type *dst_ptr = (type *)void_dst_ptr;												\
	int dst_bidx, src_bidx, sample_cnt, src_vidx, dst_vidx;								\
	int *app_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &src_vidx : &dst_vidx);		\
	int *hw_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &dst_vidx : &src_vidx);		\
	acctype accum;																		\
	int count;																			\
																						\
	for (dst_bidx=src_bidx=sample_cnt=0; sample_cnt < src_buf_size / src_voices;		\
			dst_bidx+=dst_voices, src_bidx+=src_voices, sample_cnt++)					\
	{																					\
		for (dst_vidx = 0; dst_vidx < dst_voices; dst_vidx++)							\
		{																				\
			accum = 0;																	\
			count = 0;																	\
			for (src_vidx = 0; src_vidx < src_voices; src_vidx++)						\
			{																			\
				if ( (voice_matrix[*app_vidx] >> *hw_vidx) & 0x1) {						\
					/* explicit cast to type needed here to restore sign before			\
					   implicit cast to accumtype, as ENDIAN macro casts to unsigned */	\
					accum += (type)ENDIAN_##endian##name(src_ptr[src_bidx + src_vidx]);	\
					count++;															\
				}																		\
			}																			\
			if( count > 0 ) {															\
				accum /= count;															\
				dst_ptr[dst_bidx + dst_vidx] = ENDIAN_##endian##name((type)accum);		\
			} else {																	\
				dst_ptr[dst_bidx + dst_vidx] = 0;										\
			}																			\
		}																				\
	}																					\
}

void convert_voices_24sLE(int channel, void *void_src_ptr, void *void_dst_ptr, size_t src_buf_size, int src_voices,
									 int dst_voices, uint32_t voice_matrix[])
{
	uint8_t *src_ptr = (uint8_t *)void_src_ptr;
	uint8_t *dst_ptr = (uint8_t *)void_dst_ptr;
	int dst_bidx, src_bidx, sample_cnt, src_vidx, dst_vidx;	
	int *app_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &src_vidx : &dst_vidx);
	int *hw_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &dst_vidx : &src_vidx);
	int32_t accum;
	int count;

	for (dst_bidx=src_bidx=sample_cnt=0; sample_cnt < src_buf_size / src_voices;
			dst_bidx+=(3 * dst_voices), src_bidx+=(3 * src_voices), sample_cnt++)
	{
		for (dst_vidx = 0; dst_vidx < dst_voices; dst_vidx++)
		{
			accum = 0;
			count = 0;
			for (src_vidx = 0; src_vidx < src_voices; src_vidx++)
			{
				if ( (voice_matrix[*app_vidx] >> *hw_vidx) & 0x1) {
					accum += (src_ptr[src_bidx + src_vidx * 3 + 2] << 16) + (src_ptr[src_bidx + src_vidx * 3 + 1] << 8) + src_ptr[src_bidx + src_vidx * 3] + (src_ptr[src_bidx + src_vidx * 3 + 2] & 0x80 ? 0xFF000000 : 0);
					count ++;
				}
			}
			if( count > 0 ) {
				accum /= count;
				dst_ptr[dst_bidx + dst_vidx * 3] = accum & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = (accum >> 8) & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = (accum >> 16) & 0xFF;
			} else {
				dst_ptr[dst_bidx + dst_vidx * 3] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = 0;
			}
		}
	}
}

void convert_voices_24uLE(int channel, void *void_src_ptr, void *void_dst_ptr, size_t src_buf_size, int src_voices,
									 int dst_voices, uint32_t voice_matrix[])
{
	uint8_t *src_ptr = (uint8_t *)void_src_ptr;
	uint8_t *dst_ptr = (uint8_t *)void_dst_ptr;
	int dst_bidx, src_bidx, sample_cnt, src_vidx, dst_vidx;	
	int *app_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &src_vidx : &dst_vidx);
	int *hw_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &dst_vidx : &src_vidx);
	uint32_t accum;
	int count;

	for (dst_bidx=src_bidx=sample_cnt=0; sample_cnt < src_buf_size / src_voices;
			dst_bidx+=(3 * dst_voices), src_bidx+=(3 * src_voices), sample_cnt++)
	{
		for (dst_vidx = 0; dst_vidx < dst_voices; dst_vidx++)
		{
			accum = 0;
			count = 0;
			for (src_vidx = 0; src_vidx < src_voices; src_vidx++)
			{
				if ( (voice_matrix[*app_vidx] >> *hw_vidx) & 0x1) {
					accum += (src_ptr[src_bidx + src_vidx * 3 + 2] << 16) + (src_ptr[src_bidx + src_vidx * 3 + 1] << 8) + src_ptr[src_bidx + src_vidx * 3];
					count ++;
				}
			}
			if( count > 0 ) {
				accum /= count;
				dst_ptr[dst_bidx + dst_vidx * 3] = accum & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = (accum >> 8) & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = (accum >> 16) & 0xFF;
			} else {
				dst_ptr[dst_bidx + dst_vidx * 3] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = 0;
			}
		}
	}
}


void convert_voices_24sBE(int channel, void *void_src_ptr, void *void_dst_ptr, size_t src_buf_size, int src_voices,
									 int dst_voices, uint32_t voice_matrix[])
{
	uint8_t *src_ptr = (uint8_t *)void_src_ptr;
	uint8_t *dst_ptr = (uint8_t *)void_dst_ptr;
	int dst_bidx, src_bidx, sample_cnt, src_vidx, dst_vidx;	
	int *app_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &src_vidx : &dst_vidx);
	int *hw_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &dst_vidx : &src_vidx);
	int32_t accum;
	int count;

	for (dst_bidx=src_bidx=sample_cnt=0; sample_cnt < src_buf_size / src_voices;
			dst_bidx+=(3 * dst_voices), src_bidx+=(3 * src_voices), sample_cnt++)
	{
		for (dst_vidx = 0; dst_vidx < dst_voices; dst_vidx++)
		{
			accum = 0;
			count = 0;
			for (src_vidx = 0; src_vidx < src_voices; src_vidx++)
			{
				if ( (voice_matrix[*app_vidx] >> *hw_vidx) & 0x1) {
					accum += (src_ptr[src_bidx + src_vidx * 3] << 16) + (src_ptr[src_bidx + src_vidx * 3 + 1] << 8) + src_ptr[src_bidx + src_vidx * 3 + 2] + (src_ptr[src_bidx + src_vidx * 3] & 0x80 ? 0xFF000000 : 0);
					count ++;
				}
			}
			if( count > 0 ) {
				accum /= count;
				dst_ptr[dst_bidx + dst_vidx * 3] = (accum >> 16) & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = (accum >> 8) & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = accum & 0xFF;
			} else {
				dst_ptr[dst_bidx + dst_vidx * 3] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = 0;
			}
		}
	}
}

void convert_voices_24uBE(int channel, void *void_src_ptr, void *void_dst_ptr, size_t src_buf_size, int src_voices,
									 int dst_voices, uint32_t voice_matrix[])
{
	uint8_t *src_ptr = (uint8_t *)void_src_ptr;
	uint8_t *dst_ptr = (uint8_t *)void_dst_ptr;
	int dst_bidx, src_bidx, sample_cnt, src_vidx, dst_vidx;	
	int *app_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &src_vidx : &dst_vidx);
	int *hw_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &dst_vidx : &src_vidx);
	uint32_t accum;
	int count;

	for (dst_bidx=src_bidx=sample_cnt=0; sample_cnt < src_buf_size / src_voices;
			dst_bidx+=(3 * dst_voices), src_bidx+=(3 * src_voices), sample_cnt++)
	{
		for (dst_vidx = 0; dst_vidx < dst_voices; dst_vidx++)
		{
			accum = 0;
			count = 0;
			for (src_vidx = 0; src_vidx < src_voices; src_vidx++)
			{
				if ( (voice_matrix[*app_vidx] >> *hw_vidx) & 0x1) {
					accum += (src_ptr[src_bidx + src_vidx * 3] << 16) + (src_ptr[src_bidx + src_vidx * 3 + 1] << 8) + src_ptr[src_bidx + src_vidx * 3 + 2];
					count ++;
				}
			}
			if( count > 0 ) {
				accum /= count;
				dst_ptr[dst_bidx + dst_vidx * 3] = (accum >> 16) & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = (accum >> 8) & 0xFF;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = accum & 0xFF;
			} else {
				dst_ptr[dst_bidx + dst_vidx * 3] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 1] = 0;
				dst_ptr[dst_bidx + dst_vidx * 3 + 2] = 0;
			}
		}
	}
}


/* *INDENT-OFF* */
CONVERT_VOICES_FUNC (8,s,int8_t,, int32_t)
CONVERT_VOICES_FUNC (16,s, int16_t,LE, int32_t)
CONVERT_VOICES_FUNC (32,s, int32_t,LE, int64_t)
CONVERT_VOICES_FUNC (8,u, uint8_t,, uint32_t)
CONVERT_VOICES_FUNC (16,u, uint16_t, LE, uint32_t)
CONVERT_VOICES_FUNC (32,u, uint32_t, LE, uint64_t)
CONVERT_VOICES_FUNC (16,s, int16_t, BE, int32_t)
CONVERT_VOICES_FUNC (32,s, int32_t, BE, int64_t)
CONVERT_VOICES_FUNC (16,u, uint16_t, BE, uint32_t)
CONVERT_VOICES_FUNC (32,u, uint32_t, BE, uint64_t)
/* *INDENT-ON* */

typedef void (*converter)(int, void *, void *, size_t, int, int, uint32_t[]);

converter converters[4][2][2] = {
	{
		{ convert_voices_8u, convert_voices_8u },
		{ convert_voices_8s, convert_voices_8s },
	},
	{
		{ convert_voices_16uLE, convert_voices_16uBE },
		{ convert_voices_16sLE, convert_voices_16sBE },
	},
	{
		{ convert_voices_24uLE, convert_voices_24uBE },
		{ convert_voices_24sLE, convert_voices_24sBE },
	},
	{
		{ convert_voices_32uLE, convert_voices_32uBE },
		{ convert_voices_32sLE, convert_voices_32sBE },
	},
};

static  ssize_t
voices_transfer (snd_pcm_plugin_t * plugin,
	void *src_ptr, size_t src_size, void *dst_ptr, size_t dst_size)
{
	struct voices_private_data *data;

	if (plugin == NULL || src_ptr == NULL || src_size < 0 || dst_ptr == NULL || dst_size < 0)
		return -EINVAL;
	if (src_size == 0)
		return 0;
	data = (struct voices_private_data *) snd_pcm_plugin_extra_data (plugin);
	if (data == NULL)
		return -EINVAL;
	converters[data->width-1][!!data->flg_signed][!!data->flg_endian](data->channel, src_ptr, dst_ptr, src_size / data->width, data->src_voices, data->dst_voices, data->voice_matrix);

	return (src_size * data->dst_voices) / data->src_voices;
}

static  ssize_t
voices_src_size (snd_pcm_plugin_t * plugin, size_t size)
{
	struct voices_private_data *data;

	if (plugin == NULL || size <= 0)
		return -EINVAL;
	data = (struct voices_private_data *) snd_pcm_plugin_extra_data (plugin);
	return (size * data->src_voices) / data->dst_voices;
}

static  ssize_t
voices_dst_size (snd_pcm_plugin_t * plugin, size_t size)
{
	struct voices_private_data *data;

	if (plugin == NULL || size <= 0)
		return -EINVAL;
	data = (struct voices_private_data *) snd_pcm_plugin_extra_data (plugin);
	return (size * data->dst_voices) / data->src_voices;
}

static void voices_free (snd_pcm_plugin_t * plugin, void *private_data)
{
}

int
snd_pcm_plugin_get_voice_conversion (snd_pcm_t * pcm, int channel,
	snd_pcm_voice_conversion_t * voice_conversion)
{
	snd_pcm_plugin_t *plugin;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

	if (pcm == NULL || voice_conversion == NULL || channel < 0 || channel > 1)
		return -EINVAL;

	for (plugin = pcm->plugin_first[channel]; plugin != NULL; plugin = plugin->next)
	{
		if (strcmp (plugin->name, "voices conversion") == 0)
		{
			struct voices_private_data *data =
				(struct voices_private_data *) snd_pcm_plugin_extra_data (plugin);
			if (channel == SND_PCM_CHANNEL_PLAYBACK)
			{
				voice_conversion->app_voices = data->src_voices;
				voice_conversion->hw_voices = data->dst_voices;
			}
			else
			{
				voice_conversion->app_voices = data->dst_voices;
				voice_conversion->hw_voices = data->src_voices;
			}
			memcpy (voice_conversion->matrix, data->voice_matrix,
				sizeof (voice_conversion->matrix));
			return EOK;
		}
	}
	return -ENOENT;
}

static struct voices_private_data *
snd_pcm_plugin_find_voice_plugin (snd_pcm_t * pcm, int channel)
{
	snd_pcm_plugin_t *plugin;

	if (pcm == NULL || channel < 0 || channel > 1)
		return NULL;

	for (plugin = pcm->plugin_first[channel]; plugin != NULL; plugin = plugin->next)
	{
		if (strcmp (plugin->name, "voices conversion") == 0)
		{
			return (struct voices_private_data *) snd_pcm_plugin_extra_data (plugin);
		}
	}
	return NULL;
}

int
snd_pcm_plugin_set_voice_conversion (snd_pcm_t * pcm, int channel,
	snd_pcm_voice_conversion_t * voice_conversion)
{
	int ret = EOK;
	struct voices_private_data *data;

	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);

	if (pcm == NULL || voice_conversion == NULL || channel < 0 || channel > 1)
		return -EINVAL;

	snd_pcm_plugin_set_enable(pcm, PLUGIN_VOICE);
	// Try to set the voice matrix
	data = snd_pcm_plugin_find_voice_plugin(pcm, channel);
	if (data)
	{
		memcpy (data->voice_matrix, voice_conversion->matrix, sizeof (data->voice_matrix));
	}
	else
	{
		// We don't have a voice plugin, so reparameterize and try again
		snd_pcm_channel_params(pcm, &pcm->params[(pcm->mode & SND_PCM_OPEN_PLAYBACK)?SND_PCM_CHANNEL_PLAYBACK:SND_PCM_CHANNEL_CAPTURE]);
		data = snd_pcm_plugin_find_voice_plugin(pcm, channel);
		if (data)
		{
			memcpy (data->voice_matrix, voice_conversion->matrix, sizeof (data->voice_matrix));
		}
		else
			ret = -ENOENT;
	}
	return ret;
}

static void snd_pcm_plugin_voice_initialize_matrix(uint32_t matrix[32], int channel, int src_voices, int dst_voices)
{
	int i;
	int j;

	if (src_voices <= dst_voices)
	{
		memset (matrix, 0, sizeof (uint32_t) * 32);
		/* For stereo and quad destination replicate all */
		if (dst_voices == 2 || dst_voices == 4)
		{
			int	*app_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &i : &j);
			int	*hw_vidx = (channel == SND_PCM_CHANNEL_PLAYBACK ? &j : &i);

			for (i = 0; i < src_voices; i++)
			{
				for (j = i; j < dst_voices; j += src_voices)
					matrix[*app_vidx] |= 1 << *hw_vidx;
			}
		}
		else	/* Silence fill missing channels for all other destination */
		{
			for (i = 0; i < src_voices; i++)
				matrix[i] |= 1 << i;
		}
	}
	else
	{
		memset (matrix, 0, sizeof (uint32_t) * 32);
		/* Down mix for stereo to mono conversion */
		if (src_voices == 2 && dst_voices == 1)
		{
			if (channel == SND_PCM_CHANNEL_PLAYBACK)
			{
				matrix[SND_MIXER_CHN_FRONT_LEFT] = SND_MIXER_CHN_MASK_FRONT_LEFT;
				matrix[SND_MIXER_CHN_FRONT_RIGHT] = SND_MIXER_CHN_MASK_FRONT_LEFT;
			}
			else
				matrix[SND_MIXER_CHN_FRONT_LEFT] = SND_MIXER_CHN_MASK_STEREO;
		}
		else
		{
			for (i = 0; i < dst_voices; i++)
				matrix[i] |= 1 << i;
		}
	}
}

int
snd_pcm_plugin_reset_voice_conversion (snd_pcm_t * pcm, int channel)
{
	slog2fa(NULL, 0, SLOG2_DEBUG2, "%s - %x %d", SLOG2_FA_STRING(__func__), SLOG2_FA_UNSIGNED(pcm), SLOG2_FA_UNSIGNED(channel), SLOG2_FA_END);
	struct voices_private_data *data;

	data = snd_pcm_plugin_find_voice_plugin(pcm, channel);
	// It's okay to not have a voice conversion matrix, then no need to reset
	if (data == NULL) {
		return EOK;
	}
	snd_pcm_plugin_voice_initialize_matrix(data->voice_matrix, channel, data->src_voices, data->dst_voices);

	return EOK;
}

static int voices_action (snd_pcm_plugin_t * plugin, snd_pcm_plugin_action_t action)
{
	return EOK;
}

int
snd_pcm_plugin_build_voices (snd_pcm_t *pcm, snd_pcm_format_t * src_format,
	snd_pcm_format_t * dst_format, int channel, snd_pcm_plugin_t ** r_plugin)
{
	snd_pcm_plugin_t *plugin;
	struct voices_private_data *data;

	if (r_plugin == NULL)
		return -EINVAL;
	*r_plugin = NULL;

	if (src_format->interleave != dst_format->interleave && src_format->voices > 1)
		return -EINVAL;
	if (!dst_format->interleave)
		return -EINVAL;
	if (src_format->format != dst_format->format)
		return -EINVAL;
	if (src_format->rate != dst_format->rate)
		return -EINVAL;
	if (src_format->voices < 1 || src_format->voices > 32 ||
		dst_format->voices < 1 || dst_format->voices > 32)
		return -EINVAL;

	if (src_format->format < SND_PCM_SFMT_U8 || src_format->format > SND_PCM_SFMT_S32_BE)
	{
		if (src_format->format != SND_PCM_SFMT_MU_LAW && src_format->format != SND_PCM_SFMT_A_LAW)
			return -EINVAL;
	}
	plugin = snd_pcm_plugin_build ("voices conversion", sizeof (struct voices_private_data));
	if (plugin == NULL)
		return -ENOMEM;
	data = (struct voices_private_data *) snd_pcm_plugin_extra_data (plugin);

	data->channel = channel;
	snd_pcm_plugin_voice_initialize_matrix(data->voice_matrix, channel, src_format->voices, dst_format->voices);
	data->pcm = pcm;
	data->src_voices = src_format->voices;
	data->dst_voices = dst_format->voices;
	data->width = snd_pcm_format_width (src_format->format) / _BITS_BYTE;
	data->flg_signed = snd_pcm_format_signed (src_format->format);
	data->flg_endian = snd_pcm_format_big_endian (src_format->format);
	plugin->transfer = voices_transfer;
	plugin->src_size = voices_src_size;
	plugin->dst_size = voices_dst_size;
	plugin->action = voices_action;
	plugin->private_free = voices_free;
	*r_plugin = plugin;
	return 0;
}

#ifdef __KERNEL__
EXPORT_SYMBOL (snd_pcm_plugin_build_voices);
#endif

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/pcm/plugin/voices.c $ $Rev: 815673 $")
#endif
