/*
 * $QNXLicenseC:
 * Copyright 2007, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable license fees to QNX
 * Software Systems before you may reproduce, modify or distribute this software,
 * or any work that includes all or part of this software.   Free development
 * licenses are available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review this entire
 * file for other proprietary rights or license notices, as well as the QNX
 * Development Suite License Guide at http://licensing.qnx.com/license-guide/
 * for other information.
 * $
 */





#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <math.h>
#include <errno.h>
#include <sys/termio.h>
#include <string.h>
#include <ctype.h>
#include <pthread.h>
#include <sys/asoundlib.h>
#include "../../pcm/pcm_local.h"

#define FREQ        880

bool use_mmap = 0;
bool use_plugins = 1;
bool error_state = false;
bool running = true;
long long position = 0;
int nonblock = 0;
int method = -1;
int mode = -1;
int n;
snd_mixer_group_t group;
snd_mixer_t *mixer_handle;
snd_pcm_t *pcm_handle;
char   *mSampleBfr1;
int     mSampleRate;

int
err (char *msg)
{
    perror (msg);
    return -1;
}

int
dev_raw (int fd)
{
    struct termios termios_p;

    if (tcgetattr (fd, &termios_p))
        return (-1);

    termios_p.c_cc[VMIN] = 1;
    termios_p.c_cc[VTIME] = 0;
    termios_p.c_lflag &= ~(ICANON | ECHO | ISIG);
    return (tcsetattr (fd, TCSANOW, &termios_p));
}

int
dev_unraw (int fd)
{
    struct termios termios_p;

    if (tcgetattr (fd, &termios_p))
        return (-1);

    termios_p.c_lflag |= (ICANON | ECHO | ISIG);
    return (tcsetattr (fd, TCSAFLUSH, &termios_p));
}

void
handle_keypress()
{
    int     c;
    int     rtn;
    int     vol_change = 0;

    if ((rtn = snd_mixer_group_read (mixer_handle, &group)) < 0)
        fprintf (stderr, "snd_mixer_group_read failed: %s\n", snd_strerror (rtn));

    c = getc (stdin);

    if (c != EOF)
    {
        switch (c)
        {
        case 'a':
            nonblock ^= 1;
            snd_pcm_nonblock_mode( pcm_handle, nonblock );
            break;
        case 'b':
            {
                snd_pcm_info_t info;
                memset(&info, 0, sizeof(info));
                rtn = snd_pcm_info( pcm_handle, &info );
                printf("snd_pcm_info: %d\n", rtn);
                printf("type: %d\n", info.type);
                printf("flags: %d\n", info.flags);
                printf("id: %s\n", info.id);
                printf("name: %s\n", info.name);
                printf("playback: %d\n", info.playback);
                printf("capture: %d\n", info.capture);
                printf("card: %d\n", info.card);
                printf("device: %d\n", info.device);
                printf("shared_device: %d\n", info.shared_device);
            }
            break;
        case 'c':
            {
                snd_pcm_channel_info_t info;
                memset(&info, 0, sizeof(info));
                info.channel = 0;
                rtn = snd_pcm_channel_info( pcm_handle, &info );
                printf("snd_pcm_channel_info: %d\n", rtn);
                printf("subdevice: %d\n", info.subdevice);
                printf("subname: %s\n", info.subname);
                printf("flags: %d\n", info.flags);
                printf("formats: %d\n", info.formats);
                printf("rates: %d\n", info.rates);
                printf("min_rate: %d\n", info.min_rate);
                printf("max_rate: %d\n", info.max_rate);
                printf("min_voices: %d\n", info.min_voices);
                printf("max_voices: %d\n", info.max_voices);
                printf("max_buffer_size: %d\n", info.max_buffer_size);
                printf("min_fragment_size: %d\n", info.min_fragment_size);
                printf("max_fragment_size: %d\n", info.max_fragment_size);
                printf("mixer_device: %d\n", info.mixer_device);
            }
            break;
        case 'd':
            {
                snd_pcm_channel_setup_t setup;
                memset(&setup, 0, sizeof(setup));
                rtn = snd_pcm_channel_setup( pcm_handle, &setup );
                printf("snd_pcm_channel_setup: %d\n", rtn);
                printf("format: %d\n", setup.format.format);
                printf("rate: %d\n", setup.format.rate);
                printf("voices: %d\n", setup.format.voices);
                printf("mixer_device: %d\n", setup.mixer_device);
                printf("mmap_valid: %d\n", setup.mmap_valid);
                printf("mmap_active: %d\n", setup.mmap_active);
                printf("mixer_card: %d\n", setup.mixer_card);
            }
            break;
        case'e':
            {
                snd_pcm_channel_status_t status;
                memset(&status, 0, sizeof(status));
                status.channel = 0;
                rtn = snd_pcm_channel_status(pcm_handle, &status);
                printf("snd_pcm_channel_status: %d\n", rtn);
                printf("mode: %d\n", status.mode);
                printf("status: %d\n", status.status);
                printf("scount: %d\n", status.scount);
                printf("stime: %d\n", status.stime.tv_sec);
                printf("ust_stime: %lld\n", status.ust_stime);
                printf("frag: %d\n", status.frag);
                printf("count: %d\n", status.count);
                printf("free: %d\n", status.free);
                printf("underrun: %d\n", status.underrun);
                printf("overrun: %d\n", status.overrun);
                printf("overrange: %d\n", status.overrange);
                printf("subbuffered: %d\n", status.subbuffered);
            }
            break;
        case 'f':
            rtn = snd_pcm_playback_prepare( pcm_handle );
            printf("snd_pcm_playback_prepare: %d\n", rtn);
            break;
        case 'g':
            rtn = snd_pcm_playback_go( pcm_handle );
            printf("snd_pcm_playback_go: %d\n", rtn);
            break;
        case 'h':
            rtn = snd_pcm_playback_pause( pcm_handle );
            printf("snd_pcm_playback_pause: %d\n", rtn);
            break;
        case 'i':
            rtn = snd_pcm_playback_resume( pcm_handle );
            printf("snd_pcm_playback_resume: %d\n", rtn);
            break;
        case 'j':
            rtn = snd_pcm_playback_drain( pcm_handle );
            printf("snd_pcm_playback_drain: %d\n", rtn);
            break;
        case 'k':
            rtn = snd_pcm_playback_flush( pcm_handle );
            printf("snd_pcm_playback_flush: %d\n", rtn);
            break;
        case 'l':
            rtn = snd_pcm_transfer_size( pcm_handle, 0 );
            printf("snd_pcm_transfer_size: %d\n", rtn);
            break;
        case 'm':
            rtn = snd_pcm_plugin_transfer_size( pcm_handle, 0, 1000 );
            printf("snd_pcm_plugin_transfer_size: %d\n", rtn);
            break;
        case 'n':
            rtn = snd_pcm_plugin_hardware_size( pcm_handle, 0, 1000 );
            printf("snd_pcm_plugin_hardware_size: %d\n", rtn);
            break;
        case 'o':
            {
                snd_pcm_channel_info_t info;
                memset(&info, 0, sizeof(info));
                info.channel = 0;
                rtn = snd_pcm_plugin_info( pcm_handle, &info );
                printf("snd_pcm_channel_info: %d\n", rtn);
                printf("subdevice: %d\n", info.subdevice);
                printf("subname: %s\n", info.subname);
                printf("flags: %d\n", info.flags);
                printf("formats: %d\n", info.formats);
                printf("rates: %d\n", info.rates);
                printf("min_rate: %d\n", info.min_rate);
                printf("max_rate: %d\n", info.max_rate);
                printf("min_voices: %d\n", info.min_voices);
                printf("max_voices: %d\n", info.max_voices);
                printf("max_buffer_size: %d\n", info.max_buffer_size);
                printf("min_fragment_size: %d\n", info.min_fragment_size);
                printf("max_fragment_size: %d\n", info.max_fragment_size);
                printf("mixer_device: %d\n", info.mixer_device);
            }
            break;
        case 'p':
            rtn = snd_pcm_plugin_set_src_method( pcm_handle, method );
            printf("snd_pcm_plugin_set_src_method: %d %d\n", rtn, method);
            method = (method + 1) % 4;
            break;
        case 'q':
            rtn = snd_pcm_plugin_set_src_mode( pcm_handle, mode, rand() % 100 );
            printf("snd_pcm_plugin_set_src_mode: %d %d\n", rtn, mode);
            mode = (mode + 1) % (SND_SRC_MODE_PITCH + 1);
            break;
        case 's':
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_LEFT)
                group.volume.names.front_left -= 10;
            if (group.channels & SND_MIXER_CHN_MASK_REAR_LEFT)
                group.volume.names.rear_left -= 10;
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_CENTER)
                group.volume.names.front_center -= 10;
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_RIGHT)
                group.volume.names.front_right -= 10;
            if (group.channels & SND_MIXER_CHN_MASK_REAR_RIGHT)
                group.volume.names.rear_right -= 10;
            if (group.channels & SND_MIXER_CHN_MASK_WOOFER)
                group.volume.names.woofer -= 10;
            vol_change = 1;
            break;
        case 'w':
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_LEFT)
                group.volume.names.front_left += 10;
            if (group.channels & SND_MIXER_CHN_MASK_REAR_LEFT)
                group.volume.names.rear_left += 10;
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_CENTER)
                group.volume.names.front_center += 10;
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_RIGHT)
                group.volume.names.front_right += 10;
            if (group.channels & SND_MIXER_CHN_MASK_REAR_RIGHT)
                group.volume.names.rear_right += 10;
            if (group.channels & SND_MIXER_CHN_MASK_WOOFER)
                group.volume.names.woofer += 10;
            vol_change = 1;
            break;
        // Exit the program
        case 3: // Ctrl-C
        case 27: // Escape
            running = false;
            break;
        }
        error_state = false;
        if( vol_change ) {
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_LEFT)
            {
                if (group.volume.names.front_left > group.max)
                    group.volume.names.front_left = group.max;
                if (group.volume.names.front_left < group.min)
                    group.volume.names.front_left = group.min;
            }
            if (group.channels & SND_MIXER_CHN_MASK_REAR_LEFT)
            {
                if (group.volume.names.rear_left > group.max)
                    group.volume.names.rear_left = group.max;
                if (group.volume.names.rear_left < group.min)
                    group.volume.names.rear_left = group.min;
            }
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_CENTER)
            {
                if (group.volume.names.front_center > group.max)
                    group.volume.names.front_center = group.max;
                if (group.volume.names.front_center < group.min)
                    group.volume.names.front_center = group.min;
            }
            if (group.channels & SND_MIXER_CHN_MASK_FRONT_RIGHT)
            {
                if (group.volume.names.front_right > group.max)
                    group.volume.names.front_right = group.max;
                if (group.volume.names.front_right < group.min)
                    group.volume.names.front_right = group.min;
            }
            if (group.channels & SND_MIXER_CHN_MASK_REAR_RIGHT)
            {
                if (group.volume.names.rear_right > group.max)
                    group.volume.names.rear_right = group.max;
                if (group.volume.names.rear_right < group.min)
                    group.volume.names.rear_right = group.min;
            }
            if (group.channels & SND_MIXER_CHN_MASK_WOOFER)
            {
                if (group.volume.names.woofer > group.max)
                    group.volume.names.woofer = group.max;
                if (group.volume.names.woofer < group.min)
                    group.volume.names.woofer = group.min;
            }
            printf ("Volume Now at %d:%d \n",
                (group.max - group.min) ? 100 * (group.volume.names.front_left - group.min) / (group.max - group.min) : 0,
                (group.max - group.min) ? 100 * (group.volume.names.front_right - group.min) / (group.max - group.min): 0);
            if ((rtn = snd_mixer_group_write (mixer_handle, &group)) < 0)
                fprintf (stderr, "snd_mixer_group_write failed: %s\n", snd_strerror (rtn));
        }
    }
    else
    {
        running = false;
    }

}

void handle_mixer() 
{
    fd_set  rfds;

    FD_SET (snd_mixer_file_descriptor (mixer_handle), &rfds);

    if (select (snd_mixer_file_descriptor (mixer_handle) + 1, &rfds, NULL, NULL, NULL) == -1) {
        printf ("select failed\n");
        exit(1);
    }

    snd_mixer_callbacks_t callbacks = { 0, 0, 0, 0 };

    snd_mixer_read (mixer_handle, &callbacks);
}

void write_audio_data()
{
    snd_pcm_channel_status_t status;
    int     written = 0;
    int     i;

    for(i = 0; i < n; i ++ ) {
        mSampleBfr1[i] = sin(2*M_PI*FREQ*(position+i)/mSampleRate)*127+127;
    }
    position += n;
    written = snd_pcm_plugin_write (pcm_handle, mSampleBfr1, n);

    if (written < n)
    {
        memset (&status, 0, sizeof (status));
        status.channel = SND_PCM_CHANNEL_PLAYBACK;
        if (snd_pcm_plugin_status (pcm_handle, &status) < 0)
        {
            fprintf (stderr, "underrun: playback channel status error\n");
            exit (1);
        }

        if (status.status == SND_PCM_STATUS_READY)
        {
            fprintf (stderr, "underrun: playback channel prepare error\n");
            error_state = true;
        }
        else if (status.status == SND_PCM_STATUS_UNDERRUN)
        {
            fprintf (stderr, "Underrun\n");
            error_state = true;
        }
        else if (status.status == SND_PCM_STATUS_UNSECURE)
        {
            fprintf (stderr, "Channel unsecure\n");
            error_state = true;
        }
        else if (status.status == SND_PCM_STATUS_CHANGE)
        {
            fprintf(stderr, "Channel capability change\n");
            error_state = true;
        }
        else if (status.status == SND_PCM_STATUS_ERROR)
        {
            fprintf(stderr, "error: playback channel failure\n");
            error_state = true;
        }
        else if (status.status == SND_PCM_STATUS_PREEMPTED)
        {
            fprintf(stderr, "error: playback channel preempted\n");
            error_state = true;
        }
    }
}

void *writer_thread_handler(void *data)
{
    while (running)
    {
        write_audio_data();
    }

    return NULL;
}

void *generic_thread_handler(void *data)
{
    while(1) {
        ((void (*)(void))data)();
    }

    return NULL;
}


//*****************************************************************************
/* *INDENT-OFF* */
#ifdef __USAGE
%C[Options] *

Options:
    -a[card#:]<dev#>   the card & device number to play out on
    -f<frag_size>      requested fragment size
    -s                 content is protected
    -e                 content would like to be played on a secure channel
    -r                 content can only be played on a secure channel
    -w                 use separate threads to control and write audio data
    -c<args>[,args ..] voice matrix configuration
    -n<num_frags>      requested number of fragments
    -b<num_frags>      requested number of fragments while buffering

Args:
    1=<hw_channel_bitmask> hardware channel bitmask for application voice 1
    2=<hw_channel_bitmask> hardware channel bitmask for application voice 2
    3=<hw_channel_bitmask> hardware channel bitmask for application voice 3
    4=<hw_channel_bitmask> hardware channel bitmask for application voice 4
#endif
/* *INDENT-ON* */
//*****************************************************************************


int
main (int argc, char **argv)
{
    int     card = -1;
    int     dev = 0;
    int     mSampleChannels;
    int     mSampleBits;
    int     fragsize = -1;
    int     disable_mask = 0;
    int     enable_mask = 0;

    int     rtn;
    snd_pcm_channel_info_t pi;
    snd_pcm_channel_params_t pp;
    snd_pcm_channel_setup_t setup;
    int c;
    fd_set  rfds, wfds;
    uint32_t voice_mask[] = { 0, 0, 0, 0 };
    snd_pcm_voice_conversion_t voice_conversion;
    int     voice_override = 0;
    int     num_frags = -1;
    int     num_buffered_frags = 0;
    char   *sub_opts, *value;
    char   *dev_opts[] = {
#define CHN1 0
        "1",
#define CHN2 1
        "2",
#define CHN3 2
        "3",
#define CHN4 3
        "4",
        NULL
    };
    char    name[_POSIX_PATH_MAX] = { 0 };
    int     protected_content = 0;
    int     enable_protection = 0;
    int     require_protection = 0;
    int     use_writer_thread = 0;
    void   *retval;
    pthread_t writer_thread;
    pthread_t mixer_thread;
    pthread_t keypress_thread;

    memset (&pp, 0, sizeof (pp));
    pp.start_mode = SND_PCM_START_DATA;
    pp.stop_mode = SND_PCM_STOP_STOP;

    while ((c = getopt (argc, argv, "a:b:c:efg:mno:prsw")) != EOF)
    {
        switch (c)
        {
        case 'a':
            if (strchr (optarg, ':'))
            {
                card = atoi (optarg);
                dev = atoi (strchr (optarg, ':') + 1);
            }
            else if (isalpha (optarg[0]) || optarg[0] == '/')
                strcpy (name, optarg);
            else
                dev = atoi (optarg);
            if (name[0] != '\0')
                printf ("Using device /dev/snd/%s\n", name);
            else
                printf ("Using card %d device %d \n", card, dev);
            break;
        case 'b':
            num_buffered_frags = atoi (optarg);
            break;
        case 'c':
            sub_opts = strdup (optarg);
            while (*sub_opts != '\0')
            {
                switch (getsubopt (&sub_opts, dev_opts, &value))
                {
                case CHN1:
                    voice_mask[0] = strtoul (value, NULL, 0);
                    break;
                case CHN2:
                    voice_mask[1] = strtoul (value, NULL, 0);
                    break;
                case CHN3:
                    voice_mask[2] = strtoul (value, NULL, 0);
                    break;
                case CHN4:
                    voice_mask[3] = strtoul (value, NULL, 0);
                    break;
                default:
                    break;
                }
            }
            voice_override = 1;
            break;
        case 'e':
            enable_protection = 1;
            break;
        case 'f':
            fragsize = atoi (optarg);
            break;
        case 'g':
            pp.start_mode = atoi(optarg);
            switch( pp.start_mode ) {
                case SND_PCM_START_FULL:
                    printf("Start mode: full\n");
                    break;
                case SND_PCM_START_DATA:
                    printf("Start mode: data\n");
                    break;
                case SND_PCM_START_GO:
                    printf("Start mode: go\n");
                    break;
            }
            break;
        case 'm':
            use_mmap = 1;
            break;
        case 'n':
            num_frags = atoi (optarg) - 1;
            break;
        case 'o':
            pp.stop_mode = atoi(optarg);
            switch( pp.stop_mode ) {
                case SND_PCM_STOP_STOP:
                    printf("Stop mode STOP_STOP\n");
                    break;
                case SND_PCM_STOP_ROLLOVER:
                    printf("Stop mode STOP_ROLLOVER\n");
                    break;
            }
            break;
        case 'p':
            use_plugins = 0;
            break;
        case 'r':
            require_protection = 1;
            break;
        case 's':
            protected_content = 1;
            break;
        case 'w':
            use_writer_thread = 1;
            break;
        default:
            return 1;
        }
    }

    setvbuf (stdin, NULL, _IONBF, 0);

    if (name[0] != '\0')
    {
        snd_pcm_info_t info;

        if ((rtn = snd_pcm_open_name (&pcm_handle, name, SND_PCM_OPEN_PLAYBACK)) < 0)
        {
            err ("open_name\n");
        }
        rtn = snd_pcm_info (pcm_handle, &info);
        card = info.card;
    }
    else
    {
        if (card == -1)
        {
            if ((rtn =
                    snd_pcm_open_preferred (&pcm_handle, &card, &dev, SND_PCM_OPEN_PLAYBACK)) < 0)
                return err ("device open");
        }
        else
        {
            if ((rtn = snd_pcm_open (&pcm_handle, card, dev, SND_PCM_OPEN_PLAYBACK)) < 0)
                return err ("device open");
        }
    }

    mSampleRate = 11025;
    mSampleChannels = 1;
    mSampleBits = 8;

    printf ("SampleRate = %d, Channels = %d, SampleBits = %d\n", mSampleRate, mSampleChannels,
        mSampleBits);

    if (use_mmap) {
        enable_mask |= PLUGIN_MMAP;
    } else {
        disable_mask |= PLUGIN_MMAP;
    }
    if (use_plugins) {
        enable_mask |= PLUGIN_CONVERSION;
    } else {
        disable_mask |= PLUGIN_CONVERSION;
    }

    if ((rtn = snd_pcm_plugin_set_disable (pcm_handle, disable_mask)) < 0)
    {
        fprintf (stderr, "snd_pcm_plugin_set_disable failed: %s\n", snd_strerror (rtn));
        return -1;
    }

    if ((rtn = snd_pcm_plugin_set_enable (pcm_handle, enable_mask)) < 0)
    {
        fprintf (stderr, "snd_pcm_plugin_set_disable failed: %s\n", snd_strerror (rtn));
        return -1;
    }

    memset (&pi, 0, sizeof (pi));
    pi.channel = SND_PCM_CHANNEL_PLAYBACK;
    if ((rtn = snd_pcm_plugin_info (pcm_handle, &pi)) < 0)
    {
        fprintf (stderr, "snd_pcm_plugin_info failed: %s\n", snd_strerror (rtn));
        return -1;
    }

    pp.mode = SND_PCM_MODE_BLOCK
            | (protected_content ? SND_PCM_MODE_FLAG_PROTECTED_CONTENT : 0)
            | (enable_protection ? SND_PCM_MODE_FLAG_ENABLE_PROTECTION : 0)
            | (require_protection ? SND_PCM_MODE_FLAG_REQUIRE_PROTECTION : 0);

    pp.channel = SND_PCM_CHANNEL_PLAYBACK;

    pp.buf.block.frag_size = pi.max_fragment_size;
    if (fragsize != -1)
    {
        pp.buf.block.frag_size = fragsize;
    }
    pp.buf.block.frags_max = num_frags;
    pp.buf.block.frags_buffered_max = num_buffered_frags;
    pp.buf.block.frags_min = 1;

    pp.format.interleave = 1;
    pp.format.rate = mSampleRate;
    pp.format.voices = mSampleChannels;

    if (mSampleBits == 8)
        pp.format.format = SND_PCM_SFMT_U8;
    else if (mSampleBits == 24)
        pp.format.format = SND_PCM_SFMT_S24;
    else if (mSampleBits == 32)
        pp.format.format = SND_PCM_SFMT_S32_LE;
    else
        pp.format.format = SND_PCM_SFMT_S16_LE;

    strcpy (pp.sw_mixer_subchn_name, "Asound tester");
    if ((rtn = snd_pcm_plugin_params (pcm_handle, &pp)) < 0)
    {
        fprintf (stderr, "snd_pcm_plugin_params failed: %s %d\n", snd_strerror (rtn), pp.why_failed);
        return -1;
    }

    if ((rtn = snd_pcm_plugin_prepare (pcm_handle, SND_PCM_CHANNEL_PLAYBACK)) < 0)
        fprintf (stderr, "snd_pcm_plugin_prepare failed: %s\n", snd_strerror (rtn));

    if (voice_override)
    {
        snd_pcm_plugin_get_voice_conversion (pcm_handle, SND_PCM_CHANNEL_PLAYBACK,
            &voice_conversion);
        voice_conversion.matrix[0] = voice_mask[0];
        voice_conversion.matrix[1] = voice_mask[1];
        voice_conversion.matrix[2] = voice_mask[2];
        voice_conversion.matrix[3] = voice_mask[3];
        snd_pcm_plugin_set_voice_conversion (pcm_handle, SND_PCM_CHANNEL_PLAYBACK,
            &voice_conversion);
    }

    memset (&setup, 0, sizeof (setup));
    memset (&group, 0, sizeof (group));
    setup.channel = SND_PCM_CHANNEL_PLAYBACK;
    setup.mixer_gid = &group.gid;
    if ((rtn = snd_pcm_plugin_setup (pcm_handle, &setup)) < 0)
    {
        fprintf (stderr, "snd_pcm_plugin_setup failed: %s\n", snd_strerror (rtn));
        return -1;
    }
    printf ("Format %s \n", snd_pcm_get_format_name (setup.format.format));
    printf ("Frag Size %d \n", setup.buf.block.frag_size);
    printf ("Total Frags %d \n", setup.buf.block.frags);
    printf ("Rate %d \n", setup.format.rate);
    printf ("Voices %d \n", setup.format.voices);

    if (group.gid.name[0] == 0)
    {
        printf ("Mixer Pcm Group [%s] Not Set \n", group.gid.name);
        exit (-1);
    }
    printf ("Mixer Pcm Group [%s]\n", group.gid.name);
    if ((rtn = snd_mixer_open (&mixer_handle, card, setup.mixer_device)) < 0)
    {
        fprintf (stderr, "snd_mixer_open failed: %s\n", snd_strerror (rtn));
        return -1;
    }

    if (tcgetpgrp (0) == getpid ())
        dev_raw (fileno (stdin));

    mSampleBfr1 = malloc (setup.buf.block.frag_size);
    n = setup.buf.block.frag_size;
    FD_ZERO (&rfds);
    FD_ZERO (&wfds);

    if( use_writer_thread ) {
        pthread_create( &writer_thread, NULL, writer_thread_handler, NULL );
        pthread_create( &mixer_thread, NULL, generic_thread_handler, handle_mixer );
        pthread_create( &keypress_thread, NULL, generic_thread_handler, handle_keypress );

        // First wait for feeder to complete. Any other thread will cause it to stop.
        // Then just kill the other threads
        pthread_join(writer_thread, &retval);
        pthread_cancel(keypress_thread);
        pthread_cancel(mixer_thread);
    } else {
        while (running)
        {
            FD_ZERO(&rfds);
            FD_ZERO(&wfds);
            if (tcgetpgrp (0) == getpid ())
                FD_SET (STDIN_FILENO, &rfds);
            FD_SET (snd_mixer_file_descriptor (mixer_handle), &rfds);
            if( !error_state ) {
                FD_SET (snd_pcm_file_descriptor (pcm_handle, SND_PCM_CHANNEL_PLAYBACK), &wfds);
            }

            rtn = max (snd_mixer_file_descriptor (mixer_handle),
                snd_pcm_file_descriptor (pcm_handle, SND_PCM_CHANNEL_PLAYBACK));

            if (select (rtn + 1, &rfds, &wfds, NULL, NULL) == -1)
                return err ("select");

            if (FD_ISSET (STDIN_FILENO, &rfds))
            {
                handle_keypress();
            }

            if (FD_ISSET (snd_mixer_file_descriptor (mixer_handle), &rfds))
            {
                handle_mixer();
            }

            if (FD_ISSET (snd_pcm_file_descriptor (pcm_handle, SND_PCM_CHANNEL_PLAYBACK), &wfds))
            {
                write_audio_data();
            }
        }
    }

    if (tcgetpgrp (0) == getpid ())
        dev_unraw (fileno (stdin));
    if (running) {
        snd_pcm_plugin_flush (pcm_handle, SND_PCM_CHANNEL_PLAYBACK);
    }
    snd_mixer_close (mixer_handle);
    snd_pcm_close (pcm_handle);

    return (0);
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/tests/interactive/tester.c $ $Rev: 812869 $")
#endif
