/*
 * $QNXLicenseC:
 * Copyright 2007, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable license fees to QNX
 * Software Systems before you may reproduce, modify or distribute this software,
 * or any work that includes all or part of this software.   Free development
 * licenses are available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review this entire
 * file for other proprietary rights or license notices, as well as the QNX
 * Development Suite License Guide at http://licensing.qnx.com/license-guide/
 * for other information.
 * $
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <gulliver.h>
#include <stdbool.h>
#include <string.h>

#include <sys/asoundlib.h>
#include <rate_poly.h>

typedef struct
{
    char    tag_id[4];
    int32_t tag_len;
} riff_tag_t;

typedef struct
{
    char     riff_id[4];
    uint32_t  wave_len;
    char     wave_id[4];
} riff_hdr_fixed_t;

typedef struct
{
    uint16_t   format_tag;
    uint16_t   voices;
    uint32_t   rate;
    uint32_t   char_per_sec;
    uint16_t   block_align;
    uint16_t   bits_per_sample;
} wave_format_t;

typedef struct
{
    char        data_id[4];
    uint32_t    data_len;
} wave_data_t;


/* *INDENT-OFF* */
typedef struct
{
    riff_hdr_fixed_t fixed_riff_hdr;
    struct
    {
        riff_tag_t fmt_riff_tag;
        wave_format_t fmt;
        wave_data_t data;
    }
    wave;
} riff_hdr_t;

riff_hdr_fixed_t src_riff_hdr;

riff_hdr_t dest_riff_hdr =
{
    {
         {'R', 'I', 'F', 'F' },
         sizeof (dest_riff_hdr.fixed_riff_hdr.wave_id) + sizeof (dest_riff_hdr.wave),
         { 'W', 'A', 'V', 'E'}
    },
    {
        {
            {'f', 'm', 't', ' '},
            sizeof (dest_riff_hdr.wave.fmt)
        },
        {
            1, 0, 0, 0, 0, 0
        },
        {
            {'d', 'a', 't', 'a' },
            0,
        }
    }
};
/* *INDENT-ON* */

int
err (char *msg)
{
    perror (msg);
    return -1;
}

static int bigEndian = 0;
static int running = 0;

/* return length associated with the tag on success, -1 otherwise */
int
find_riff_tag (FILE * fp, const char *tag)
{
    riff_tag_t riff_tag = { "", 0 };

    // Keep reading until we find the tag or hit the EOF.
    while (fread ((unsigned char *) &riff_tag, sizeof (riff_tag), 1, fp) > 0)
    {

        if( bigEndian ) {
            riff_tag.tag_len = ENDIAN_BE32 (riff_tag.tag_len);
        } else {
            riff_tag.tag_len = ENDIAN_LE32 (riff_tag.tag_len);
        }
        // If this is our tag, set the length and break.
        if (strncmp (tag, riff_tag.tag_id, sizeof(riff_tag.tag_id)) == 0)
        {
            return riff_tag.tag_len;
        }

        // Skip ahead the specified number of bytes in the stream
        fseek (fp, riff_tag.tag_len, SEEK_CUR);
    }

    return -1;
}

/* return endianness on success, -1 otherwise */
int
check_fixed_riff_hdr (FILE * fp)
{
    const char* RIFF_ID = "RIFF";
    const char* RIFX_ID = "RIFX";
    const char* WAVE_ID = "WAVE";

    int big_endian;

    riff_hdr_fixed_t fixed_riff_hdr = { "", 0, "" };

    if (fread ((unsigned char *) &fixed_riff_hdr, sizeof (fixed_riff_hdr), 1, fp) <= 0 ) {
        printf("Failed reading fixed_riff_hdr\n");
        return -1;
    }
    if (!strncmp (fixed_riff_hdr.riff_id, RIFF_ID, strlen (RIFF_ID))) {
        big_endian = 0;
    } else if (!strncmp (fixed_riff_hdr.riff_id, RIFX_ID, strlen (RIFX_ID))) {
        big_endian = 1;
    } else {
        printf("Unexpected RIFF id\n");
        return -1;
    }
    if (strncmp (fixed_riff_hdr.wave_id, WAVE_ID, strlen (WAVE_ID))) {
        printf("Unexpected WAVE id\n");
        return -1;
    }
    return big_endian;
}

//*****************************************************************************
/* *INDENT-OFF* */
#ifdef __USAGE
%C[Options] *

Options:
    -i<file>           input .wav file
    -o<file>           output .wav file
    -s<n>              output sample rate in Hz
    -p                 playback specific conversion
    -c                 capture specific conversion
#endif
/* *INDENT-ON* */
//*****************************************************************************

void sig_handler( int sig_no )
{
	running = false;
	return;
}

int
main (int argc, char **argv)
{
    char c;
    wave_format_t srcWaveFormat;
    FILE*   srcFile;
    FILE*   destFile;
    int     bigEndian;
    int     playback = 1;
    int     len;
    int     srcSamples, destSamples;
    int     srcSampleRate, destSampleRate = 0;
    int     srcSampleBufferSize, destSampleBufferSize;
    char    *srcSampleBuffer, *destSampleBuffer;
    int     channels;
    int     bitsPerSample;
    int     fragSize;
    int     nread, nreadTotal;
    int     nwritten, nwrittenTotal;
    void    *mem;
    int     memSize;
    static char    srcFileName[_POSIX_PATH_MAX] = { 0 };
    static char    destFileName[_POSIX_PATH_MAX] = { 0 };

    while ((c = getopt (argc, argv, "i:o:s:pc")) != EOF)
    {
        switch (c)
        {
        case 'i':
            strncpy (srcFileName, optarg, sizeof(srcFileName) - 1);
            break;
        case 'o':
            strncpy (destFileName, optarg, sizeof(destFileName) - 1);
            break;
        case 's':
            destSampleRate = atoi (optarg);
            break;
        case 'p':
            playback = 1;
            break;
        case 'c':
            playback = 0;
            break;
        default:
            return 1;
        }
    }
    if (destSampleRate == 0) {
        return err ("destSampleRate param missing");
    }
    if (srcFileName[0] != '\0')
    {
        if ((srcFile = fopen (srcFileName, "r")) == 0) {
            return err ("input file open");
        }
    }
    else
    {
        return err ("no input file specified");
    }
    if (destFileName[0] != '\0')
    {
        if ((destFile = fopen (destFileName, "w")) == 0) {
            fclose(srcFile);
            return err ("output file open");
        }
    }
    else
    {
        fclose(srcFile);
        return err ("no output file specified");
    }

    if ((bigEndian = check_fixed_riff_hdr (srcFile)) == -1) {
        fclose(srcFile);
        fclose(destFile);
        return err ("check_fixed_riff_hdr");
    }

    len = find_riff_tag (srcFile, "fmt ");
    if( len < sizeof (srcWaveFormat) )
    {
        fclose(srcFile);
        fclose(destFile);
        return err ("find_riff_tag");
    }
    fread (&srcWaveFormat, sizeof (srcWaveFormat), 1, srcFile);
    fseek (srcFile, (len - sizeof (srcWaveFormat)), SEEK_CUR);

    if( bigEndian ) {
        srcSampleRate = ENDIAN_BE32 (srcWaveFormat.rate);
        channels = ENDIAN_BE16 (srcWaveFormat.voices);
        bitsPerSample = ENDIAN_BE16 (srcWaveFormat.bits_per_sample);
        srcWaveFormat.format_tag = ENDIAN_BE16 (srcWaveFormat.format_tag);
    } else {
        srcSampleRate = ENDIAN_LE32 (srcWaveFormat.rate);
        channels = ENDIAN_LE16 (srcWaveFormat.voices);
        bitsPerSample = ENDIAN_LE16 (srcWaveFormat.bits_per_sample);
        srcWaveFormat.format_tag = ENDIAN_LE16 (srcWaveFormat.format_tag);
    }

    printf ("Source sampleRate = %d, dest samplerate = %d, channels = %d, bitsPerSample = %d\n",
            srcSampleRate, destSampleRate, channels, bitsPerSample);

    if( srcWaveFormat.format_tag == 6 || srcWaveFormat.format_tag == 7 ) {
        fclose(srcFile);
        fclose(destFile);
        return err ("incorrect format");
    }

    if( bitsPerSample != 16 ) {
        fclose(srcFile);
        fclose(destFile);
        return err ("unsupported bit per sample");
    }

    if( bigEndian ) {
        fclose(srcFile);
        fclose(destFile);
        return err ("unsupported endianness");
    }

    len = find_riff_tag (srcFile, "data");
    if( bigEndian ) {
        len = ENDIAN_BE32( len );
    } else {
        len = ENDIAN_LE32( len );
    }
    printf("src file data len = %d\n", len);

    if( (srcSampleRate % 4000) == 0 && (destSampleRate % 4000) == 0 ) {
        if( srcSampleRate > destSampleRate ) {
            srcSamples = 300;
            destSamples = srcSamples * destSampleRate / srcSampleRate;
        } else {
            destSamples = 300;
            srcSamples = destSamples * srcSampleRate / destSampleRate;
        }
    } else if( (srcSampleRate % 4000) != 0 && (destSampleRate % 4000) != 0 ) {
        if( srcSampleRate > destSampleRate ) {
            srcSamples = 320;
            destSamples = srcSamples * destSampleRate / srcSampleRate;
        } else {
            destSamples = 320;
            srcSamples = destSamples * srcSampleRate / destSampleRate;
        }
    } else if( (srcSampleRate % 4000) == 0 ) {
        srcSamples = 320;
        destSamples = srcSamples * destSampleRate / srcSampleRate;
    } else {
        destSamples = 320;
        srcSamples = destSamples * srcSampleRate / destSampleRate;
    }

    // exception for 11025 to 48000 - rate_poly_core table uses 11 to 48 ratio in this case
    if( srcSampleRate == 11025 && destSampleRate == 48000 ) {
        destSamples = 480;
        srcSamples = 110;
        //destSamples = 640;
        //srcSamples = 147;
    }

    srcSampleBufferSize = srcSamples * channels * bitsPerSample / 8;
    destSampleBufferSize = destSamples * channels * bitsPerSample / 8;

    printf("srcSampleBufferSize = %d, destSampleBufferSize = %d\n",
           srcSampleBufferSize, destSampleBufferSize);
    srcSampleBuffer = malloc (srcSampleBufferSize);
    if( srcSampleBuffer == NULL ) {
        fclose(srcFile);
        fclose(destFile);
        return err ("malloc srcSampleBuffer");
    }
    destSampleBuffer = malloc (destSampleBufferSize);
    if( destSampleBuffer == NULL ) {
        fclose(srcFile);
        fclose(destFile);
        free(srcSampleBuffer);
        return err ("malloc destSampleBuffer");
    }
    dest_riff_hdr.wave.fmt.voices = ENDIAN_LE16 (channels);
    dest_riff_hdr.wave.fmt.rate = ENDIAN_LE32 (destSampleRate);
    dest_riff_hdr.wave.fmt.char_per_sec =
        ENDIAN_LE32 (destSampleRate * channels * bitsPerSample / 8);
    dest_riff_hdr.wave.fmt.block_align = ENDIAN_LE16 (channels * bitsPerSample / 8);
    dest_riff_hdr.wave.fmt.bits_per_sample = ENDIAN_LE16 (bitsPerSample);
    dest_riff_hdr.wave.data.data_len = ENDIAN_LE32 (len * destSamples / srcSamples);
    dest_riff_hdr.fixed_riff_hdr.wave_len = ENDIAN_LE32 (len * destSamples / srcSamples + sizeof (dest_riff_hdr) -
                                                         sizeof(dest_riff_hdr.fixed_riff_hdr.wave_len) -
                                                         sizeof(dest_riff_hdr.fixed_riff_hdr.riff_id));
    fwrite (&dest_riff_hdr, 1, sizeof (dest_riff_hdr), destFile);

    memSize = rate_poly_mem_required();
    mem = malloc(memSize);

    fragSize = max( srcSampleBufferSize, destSampleBufferSize );

    if( rate_poly_create(mem, srcSampleRate, channels, destSampleRate, fragSize, 0, playback) != 0 ) {
        fclose(srcFile);
        fclose(destFile);
        free(mem);
        free(srcSampleBuffer);
        free(destSampleBuffer);
        return err( "rate_poly_create" );
    }

    ResampleState_t *resample_state = (ResampleState_t*) mem;

    printf("rate_poly_create success: chanOptimization = %s, convFuncVersion = %s\n",
           resample_state->channelOptimization == RATE_POLY_MONO ? "mono" : "stereo",
           resample_state->conversionFunctionVersion == RATE_POLY_NEON_I_CONV_FUNC ? "neon_i" :
               (resample_state->conversionFunctionVersion == RATE_POLY_NEON_I_24T_CONV_FUNC ? "neon_i_24t" : "generic"));

    if( playback ) {
        if( rate_poly_reset(mem, destSampleBufferSize) < 0 ) {
            fclose(srcFile);
            fclose(destFile);
            free(mem);
            free(srcSampleBuffer);
            free(destSampleBuffer);
            return err( "rate_poly_reset" );
        }
    } else {
        if( rate_poly_reset(mem, srcSampleBufferSize) < 0 ) {
            fclose(srcFile);
            fclose(destFile);
            free(mem);
            free(srcSampleBuffer);
            free(destSampleBuffer);
            return err( "rate_poly_reset" );
        }
    }

    printf("rate_poly_reset success: taps=%d, dlySize=%d, dlyIdx=%d, firPhase=%d, dlyUpdate=%d\n",
            resample_state->taps, resample_state->dlySize, resample_state->dlyIdx, resample_state->firPhase, resample_state->dlyUpdate);

    signal(SIGINT, sig_handler);
    signal(SIGTERM, sig_handler);

    running = 1;
    nreadTotal = 0;
    nwrittenTotal = 0;

    while (running && nreadTotal < len)
    {
        if( (nread = fread (srcSampleBuffer, 1, min (len - nreadTotal, srcSampleBufferSize), srcFile)) <= 0 ) {
            fclose(srcFile);
            fclose(destFile);
            free(mem);
            free(srcSampleBuffer);
            free(destSampleBuffer);
            return err( "fread" );
        }

        if( rate_poly_process(mem, (int16_t*)srcSampleBuffer, nread,
                                   (int16_t*)destSampleBuffer, nread * destSamples / srcSamples) != 0) {
            fclose(srcFile);
            fclose(destFile);
            free(mem);
            free(srcSampleBuffer);
            free(destSampleBuffer);
            return err( "rate_poly_process" );
        }

        if( (nwritten = fwrite( destSampleBuffer, 1, nread * destSamples / srcSamples, destFile )) <= 0 ) {
            fclose(srcFile);
            fclose(destFile);
            free(mem);
            free(srcSampleBuffer);
            free(destSampleBuffer);
            return err( "fwrite" );
        }
        nreadTotal += nread;
        nwrittenTotal += nwritten;
    }

    rate_poly_free(mem);

    if( len * destSamples / srcSamples != nwrittenTotal ) {
        printf( "%d bytes written to destination file, %d calculated\n", nwrittenTotal, len * destSamples / srcSamples );
    }
    printf("Exiting...\n");

    fclose(srcFile);
    fclose(destFile);
    free(mem);
    free(srcSampleBuffer);
    free(destSampleBuffer);
    return (0);
}


#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn/product/branches/7.0.0/trunk/lib/asound/tests/rateconvert/rateconvert.c $ $Rev: 798872 $")
#endif
