/*
 *    Linux DVB USB Interface Driver for TS2USB Bridge device
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* #define SHOW_ERR_INFO_RAW */
#define FIRST_512_ERR_INFO 512

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/device.h>		
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/dvb/dmx.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include "usb2ts_usb.h"

#define TS2USB_FPGA_HAS_FW_INFO_RETRIEVAL_SUPPORT
/* #undef TS2USB_FPGA_HAS_FW_INFO_RETRIEVAL_SUPPORT */

/* #define EN_USB2TS_DBG_LOGS */
 
DVB_DEFINE_MOD_OPT_ADAPTER_NR(adapter_nr);

#define TS2USB_DEVNAME "ts2usb" /* device name that appear under /dev */
#define USB2TS_PROCFS_NAME "err-statistics-ts2usb"
#define NUM_TS_STREAMS 2
#define MAX_SYNC_CNTR 65535
//#define CHECK_PARITY_ERR

enum TS_CHANNEL {TS0, TS1};

/* let it be turn ON by default (we have test pgm on user-space but that will only work for payload-only-seq-counter input TS files) */
static int usb2ts_usb_iso_debug = 1;
module_param(usb2ts_usb_iso_debug, int, 0644);
MODULE_PARM_DESC(usb2ts_usb_iso_debug, "Enable or disable USB2TS USB debug mode, (default is 1)" );

static char usb2ts_fpga_rtl_fw_ver[80];

/* for the char device node stuff */
static int usb2ts_char_dev_major_num;
static struct device *usb2ts_char_dev_device;
static struct class *usb2ts_char_dev_cl = NULL;
static struct dvb_usb_device *copy_d = NULL; /* to clear out priv cb ptr (i.e. fn usb2ts_process_isopkt_err_info) on module exit */		

static struct proc_dir_entry *usb2ts_proc_file = NULL;

/* global object to store the TS2USB bridge device's error statistics */
static struct usb2ts_err_statistics usb2ts_err_statistics[NUM_TS_STREAMS]; /* two instances required as we have two TS streams */

static void clear_err_statistics(struct usb2ts_err_statistics *ptr_usb2ts_err_statistics) 
{
#ifdef SHOW_ERR_INFO_RAW
	int j;
#endif

	atomic_long_set(&ptr_usb2ts_err_statistics->num_sync_counter_err, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->num_fifo_overflow, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->num_parity_err, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->val_counter_in_isopkt_last_err, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->val_counter_in_driver_last_err, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->num_total_ts_pkt_lost, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->num_total_ts_pkts_received, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->num_fragmented_ts_pkt, 0);
	atomic_long_set(&ptr_usb2ts_err_statistics->num_received_iso_pkt, 0);
	
#ifdef SHOW_ERR_INFO_RAW
	for (j = 0; j < FIRST_512_ERR_INFO; j++)
		ptr_usb2ts_err_statistics->err_info_raw[j] = 0x0;
#endif
}

static int usb2ts_proc_show(struct seq_file *m, void *v)
{
	int i;
#ifdef SHOW_ERR_INFO_RAW
	int j;
#endif
	if (usb2ts_usb_iso_debug) {
		for (i = 0; i < NUM_TS_STREAMS; i++) {
			seq_printf(m, "\nStatistics for Transport Stream %d (TS%d):\n", i, i);
			
			seq_printf(m, "Number of SYNC count miss occurrence of ISO packets = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_sync_counter_err));
			seq_printf(m, "Number of occurrence of FIFO overflow = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_fifo_overflow));
			seq_printf(m, "Number of occurrence of parity error = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_parity_err));
			
			seq_printf(m, "Total number of TS packets lost = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_total_ts_pkt_lost));
			seq_printf(m, "Total number of TS packets received = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_total_ts_pkts_received));
			seq_printf(m, "The received very first TS packet number = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_very_first_ts_pkt));
			seq_printf(m, "Total number of ISO packets received = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_received_iso_pkt));
			seq_printf(m, "The number of received TS packets that are fragmented = %ld\n",
				atomic_long_read(&usb2ts_err_statistics[i].num_fragmented_ts_pkt));
	
			if (atomic_long_read(&usb2ts_err_statistics[i].num_sync_counter_err) != 0) {
				seq_printf(m, "Very recent detailed log:\n");
				seq_printf(m, "Number of SYNC count miss occurrence = %ld => val_counter_in_isopkt at the time = %ld, val_counter_in_driver at the time = %ld\n",
					atomic_long_read(&usb2ts_err_statistics[i].num_sync_counter_err),
					atomic_long_read(&usb2ts_err_statistics[i].val_counter_in_isopkt_last_err),
					atomic_long_read(&usb2ts_err_statistics[i].val_counter_in_driver_last_err));
			}
#ifdef SHOW_ERR_INFO_RAW			
			seq_printf(m, "Error info raw bytes (first 512 numbers):\n");
			for (j = 0; j < FIRST_512_ERR_INFO; j++)
				seq_printf(m, "0x%08x\n", usb2ts_err_statistics[i].err_info_raw[j]);
#endif
		}
	} else {
		seq_printf(m, "The USB2TS DVB USB module parameter: 'usb2ts_usb_iso_debug' has not been set\n");
	}
	return 0;
}

ssize_t usb2ts_proc_write(struct file *filp, const char __user *buf, size_t count, loff_t *data)
{
	char b[80];
	unsigned long command;
	int i;
	int retval;
	
	if (copy_from_user(b, (void __user*)buf, sizeof (b)))
		return -EFAULT;

	retval = kstrtoul(b, 0, &command);
	if (retval < 0) {
		printk(KERN_DEBUG "Could not parse RESET command for %s\n", TS2USB_DEVNAME);
		return retval;
	}
	
	if (command == 0) { /* clear the statistics; never clear the 'val_counter_in_driver' with this operation */
		for (i = 0; i < NUM_TS_STREAMS; i++) {
			clear_err_statistics(&usb2ts_err_statistics[i]);
		}
	}
	return count;
}

static int usb2ts_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, usb2ts_proc_show, NULL);
}

static const struct file_operations usb2ts_proc_fops = {
	.owner		= THIS_MODULE,
    .open		= usb2ts_proc_open,
    .read		= seq_read,
    .write		= usb2ts_proc_write,
    .llseek		= seq_lseek,
    .release	= single_release,
};

#ifdef CHECK_PARITY_ERR
static bool usb2ts_compute_parity(u8 *data_p, size_t length)
{
	bool parity = 0;
	unsigned long count = 0, i, j, b = 1;

	for(j = 0; j < length; j++)	{
		for(i = 0; i < 8; i++) {
			if( (*(data_p + j)) & (b << i) ) {
				count++;
			}
		}
	}
	parity = (count % 2) ? 1 : 0;
	return parity;
}
#endif

static void usb2ts_process_isopkt_err_info(u8 *buf, size_t len, struct dvb_usb_adapter *adap, bool *err_det)
{
	u32 num_ts_pkts_in_curr_iso;
	u32 err_info_field;
#ifdef CHECK_PARITY_ERR
	bool sw_parity;
	bool hw_parity;
	bool parity_result;
#endif
	bool fifo_overflow_status;
 
	struct usb2ts_err_statistics *ptr_usb2ts_err_statistics = &usb2ts_err_statistics[adap->id];
	if(err_det)     *err_det = false;

	if (usb2ts_usb_iso_debug) {
		if (len > 12) {
			err_info_field = *((u32*)(buf + (len - 4)));
			/*FPGA sets (12Bit)0xAB,0xC0 at the start of debug info */
			if(((err_info_field >> 16) & 0xFFF0) != 0xABC0){
				if(err_det)	*err_det = true;
				return;
			}
			if ((len - 4) % 188) {
				atomic_long_inc(&ptr_usb2ts_err_statistics->num_fragmented_ts_pkt);
			}
			num_ts_pkts_in_curr_iso = (len - 4) / 188;
			atomic_long_add(num_ts_pkts_in_curr_iso, &ptr_usb2ts_err_statistics->num_total_ts_pkts_received);
		#ifdef CHECK_PARITY_ERR
			sw_parity = usb2ts_compute_parity(buf, len - 4);  
		      	hw_parity = err_info_field >> 17 & 0x01;
			parity_result = (hw_parity == sw_parity) ? 1 : 0;
		#endif
			*((u32*)(buf + (len - 4))) = 0xffffffff;
			
			fifo_overflow_status = err_info_field >> 16 & 0x01;
			ptr_usb2ts_err_statistics->val_counter_in_isopkt = err_info_field & 0xFFFF;
			
			if (ptr_usb2ts_err_statistics->init) {
				atomic_long_set(&ptr_usb2ts_err_statistics->num_very_first_ts_pkt, 
					(ptr_usb2ts_err_statistics->val_counter_in_isopkt - num_ts_pkts_in_curr_iso) + 1);
					
				/* the driver's initial expectancy in sequence counter must be the number of TS packets in the first ISO */
				ptr_usb2ts_err_statistics->val_counter_in_driver = num_ts_pkts_in_curr_iso;
				ptr_usb2ts_err_statistics->init = 0;
#ifdef SHOW_ERR_INFO_RAW				
				ptr_usb2ts_err_statistics->index_err_info_raw = 0;
#endif
			} else {
				/* except for the first invocation, the expectancy in sequence counter must be a value which is the current value plus the number of TS available in the current ISO */
				ptr_usb2ts_err_statistics->val_counter_in_driver += num_ts_pkts_in_curr_iso;
			}
			atomic_long_inc(&ptr_usb2ts_err_statistics->num_received_iso_pkt);
#ifdef SHOW_ERR_INFO_RAW
			if (ptr_usb2ts_err_statistics->index_err_info_raw < 512)
				ptr_usb2ts_err_statistics->err_info_raw[ptr_usb2ts_err_statistics->index_err_info_raw++] = err_info_field;
#endif
			
#ifdef EN_USB2TS_DBG_LOGS		
			printk(KERN_DEBUG "val_counter_in_isopkt = %d, adap ID = %d, val_counter_in_driver = %d\n",  
				ptr_usb2ts_err_statistics->val_counter_in_isopkt, 
				adap->id, 
				ptr_usb2ts_err_statistics->val_counter_in_driver);
#endif   
		
			if (fifo_overflow_status == 0x1) {
				atomic_long_inc(&ptr_usb2ts_err_statistics->num_fifo_overflow);
				if(err_det)	*err_det = true;
#if 1	//def EN_USB2TS_DBG_LOGS			
				printk(KERN_DEBUG "[%d]Number of FIFO overflow occurred = %ld\n", 
					adap->id, atomic_long_read(&ptr_usb2ts_err_statistics->num_fifo_overflow));
#endif
			}
#ifdef CHECK_PARITY_ERR
			if (parity_result == 0) { /* we have a non-match; we use odd-parity */
				atomic_long_inc(&ptr_usb2ts_err_statistics->num_parity_err);
				if(err_det)     *err_det = true;
#ifdef EN_USB2TS_DBG_LOGS			
				printk(KERN_DEBUG "[%d]Number of parity error occurred = %ld\n", 
					adap->id, atomic_long_read(&ptr_usb2ts_err_statistics->num_parity_err));
#endif
			}
#endif

			if (ptr_usb2ts_err_statistics->val_counter_in_isopkt != ptr_usb2ts_err_statistics->val_counter_in_driver) {
				atomic_long_inc(&ptr_usb2ts_err_statistics->num_sync_counter_err);
#ifdef EN_USB2TS_DBG_LOGS	
				printk(KERN_DEBUG "[%d]Number of SYNC count misses = %ld, val_counter_in_isopkt = %d, val_counter_in_driver = %d\n", 
					adap->id, atomic_long_read(&ptr_usb2ts_err_statistics->num_sync_counter_err), 
					ptr_usb2ts_err_statistics->val_counter_in_isopkt,
					ptr_usb2ts_err_statistics->val_counter_in_driver);
#endif
				atomic_long_set(&ptr_usb2ts_err_statistics->val_counter_in_isopkt_last_err, ptr_usb2ts_err_statistics->val_counter_in_isopkt);
				atomic_long_set(&ptr_usb2ts_err_statistics->val_counter_in_driver_last_err, ptr_usb2ts_err_statistics->val_counter_in_driver);
			
				if (ptr_usb2ts_err_statistics->val_counter_in_isopkt > ptr_usb2ts_err_statistics->val_counter_in_driver)
					atomic_long_add(ptr_usb2ts_err_statistics->val_counter_in_isopkt -  
						ptr_usb2ts_err_statistics->val_counter_in_driver, 
						&ptr_usb2ts_err_statistics->num_total_ts_pkt_lost);
				else
					atomic_long_add(MAX_SYNC_CNTR + 1 +
						ptr_usb2ts_err_statistics->val_counter_in_isopkt - 
						ptr_usb2ts_err_statistics->val_counter_in_driver,
						&ptr_usb2ts_err_statistics->num_total_ts_pkt_lost);
			
				/* assign the val_counter_in_driver to the currently obtained count value or else we keep hitting here */
				ptr_usb2ts_err_statistics->val_counter_in_driver = ptr_usb2ts_err_statistics->val_counter_in_isopkt;
			}
		}
	}
}

static void fill_with_err_statistics_data(struct usb2ts_error_statistics *errst, enum TS_CHANNEL ts_channel)
{
	errst->num_total_ts_pkts_received		= atomic_long_read(&usb2ts_err_statistics[ts_channel].num_total_ts_pkts_received);
	errst->num_total_ts_pkt_lost			= atomic_long_read(&usb2ts_err_statistics[ts_channel].num_total_ts_pkt_lost);
	errst->num_sync_counter_err				= atomic_long_read(&usb2ts_err_statistics[ts_channel].num_sync_counter_err);
	errst->num_fifo_overflow				= atomic_long_read(&usb2ts_err_statistics[ts_channel].num_fifo_overflow);
	errst->num_parity_err					= atomic_long_read(&usb2ts_err_statistics[ts_channel].num_parity_err);
	errst->num_very_first_ts_pkt			= atomic_long_read(&usb2ts_err_statistics[ts_channel].num_very_first_ts_pkt);
	errst->val_counter_in_isopkt_last_err	= atomic_long_read(&usb2ts_err_statistics[ts_channel].val_counter_in_isopkt_last_err);
	errst->val_counter_in_driver_last_err	= atomic_long_read(&usb2ts_err_statistics[ts_channel].val_counter_in_driver_last_err);
	errst->num_fragmented_ts_pkt			= atomic_long_read(&usb2ts_err_statistics[ts_channel].num_fragmented_ts_pkt);
	errst->val_counter_in_driver 			= usb2ts_err_statistics[ts_channel].val_counter_in_driver;
}

long usb2ts_ioctl(struct file *f, unsigned int cmd, unsigned long buf)
{
	struct usb2ts_error_statistics errst;
	long ret;
	
	switch (cmd) {
		case USB2TS_GET_ERR_STATISTICS_TS0:
			fill_with_err_statistics_data(&errst, TS0);
			ret = copy_to_user((void __user*) buf, &errst, sizeof(errst));
			if (ret != 0) {
			//	printk(KERN_DEBUG, "Error in copying error statistics to application\n");
				return -EFAULT;
			}
			break;
		case USB2TS_GET_ERR_STATISTICS_TS1:
			fill_with_err_statistics_data(&errst, TS1);
			ret = copy_to_user((void __user*) buf, &errst, sizeof(errst));
			if (ret != 0) {
			//	printk(KERN_DEBUG, "Error in copying error statistics to application\n");
				return -EFAULT;
			}
			break;
		case USB2TS_RESET_ERR_STATISTICS_TS0:
			clear_err_statistics(&usb2ts_err_statistics[0]);
			break;
		case USB2TS_RESET_ERR_STATISTICS_TS1:
			clear_err_statistics(&usb2ts_err_statistics[1]);
			break;
#ifdef TS2USB_FPGA_HAS_FW_INFO_RETRIEVAL_SUPPORT
		case USB2TS_GET_FPGA_RTL_FW_VER:
			ret = copy_to_user((void __user*) buf, usb2ts_fpga_rtl_fw_ver, 45);
			if (ret != 0) {
				printk(KERN_DEBUG "Error in copying FPGA RTL FW version information to application\n");
				return -EFAULT;
			}
			break;
#endif
		default:
		//	printk(KERN_DEBUG, "Invalid IOCTL request\n");
			return -EFAULT;
		}
	return 0;
}

static int usb2ts_char_dev_open(struct inode *i, struct file *f)
{
	return 0;
}

static int usb2ts_char_dev_close(struct inode *i, struct file *f)
{
	return 0;
}

static struct file_operations usb2ts_char_dev_fops = {
    .owner			= THIS_MODULE,
    .open			= usb2ts_char_dev_open,
    .release		= usb2ts_char_dev_close,
    .unlocked_ioctl	= usb2ts_ioctl
};

/* set device permissions to something useful */
static int uevent(struct device *dev, struct kobj_uevent_env *env)
{
    add_uevent_var(env, "DEVMODE=%#o", 0766);
    return 0;
}

static int init_usb2ts_char_dev(void)
{
	int ret;

	if ((usb2ts_char_dev_major_num = register_chrdev(0, TS2USB_DEVNAME, &usb2ts_char_dev_fops)) < 0) {
	//	printk(KERN_INFO, "Failed to allocate major number for the device: /dev/%s\n", TS2USB_DEVNAME);
		goto err_one;
	}

    if (IS_ERR(usb2ts_char_dev_cl = class_create(THIS_MODULE, "char"))) {
		printk(KERN_INFO "Failure in creating char class\n");
		ret = PTR_ERR(usb2ts_char_dev_cl);
		goto err_two;
    }
	
	usb2ts_char_dev_cl->dev_uevent = uevent; /* access permission regarding */
	
    if (IS_ERR(usb2ts_char_dev_device = device_create(usb2ts_char_dev_cl, NULL, MKDEV(usb2ts_char_dev_major_num, 0), NULL, TS2USB_DEVNAME))) {
		printk(KERN_INFO "Device file creation (/dev/%s) failed\n", TS2USB_DEVNAME);	
		ret = PTR_ERR(usb2ts_char_dev_device);
		goto err_three;
    }
	
	return 0;
	
err_three:
	class_unregister(usb2ts_char_dev_cl);
	class_destroy(usb2ts_char_dev_cl);
err_two:
	unregister_chrdev(usb2ts_char_dev_major_num, TS2USB_DEVNAME);
err_one:
	return ret;
}

static void destroy_usb2ts_char_dev(void)
{
	device_destroy(usb2ts_char_dev_cl, MKDEV(usb2ts_char_dev_major_num, 0));
	class_unregister(usb2ts_char_dev_cl);
	class_destroy(usb2ts_char_dev_cl);
	unregister_chrdev(usb2ts_char_dev_major_num, TS2USB_DEVNAME);
}

static int usb2ts_usb_msg(struct dvb_usb_device *d, u8 operation, u8 *ret_buf, size_t len)
{
	int ret;

	ret = usb_control_msg(d->udev, usb_rcvctrlpipe(d->udev, 0), operation,
			      USB_TYPE_VENDOR | USB_DIR_IN | USB_RECIP_INTERFACE, 1, 1,
			      ret_buf, len, USB2TS_USB_TIMEOUT);

	dvb_usb_dbg_usb_control_msg(d->udev, operation,
			(USB_TYPE_VENDOR | USB_DIR_IN | USB_RECIP_INTERFACE), 1, 1,
			ret_buf, (unsigned int)len);

	return ret;
}

static int usb2ts_init(struct dvb_usb_device *d)
{
	int i;
	int ret;
	char *usb_buff = NULL;
	struct dvb_usb_adapter *adap;
	
	copy_d = d;
	
	/*
	 * operation = 1; //swrst
     * operation = 2; //debug_en
	 * 
	 */
	

	if (usb_buff == NULL) {
		usb_buff = kmalloc(80, GFP_KERNEL);

		if (!usb_buff)
			return -ENOMEM;
	}


#ifdef TS2USB_FPGA_HAS_FW_INFO_RETRIEVAL_SUPPORT /* in this version, the below code is not issued; not sure why */
	ret = usb2ts_usb_msg(d, 1, usb_buff, 1);  //Issue a software reset on init(only reset the ts handling logic and endpfifo ,not usb)
	if (ret < 0) 
		printk(KERN_INFO "Error in issuing software reset to TS2USB bridge device\n");
#endif

	/*
	 * Why is it commented?
	 *
	 *
	 * // if (usb2ts_usb_debug_mode_en) //Always Enable it for now
	 *
	 */
	 
	ret = usb2ts_usb_msg(d, 2, usb_buff, 1);  //Issue debug enable
	if (ret < 0)
		printk(KERN_INFO "Error in issuing debug enable to TS2USB bridge device\n");

#ifdef TS2USB_FPGA_HAS_FW_INFO_RETRIEVAL_SUPPORT
	ret = usb2ts_usb_msg(d, 3, usb_buff, 45);  //Read firmware version
	strncpy(usb2ts_fpga_rtl_fw_ver, usb_buff, 45);
	printk(KERN_DEBUG "FPGA_RTL_FW_Info: %s\n", usb2ts_fpga_rtl_fw_ver);
#endif

   	if (!usb2ts_proc_file) {	
		usb2ts_proc_file = proc_create(USB2TS_PROCFS_NAME, 0666, NULL, &usb2ts_proc_fops);
		if (!usb2ts_proc_file) {
			printk(KERN_INFO "Create %s failed\n", USB2TS_PROCFS_NAME);
		}
	}
	
	/* need to clear error statistics on every USB disconnect-reconnect */
	for (i = 0; i < NUM_TS_STREAMS; i++) {
		clear_err_statistics(&usb2ts_err_statistics[i]);
		usb2ts_err_statistics[i].init = 1;
	}
	
	for (i = 0; i < d->props->num_adapters; i++) {
		adap = &d->adapter[i];
		adap->usb2ts_priv_fn = usb2ts_process_isopkt_err_info;
	}

	kfree(usb_buff);
	
	return ret;
}

static int usb2ts_dummy_frontend_attach(struct dvb_usb_adapter *adap)
{
	int ret = 0;
	int i;
	
	for (i = 0; i < MAX_NO_OF_FE_PER_ADAP; i++) {
		adap->fe[i] = dvb_attach(dvb_dummy_fe_ofdm_attach);
		if (adap->fe[i] == NULL) {
			printk(KERN_INFO
				": Failed to attach dummy front end\n");
				ret = -EINVAL;
				goto err_free;
		}
	}
	adap->active_fe = 0;
	return ret;
	
err_free:
	for (i = MAX_NO_OF_FE_PER_ADAP - 1; i >= 0; i--) {
		if (adap->fe[i]) {
			dvb_frontend_detach(adap->fe[i]);
			adap->fe[i] = NULL;
		}
	}
	
	return ret;
}

static int usb2ts_frontend_attach(struct dvb_usb_adapter *adap)
{
	return usb2ts_dummy_frontend_attach(adap);
}

static struct dvb_usb_device_properties usb2ts_props = {
	.driver_name = KBUILD_MODNAME,
	.owner = THIS_MODULE,
	.adapter_nr = adapter_nr,
	.init = usb2ts_init,
	.frontend_attach = usb2ts_frontend_attach,
	.num_adapters = 2,
	.adapter = {
		{
			.stream = DVB_USB_STREAM_BULK(0x81, 100, 24576),	//24576 => (8TS + Debug Info + Padding)* 16
		},
               {
			.stream = DVB_USB_STREAM_BULK(0x82, 100, 24576),
		},
	},
};

static const struct usb_device_id usb2ts_id_table[] = {
	{ DVB_USB_DEVICE(0x0559, 0x4012,
		&usb2ts_props, "Cadence Design Systems, Inc", NULL) },
	{ }
};
MODULE_DEVICE_TABLE(usb, usb2ts_id_table);

static struct usb_driver usb2ts_driver = {
	.name = KBUILD_MODNAME,
	.id_table = usb2ts_id_table,
	.probe = dvb_usbv2_probe,
	.disconnect = dvb_usbv2_disconnect,
	.suspend = NULL, /* not planning this now */
	.resume = NULL, /* not planning this now */
	.reset_resume = NULL, /* not planning this now */
	.no_dynamic_id = 1,
	.soft_unbind = 1,
};

/* not following the 'module_usb_driver' style as we have some work here */
static int __init usb2ts_module_init(void)
{
	if (init_usb2ts_char_dev() < 0)
		printk(KERN_INFO "The USB2TS driver's character device initialization failed\n");
	else
		printk(KERN_INFO "USB2TS character device creation succeeded\n");
	
	printk(KERN_INFO "USB2TS Driver (TS Driver) version 2.4\n");
	return usb_register(&usb2ts_driver);
}

static void __exit usb2ts_module_exit(void)
{
	int i;
	struct dvb_usb_adapter *adap;

	destroy_usb2ts_char_dev();
	
	if (copy_d != NULL) {
		for (i = 0; i < copy_d->props->num_adapters; i++) {
			adap = &copy_d->adapter[i];
			adap->usb2ts_priv_fn = NULL;
		}
	}
	
	remove_proc_entry(USB2TS_PROCFS_NAME, NULL);
	usb_deregister(&usb2ts_driver);
}
module_init(usb2ts_module_init);
module_exit(usb2ts_module_exit);

MODULE_AUTHOR("M's STYLE TECHNOLOGY Corp.");
MODULE_DESCRIPTION("DVB USB Interface Driver for TS2USB Bridge device");
MODULE_VERSION("2.4");
MODULE_LICENSE("GPL");
