/* -*- pse-c -*-
 *-----------------------------------------------------------------------------
 * Filename: ovl_tnc.c
 * $Revision: 1.23.46.2.4.8.2.10 $
 *-----------------------------------------------------------------------------
 * Copyright (c) 2002-2010, Intel Corporation.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *-----------------------------------------------------------------------------
 * Description:
 *  This file contains function that actually programs the overlay
 *  register back buffer with the bits to properly configure
 *  the overlay
 *  Also includes functions to execute the overlay flip instruction,
 *  and query the overlay flip status.
 *  Also contains some hardware capabilities querrying functions
 *  for upper overlay layer to get this chips overlay capabilities
 *-----------------------------------------------------------------------------
 */

#define MODULE_NAME hal.overlay

#include <math_fix.h>
#include <tnc/regs.h>
#include <tnc/cmd.h>
#include <tnc/context.h>
#include "ovl_tnc_cache.h"
#include "../cmn/ovl_dispatch.h"
#include "../cmn/ovl_virt.h"
#include "../cmn/ovl_coeff.h"
#include "ovl2_tnc.h"
#include <linux/module.h>
#include <emgdhmi_2fb.h>


/*-----------------------------------------------------------------------------
 * Common dispatch functions
 *---------------------------------------------------------------------------*/
static int alter_ovl_tnc(igd_display_context_t *display,
	igd_surface_t       *src_surf,
	igd_rect_t          *src_rect,
	igd_rect_t          *dest_rect,
	igd_ovl_info_t      *ovl_info,
	unsigned int         flags);
static int query_ovl_tnc(igd_display_h display_h,
	unsigned int flags);
static int query_max_size_ovl_tnc(igd_display_h display_h,
	unsigned long pf,
	unsigned int *max_width,
	unsigned int *max_height);
static int configure_buffers_tnc(igd_display_h primary,
	igd_display_h secondary,
	igd_buffer_config_t buf_cfg[2][3]);


ovl_dispatch_t ovl_dispatch_tnc[] = {
	/* Dispatch for the hardware overlay */
	{
		NULL, /* blend_surf_needed_tnc, */
		alter_ovl_tnc,
		query_ovl_tnc,
		query_max_size_ovl_tnc,
		configure_buffers_tnc,
	},
	/* Dispatch for the software overlay */
	{
		NULL, /* blend2_surf_needed_tnc, */
		alter_ovl2_tnc,
		query_ovl2_tnc,
		query_max_size_ovl2_tnc,
		configure_buffers_tnc,
	},
};


typedef struct _ovl_chipset_tnc {
	unsigned int	num_linebuf;
	unsigned int	pixel_format;
	unsigned int	max_width;
	unsigned int	max_height;
} ovl_chipset_tnc_t;

static ovl_chipset_tnc_t ovl_chipset_tnc[] = {
	{OVL_CONFIG_THREE_LINE_BUFF,
		(PF_DEPTH_16 | PF_TYPE_YUV_PACKED), 1280, 1080},
	{OVL_CONFIG_THREE_LINE_BUFF,
		(PF_DEPTH_8 | PF_TYPE_YUV_PLANAR), 1024, 1080},
	{OVL_CONFIG_TWO_LINE_BUFF,
		(PF_DEPTH_16 | PF_TYPE_YUV_PACKED), 1920, 1088},
	{OVL_CONFIG_TWO_LINE_BUFF,
		(PF_DEPTH_8 | PF_TYPE_YUV_PLANAR), 1920, 1088},
	{OVL_CONFIG_TWO_LINE_BUFF,
		(PF_DEPTH_32 | PF_TYPE_RGB), 1920, 1088},
	{OVL_CONFIG_NO_LINE_BUFF, 0, 0, 0}
};

typedef struct _dd_context_tnc {
	igd_dd_context_t        bridge;
	igd_surface_t           surf;
	igd_ovl_info_t          ovl;
	igd_display_context_t  *dpy;
} dd_context_tnc_t;

/* Pointers to allocated memory for the Direct Display context info */
static dd_context_tnc_t *dd_context_ovl;
static dd_context_tnc_t *dd_context_ovl2;

/* Local context for the Pop-up surfaces */
static dd_context_tnc_t popup_context_ovl;
static dd_context_tnc_t popup_context_ovl2;

/* The buffer configuration is maintained by emgd_interface.c */
extern igd_buffer_config_t igd_buf_cfg[2][3];

/* For Sprite C updates for pop-ups when surfaces attributes aren't changing */
extern void force_ovl2_cache_update(void);

/* Used to enable/disable Constant Alpha for pop-ups on Sprite C */
extern int ovl2_update_constant_alpha_tnc(
	igd_display_context_t *ctx,
	int enable,
	int alpha);


#ifdef DEBUG_BUILD_TYPE
static void ovl_dump_regs_tnc(
	ovl_reg_image_tnc_t *ovl_regs_tnc)
{
	{
		int i;
		unsigned long *ovl_buf = (unsigned long *)ovl_regs_tnc;
		for (i=0; i<0x10; i++) {
			EMGD_DEBUG_S("0x%2x: 0x%8lx 0x%8lx 0x%8lx 0x%8lx\n",
				i*0x10,
				*(ovl_buf+(i*4)), *(ovl_buf+(i*4+1)),
				*(ovl_buf+(i*4+2)), *(ovl_buf+(i*4+3)));
		}
	}


	EMGD_DEBUG_S("************************************************\n");
	EMGD_DEBUG_S("OVERLAY REGISTER LISTING\n");

	EMGD_DEBUG_S("RGB LOFF 0       =     0x%x  \n",
		ovl_regs_tnc->buffer0_yrgb_loff);
	EMGD_DEBUG_S("RGB LOFF 1       =     0x%x  \n",
		ovl_regs_tnc->buffer1_yrgb_loff);
	EMGD_DEBUG_S("U LOFF 0         =     0x%x  \n",
		ovl_regs_tnc->buffer0_u_loff);
	EMGD_DEBUG_S("V LOFF 0         =     0x%x  \n",
		ovl_regs_tnc->buffer0_v_loff);
	EMGD_DEBUG_S("U LOFF 1         =     0x%x  \n",
		ovl_regs_tnc->buffer1_u_loff);
	EMGD_DEBUG_S("V LOFF 1         =     0x%x  \n",
		ovl_regs_tnc->buffer1_v_loff);

	EMGD_DEBUG_S("RGB STRIDE       =     0x%x  \n",
		ovl_regs_tnc->yrgb_stride);
	EMGD_DEBUG_S("UV STRIDE        =     0x%x  \n",
		ovl_regs_tnc->uv_stride);
	EMGD_DEBUG_S("DST POS X        =     %d  \n",
		ovl_regs_tnc->dest_pos_x_left);
	EMGD_DEBUG_S("DST POS Y        =     %d  \n",
		ovl_regs_tnc->dest_pos_y_top);
	EMGD_DEBUG_S("DST WIDTH        =     %d  \n",
		ovl_regs_tnc->dest_width_x);
	EMGD_DEBUG_S("DST HEIGHT       =     %d  \n",
		ovl_regs_tnc->dest_height_y);
	EMGD_DEBUG_S("SRC WIDTH        =     %d  \n",
		ovl_regs_tnc->source_yrgb_width);
	EMGD_DEBUG_S("SRC SWWIDTH      =     0x%x  \n",
		ovl_regs_tnc->source_yrgb_width_swords);
	EMGD_DEBUG_S("SRC HEIGHT       =     %d  \n",
		ovl_regs_tnc->source_yrgb_height);
	EMGD_DEBUG_S("UV SRC WIDTH     =     %d  \n",
		ovl_regs_tnc->source_uv_width);
	EMGD_DEBUG_S("UV SRC SWWIDTH   =     %d  \n",
		ovl_regs_tnc->source_uv_width_swords);
	EMGD_DEBUG_S("UV SRC HEIGHT    =     %d  \n",
		ovl_regs_tnc->source_uv_height);
	EMGD_DEBUG_S("RGB SCALE        =     0x%x  \n",
		ovl_regs_tnc->yrgb_scale);
	EMGD_DEBUG_S("UV SCALE         =     0x%x  \n",
		ovl_regs_tnc->uv_scale);
	EMGD_DEBUG_S("COL CTL BRT CON  =     0x%x  \n",
		ovl_regs_tnc->col_ctl_brt_con);
	EMGD_DEBUG_S("COL CTL SAT HUE  =     0x%x  \n",
		ovl_regs_tnc->col_ctl_sat_hue);
	EMGD_DEBUG_S("DST COLOR KEY    =     0x%x  \n",
		ovl_regs_tnc->dest_ckey_val);
	EMGD_DEBUG_S("DST COLOR KEY MASK =   0x%x  \n",
		ovl_regs_tnc->dest_ckey_mask);
	EMGD_DEBUG_S("SRC COLOR KEY HI =     0x%x  \n",
		ovl_regs_tnc->source_ckey_high);
	EMGD_DEBUG_S("SRC COLOR KEY LO =     0x%x  \n",
		ovl_regs_tnc->source_ckey_low);
	EMGD_DEBUG_S("SRC COLOR KEY MASK =   0x%x  \n",
		ovl_regs_tnc->source_ckey_mask);
	EMGD_DEBUG_S("OVL CONFIG       =     0x%x  \n", ovl_regs_tnc->config);
	EMGD_DEBUG_S("OVL CMD          =     0x%x  \n", ovl_regs_tnc->command);
	EMGD_DEBUG_S("Y START 0        =     0x%x  \n",
		ovl_regs_tnc->buffer0_yrgb_start);
	EMGD_DEBUG_S("Y START 1        =     0x%x  \n",
		ovl_regs_tnc->buffer1_yrgb_start);
	EMGD_DEBUG_S("U START 0        =     0x%x  \n",
		ovl_regs_tnc->buffer0_u_start);
	EMGD_DEBUG_S("V START 0        =     0x%x  \n",
		ovl_regs_tnc->buffer0_v_start);
	EMGD_DEBUG_S("U START 1        =     0x%x  \n",
		ovl_regs_tnc->buffer1_u_start);
	EMGD_DEBUG_S("V START 1        =     0x%x  \n",
		ovl_regs_tnc->buffer1_v_start);
	EMGD_DEBUG_S("Y X TOFF 0       =     0x%x  \n",
		ovl_regs_tnc->buffer0_yrgb_x_toff);
	EMGD_DEBUG_S("Y Y TOFF 0       =     0x%x  \n",
		ovl_regs_tnc->buffer0_yrgb_y_toff);
	EMGD_DEBUG_S("Y X TOFF 1       =     0x%x  \n",
		ovl_regs_tnc->buffer1_yrgb_x_toff);
	EMGD_DEBUG_S("Y Y TOFF 1       =     0x%x  \n",
		ovl_regs_tnc->buffer1_yrgb_y_toff);
	EMGD_DEBUG_S("U X TOFF 0       =     0x%x  \n",
		ovl_regs_tnc->buffer0_u_x_toff);
	EMGD_DEBUG_S("U Y TOFF 0       =     0x%x  \n",
		ovl_regs_tnc->buffer0_u_y_toff);
	EMGD_DEBUG_S("V X TOFF 0       =     0x%x  \n",
		ovl_regs_tnc->buffer0_v_x_toff);
	EMGD_DEBUG_S("V Y TOFF 0       =     0x%x  \n",
		ovl_regs_tnc->buffer0_v_y_toff);
	EMGD_DEBUG_S("U X TOFF 1       =     0x%x  \n",
		ovl_regs_tnc->buffer1_u_x_toff);
	EMGD_DEBUG_S("U Y TOFF 1       =     0x%x  \n",
		ovl_regs_tnc->buffer1_u_y_toff);
	EMGD_DEBUG_S("V X TOFF 1       =     0x%x  \n",
		ovl_regs_tnc->buffer1_v_x_toff);
	EMGD_DEBUG_S("V Y TOFF 1       =     0x%x  \n",
		ovl_regs_tnc->buffer1_v_y_toff);
	EMGD_DEBUG_S("FAST_V_DSCALE    =     0x%x  \n",
		ovl_regs_tnc->vert_downscale);
	EMGD_DEBUG_S("************************************************\n");
}
#endif

/*----------------------------------------------------------------------
 * Function: ovl_check_pf_tnc()
 * Parameters: unsigned int requested_pixel_format -
 *             according to definitions in igd_mode.h
 *
 * Description:
 *
 * Returns:
 *   TRUE on Success
 *   FALSE on The first pixel format that is supported
 *----------------------------------------------------------------------*/
static unsigned int ovl_check_pf_tnc(
	igd_display_context_t *display,
	unsigned int requested_pixel_format)
{
	unsigned long *overlay_pfs;
	int temp_loop = 0;

	display->context->dispatch.get_pixelformats(
		(igd_display_h)display, NULL, NULL, &overlay_pfs, NULL, NULL);

	while(overlay_pfs[temp_loop]) {
		if(overlay_pfs[temp_loop] == requested_pixel_format) {
			return TRUE;
		}
		++temp_loop;
	}

	return FALSE;
}

static unsigned int get_uv_shift_x (unsigned long pf)
{

	switch(pf) {
	case IGD_PF_YUV422_PACKED_YUY2:
	case IGD_PF_YUV422_PACKED_UYVY:
	case IGD_PF_YUV420_PLANAR_I420: /* same as IYUV */
	case IGD_PF_YUV420_PLANAR_YV12:
	case IGD_PF_YUV420_PLANAR_NV12:
		return 1;
		break;
	case IGD_PF_YUV410_PLANAR_YVU9:
		return 2;
		break;
	default:
		return 0;
	}

}

static unsigned int get_uv_shift_y (unsigned long pf)
{

	switch(pf) {
	case IGD_PF_YUV420_PLANAR_I420: /* same as IYUV */
	case IGD_PF_YUV420_PLANAR_YV12:
	case IGD_PF_YUV420_PLANAR_NV12:
		return 1;
		break;
	case IGD_PF_YUV410_PLANAR_YVU9:
		return 2;
		break;
	default:
		return 0;
	}

}

static unsigned int ovl_check_tnc(igd_display_context_t *display,
	igd_surface_t       *src_surf,
	igd_rect_t          *src_rect,
	igd_rect_t          *dest_rect,
	igd_ovl_info_t      *ovl_info,
	unsigned int         flags)
{
	igd_timing_info_t *timing;
	ovl_chipset_tnc_t *ovl_chip;
	unsigned int min_w, min_h;

	EMGD_TRACE_ENTER;

	if (!display){
		EMGD_ERROR_EXIT("display is null");
		return -IGD_ERROR_INVAL;
	}
	if (!PIPE(display)){
		EMGD_ERROR_EXIT("PIPE(display) is null");
		return -IGD_ERROR_INVAL;
	}

	timing = PIPE(display)->timing;

	if(!timing) {
		EMGD_ERROR_EXIT("timing is null\n");
		return -IGD_ERROR_INVAL;
	}

	/*************************************************************************
	 * Ensure the framebuffer dotclock does not exceed the board SKU
	 * max dotclock
	 *************************************************************************/
	/* DCT-PC99TA crashes with dotclock > 300MHz */
	/* FIXME: This is using the dclk from Napa */
	if(timing->dclk >= 340000){
		EMGD_ERROR_EXIT("Cannot support dotclock > 340MHz for this SKU");
		return -IGD_ERROR_HWERROR;
	}

	/* The following parameters are only valid if the overlay is on, so
	 * return success if the overlay is being turned off. */
	if ((flags & IGD_OVL_ALTER_ON) == IGD_OVL_ALTER_OFF) {
		EMGD_TRACE_EXIT;
		return IGD_SUCCESS;
	}

	/*************************************************************************
	 * Ensure the overlay surface is ok and can be properly displayed.
	 * This ensures the following is valid:
	 *    - Ensure x1, x2, y1, y2 are pixel aligned
	 *    - 2 pixels or greater in width and height
	 *    - Pixel format is supported by the overlay
	 *    - Pitch is <= 8KB
	 *    - Based on the pixel format, the width is supported
	 *************************************************************************/
	if (!src_surf){
		EMGD_ERROR_EXIT("src_surf is null");
		return -IGD_ERROR_INVAL;
	}
	if (!src_rect){
		EMGD_ERROR_EXIT("src_rect is null");
		return -IGD_ERROR_INVAL;
	}
	/* Get the minimum size of 1 pixel in width and height for y, u, and v.
	 */
	min_w = 1 << get_uv_shift_x(src_surf->pixel_format);
	min_h = 1 << get_uv_shift_y(src_surf->pixel_format);

	if (((src_rect->x2 - src_rect->x1) < min_w*2) ||
		((src_rect->y2 - src_rect->y1) < min_h*2)) {
		EMGD_ERROR_EXIT(
			"Overlay source width or height is < 2 pixels (%dx%d)\n",
			src_rect->x2 - src_rect->x1, src_rect->y2 - src_rect->y1);
		return -IGD_ERROR_INVAL;
	}

	if (FALSE == ovl_check_pf_tnc(display, src_surf->pixel_format)) {
		EMGD_ERROR_EXIT("Overlay source pixel format unsupported (pf:0x%lx)",
			src_surf->pixel_format);
		return -IGD_ERROR_HWERROR;
	}

	if (src_surf->pitch > 8192) {
		EMGD_ERROR_EXIT("Overlay source pitch (%d) > 8KB",
			src_surf->pitch);
		return -IGD_ERROR_HWERROR;
	}

	ovl_chip = ovl_chipset_tnc;
	while(ovl_chip->num_linebuf != OVL_CONFIG_NO_LINE_BUFF){
		if(((src_surf->pixel_format & IGD_PF_MASK) ==
				ovl_chip->pixel_format) &&
			(src_surf->width <= ovl_chip->max_width)) {
			break;
		}
		ovl_chip++;
	}
	if (ovl_chip->num_linebuf == OVL_CONFIG_NO_LINE_BUFF) {
		EMGD_ERROR_EXIT("Overlay source width (%d) > max supported",
			src_surf->width);
		return -IGD_ERROR_HWERROR;
	}

	/*************************************************************************
	 * Ensure the location on the framebuffer is ok and can be properly
	 * displayed
	 * This ensures the following is valid:
	 *    - Greater than 1 pixel width and height
	 *    - Will be displayed on screen (not panned off)
	 *************************************************************************/
	if (!dest_rect){
		EMGD_ERROR_EXIT("dest_rect is null");
		return -IGD_ERROR_INVAL;
	}
	if (((dest_rect->x2 - dest_rect->x1) <= 1) ||
		((dest_rect->y2 - dest_rect->y1) <= 1)) {
		EMGD_ERROR_EXIT(
			"Overlay dest width or height is single pixel (%dx%d)\n",
			dest_rect->x2 - dest_rect->x1, dest_rect->y2 - dest_rect->y1);
		return -IGD_ERROR_INVAL;
	}

	if ((dest_rect->x1 >= timing->width) ||
		(dest_rect->y1 >= timing->height)) {
		EMGD_ERROR_EXIT(
			"Overlay dest is panned off the screen (%d,%d)\n",
			dest_rect->x1, dest_rect->y1);
		return -IGD_ERROR_INVAL;
	}

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}



static unsigned int ovl_update_src_tnc(igd_display_context_t *display,
	ovl_reg_image_tnc_t *ovl_regs_tnc,
	igd_surface_t *src_surf,
	igd_rect_t    *src_rect)
{
	ovl_chipset_tnc_t *ovl_chip;
	unsigned int      src_Bpp;
	unsigned int      src_uv_shift_x, src_uv_shift_y;
	unsigned short    src_w;
	unsigned short    src_h;

	EMGD_TRACE_ENTER;

	/* This is in Bytes per pixel */
	src_Bpp = IGD_PF_BPP(src_surf->pixel_format)/8;

	src_uv_shift_x = get_uv_shift_x(src_surf->pixel_format);
	src_uv_shift_y = get_uv_shift_y(src_surf->pixel_format);

	src_w = src_rect->x2 - src_rect->x1;
	src_h = src_rect->y2 - src_rect->y1;

	/* Surface pitch */
	ovl_regs_tnc->yrgb_stride      = (unsigned short)src_surf->pitch;
	ovl_regs_tnc->uv_stride        = (unsigned short)src_surf->u_pitch;


	/* src width */
	ovl_regs_tnc->source_yrgb_width      = src_w;
	ovl_regs_tnc->source_uv_width        = src_w >> src_uv_shift_x;

	/* src width swords - This equation follows the B-Spec */
	/* Equation in B-spec is cancelling out fixed point round-up when -1.
	   Explain below:
	   [[(offset + width + 63)/128 - offset/128] * 2 ] - 1
	   [(width/128 + 63/128) * 2] - 1
	   (width/128)*2 + (0.249*2) - 1
	   width/64

	   Modify the formula witihout -1 to preserve fix point round-up.
           Equation = width/64 + 0.4999
	   Also modified 63/128(0x3F) to 31/128(0x1F) to have 0.5 when *2.
        */

	ovl_regs_tnc->source_yrgb_width_swords =
		(unsigned short)(((((src_surf->offset +
							(ovl_regs_tnc->source_yrgb_width * src_Bpp) +
							0x1F) >> 6) -
					(src_surf->offset >> 6)) << 1) << 2);
	ovl_regs_tnc->source_uv_width_swords   =
		(unsigned short)(((((src_surf->u_offset +
							(ovl_regs_tnc->source_uv_width * src_Bpp) +
							0x1F) >> 6) -
					(src_surf->u_offset >> 6)) << 1) << 2 );

	/* src height */
	ovl_regs_tnc->source_yrgb_height     = src_h;
	ovl_regs_tnc->source_uv_height       = src_h >> src_uv_shift_y;

	/* src pixel format */
	switch(src_surf->pixel_format){
	case IGD_PF_YUV422_PACKED_YUY2:
		ovl_regs_tnc->command |= OVL_CMD_YUV_422;
		break;
	case IGD_PF_YUV422_PACKED_UYVY:
		ovl_regs_tnc->command |= OVL_CMD_YUV_422 | OVL_CMD_Y_SWAP;
		break;
	case IGD_PF_YUV420_PLANAR_I420: /* same as IYUV */
		ovl_regs_tnc->command |= OVL_CMD_YUV_420P;
		break;
	case IGD_PF_YUV420_PLANAR_YV12:
		ovl_regs_tnc->command |= OVL_CMD_YUV_420P | OVL_CMD_UV_SWAP;
		break;
	case IGD_PF_YUV420_PLANAR_NV12:
		ovl_regs_tnc->command |= OVL_CMD_YUV_NV12;
		break;
	case IGD_PF_YUV410_PLANAR_YVU9:
		ovl_regs_tnc->command |= OVL_CMD_YUV_410P;
		break;
	case IGD_PF_ARGB32_8888:
	case IGD_PF_xRGB32_8888:
		ovl_regs_tnc->command |= OVL_CMD_RGB_8888;
		break;
	case IGD_PF_RGB16_565:
		ovl_regs_tnc->command |= OVL_CMD_RGB_565;
		break;
	case IGD_PF_xRGB16_555:
	case IGD_PF_ARGB16_1555:
		ovl_regs_tnc->command |= OVL_CMD_RGB_555;
		break;
	default:
		EMGD_ERROR_EXIT("Invalid pixel format: 0x%lx", src_surf->pixel_format);
		return -IGD_ERROR_HWERROR;
	}

	/* Turn off YUV to RGB conversion if the src is RGB */
	if (!(src_surf->pixel_format & PF_TYPE_YUV)) {
		ovl_regs_tnc->config |= (1<<4);
	}

	ovl_chip = ovl_chipset_tnc;
	ovl_regs_tnc->config &= ~OVL_CONFIG_LINE_BUFF_MASK;
	while(ovl_chip->num_linebuf != OVL_CONFIG_NO_LINE_BUFF){
		if(((src_surf->pixel_format & IGD_PF_MASK) ==
				ovl_chip->pixel_format) &&
			(src_w <= ovl_chip->max_width)) {
			ovl_regs_tnc->config |= ovl_chip->num_linebuf;
			break;
		}
		ovl_chip++;
	}

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}

static unsigned int ovl_update_src_ptr_tnc(igd_display_context_t *display,
	ovl_reg_image_tnc_t *ovl_regs_tnc,
	igd_surface_t *src_surf,
	igd_rect_t    *src_rect)
{
	unsigned int      src_Bpp;
	unsigned int      src_uv_shift_x, src_uv_shift_y;

	EMGD_TRACE_ENTER;

	/* This is in Bytes per pixel */
	src_Bpp = IGD_PF_BPP(src_surf->pixel_format)/8;

	src_uv_shift_x = get_uv_shift_x(src_surf->pixel_format);
	src_uv_shift_y = get_uv_shift_y(src_surf->pixel_format);

	/* Surface offset */
	ovl_regs_tnc->buffer0_yrgb_start =
		ovl_regs_tnc->buffer1_yrgb_start =
		src_surf->offset;
	ovl_regs_tnc->buffer0_u_start =
		ovl_regs_tnc->buffer1_u_start =
		src_surf->u_offset;
	ovl_regs_tnc->buffer0_v_start =
		ovl_regs_tnc->buffer1_v_start =
		src_surf->v_offset;

	/* Linear surface information */
	ovl_regs_tnc->buffer0_yrgb_loff     =
		ovl_regs_tnc->buffer1_yrgb_loff =
		(src_rect->y1 * src_surf->pitch) + (src_rect->x1 * src_Bpp);

	/*
	 * The NV12 format has the UV pixels interleaved so the total
	 * width of the UV portion of the surface is the same as the
	 * Y width. Thus, don't do any shifting of the UV plane in the
	 * X direction.
	 */
	if (src_surf->pixel_format == IGD_PF_YUV420_PLANAR_NV12) {
		ovl_regs_tnc->buffer0_u_loff     =
			ovl_regs_tnc->buffer1_u_loff =
			((src_rect->y1>>src_uv_shift_y) *
			 src_surf->u_pitch) + src_rect->x1;

		ovl_regs_tnc->buffer0_v_loff     =
			ovl_regs_tnc->buffer1_v_loff =
			((src_rect->y1>>src_uv_shift_y) *
			 src_surf->v_pitch) + src_rect->x1;
	} else {
		ovl_regs_tnc->buffer0_u_loff     =
			ovl_regs_tnc->buffer1_u_loff =
			((src_rect->y1>>src_uv_shift_y) *
			 src_surf->u_pitch) +
			(src_rect->x1>>src_uv_shift_x);

		ovl_regs_tnc->buffer0_v_loff     =
			ovl_regs_tnc->buffer1_v_loff =
			((src_rect->y1>>src_uv_shift_y) * src_surf->v_pitch) +
			(src_rect->x1>>src_uv_shift_x);
	}


	/* The B-Spec states that these values are ignored with a linear
	 * surface.  However on the 965G, these values are not ignored,
	 * so zero them. */
	ovl_regs_tnc->buffer0_yrgb_x_toff =
		ovl_regs_tnc->buffer1_yrgb_x_toff =
		ovl_regs_tnc->buffer0_yrgb_y_toff =
		ovl_regs_tnc->buffer1_yrgb_y_toff =
		ovl_regs_tnc->buffer0_u_x_toff =
		ovl_regs_tnc->buffer0_v_x_toff =
		ovl_regs_tnc->buffer1_u_x_toff =
		ovl_regs_tnc->buffer1_v_x_toff =
		ovl_regs_tnc->buffer0_u_y_toff =
		ovl_regs_tnc->buffer0_v_y_toff =
		ovl_regs_tnc->buffer1_u_y_toff =
		ovl_regs_tnc->buffer1_v_y_toff =
		0;

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}

static unsigned int ovl_update_phase_tnc(
	ovl_reg_image_tnc_t *ovl_regs_tnc,
	igd_surface_t       *src_surf,
	igd_rect_t          *src_rect)
{

	EMGD_TRACE_ENTER;

	/*
	 * Set the Vertical/Horizontal Phase Registers.  Both Field0
	 * and Field1 are set, although Field1 is only used when
	 * interleaved.
	 */
	switch (src_surf->pixel_format) {
	case IGD_PF_YUV422_PACKED_YUY2:
	case IGD_PF_YUV422_PACKED_UYVY:
		/* YUV 422 */
		ovl_regs_tnc->init_phase_shift = 0;

		/* Vertical Phase */
		if (ovl_regs_tnc->config & OVL_CONFIG_THREE_LINE_BUFF) {
			ovl_regs_tnc->yrgb_vert_phase_field0 = 0;
			ovl_regs_tnc->yrgb_vert_phase_field1 = 0;
			ovl_regs_tnc->uv_vert_phase_field0 = 0;
			ovl_regs_tnc->uv_vert_phase_field1 = 0;
		} else {
			ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000; /*.5*/
			ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000; /*.5*/
			ovl_regs_tnc->uv_vert_phase_field0   = 0x8000; /*.5*/
			ovl_regs_tnc->uv_vert_phase_field1   = 0x8000; /*.5*/
		}

		/* Horizontal Phase */
		if (!(src_rect->x1 & 1)) {
			ovl_regs_tnc->yrgb_hphase = 0;
			ovl_regs_tnc->uv_hphase = 0;
		} else {
			ovl_regs_tnc->init_phase_shift |= Y_HPP_PLUS1;
			ovl_regs_tnc->yrgb_hphase = 0; /*1*/
			ovl_regs_tnc->uv_hphase = 0x8000; /*.5*/
		}
		break;

	case IGD_PF_YUV420_PLANAR_I420:
	case IGD_PF_YUV420_PLANAR_YV12:
	case IGD_PF_YUV420_PLANAR_NV12:
		/* YUV 420 */
		ovl_regs_tnc->init_phase_shift = 0;

		/* Vertical Phase */
		if (ovl_regs_tnc->config & OVL_CONFIG_THREE_LINE_BUFF) {
			if (!(src_rect->y1 & 1)) {
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0; /*0*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0;/*1*/

				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD0_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field0 = 0xc000; /*-.25*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0x4000; /*.25*/
			} else {
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0; /*1*/
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0;/*0*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0x4000; /*.25*/
				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD1_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field1 = 0xc000; /*-.25*/
			}
		} else {
			if (!(src_rect->y1 & 1)) {
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000; /*.5*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000;/*1.5*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0x4000; /*.25*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0xc000; /*.75*/
			} else {
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000;/*1.5*/
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000; /*.5*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0xc000; /*.75*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0x4000; /*.25*/
			}
		}

		/* Horizontal Phase */
		if (!(src_rect->x1 & 1)) {
			ovl_regs_tnc->yrgb_hphase = 0;
			ovl_regs_tnc->uv_hphase = 0;
		} else {
			ovl_regs_tnc->init_phase_shift |= Y_HPP_PLUS1;
			ovl_regs_tnc->yrgb_hphase = 0; /*1*/
			ovl_regs_tnc->uv_hphase = 0x8000; /*.5*/
		}
		break;

	case IGD_PF_YUV410_PLANAR_YVU9:
		/* YUV 410 */
		ovl_regs_tnc->init_phase_shift = 0;

		/* Vertical Phase */
		if (ovl_regs_tnc->config & OVL_CONFIG_THREE_LINE_BUFF) {
			switch (src_rect->y1 & 3) {
			default:
			case 0:
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0; /*0*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0;/*1*/

				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD0_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field0 = 0xa000; /*-.375*/
				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD1_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field1 = 0xe000; /*-.125*/
				break;

			case 1:
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0; /*1*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0;/*1*/

				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD0_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field0 = 0xe000; /*-.125*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0x2000; /*.125*/
				break;

			case 2:
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0; /*1*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0;/*1*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0x2000; /*.125*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0x6000; /*.375*/
				break;

			case 3:
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0; /*1*/
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0;/*0*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0x6000; /*.375*/
				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD1_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field1 = 0xa000; /*-.375*/
				break;
			}
		} else {
			switch (src_rect->y1 & 3) {
			default:
			case 0:
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000; /*.5*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000;/*1.5*/

				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD0_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field0 = 0xc000; /*-.25*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0; /*0*/
				break;

			case 1:
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000;/*1.5*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000; /*1.5*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0x0; /*0*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0x4000; /*.25*/
				break;

			case 2:
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000;/*1.5*/
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD1_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000; /*1.5*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0x4000; /*.25*/
				ovl_regs_tnc->uv_vert_phase_field1 = 0x8000; /*.5*/
				break;

			case 3:
				ovl_regs_tnc->init_phase_shift |= Y_VPP_FLD0_PLUS1;
				ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000;/*1.5*/
				ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000; /*.5*/

				ovl_regs_tnc->uv_vert_phase_field0 = 0x8000; /*.5*/
				ovl_regs_tnc->init_phase_shift |= UV_VPP_FLD1_MINUS1;
				ovl_regs_tnc->uv_vert_phase_field1 = 0xc000; /*-.25*/
				break;
			}
		}

		/* Horizontal Phase */
		switch (src_rect->x1 & 3) {
		default:
		case 0:
			ovl_regs_tnc->yrgb_hphase = 0;
			ovl_regs_tnc->init_phase_shift |= UV_HPP_MINUS1;
			ovl_regs_tnc->uv_hphase = 0xa000; /*-.375*/
			break;

		case 1:
			ovl_regs_tnc->init_phase_shift |= Y_HPP_PLUS1;
			ovl_regs_tnc->yrgb_hphase = 0; /*1*/
			ovl_regs_tnc->init_phase_shift |= UV_HPP_MINUS1;
			ovl_regs_tnc->uv_hphase = 0xe000; /*-.125*/
			break;

		case 2:
			ovl_regs_tnc->init_phase_shift |= Y_HPP_PLUS2;
			ovl_regs_tnc->yrgb_hphase = 0; /*2*/
			ovl_regs_tnc->uv_hphase = 0x2000; /*.125*/
			break;

		case 3:
			ovl_regs_tnc->init_phase_shift |= Y_HPP_PLUS2;
			ovl_regs_tnc->yrgb_hphase = 0xffff; /*3*/
			ovl_regs_tnc->uv_hphase = 0x6000; /*.375*/
			break;
		}
		break;

	default:
		/* RGB format */
		ovl_regs_tnc->init_phase_shift = 0;

		/* Vertical Phase */
		if (ovl_regs_tnc->config & OVL_CONFIG_THREE_LINE_BUFF) {
			ovl_regs_tnc->yrgb_vert_phase_field0 = 0;
			ovl_regs_tnc->yrgb_vert_phase_field1 = 0;
			ovl_regs_tnc->uv_vert_phase_field0 = 0;
			ovl_regs_tnc->uv_vert_phase_field1 = 0;
		} else {
			ovl_regs_tnc->yrgb_vert_phase_field0 = 0x8000;
			ovl_regs_tnc->yrgb_vert_phase_field1 = 0x8000;
			ovl_regs_tnc->uv_vert_phase_field0   = 0x8000;
			ovl_regs_tnc->uv_vert_phase_field1   = 0x8000;
		}

		/* Horizontal Phase */
		ovl_regs_tnc->yrgb_hphase = 0;
		ovl_regs_tnc->uv_hphase = 0;
		break;
	}

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}



/*----------------------------------------------------------------------
 * Function: ovl_update_scale_tnc()
 *
 * Description: Will update only the scaling registers for the Atom E6xx core
 *
 * Returns:
 *   N/A
 *----------------------------------------------------------------------*/

static unsigned int ovl_update_scale_tnc(
	ovl_reg_image_tnc_t *ovl_regs_tnc,
	igd_surface_t *src_surf,
	igd_rect_t   *src_rect,
	igd_rect_t   *dest_rect,
	unsigned int flags)
{
	unsigned int uv_shift;
	unsigned int xscale, xscale_int, xscale_fract;
	unsigned int yscale, yscale_int, yscale_fract;
	unsigned int xscale_int_uv, xscale_fract_uv;
	unsigned int yscale_int_uv, yscale_fract_uv;

	EMGD_TRACE_ENTER;

	xscale = ((src_rect->x2 - src_rect->x1)<<12) /
		(dest_rect->x2 - dest_rect->x1);
	yscale = ((src_rect->y2 - src_rect->y1)<<12) /
		(dest_rect->y2 - dest_rect->y1);

	/* In interleaved mode, the y scale is /2 */
	if (flags & IGD_OVL_ALTER_INTERLEAVED) {
		yscale >>= 1;
	}

	xscale_int = (xscale & 0x7000) >> 12;
	xscale_fract = xscale & 0xfff;
	yscale_int = (yscale & 0x7ff000) >> 12;
	yscale_fract = yscale & 0xfff;

	/* The uv scale is used for both YUV Planar as well as YUV Packed */
	uv_shift = get_uv_shift_x(src_surf->pixel_format);
	xscale_int_uv = ((xscale>>uv_shift) & 0x7000) >> 12;
	xscale_fract_uv = (xscale>>uv_shift) & 0xfff;

	uv_shift = get_uv_shift_y(src_surf->pixel_format);
	yscale_int_uv = ((yscale>>uv_shift) & 0x7ff000) >> 12;
	yscale_fract_uv = (yscale>>uv_shift) & 0xfff;

	ovl_regs_tnc->yrgb_scale =
		(yscale_fract  << 20) |   /* Vert  Scale Fraction */
		(xscale_int    << 16) |   /* Horiz Scale Int */
		(xscale_fract  << 3);     /* Horiz Scale Fraction */

	ovl_regs_tnc->uv_scale =
		(yscale_fract_uv << 20) | /* UV Vert  Scale Fraction */
		(xscale_int_uv   << 16) | /* UV Horiz Scale Int */
		(xscale_fract_uv << 3 );  /* UV Horiz Scale Fraction */
	ovl_regs_tnc->vert_downscale =
		(yscale_int    << 16) |   /* Vert Scale Factor */
		yscale_int_uv;            /* UV Vert Scale Factor */

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}



/*----------------------------------------------------------------------
 * Function: ovl_update_video_quality_tnc()
 *
 * Description:
 *   This function updates the contrast, brightness, and saturation of
 *   the overlay using the values specified in overlay_info.
 *
 * Returns:
 *   != 0 on Error
 *   0 on Success
 *----------------------------------------------------------------------*/

static int ovl_update_video_quality_tnc(
	ovl_reg_image_tnc_t *ovl_regs_tnc,
	igd_surface_t       *src_surf,
	igd_ovl_video_quality_info_t *video_quality)
{
	int                          calc_brightness_tmp = 0;
	int                          calc_brightness     = 0;
	unsigned int                 calc_contrast_tmp   = 0;
	unsigned int                 calc_contrast       = 0;
	unsigned int                 calc_saturation_tmp = 0;
	unsigned int                 calc_saturation     = 0;

	EMGD_TRACE_ENTER;

	/* If the src_surf pixel format is RGB, then brightness, contrast,
	 * and saturation should all be set to the exact default */
	if (src_surf->pixel_format & PF_TYPE_RGB) {
		if (video_quality->brightness != 0x8000) {
			EMGD_DEBUG("RGB surfaces must set brightness to default");
		}
		if (video_quality->contrast != 0x8000) {
			EMGD_DEBUG("RGB surfaces must set contrast to default");
		}
		if (video_quality->saturation != 0x8000) {
			EMGD_DEBUG("RGB surfaces must set saturation to default");
		}

		ovl_regs_tnc->col_ctl_brt_con = OVL_RGB_COLOR_DEF_CONT_BRGHT;
		ovl_regs_tnc->col_ctl_sat_hue = OVL_RGB_COLOR_DEF_SATN_HUE;

		EMGD_TRACE_EXIT;
		return IGD_SUCCESS;
	}

	/*************************************************************************
	 * Brightness
	 *************************************************************************/
	if (0x8000 == video_quality->brightness) {
		calc_brightness = MID_BRIGHTNESS_YUV;
	} else if (video_quality->brightness < 0x8000) {
		/*
		 * we have here a brightness that is less than the default
		 * mid point
		 */
		calc_brightness_tmp = 0x8000 - video_quality->brightness;
		calc_brightness_tmp <<= 14;
		calc_brightness_tmp /= 0x8000;
		calc_brightness     = -128 - MID_BRIGHTNESS_YUV;
		/*
		 * more range if the midpoint is positive but less range
		 * if midpoint is negative
		 */

		calc_brightness *= calc_brightness_tmp;
		calc_brightness += BIT13;
		calc_brightness >>= 14;

		if (calc_brightness < -128) {
			calc_brightness = -128;
		}
		if (calc_brightness > MID_BRIGHTNESS_YUV) {
			calc_brightness = MID_BRIGHTNESS_YUV;
		}
	} else {
		/*
		 * we have here a brightness that is more than the default
		 * mid point
		 */
		calc_brightness_tmp = video_quality->brightness - 0x8000;
		calc_brightness_tmp <<= 14;
		calc_brightness_tmp /= 0x8000;
		calc_brightness     = 127 - MID_BRIGHTNESS_YUV;
		/*
		 * less range if the midpoint is positive but more range
		 * if midpoint is negative
		 */
		calc_brightness *= calc_brightness_tmp;
		calc_brightness += BIT13;
		calc_brightness >>= 14;

		if (calc_brightness > 127) {
			calc_brightness = 127;
		}
		if (calc_brightness < MID_BRIGHTNESS_YUV) {
			calc_brightness = MID_BRIGHTNESS_YUV;
		}
	}

	ovl_regs_tnc->col_ctl_brt_con =
		(ovl_regs_tnc->col_ctl_brt_con & 0xFFFFFF00) |
		(calc_brightness & 0xFF);

	/*************************************************************************
	 * Contrast
	 *************************************************************************/
	if (0x8000 == video_quality->contrast ){
		calc_contrast = MID_CONTRAST_YUV;
	} else if (video_quality->contrast < 0x8000) {
		/* we have here a contrast that is less than the
		 * default mid point */
		calc_contrast_tmp = video_quality->contrast;
		calc_contrast_tmp <<= 12;
		calc_contrast_tmp /= 0x8000;
		calc_contrast     = MID_CONTRAST_YUV;
		calc_contrast     *= calc_contrast_tmp;
		calc_contrast     += BIT11;
		calc_contrast     >>= 12;
		if (calc_contrast > 0x3F) {
			calc_contrast = 0x3F;
		}
	} else {
		/* we have here a contrast that is more than the
		 * default mid point */
		calc_contrast_tmp = video_quality->contrast - 0x8000;
		calc_contrast_tmp <<= 12;
		calc_contrast_tmp /= 0x8000;
		calc_contrast     = (0x1FF - MID_CONTRAST_YUV);
		calc_contrast     *= calc_contrast_tmp;
		calc_contrast     += BIT11;
		calc_contrast     >>= 12;
		calc_contrast     += MID_CONTRAST_YUV;
		if (calc_contrast > 0x1FF) {
			calc_contrast = 0x1FF;
		}
	}

	ovl_regs_tnc->col_ctl_brt_con =
		(ovl_regs_tnc->col_ctl_brt_con & 0xF803FFFF) |
		((calc_contrast & 0x1FF) << 18);

	/*************************************************************************
	 * Saturation
	 *************************************************************************/
	if (video_quality->saturation == 0x8000) {
		calc_saturation = MID_SATURATION_YUV;
	} else if (video_quality->saturation < 0x8000) {
		/* we have here a saturation that is less than the default
		 * mid point */
		calc_saturation_tmp = video_quality->saturation;
		calc_saturation_tmp <<= 12;
		calc_saturation_tmp /= 0x8000;
		calc_saturation     = MID_SATURATION_YUV;
		calc_saturation     *= calc_saturation_tmp;
		calc_saturation     += BIT11;
		calc_saturation     >>= 12;
		if (calc_saturation > 0x7F) {
			calc_saturation = 0x7F;
		}
	} else {
		/* we have here a saturation that is more than the default
		 * mid point*/
		calc_saturation_tmp = video_quality->saturation - 0x8000;
		calc_saturation_tmp <<= 12;
		calc_saturation_tmp /= 0x8000;
		calc_saturation     = (0x3FF - MID_SATURATION_YUV);
		calc_saturation     *= calc_saturation_tmp;
		calc_saturation     += BIT11;
		calc_saturation     >>= 12;
		calc_saturation     += MID_SATURATION_YUV;

		if (calc_saturation > 0x3FF) {
			calc_saturation = 0x3FF;
		}
	}

	ovl_regs_tnc->col_ctl_sat_hue =
		(ovl_regs_tnc->col_ctl_sat_hue & 0xFFFFFC00) |
		(calc_saturation & 0x3FF);

	/*************************************************************************
	 * Hue
	 *************************************************************************/
	/* Hue is always set to the default value.  It is based on the saturation
	 * value, and having a separate hue is of minimal value. */
	ovl_regs_tnc->col_ctl_sat_hue =
		(ovl_regs_tnc->col_ctl_sat_hue & 0xF800FFFF);

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}

static void check_gamma(unsigned int *gamma)
{

	if (*gamma < IGD_OVL_GAMMA_MIN) {
		EMGD_ERROR("Gamma to small (0x%x in 24i.8f format), "
			"changing to Min Gamma (0.6)",
			*gamma);
		*gamma = IGD_OVL_GAMMA_MIN;
	}
	if (*gamma > IGD_OVL_GAMMA_MAX) {
		EMGD_ERROR("Gamma to large (0x%x in 24i.8f format), "
			"changing to Max Gamma (6.0)",
			*gamma);
		*gamma = IGD_OVL_GAMMA_MAX;
	}

	return;
}

/*-----------------------------------------------------------------------------
 * Function: ovl_update_gamma_tnc()
 *
 * Description:
 *    This function sets the gamma correction values for the overlays.
 *
 * Returns:
 *   != 0 on Error
 *   IGD_SUCCESS on Success
 *---------------------------------------------------------------------------*/
static int ovl_update_gamma_tnc(
	igd_display_context_t *display,
	igd_ovl_gamma_info_t * ovl_gamma)
{
	const int gamma_reg_input[OVL_TOTAL_GAMMA_REG] = {8, 16, 32, 64, 128, 192};
	const int gamma_reg_offset[OVL_TOTAL_GAMMA_REG] = {
		OVL_REG_ADDR_GAMMA0,
		OVL_REG_ADDR_GAMMA1,
		OVL_REG_ADDR_GAMMA2,
		OVL_REG_ADDR_GAMMA3,
		OVL_REG_ADDR_GAMMA4,
		OVL_REG_ADDR_GAMMA5
	};
	const unsigned int gamma_def[OVL_TOTAL_GAMMA_REG] = {
		0x00080808,
		0x00101010,
		0x00202020,
		0x00404040,
		0x00808080,
		0x00c0c0c0
	};
	unsigned int          new_gamma_red_24i_8f, new_gamma_green_24i_8f;
	unsigned int          new_gamma_blue_24i_8f;
	unsigned int          gamma_normal_r_24i_8f;
	unsigned int          gamma_normal_g_24i_8f;
	unsigned int          gamma_normal_b_24i_8f;
	unsigned int          gamma_reg, gamma_reg_24i_8f;
	unsigned int          i;

	EMGD_TRACE_ENTER;

	/* FIXME: The gamma values are re-written for every alter_ovl call.
	 * This may cause issues or may be to slow?  If so, store the previous
	 * values and only re-write when they change. */

	/* If the overlay gamma is disabled, set it to the default */
	if ((ovl_gamma->flags & IGD_OVL_GAMMA_ENABLE) == IGD_OVL_GAMMA_DISABLE) {
		for (i = 0; i < OVL_TOTAL_GAMMA_REG; i++) {
			/* program register */
			EMGD_WRITE32(gamma_def[i], MMIO(display) + gamma_reg_offset[i]);
		}
		EMGD_TRACE_EXIT;
		return IGD_SUCCESS;
	}

	/* It is assumed that the input value is a 24-bit number */
	new_gamma_red_24i_8f   = ovl_gamma->red;
	new_gamma_green_24i_8f = ovl_gamma->green;
	new_gamma_blue_24i_8f  = ovl_gamma->blue;

	/* Ensure the gamma values are between MIN and MAX */
	check_gamma(&new_gamma_red_24i_8f);
	check_gamma(&new_gamma_green_24i_8f);
	check_gamma(&new_gamma_blue_24i_8f);

	/*
	 * Program RGB for each of the 6 gamma registers
	 */

	/* Since the OS_POW_FIX function can only take an integer base,
	 * we need to normalize the result by gamma_normal_x
	 */
	gamma_normal_r_24i_8f =  OS_POW_FIX(255, (1<<16)/new_gamma_red_24i_8f);
	gamma_normal_g_24i_8f =  OS_POW_FIX(255, (1<<16)/new_gamma_green_24i_8f);
	gamma_normal_b_24i_8f =  OS_POW_FIX(255, (1<<16)/new_gamma_blue_24i_8f);

	for( i = 0; i < OVL_TOTAL_GAMMA_REG; i++ )
	{
		/* red */
		gamma_reg_24i_8f = OS_POW_FIX(gamma_reg_input[i],
								(1<<16)/new_gamma_red_24i_8f);
		gamma_reg        =
			((255 * gamma_reg_24i_8f) / gamma_normal_r_24i_8f) << 16;

		/* green */
		gamma_reg_24i_8f = OS_POW_FIX(gamma_reg_input[i],
							(1<<16)/new_gamma_green_24i_8f);
		gamma_reg        |=
			((255 * gamma_reg_24i_8f) / gamma_normal_g_24i_8f) << 8;

		/* blue */
		gamma_reg_24i_8f = OS_POW_FIX(gamma_reg_input[i],
							(1<<16)/new_gamma_blue_24i_8f);
		gamma_reg        |=
			((255 * gamma_reg_24i_8f) / gamma_normal_b_24i_8f);

		/* turn overlay off (TBD) */

		/* program register */
		EMGD_WRITE32(gamma_reg, MMIO(display) + gamma_reg_offset[i]);

		/* turn overlay on (TBD) */
	}

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}

/*----------------------------------------------------------------------
 * Function: ovl_update_coeff_tnc()
 * Description: Function to calculate the filter coeffcient
 *              registers for tnc
 * Notes in Usage:
 *
 *----------------------------------------------------------------------*/
static unsigned int ovl_update_coeff_tnc(
	ovl_reg_image_tnc_t	*ovl_regs_tnc,
	igd_surface_t       *src_surf,
	igd_rect_t          *src_rect,
	igd_rect_t          *dest_rect,
	unsigned int        flags)
{
	unsigned int scale_int, scale_fpint;

	unsigned int dest_h = dest_rect->y2 - dest_rect->y1;
	unsigned int dest_w = dest_rect->x2 - dest_rect->x1;
	unsigned int src_h  = src_rect->y2  - src_rect->y1;
	unsigned int src_w  = src_rect->x2  - src_rect->x1;

	EMGD_TRACE_ENTER;

	/* FIXME: The coeff values are re-written for every alter_ovl call.
	 * This may cause issues or may be to slow?  If so, store the previous
	 * values and only re-write when they change. */

        /*
	ovl_regs_tnc =
		(ovl_reg_image_tnc_t *)(display->context->device_context.virt_fb_adr +
			ovl_context->reg_update_offset);
        ovl_regs_tnc = phys_to_virt(ovl_context->reg_update_phys);
        */

	/* In interleaved mode, the src_h is /2 */
	if (flags & IGD_OVL_ALTER_INTERLEAVED) {
		src_h >>= 1;
	}

	/* Y Horizontal */
	scale_int = ((ovl_regs_tnc->yrgb_scale) >> 16) & 0x7;
	if (!scale_int) {
		/* upscale - clamp to 1.0 */
		scale_fpint = 1<<20;
	} else {
		scale_fpint = ((src_w << 20) / dest_w) / scale_int;
	}
	ovl_update_coeff_regs(5, scale_fpint, 1, 1,
		(unsigned short *)ovl_regs_tnc->y_horz_coeff_single);

	/* Y Vertical */
	scale_int = ((ovl_regs_tnc->vert_downscale) >> 16) & 0x7ff;
	if (!scale_int) {
		/* upscale - clamp to 1.0 */
		scale_fpint = 1<<20;
	} else {
		scale_fpint = ((src_h << 20) / dest_h) / scale_int;
	}
	ovl_update_coeff_regs(3, scale_fpint, 0, 1,
		(unsigned short *)ovl_regs_tnc->y_vert_coeff_single);

	/* UV Horizontal */
	scale_int = ((ovl_regs_tnc->uv_scale) >> 16) & 0x7;
	if (!scale_int) {
		/* upscale - clamp to 1.0 */
		scale_fpint = 1<<20;
	} else {
		scale_fpint = ((src_w << 20) / dest_w) / scale_int;
		scale_fpint >>= get_uv_shift_x(src_surf->pixel_format);
	}
	ovl_update_coeff_regs(3, scale_fpint , 1, 0,
		(unsigned short *)ovl_regs_tnc->uv_horz_coeff_single);

	/* UV Vertical */
	scale_int = (ovl_regs_tnc->vert_downscale) & 0x7ff;
	if (!scale_int) {
		/* upscale - clamp to 1.0 */
		scale_fpint = 1<<20;
	} else {
		scale_fpint = ((src_h << 20) / dest_h) / scale_int;
		scale_fpint >>= get_uv_shift_y(src_surf->pixel_format);
	}
	ovl_update_coeff_regs(3, scale_fpint, 0, 0,
		(unsigned short *)ovl_regs_tnc->uv_vert_coeff_single);

	/* Adjust for 2-line Vertical Buffer */
	if((ovl_regs_tnc->config & OVL_CONFIG_LINE_BUFF_MASK)==
		OVL_CONFIG_TWO_LINE_BUFF){
		ovl_update_coeff_regs(2, 0x10, 0, 1,
			(unsigned short *)ovl_regs_tnc->y_vert_coeff_single);
		ovl_update_coeff_regs(2, 0x10, 0, 0,
			(unsigned short *)ovl_regs_tnc->uv_vert_coeff_single);
	}

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}

static unsigned int convert_color_key_to_hw (
	unsigned long pf,
	unsigned int input)
{
	unsigned int output;

	switch (pf) {
	case IGD_PF_ARGB32:
	case IGD_PF_xRGB32:
	case IGD_PF_ARGB8_INDEXED:
	default:
		output = input;
		break;
	case IGD_PF_RGB16_565:
		output =
			((((input & 0xf800)>>11)<<3)<<16) |
			((((input & 0x07e0)>>5 )<<2)<<8 ) |
			((((input & 0x001f)>>0 )<<3)<<0 );
		break;
	case IGD_PF_ARGB16_1555:
		output =
			((((input & 0x7c00)>>10)<<3)<<16) |
			((((input & 0x03e0)>>5 )<<3)<<8 ) |
			((((input & 0x001f)>>0 )<<3)<<0 );
		break;
	}

	return output;
}
static unsigned int convert_color_key_to_mask (
	unsigned long pf,
	unsigned int input)
{
	unsigned int output;

	switch (pf) {
	case IGD_PF_ARGB32:
	case IGD_PF_xRGB32:
	default:
		output = 0x00000000;
		break;
	case IGD_PF_RGB16_565:
		output = 0x00070307;
		break;
	case IGD_PF_ARGB16_1555:
		output = 0x00070707;
		break;
	case IGD_PF_ARGB8_INDEXED:
		output = 0x00ffff00;
		break;
	}

	return output;
}


#ifndef OVL_TNC_CACHE_QUICK_SWAP

static unsigned int ovl_update_regs_tnc(
	igd_display_context_t *display,
	igd_surface_t       *src_surf,
	igd_rect_t          *src_rect,
	igd_rect_t          *dest_rect,
	igd_ovl_info_t      *ovl_info,
	unsigned int         flags)
{
	ovl_reg_image_tnc_t *ovl_regs_tnc, *ovl_cache_tnc;
	int ret;

	EMGD_TRACE_ENTER;

        /*
	ovl_regs_tnc =
		(ovl_reg_image_tnc_t *)(display->context->device_context.virt_fb_adr +
			ovl_context->reg_update_offset);
        */
	ovl_regs_tnc = phys_to_virt(ovl_context->reg_update_phys);
	ovl_cache_tnc = OS_ALLOC(sizeof(ovl_reg_image_tnc_t));
	OS_MEMSET(ovl_cache_tnc, 0, sizeof(ovl_reg_image_tnc_t));

	if ((flags & IGD_OVL_ALTER_ON) == IGD_OVL_ALTER_OFF) {
		/* Turn the overlay Off */
		ovl_regs_tnc->command = 0;
		/* Always use buf 0 when turning the overlay off. */
		ovl_context->ovl_buff = 0;
		OS_FREE(ovl_cache_tnc);
		EMGD_TRACE_EXIT;
		return IGD_SUCCESS;
	}

	/* Force value to even due hardware expects even number */
	dest_rect->y1 &= ~1;
	dest_rect->y2 = (dest_rect->y2 + 1) & ~1;
	dest_rect->x1 &= ~1;
	dest_rect->x2 = (dest_rect->x2 + 1) & ~1;

	/*************************************************************************
	 * Copy the information passed in to the HW overlay structure
	 *************************************************************************/
	/* Zero the config and command, since they will be OR'ed in with data
	 * below */
	ovl_cache_tnc->config = 0;
	ovl_cache_tnc->command = 0;

	/* Set overlay to the proper pipe */
	if (1 == PIPE(display)->pipe_num) {
		/* This is pipe B */
		ovl_cache_tnc->config |= 1 << 18;
	}

	/* Interleaved/progressive and Odd/Even if interleaved */
	if (flags & IGD_OVL_ALTER_INTERLEAVED) {
		ovl_cache_tnc->command |= OVL_CMD_FIELD_MODE;
		/* Need to enable FIELD SYNC OVERLAY FLIP in field mode. */
		ovl_cache_tnc->command |= OVL_CMD_FIELD_SYNC_FLIP;
		if (flags & IGD_OVL_ALTER_FLIP_ODD) {
			ovl_cache_tnc->command |= OVL_CMD_ACT_FLD1;
		} else {
			ovl_cache_tnc->command |= OVL_CMD_ACT_FLD0;
		}
	} else {
		ovl_cache_tnc->command |= OVL_CMD_FRAME_MODE;
	}

	/* Dest rect information */
	ovl_cache_tnc->dest_pos_x_left        = (unsigned short)dest_rect->x1;
	ovl_cache_tnc->dest_pos_y_top         = (unsigned short)dest_rect->y1;
	ovl_cache_tnc->dest_width_x           =
		(unsigned short)(dest_rect->x2 - dest_rect->x1);
	ovl_cache_tnc->dest_height_y          =
		(unsigned short)(dest_rect->y2 - dest_rect->y1);

	/* Src surface offset information */
	ret = ovl_update_src_ptr_tnc(display, ovl_cache_tnc, src_surf, src_rect);
	if (ret) {
		OS_FREE(ovl_cache_tnc);
		EMGD_ERROR_EXIT("Overlay updating src failed");
		return ret;
	}

	/* Src rect and surface information */
	ret = ovl_update_src_tnc(display, ovl_cache_tnc, src_surf, src_rect);
	if (ret) {
		OS_FREE(ovl_cache_tnc);
		EMGD_ERROR_EXIT("Overlay updating src failed");
		return ret;
	}

	/* Scaling information including Vertical downscaling.
	 * Scaling should be guaranteed to work, since if the scale is not
	 * supported, it should have already been blended to a supported scale. */
	ret = ovl_update_scale_tnc(ovl_cache_tnc, src_surf, src_rect, dest_rect,
		flags);
	if (ret) {
		OS_FREE(ovl_cache_tnc);
		EMGD_ERROR_EXIT("Overlay updating scaling failed");
		return ret;
	}

	/* Color control information */
	ret = ovl_update_video_quality_tnc(ovl_cache_tnc, src_surf,
		&ovl_info->video_quality);
	if (ret) {
		OS_FREE(ovl_cache_tnc);
		EMGD_ERROR_EXIT("Overlay video quality failed");
		return ret;
	}
	ret = ovl_update_gamma_tnc(display, &ovl_info->gamma);
	if (ret) {
		OS_FREE(ovl_cache_tnc);
		EMGD_ERROR_EXIT("Overlay gamma failed");
		return ret;
	}

	/* Destination color key */
	EMGD_DEBUG("Color key.flags: 0x%lx", ovl_info->color_key.flags);
	if (ovl_info->color_key.flags & IGD_OVL_DST_COLOR_KEY_ENABLE) {
		EMGD_DEBUG("Overlay Enable Dest Color Key");
		/* The mask and color key are different for the different
		 * pixel formats */
		ovl_cache_tnc->dest_ckey_val = convert_color_key_to_hw(
			PLANE(display)->fb_info->pixel_format,
			ovl_info->color_key.dest);
		ovl_cache_tnc->dest_ckey_mask = convert_color_key_to_mask(
			PLANE(display)->fb_info->pixel_format,
			ovl_info->color_key.dest);
		ovl_cache_tnc->dest_ckey_mask |= 0x80000000;
	} else {
		EMGD_DEBUG("Overlay Disable Dest Color Key");
		ovl_cache_tnc->dest_ckey_mask = 0x00000000;
	}

	/* Source Color key */
	if (ovl_info->color_key.flags & IGD_OVL_SRC_COLOR_KEY_ENABLE) {
		EMGD_DEBUG("Overlay Enable Src Color Key");
		ovl_cache_tnc->source_ckey_high = ovl_info->color_key.src_hi;
		ovl_cache_tnc->source_ckey_low = ovl_info->color_key.src_lo;

		ovl_cache_tnc->source_ckey_mask = 0x07000000;
	} else {
		EMGD_DEBUG("Overlay Disable Src Color Key");
		ovl_cache_tnc->source_ckey_mask = 0x00000000;
	}

	/* Coefficients - Must be after Scaling */
//	ret = ovl_update_coeff_tnc(display, src_surf, src_rect, dest_rect, flags);
	ret = ovl_update_coeff_tnc(ovl_cache_tnc, src_surf, src_rect, dest_rect, flags);
	if (ret) {
		OS_FREE(ovl_cache_tnc);
		EMGD_ERROR_EXIT("Overlay updating coefficient failed");
		return ret;
	}

	/* Phase information - Must be after Coefficients */
	ret = ovl_update_phase_tnc(ovl_cache_tnc, src_surf, src_rect);
	if (ret) {
		OS_FREE(ovl_cache_tnc);
		EMGD_ERROR_EXIT("Overlay updating phase failed");
		return ret;
	}

	/* General overlay information.  Turn the overlay on and alternate
	 * between Buffer 0 and Buffer 1. */
	ovl_cache_tnc->command = (ovl_cache_tnc->command & 0xfffffff3) |
		ovl_context->ovl_buff | 1;
	ovl_context->ovl_buff ^= OVL_CMD_ACT_BUF1;

	/* Dump out the Overlay Update Registers if debugging */
	EMGD_VERBOSE(hal.dump_overlay_regs, ovl_dump_regs_tnc(ovl_regs_tnc));

	OS_MEMCPY(ovl_regs_tnc, ovl_cache_tnc, sizeof(ovl_reg_image_tnc_t));
	OS_FREE(ovl_cache_tnc);


	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}

#else /* new version */


/* Atom E6xx cache structure */
static ovl_tnc_cache_t ovl_cache;

/* Flag to signal the cache is invalid and needs
 * to be re-initialized */
static int ovl_cache_needs_init = TRUE;




static unsigned int ovl_update_regs_tnc(
	igd_display_context_t *display,
	igd_surface_t       *src_surf,
	igd_rect_t          *src_rect,
	igd_rect_t          *dest_rect,
	igd_ovl_info_t      *ovl_info,
	unsigned int         flags)
{
	ovl_reg_image_tnc_t *ovl_regs_tnc, *ovl_cache_regs;
	unsigned int cache_changed;
	int ret;

	EMGD_TRACE_ENTER;

	/* get the pointers to the real regs, and our cached copy of them */
	ovl_regs_tnc = phys_to_virt(ovl_context->reg_update_phys);
	ovl_cache_regs = &ovl_cache.ovl_regs;

	/* Fast path for turning off overlay. No need for cache */
	if ((flags & IGD_OVL_ALTER_ON) == IGD_OVL_ALTER_OFF) {

		/* Turn the overlay Off */
		ovl_regs_tnc->command = 0;

		/* if we were using the cache, turn it off there too */
		if (!ovl_cache_needs_init) {
			ovl_cache.ovl_regs.command = 0;
		}

		/* Reset the cache */
		ovl_cache_needs_init = TRUE;

		/* Always use buf 0 when turning the overlay off. */
		ovl_context->ovl_buff = 0;
		EMGD_TRACE_EXIT;
		return IGD_SUCCESS;
	}

	/* Init the cache if necessary */
	if (ovl_cache_needs_init) {
		/* Force every cache check to miss */
		OS_MEMSET(&ovl_cache, 0, sizeof(ovl_tnc_cache_t));

		/* We just set our cached flags to 0, which might accidently
		 * match up with "OFF" for some important incoming flag
		 * bits, causing us to think we already handled them when
		 * we didn't.  So set our cached flags to the exact
		 * opposite of the incoming flags, which will force
		 * us to test and handle every single bit, regardless
		 * of whether it is on or off. */
		ovl_cache.flags = ~flags;

		/* init our cached registers */
		OS_MEMCPY(ovl_cache_regs,
			  ovl_regs_tnc,
			  sizeof(ovl_reg_image_tnc_t));

		/* initialization complete */
		ovl_cache_needs_init = FALSE;
	}

	/* See what has changed in the cache */
	cache_changed = get_cache_changes_tnc(src_surf,
		src_rect,
		dest_rect,
		ovl_info,
		flags,
		&ovl_cache);

	/* Perhaps the biggest challenge of caching the overlay
	 * state is what to do with the command and config regs.
	 * Normally we would clear command and config to 0 here,
	 * and let the update process set only the bits that are
	 * needed.  But doing this would invalidate our cache.
	 * Instead we are relying on the above call to
	 * get_cache_changes() to clear those bits in command
	 * and config that will be changing */

	/* Set overlay to the proper pipe */
	/* it is cheaper to just set this, than to test it and set it. */
	if (1 == PIPE(display)->pipe_num) {
		/* This is pipe B */
		ovl_cache_regs->config |= 1 << 18;
	} else {
		ovl_cache_regs->config &= ~(1 << 18);
	}

	if (cache_changed & IGD_OVL_TNC_UPDATE_FLAGS) {
		/* Interleaved/progressive and Odd/Even if interleaved. */
		if (flags & IGD_OVL_ALTER_INTERLEAVED) {
			ovl_cache_regs->command |= OVL_CMD_FIELD_MODE;
			/* enable FIELD SYNC OVERLAY FLIP in field mode. */
			ovl_cache_regs->command |= OVL_CMD_FIELD_SYNC_FLIP;
			if (flags & IGD_OVL_ALTER_FLIP_ODD) {
				ovl_cache_regs->command |= OVL_CMD_ACT_FLD1;
			} else {
				ovl_cache_regs->command |= OVL_CMD_ACT_FLD0;
			}
		} else {
			ovl_cache_regs->command |= OVL_CMD_FRAME_MODE;
		}
	}

	/* Has our destination rectangle changed? */
	if (cache_changed & IGD_OVL_TNC_UPDATE_DEST) {
		ovl_cache_regs->dest_pos_x_left  =
			(unsigned short) dest_rect->x1;
		ovl_cache_regs->dest_pos_y_top   =
			(unsigned short) dest_rect->y1;
		ovl_cache_regs->dest_width_x     =
			(unsigned short) (dest_rect->x2 - dest_rect->x1);
		ovl_cache_regs->dest_height_y    =
			(unsigned short) (dest_rect->y2 - dest_rect->y1);
	}

	/* Always update the source pointers every frame */
	ret = ovl_update_src_ptr_tnc(display, ovl_cache_regs, src_surf, src_rect);
	if (ret) {
		/* Not good. Invalidate the entire cache and bail. */
		ovl_cache_needs_init = TRUE;
		EMGD_ERROR_EXIT("Overlay updating src pointers failed");
		return ret;
	}

	/* Did either the Src rect or surface change? */
	if (cache_changed & (IGD_OVL_TNC_UPDATE_SURF | IGD_OVL_TNC_UPDATE_SRC)) {
		ret = ovl_update_src_tnc(display, ovl_cache_regs, src_surf, src_rect);
		if (ret) {
			/* Not good. Invalidate the entire cache and bail. */
			ovl_cache_needs_init = TRUE;
			EMGD_ERROR_EXIT("Overlay updating src failed");
			return ret;
		}
	}

	/* Scaling information including Vertical downscaling.
	 * Scaling should be guaranteed to work, since if the scale
	 * is not supported, it should have already been blended
	 * to a supported scale. */
	if ( cache_changed & (IGD_OVL_TNC_UPDATE_SRC |
		IGD_OVL_TNC_UPDATE_SURF | IGD_OVL_TNC_UPDATE_DEST |
		IGD_OVL_TNC_UPDATE_FLAGS) ) {

		ret = ovl_update_scale_tnc(ovl_cache_regs, src_surf, src_rect,
			dest_rect, flags);
		if (ret) {
			/* Not good. Invalidate the entire cache and bail. */
			ovl_cache_needs_init = TRUE;
			EMGD_ERROR_EXIT("Overlay updating scaling failed");
			return ret;
		}
	}

	/* Did video quality change? */
	if (cache_changed & (IGD_OVL_TNC_UPDATE_VQ |
		IGD_OVL_TNC_UPDATE_SURF ) ) {

		/* Color control information */
		ret = ovl_update_video_quality_tnc(ovl_cache_regs, src_surf,
			&ovl_info->video_quality);

		if (ret) {
			/* Not good. Invalidate the entire cache and bail. */
			ovl_cache_needs_init = TRUE;
			EMGD_ERROR_EXIT("Overlay video quality failed");
			return ret;
		}
	}

	/* Did gamma change? */
	if (cache_changed & IGD_OVL_TNC_UPDATE_GAMMA) {
		ret = ovl_update_gamma_tnc(display, &ovl_info->gamma);
		if (ret) {
			/* Not good. Invalidate the entire cache and bail. */
			ovl_cache_needs_init = TRUE;
			EMGD_ERROR_EXIT("Overlay gamma failed");
			return ret;
		}
	}

	/* Did color key change? */
	if (cache_changed & IGD_OVL_TNC_UPDATE_COLORKEY) {
		/* Destination color key */
		if (ovl_info->color_key.flags & IGD_OVL_DST_COLOR_KEY_ENABLE) {
			/* The mask and color key are different for the
			 * different pixel formats */
			ovl_cache_regs->dest_ckey_val =
				convert_color_key_to_hw(
				    PLANE(display)->fb_info->pixel_format,
				    ovl_info->color_key.dest);
			ovl_cache_regs->dest_ckey_mask =
				convert_color_key_to_mask(
				    PLANE(display)->fb_info->pixel_format,
				    ovl_info->color_key.dest);
			ovl_cache_regs->dest_ckey_mask |= 0x80000000;
		} else {
			ovl_cache_regs->dest_ckey_mask = 0x00000000;
		}

		/* Source Color key */
		if (ovl_info->color_key.flags & IGD_OVL_SRC_COLOR_KEY_ENABLE) {
			ovl_cache_regs->source_ckey_high = ovl_info->color_key.src_hi;
			ovl_cache_regs->source_ckey_low = ovl_info->color_key.src_lo;
			ovl_cache_regs->source_ckey_mask = 0x07000000;
		} else {
			ovl_cache_regs->source_ckey_mask = 0x00000000;
		}

		/* Constant Alpha */
		if (ovl_info->color_key.flags & IGD_OVL_CONST_ALPHA_ENABLE) {
			/* An active change to the alpha setting is required */
			if (ovl_info->alpha.enable) {
				/* Set the alpha value and the enable bit */
				ovl_cache_regs->source_ckey_mask |=
					ovl_info->alpha.value & 0x000000FF;
				ovl_cache_regs->dest_ckey_mask |= OVL_CONST_ALPHA_ENA;
			} else {
				/* Clear the alpha value and the enable bit */
				ovl_cache_regs->source_ckey_mask &= 0xFFFFFF00;
				ovl_cache_regs->dest_ckey_mask &= ~OVL_CONST_ALPHA_ENA;
			}
		}
	} /* end color key changes */

	/* Coefficients - Must be after Scaling */
	if (cache_changed & (IGD_OVL_TNC_UPDATE_SRC |
		IGD_OVL_TNC_UPDATE_SURF | IGD_OVL_TNC_UPDATE_DEST |
		IGD_OVL_TNC_UPDATE_FLAGS ) ) {

		ret = ovl_update_coeff_tnc(ovl_cache_regs, src_surf, src_rect,
			dest_rect, flags);
		if (ret) {
			/* Not good. Invalidate entire cache and bail. */
			ovl_cache_needs_init = TRUE;
			EMGD_ERROR_EXIT("Overlay update coefficient failed");
			return ret;
		}
	}

	/* Phase information - Must be after Coefficients */
	if (cache_changed & (IGD_OVL_TNC_UPDATE_SRC |
		IGD_OVL_TNC_UPDATE_SURF ) ) {

		ret = ovl_update_phase_tnc(ovl_cache_regs, src_surf, src_rect);
		if (ret) {
			/* Not good. Invalidate entire cache and bail. */
			ovl_cache_needs_init = TRUE;
			EMGD_ERROR_EXIT("Overlay updating phase failed");
			return ret;
		}
	}

	/* General overlay information.  Turn the overlay on and alternate
	 * between Buffer 0 and Buffer 1. */
	ovl_cache_regs->command = (ovl_cache_regs->command & 0xfffffff3) |
		ovl_context->ovl_buff | 1;
	ovl_context->ovl_buff ^= OVL_CMD_ACT_BUF1;

	/* Finally, transfer the cached regs to the real regs */
	OS_MEMCPY(ovl_regs_tnc, ovl_cache_regs, sizeof(ovl_reg_image_tnc_t));

	/* Dump out the Overlay Update Registers if debugging */
	EMGD_VERBOSE(hal.dump_overlay_regs, ovl_dump_regs_tnc(ovl_regs_tnc));

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;

}
#endif

static unsigned int ovl_send_instr_tnc(
	igd_display_context_t *display,
	unsigned int      flags)
{
	unsigned char * mmio = MMIO(display);
	unsigned long tmp;

	EMGD_TRACE_ENTER;

	/* We dont need the CMD_WAIT_OVL_TNC instruction coz
	 * our alter_ovl code already querried status
	 * for last flip completion before getting here. See
	 * alter_ovl_tnc calling query
	 */

	if ((flags & IGD_OVL_ALTER_ON) == IGD_OVL_ALTER_ON) {
		ovl_context->state = OVL_STATE_ON;
		/*
		 * If Overlay + FB Blend is requested and the FB is xRGB
		 * turn on the ARGB format.
		 */
		if(ovl_context->fb_blend_ovl) {
			tmp = EMGD_READ32(mmio + PLANE(display)->plane_reg);
			if((tmp & 0x3c000000) == 0x18000000) {
				tmp = tmp & 0xc3FFFFFF;
				EMGD_WRITE32(tmp | 0x1c000000, mmio + PLANE(display)->plane_reg);
				EMGD_READ32(mmio + PLANE(display)->plane_reg);
				tmp = EMGD_READ32(mmio + PLANE(display)->plane_reg + 0x1c);
				EMGD_WRITE32(tmp, mmio + PLANE(display)->plane_reg + 0x1c);
			}
		}

	} else {
		if(ovl_context->fb_blend_ovl) {
			tmp = EMGD_READ32(mmio +  PLANE(display)->plane_reg);
			if((tmp & 0x3c000000) == 0x1c000000) {
				tmp = tmp & 0xc3FFFFFF;
				EMGD_WRITE32(tmp | 0x18000000, mmio +  PLANE(display)->plane_reg);
				EMGD_READ32(mmio + PLANE(display)->plane_reg);
				tmp = EMGD_READ32(mmio + PLANE(display)->plane_reg + 0x1c);
				EMGD_WRITE32(tmp, mmio + PLANE(display)->plane_reg + 0x1c);
				OS_SLEEP(100);
			}
		}
		OS_SLEEP(1);

		/* if overlay is being turned OFF - ensure it's ON first */
		if (ovl_context->state == OVL_STATE_OFF) {
			/* Overlay is already off, no need to turn it off again */
			EMGD_TRACE_EXIT;
			return IGD_SUCCESS;
		}
		ovl_context->state = OVL_STATE_OFF;
	}

	/* Write the address of the memory buffer to the Overlay Update
	 * Address Register causes the HW to load the new values from the
	 * memory on the next VBLANK */

	EMGD_WRITE32((ovl_context->reg_update_offset | 0x01), mmio + 0x30000);
	ovl_context->sync = 0;

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}

static int alter_ovl_tnc(igd_display_context_t *display,
	igd_surface_t       *src_surf,
	igd_rect_t          *src_rect,
	igd_rect_t          *dest_rect,
	igd_ovl_info_t      *ovl_info,
	unsigned int         flags)
{
	int ret, query_mode;

	EMGD_TRACE_ENTER;

	/* Check to ensure the overlay can be used given the current mode as
	 * well as what the IAL is asking for.  If not return an error. */
	ret = ovl_check_tnc(display, src_surf, src_rect, dest_rect, ovl_info,
			flags);
	if (ret) {
		EMGD_ERROR_EXIT("Overlay Check failed");
		return ret;
	}

	/* Check if last flip is still pending.
	 * This is necessary for the following reasons:
	 *    - If the previous instructions have not been processed, then the
	 *      ovl_regs_tnc is still in use and can not be overwritten.
	 */
	query_mode = (flags & IGD_OVL_ALTER_DROP_IF_BUSY
		      ? IGD_OVL_QUERY_IS_LAST_FLIP_DONE
		      : IGD_OVL_QUERY_WAIT_LAST_FLIP_DONE);
	if (!query_ovl_tnc((igd_display_h)display, query_mode)) {
		if (flags & IGD_OVL_ALTER_DROP_IF_BUSY) {
			/* The HW isn't ready; wait up to 250 usec before dropping a frame */
			volatile int i, j;
			int retry = 0;
			while (1) {
				/*delay ~1 usec, then query the HW again*/
				for (i = 0; i < 1; i++) {
					for(j = 0; j < 1000; j++);
				}
				if (query_ovl_tnc((igd_display_h)display, query_mode)) {
					/*HW is ready now, so we can continue with the display*/
					break;
				} else if (++retry == 250) {
					/*skip this frame so we don't miss the next interval*/
					return -IGD_ERROR_BUSY;
				}
			}
		} else if (flags & IGD_OVL_ALTER_ON) {
			/* Only return an error if the overlay is on.
			 * If turning it off, allow it to continue,
			 * since something may have failed and we
			 * should try our best to turn the overlay
			 * off. */
			EMGD_ERROR_EXIT("Query Overlay failed");
			return -IGD_ERROR_HWERROR;
		}
	}

	/* Update all Overlay Update Registers */
	ret = ovl_update_regs_tnc(display, src_surf, src_rect, dest_rect, ovl_info,
		flags);
	if (ret) {
		EMGD_ERROR_EXIT("Overlay Update Registers failed");
		return ret;
	}

	/* Send the instructions to the command queue */
	ret = ovl_send_instr_tnc(display, flags);

	EMGD_TRACE_EXIT;
	return ret;
}



static int query_ovl_tnc(igd_display_h display_h,
	unsigned int flags)
{
	igd_display_context_t *display = (igd_display_context_t *)display_h;
	os_alarm_t timeout;

	EMGD_TRACE_ENTER;

	switch (flags) {
	case IGD_OVL_QUERY_IS_HW_SUPPORTED:
		/* This is the first overlay, so HW overlay is supported */
		break;
	case IGD_OVL_QUERY_IS_LAST_FLIP_DONE:
		/* Check to see if the register update is complete.  If not return
		 * FALSE (Flip not done). */
		if(!(EMGD_READ32(MMIO(display) + 0x30008) & 0x80000000)) {
			/* This is not an error */
			return FALSE;
		}
		/* Now that we know the last flip is done and the register update is
		 * complete, set the sync to 0 and return TRUE (Flip done). */
		ovl_context->sync = 0;
		break;
	case IGD_OVL_QUERY_WAIT_LAST_FLIP_DONE:
		/* Wait for 200 milliseconds for the last flip to complete.  If not
		 * done in that time, there is likely a hardware problem so return
		 * FALSE. */
		timeout = OS_SET_ALARM(200);
		do {
			if (TRUE ==
				query_ovl_tnc(display_h, IGD_OVL_QUERY_IS_LAST_FLIP_DONE)) {
				EMGD_TRACE_EXIT;
				return TRUE;
			}
		} while (!OS_TEST_ALARM(timeout));
		EMGD_ERROR_EXIT("Timeout waiting for last flip done");
		return FALSE;
		break;
	case IGD_OVL_QUERY_IS_GAMMA_SUPPORTED:
		return TRUE;
		break;
	case IGD_OVL_QUERY_IS_VIDEO_PARAM_SUPPORTED:
		return TRUE;
		break;
	}

	EMGD_TRACE_EXIT;
	return TRUE;
}

static int query_max_size_ovl_tnc(
	igd_display_h display_h,
	unsigned long pf,
	unsigned int *max_width,
	unsigned int *max_height)
{
	ovl_chipset_tnc_t *ovl_chip;

	EMGD_TRACE_ENTER;

	ovl_chip = ovl_chipset_tnc;
	*max_width = 0;
	*max_height = 0;
	while(ovl_chip->num_linebuf != OVL_CONFIG_NO_LINE_BUFF){
		if(((pf & IGD_PF_MASK) == ovl_chip->pixel_format) &&
			(ovl_chip->max_width > *max_width)) {
			*max_width = ovl_chip->max_width;
			*max_height = ovl_chip->max_height;
		}
		ovl_chip++;
	}

	EMGD_TRACE_EXIT;
	return IGD_SUCCESS;
}


/*
 * enable_direct_display_tnc(): Enables direct display of video DMA input
 * buffers on the Overlay or Sprite C plane of the specified screen.
 */
int enable_direct_display_tnc(
	void *arg,
	igd_dd_context_t dd_context)
{
	igd_display_context_t *display = (igd_display_context_t *)arg;
	dd_context_tnc_t *dd_context_ptr;
	int ret;

	EMGD_TRACE_ENTER;

	/* Tell the EMGDHMI-specific buffer flip/sync management that we are using it. */
	emgdhmi_enable();

	/* Ensure the display context has been set */
	if (display == NULL) {
		EMGD_ERROR_EXIT("display context (arg) cannot be NULL !\n");
		return -EINVAL;
	}
	/* Ensure the igd context has been set */
	if (display->context == NULL) {
		EMGD_DEBUG("igd context cannot be NULL !\n");
		return -EINVAL;
	}
	switch (dd_context.usage) {
	case IGD_PLANE_OVERLAY_VIDEO:
		/* Check to see if direct display on Overlay plane is available */
		if (dd_context_ovl) {
			EMGD_ERROR_EXIT("Overlay direct display already enabled !\n");
			return -EBUSY;
		}
		break;
	case IGD_PLANE_SPRITE_VIDEO:
		/* Check to see if direct display on Sprite C plane is available */
		if (dd_context_ovl2) {
			EMGD_ERROR_EXIT("Overlay direct display already enabled !\n");
			return -EBUSY;
		}
		break;
	default:
		EMGD_ERROR_EXIT("Invalid direct display usage type (%d) !\n",
				dd_context.usage);
		return -EINVAL;
	}
	/* Attempt to allocate the direct display context block */
	dd_context_ptr = OS_ALLOC(sizeof(dd_context_tnc_t));
	if (dd_context_ptr == NULL) {
		EMGD_ERROR_EXIT("Unable to allocate DD context block !\n");
		return -ENOMEM;
	}
	/* Clear the DD context block and then copy the bridge data into it */
	OS_MEMSET(dd_context_ptr, 0, sizeof(dd_context_tnc_t));
	dd_context_ptr->bridge = dd_context;

	/* Initialize color keying, video quality, and gamma */
	dd_context_ptr->ovl.color_key.src_lo = 0x00000000;
	dd_context_ptr->ovl.color_key.src_hi = 0x00000000;
	dd_context_ptr->ovl.color_key.dest = 0xFF00FF; /* customer specified */
	dd_context_ptr->ovl.color_key.flags = IGD_OVL_DST_COLOR_KEY_ENABLE
						| IGD_OVL_CONST_ALPHA_ENABLE;
	dd_context_ptr->ovl.alpha.enable = 0; /* This disables Constant Alpha */
	dd_context_ptr->ovl.alpha.value = 0;
	dd_context_ptr->ovl.video_quality.contrast = 0x8000;
	dd_context_ptr->ovl.video_quality.brightness = 0x8000;
	dd_context_ptr->ovl.video_quality.saturation = 0x8000;
	dd_context_ptr->ovl.gamma.red = 0x100;
	dd_context_ptr->ovl.gamma.green = 0x100;
	dd_context_ptr->ovl.gamma.blue = 0x100;
	dd_context_ptr->ovl.gamma.flags = IGD_OVL_GAMMA_ENABLE;

	/* Initialize the generic rendering surface attributes */
	dd_context_ptr->surf.width = dd_context_ptr->bridge.src.x2 -
		dd_context_ptr->bridge.src.x1;
	dd_context_ptr->surf.height = dd_context_ptr->bridge.src.y2 -
		dd_context_ptr->bridge.src.y1;
	dd_context_ptr->surf.flags = IGD_OVL_ALTER_ON;
	dd_context_ptr->surf.pitch = dd_context.video_w * 2;

	if (dd_context.usage == IGD_PLANE_OVERLAY_VIDEO) {
		/* Initialize the YUV422-specific surface attributes */
		dd_context_ptr->surf.pixel_format = IGD_PF_YUV422_PACKED_UYVY;

		/* Save this pointer as our Overlay context */
		dd_context_ovl = dd_context_ptr;
	} else {
		/* Immediately disable Constant Alpha on Sprite C */
		ret = ovl2_update_constant_alpha_tnc(display, 0, 0);

		/* Initialize the RGB565-specific surface attributes */
		dd_context_ptr->surf.pixel_format = IGD_PF_RGB16_565;

		/* Save this pointer as our Overlay 2 (Sprite C) context */
		dd_context_ovl2 = dd_context_ptr;
	}
	/* Save the display handle argument that we received */
	dd_context_ptr->dpy = display;

	EMGD_TRACE_EXIT;
	return 0;
}
EXPORT_SYMBOL(enable_direct_display_tnc);


/*
 * disable_direct_display_tnc(): Disables direct display of video on the
 * Overlay plane or Sprite C plane.
 */
int disable_direct_display_tnc(
	void *arg,
	int usage)
{
	igd_display_context_t *display = (igd_display_context_t *)arg;
	dd_context_tnc_t *dd_context_ptr;
	int ret;

	EMGD_TRACE_ENTER;

	/* Ensure the display context has been set */
	if (display == NULL) {
		EMGD_ERROR_EXIT("Display context (arg) cannot be NULL !\n");
		return -EINVAL;
	}
	/* Set the direct display context pointer according to the usage arg */
	dd_context_ptr =
		(usage == IGD_PLANE_OVERLAY_VIDEO) ? dd_context_ovl : dd_context_ovl2;

	/* Ensure this direct display context pointer is set */
	if (dd_context_ptr == NULL) {
		EMGD_ERROR_EXIT("%s direct display not currently enabled !\n",
			(usage == IGD_PLANE_OVERLAY_VIDEO) ? "Overlay" : "Sprite C");
		return -EINVAL;
	}
	/* Set the flag to turn Overlay/Sprite C off */
	dd_context_ptr->surf.flags = IGD_OVL_ALTER_OFF;

	/* Check the DD context being referenced */
	if (usage == IGD_PLANE_OVERLAY_VIDEO) {
		/* Disable the display of the Overlay plane */
		ret = alter_ovl_tnc(display, &dd_context_ovl->surf,
				&dd_context_ovl->bridge.src, &dd_context_ovl->bridge.dest,
				&dd_context_ovl->ovl, dd_context_ovl->surf.flags);

		/* Mark the Overlay direct display context as invalid */
		dd_context_ovl = NULL;
	} else {
		/* Disable the display of the Sprite C plane */
		ret = alter_ovl2_tnc(display, &dd_context_ovl2->surf,
				&dd_context_ovl2->bridge.src, &dd_context_ovl2->bridge.dest,
				&dd_context_ovl2->ovl, dd_context_ovl2->surf.flags);

		/* Mark the Sprite C direct display context as invalid */
		dd_context_ovl2 = NULL;
	}
	/* Deallocate the DD context block for the direct display plane */
	OS_FREE(dd_context_ptr);

	EMGD_TRACE_EXIT;
	return ret;
}
EXPORT_SYMBOL(disable_direct_display_tnc);

/*
 * direct_display_frame_tnc(): Performs the direct display of the new frame
 * of video data on the Overlay plane or Sprite C plane.
 */
int direct_display_frame_tnc(
	void *arg,
	int usage,
	unsigned long offset)
{
	igd_display_context_t *display = (igd_display_context_t *)arg;
	dd_context_tnc_t *dd_context_ptr;
	platform_context_tnc_t *platform;
	int ret;

	EMGD_TRACE_ENTER;

	/* Ensure the display context is set */
	if (!display) {
		EMGD_ERROR_EXIT("direct_display_frame_tnc: display context is NULL!\n");
		return -EINVAL;
	}
	/* Set the direct display context pointer according to the usage arg */
	dd_context_ptr = (usage == IGD_PLANE_OVERLAY_VIDEO)
		? dd_context_ovl : dd_context_ovl2;

	/* Ensure this direct display context pointer is set */
	if (!dd_context_ptr) {
		EMGD_ERROR_EXIT("direct_display_frame_tnc: %s DD context is NULL !\n",
			(usage == IGD_PLANE_OVERLAY_VIDEO) ? "Overlay" : "Sprite C");
		return -EINVAL;
	}
	/* Skip the frame if video not in the current buffer config or CB pending */
	if ((igd_buf_cfg[dd_context_ptr->bridge.screen][0].plane != usage) ||
		(display->context->device_context.dd_suspended)) {
		return IGD_SUCCESS;
	}
	/* Grab the mutex used to control the Z-stack changes */
	platform = (platform_context_tnc_t *)display->context->platform_context;
	ret = OS_PTHREAD_MUTEX_LOCK(&display->context->device_context.z_stack_mutex);

	/* Update the surface offset corresponding to this new frame */
	dd_context_ptr->surf.offset = offset;

	/* Obtain the current handle (it may have changed since video started) */
	display = dd_context_ptr->dpy;

	if (usage == IGD_PLANE_OVERLAY_VIDEO) {
		/* display the new video frame on the Overlay plane */
		int flags = dd_context_ovl->surf.flags;
		ret = alter_ovl_tnc(display, &dd_context_ovl->surf,
				&dd_context_ovl->bridge.src, &dd_context_ovl->bridge.dest,
				&dd_context_ovl->ovl, flags | IGD_OVL_ALTER_DROP_IF_BUSY);
	} else {
		/* display the new video frame on the Sprite C plane */
		int flags = dd_context_ovl2->surf.flags;
		ret = alter_ovl2_tnc(display, &dd_context_ovl2->surf,
				&dd_context_ovl2->bridge.src, &dd_context_ovl2->bridge.dest,
				&dd_context_ovl2->ovl, flags | IGD_OVL_ALTER_DROP_IF_BUSY);
	}
	OS_PTHREAD_MUTEX_UNLOCK(&display->context->device_context.z_stack_mutex);
	EMGD_TRACE_EXIT;
	return ret;
}
EXPORT_SYMBOL(direct_display_frame_tnc);


/*
 * create_popup_surface(): Initializes the "popup" context information for
 * a displayable surface on the Overlay plane or Sprite C plane.
 */
static int create_popup_surface(
	igd_display_context_t *ctx,
	int screen,
	igd_buffer_config_t buf)
{
	dd_context_tnc_t *p_context;
	int ret = 0;
	int ckey;

	EMGD_TRACE_ENTER;

	p_context = (buf.plane == IGD_PLANE_OVERLAY_POPUP)
		? &popup_context_ovl : &popup_context_ovl2;
	OS_MEMSET(p_context, 0, sizeof(popup_context_ovl));

	/* Setup the basic information for this pop-up surface */
	p_context->bridge.usage = buf.plane;
	p_context->bridge.screen = screen;

	/* Copy the src & dest rects directly from input buffer config */
	OS_MEMCPY(&p_context->bridge.src, &buf.src, sizeof(igd_rect_t));
	OS_MEMCPY(&p_context->bridge.dest, &buf.dest, sizeof(igd_rect_t));

	/* Initialize video quality and gamma */
	p_context->ovl.video_quality.contrast = 0x8000;
	p_context->ovl.video_quality.brightness = 0x8000;
	p_context->ovl.video_quality.saturation = 0x8000;
	p_context->ovl.gamma.red = 0x100;
	p_context->ovl.gamma.green = 0x100;
	p_context->ovl.gamma.blue = 0x100;
	p_context->ovl.gamma.flags = IGD_OVL_GAMMA_ENABLE;

	/* Initialize color keying */
	p_context->ovl.color_key.flags = IGD_OVL_DST_COLOR_KEY_DISABLE;
	p_context->ovl.color_key.dest = 0x000000;
	if (buf.ckey_ena) {
		p_context->ovl.color_key.flags |= IGD_OVL_SRC_COLOR_KEY_ENABLE;
		ckey = buf.ckey_val;
		/* Convert the RGB key value to GBR for Overlay surfaces */
		if (buf.plane == IGD_PLANE_OVERLAY_POPUP) {
			ckey =  ((ckey & 0x00ff0000) >> 16) |
				((ckey & 0x0000ff00) << 8)  |
				((ckey & 0x000000ff) << 8);
		}
	} else {
		ckey = 0;
	}
	/* Only allow one specific key value */
	p_context->ovl.color_key.src_lo = ckey;
	p_context->ovl.color_key.src_hi = ckey;

	/* Initialize the rendering surface attributes */
	p_context->surf.width = buf.src.x2 - buf.src.x1 + 1;
	p_context->surf.height = buf.src.y2 - buf.src.y1 + 1;
	p_context->surf.pitch = buf.stride;
	p_context->surf.pixel_format = (buf.plane == IGD_PLANE_OVERLAY_POPUP)
					? IGD_PF_xRGB32 : IGD_PF_ARGB32;
	p_context->surf.flags = IGD_OVL_ALTER_ON;
	p_context->surf.offset = buf.offset;

	/* Enable/disable Constant Alpha, as requested */
	p_context->ovl.color_key.flags |= IGD_OVL_CONST_ALPHA_ENABLE;
	p_context->ovl.alpha.enable = buf.alpha_ena;
	p_context->ovl.alpha.value = buf.alpha_val;

	EMGD_TRACE_EXIT;
	return ret;
}

/*
 * disable_affected_planes(): Walks the current buffer configuration for each
 * screen (stored in the external igd_buf_cfg array), and disables those planes
 * which are changing in the "new" buffer configuration.
 */
static int disable_affected_planes(
	igd_display_context_t *handles[2],
	igd_buffer_config_t old[2][3],
	igd_buffer_config_t new[2][3])
{
	igd_display_context_t *ctx;
	int ret = 0;
	int i, j;
	int new_ovl_popup=0, new_ovl2_popup=0, new_ovl_video=0, new_ovl2_video=0;

	EMGD_TRACE_ENTER;

	/* detect planes which will later be enabled, we don't want to
	 * disable these because the hardware will stuff a 1-frame
	 * delay between the register writes and we'll flicker */
	for (i = 0; i < 2; i++) {
		for (j = 0; j < 3 && new[i][j].plane != IGD_PLANE_NONE; j++) {
			switch (new[i][j].plane) {
			case IGD_PLANE_OVERLAY_POPUP:
				new_ovl_popup = 1;
				break;
			case IGD_PLANE_OVERLAY_VIDEO:
				new_ovl_video = 1;
				break;
			case IGD_PLANE_SPRITE_POPUP:
				new_ovl2_popup = 1;
				break;
			case IGD_PLANE_SPRITE_VIDEO:
				new_ovl2_video = 1;
				break;
			default:
				break;
			}
		}
	}

	/* scan the previous & new buffer configs for planes that are changing */
	for (i = 0; i < 2; i++) {
		/* skip this screen if the handle hasn't been set */
		if (!(ctx = handles[i])) {
			continue;
		}
		for (j = 0; j < 3 && old[i][j].plane != IGD_PLANE_NONE; j++) {
			if (old[i][j].plane != new[i][j].plane) {
				/* This plane is changing; disable it first */
				switch (old[i][j].plane) {
				case IGD_PLANE_HMI:
				case IGD_PLANE_X11:
					/* The A/B planes were already handled */
					break;

				case IGD_PLANE_OVERLAY_POPUP:
					/* Turn the Overlay plane off */
					if(!new_ovl_popup) {
						popup_context_ovl.surf.offset = old[i][j].offset;
						popup_context_ovl.surf.flags = IGD_OVL_ALTER_OFF;
						ret = alter_ovl_tnc(ctx, &popup_context_ovl.surf,
							&popup_context_ovl.bridge.src,
							&popup_context_ovl.bridge.dest,
							&popup_context_ovl.ovl,
							popup_context_ovl.surf.flags);
					}
					break;

				case IGD_PLANE_SPRITE_POPUP:
					/* Turn the Sprite C plane off */
					if(!new_ovl2_popup) {
						popup_context_ovl2.surf.offset = old[i][j].offset;
						popup_context_ovl2.surf.flags = IGD_OVL_ALTER_OFF;
						ret = alter_ovl2_tnc(ctx, &popup_context_ovl2.surf,
							&popup_context_ovl2.bridge.src,
							&popup_context_ovl2.bridge.dest,
							&popup_context_ovl2.ovl,
							popup_context_ovl2.surf.flags);
						break;
					}

				case IGD_PLANE_OVERLAY_VIDEO:
					/* Disable Direct Display on Overlay */
					if (!new_ovl_video && dd_context_ovl) {
						dd_context_ovl->surf.flags = IGD_OVL_ALTER_OFF;
						ret = alter_ovl_tnc(ctx, &dd_context_ovl->surf,
							&dd_context_ovl->bridge.src,
							&dd_context_ovl->bridge.dest,
							&dd_context_ovl->ovl,
							dd_context_ovl->surf.flags);
					}
					break;

				case IGD_PLANE_SPRITE_VIDEO:
					/* Disable Direct Display on the Sprite C plane */
					if (!new_ovl2_video && dd_context_ovl2) {
						dd_context_ovl2->surf.flags = IGD_OVL_ALTER_OFF;
						ret = alter_ovl2_tnc(ctx, &dd_context_ovl2->surf,
							&dd_context_ovl2->bridge.src,
							&dd_context_ovl2->bridge.dest,
							&dd_context_ovl2->ovl,
							dd_context_ovl2->surf.flags);
					}
					break;
				default:
					/* Continue disabling other planes */
					EMGD_ERROR_EXIT("Unknown plane type %d\n", old[i][j].plane);
					ret = -EINVAL;
					break;
				}
			}
		}
	}
	EMGD_TRACE_EXIT;
	return ret;
}


/*
 * enable_affected_planes(): Walks the "new" buffer configuration for each
 * screen, and enables the specified planes.
 */
static int enable_affected_planes(
	igd_display_context_t *ctx,
	int screen,
	igd_buffer_config_t new[3])
{
	int j;
	int ret = 0;

	EMGD_TRACE_ENTER;

	/* scan both "new" Z-stack arrays for planes that are changing */
	for (j = 0; j < 3 && new[j].plane != IGD_PLANE_NONE; j++) {
		/* This plane is in the new config; enable it */
		switch (new[j].plane) {
		case IGD_PLANE_HMI:
		case IGD_PLANE_X11:
			/* The A/B planes were already handled */
			break;

		case IGD_PLANE_OVERLAY_POPUP:
			/* Enable Direct Display on Overlay */
			ret = alter_ovl_tnc(ctx, &popup_context_ovl.surf,
					&popup_context_ovl.bridge.src,
					&popup_context_ovl.bridge.dest,
					&popup_context_ovl.ovl, popup_context_ovl.surf.flags);
			break;

		case IGD_PLANE_SPRITE_POPUP:
			/* Enable Direct Display on Sprite C */
			ret = alter_ovl2_tnc(ctx, &popup_context_ovl2.surf,
					&popup_context_ovl2.bridge.src,
					&popup_context_ovl2.bridge.dest,
					&popup_context_ovl2.ovl, popup_context_ovl2.surf.flags);
			break;

		case IGD_PLANE_OVERLAY_VIDEO:
			if (dd_context_ovl) {
				/* Update Overlay's DD context */
				dd_context_ovl->dpy = ctx;
				dd_context_ovl->bridge.screen = screen;
				dd_context_ovl->surf.flags = IGD_OVL_ALTER_ON;
				/* alter_ovl_tnc called when next frame arrives */
			}
			break;

		case IGD_PLANE_SPRITE_VIDEO:
			if (dd_context_ovl2) {
				/* Update Sprite C's DD context */
				dd_context_ovl2->dpy = ctx;
				dd_context_ovl2->bridge.screen = screen;
				dd_context_ovl2->surf.flags = IGD_OVL_ALTER_ON;
				/* alter_ovl2_tnc called when next frame arrives */
			}
			break;

		default:
			/* Continue processing other planes */
			EMGD_ERROR_EXIT("Unknown bottom plane (type %d) on screen %d\n",
				new[j].plane, screen);
			break;
		}
	}
	EMGD_TRACE_EXIT;
	return ret;
}


/*
 * adjust_z_stacks(): Walks the "new" buffer configuration for each screen, and
 * sets the relative Z-order for the specified planes.
 */
static int adjust_z_stacks(
	igd_display_context_t *ctx,
	int screen,
	igd_buffer_config_t new[3])
{
	int ret = 0;
	int popup_depth = 0;  /* Prevent two pop-ups on the same pipe */
	int stack_depth;

	EMGD_TRACE_ENTER;

	/* Determine the new Z-stack depth for this screen */
	for (stack_depth = 0; stack_depth < 3; stack_depth++) {
		if (new[stack_depth].plane == IGD_PLANE_NONE) {
			break;
		}
	}

	/* Adjust the HW attributes for Z-depth of the planes in this stack */
	/* Video, if enabled, is always on the bottom layer of the Z-stack */
	/* A Pop-up will always be on the top-most layer of the Z-stack */
	switch (stack_depth) {
	case 3:
		if ((new[2].plane == IGD_PLANE_OVERLAY_POPUP) ||
		    (new[2].plane == IGD_PLANE_SPRITE_POPUP)) {
			/* Pop-up on Overlay or Sprite C as top-most plane. */
			/* Create a pseudo-surface for this popup plane. */
			ret = create_popup_surface(ctx, screen, new[2]);
			popup_depth = 3;
		} else {
			EMGD_ERROR_EXIT("Depth 3 top plane (%d) not a pop-up !\n",
				new[2].plane);
		}
		/* Fall through to case 2... */
	case 2:
		if ((new[1].plane == IGD_PLANE_OVERLAY_POPUP) ||
		    (new[1].plane == IGD_PLANE_SPRITE_POPUP)) {
			if (popup_depth == 3) {
				EMGD_ERROR_EXIT("Already have a popup on pipe %d !\n", screen);
			} else {
				/* Create a pseudo-surface for this popup plane */
				ret = create_popup_surface(ctx, screen, new[1]);
				popup_depth = 2;
			}
		}
		/* Fall through to case 1... */
	case 1:
		switch (new[0].plane) {
		case IGD_PLANE_HMI:
		case IGD_PLANE_X11:
			/* Graphics plane visibility already adjusted */
			break;

		case IGD_PLANE_OVERLAY_POPUP:
		case IGD_PLANE_SPRITE_POPUP:
			/* A pop-up on the bottom layer... */
			if (popup_depth != 0) {
				EMGD_ERROR_EXIT("Already have a popup on pipe %d !\n", screen);
			} else {
				/* Create a pseudo-surface for this popup plane */
				ret = create_popup_surface(ctx, screen, new[0]);
				popup_depth = 1;
			}
			break;

		case IGD_PLANE_OVERLAY_VIDEO:
			if (dd_context_ovl) {
				/* Enable destination color key */
				dd_context_ovl->ovl.color_key.flags = IGD_OVL_DST_COLOR_KEY_ENABLE;
				dd_context_ovl->ovl.color_key.dest = 0xFF00FF;
			}
			break;

		case IGD_PLANE_SPRITE_VIDEO:
			if (dd_context_ovl2) {
				/* Enable destination color key */
				dd_context_ovl2->ovl.color_key.flags = IGD_OVL_DST_COLOR_KEY_ENABLE;
				dd_context_ovl2->ovl.color_key.dest = 0xFF00FF;
			}
			break;

		default:
			/* Continue processing the other planes */
			EMGD_ERROR_EXIT("Unknown plane (%d) on screen %d\n",
				new[0].plane, screen);
			break;
		}
		break;
	default:
		break;
	}
	EMGD_TRACE_EXIT;
	return ret;
}

int emgdhmi_set_ovl_planes(
	igd_display_context_t *ctx,
	int screen,
	igd_buffer_config_t buf_cfg[3])
{
	int err;

	/* If the handle hasn't been set, we have nothing to do */
	if (!ctx) {
		return 0;
	}
	/* Set the Z attributes and enable the planes on this pipe */
	if ((err = adjust_z_stacks(ctx, screen, buf_cfg))) {
		EMGD_ERROR_EXIT("adjust_z_stacks() failed %d !\n", err);
	} else if ((err = enable_affected_planes(ctx, screen, buf_cfg))) {
		EMGD_ERROR_EXIT("enable_affected_planes() failed %d !\n", err);
	}
	return err;
}

/*
 * configure_buffers_tnc(): Manages the changing of the "current" buffer
 * configuration for each screen (maintained externally by emgd_interface.c
 * in the igd_buf_cfg array) into the newly-specified buffer configuration,
 * and the corresponding display updates.
 */
static int configure_buffers_tnc(
	igd_display_h primary,
	igd_display_h secondary,
	igd_buffer_config_t buf_cfg[2][3])
{
	igd_display_context_t *pri_context = (igd_display_context_t *)primary;
	igd_display_context_t *sec_context = (igd_display_context_t *)secondary;
	igd_display_context_t *dpy_handles[2];
	platform_context_tnc_t *platform;
	int result;

	EMGD_TRACE_ENTER;

	/* Note the entrance to the alternative flip architecture */
	emgdhmi_enable();

	/* Check the display handles that were sent down to us */
	if (!primary && !secondary) {
		EMGD_ERROR_EXIT("configure_buffers_tnc: no valid handle !\n");
		EMGD_TRACE_EXIT;
		return -EINVAL;
	}
	/* Direct Display should be suspended until buffer reconfiguration completes */
	platform = (platform_context_tnc_t *)pri_context->context->platform_context;
	result = OS_PTHREAD_MUTEX_LOCK(&pri_context->context->device_context.z_stack_mutex);
	pri_context->context->device_context.dd_suspended = 1;

	/* Show the configuration of the previous state */
	EMGD_DEBUG("configure_buffers_tnc: prev pri: bot=%d, mid=%d, top=%d\n",
		igd_buf_cfg[0][0].plane, igd_buf_cfg[0][1].plane, igd_buf_cfg[0][2].plane);
	EMGD_DEBUG("configure_buffers_tnc: prev sec: bot=%d, mid=%d, top=%d\n",
		igd_buf_cfg[1][0].plane, igd_buf_cfg[1][1].plane, igd_buf_cfg[1][2].plane);

	/* Show the requested buffer configuration */
	EMGD_DEBUG("configure_buffers_tnc: new pri: bot=%d, mid=%d, top=%d\n",
		buf_cfg[0][0].plane, buf_cfg[0][1].plane, buf_cfg[0][2].plane);
	EMGD_DEBUG("configure_buffers_tnc: new sec: bot=%d, mid=%d, top=%d\n",
		buf_cfg[1][0].plane, buf_cfg[1][1].plane, buf_cfg[1][2].plane);

	/* Planes changing across pipes must be disabled up front */
	dpy_handles[0] = primary;
	dpy_handles[1] = secondary;
	if ((result = disable_affected_planes(dpy_handles, igd_buf_cfg, buf_cfg))) {
		EMGD_ERROR_EXIT("disable_affected_planes() failed %d !\n", result);
		pri_context->context->device_context.dd_suspended = 0;
		OS_PTHREAD_MUTEX_UNLOCK(&pri_context->context->device_context.z_stack_mutex);
		return result;
	}
	/* Prepare the new buffer configuration to be enabled */
	emgdhmi_set_buf_cfg(pri_context, sec_context, buf_cfg);
	result = emgdhmi_wait_flips();

	/* Okay to resume direct display updates now */
	pri_context->context->device_context.dd_suspended = 0;
	OS_PTHREAD_MUTEX_UNLOCK(&pri_context->context->device_context.z_stack_mutex);

	EMGD_TRACE_EXIT;
	return result;
}


