/**********************************************************************
 * Copyright (c) 2011, Intel Corporation.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 ******************************************************************************/

/* The EMGDHMI vertical extended mode with second framebuffer requires
 * a custom flip architecture to manage async flips across the two
 * physical displays, and to properly synchronize the sprite/overlay
 * plane flips with the color buffer flips via the
 * emgdHmiConfigureBuffers API */

#include <linux/spinlock.h>
#include <linux/wait.h>
#include <linux/list.h>

#include "drm_emgd_private.h"
#include "img_defs.h"
#include "servicesext.h"
#include "kerneldisplay.h"

#include "emgd_dc.h"
#include "emgdhmi_2fb.h"

static DEFINE_SPINLOCK(bufcfg_lock);
static DECLARE_WAIT_QUEUE_HEAD(flip_waitq);
static int check_swaps[2];
static int flip_result;
static int flip_finished; /* condition for the wait queue */
static igd_buffer_config_t next_bufcfg[2][3]; /* [screen][stack_idx] */
static igd_display_context_t *ovl_ctx[2]; /* [screen] */
static igd_context_t *next_fb_ctx;
static igd_surface_t next_fb_surfs[2]; /* [owner] */
static igd_display_h next_fb_displays[2][2]; /* [owner][screen] */
static int last_owners[2];
static int emgdhmi_active;
static int secondary_active;

/* PVR defines a command complete callback that we must emit from the
 * ISR.  Handling is a little complicated because we can't free it
 * there, so there's a "finished" queue that gets purged at the end of
 * wait_flips(). */
static LIST_HEAD(pvr_queued);
static LIST_HEAD(pvr_finished);
typedef struct {
	PVRSRV_DC_DISP2SRV_KMJTABLE *funcs;
	IMG_HANDLE cookie;
	struct list_head list;
} pvr_flip_t;

void emgdhmi_set_buf_cfg(igd_display_context_t *pri, igd_display_context_t *sec,
			 igd_buffer_config_t bc[2][3])
{
	unsigned long flags;
	int err, s;

	emgdhmi_enable();

	spin_lock_irqsave(&bufcfg_lock, flags);
	ovl_ctx[0] = pri;
	ovl_ctx[1] = sec;
	memcpy(next_bufcfg, bc, 2*3*sizeof(bc[0][0]));
	check_swaps[0] = 1;
	check_swaps[1] = secondary_active;
	flip_finished = 0;

        /* Set up the sprite/overlay planes: */
	for (s = 0; s < 2; s++) {
        	err = emgdhmi_set_ovl_planes(ovl_ctx[s], s, next_bufcfg[s]);
        	if(err) {
                	flip_result = err;
        	}
	}
	spin_unlock_irqrestore(&bufcfg_lock, flags);
}

/* May be called from ISR context.  Assumes bufcfg_lock is held. */
int emgdhmi_flip_framebuffer(int screen)
{
	int i, o, err, result = 0, owners[2] = {0,0};
	igd_dispatch_t *d = &next_fb_ctx->dispatch;

	for(i=0; i<3; i++)
		owners[screen] |= (next_bufcfg[screen][i].plane == IGD_PLANE_X11);

	o = owners[screen];
	if(next_fb_displays[o][screen]) {
		err = d->set_surface(next_fb_displays[o][screen], IGD_PRIORITY_NORMAL,
				     IGD_BUFFER_DISPLAY, &next_fb_surfs[o], NULL, 0);
		if(err) {
			printk(KERN_ERR "Error %d in set_surface for screen %d (%s)\n",
				   err, screen, o ? "HMI" : "X11");
			result = err;
		}
	}
	last_owners[screen] = o;
	return result;
}

void emgdhmi_set_framebuffer(int owner, emgddc_buffer_t *buf, int sync,
                             PVRSRV_DC_DISP2SRV_KMJTABLE *funcs,
                             IMG_HANDLE cookie)
{
	unsigned long flags;
	drm_emgd_private *priv = buf->priv;
	igd_display_h *dh;
	pvr_flip_t *flip, *tmp;

	spin_lock_irqsave(&bufcfg_lock, flags);

	memset(&next_fb_surfs[owner], 0, sizeof(igd_surface_t));
	next_fb_surfs[owner].offset = buf->offset;
	next_fb_surfs[owner].pitch = buf->pitch;
	next_fb_surfs[owner].width = buf->width;
	next_fb_surfs[owner].height = buf->height;
	next_fb_surfs[owner].pixel_format = buf->pixel_format;
	next_fb_surfs[owner].flags = IGD_SURFACE_RENDER | IGD_SURFACE_DISPLAY;

	next_fb_ctx = priv->context;

	dh = (igd_display_h*)next_fb_ctx->mod_dispatch.dsp_display_list;
	next_fb_displays[owner][0] = dh[IGD_DC_PRIMARY(priv->dc)];
	next_fb_displays[owner][1] = dh[IGD_DC_SECONDARY(priv->dc)];

	/* Purge any finished flips */
	list_for_each_entry_safe(flip, tmp, &pvr_finished, list) {
		list_del(&flip->list);
		OS_FREE(flip);
	}

	if(funcs) {
		flip = OS_ALLOC(sizeof(*flip));
		flip->funcs = funcs;
		flip->cookie = cookie;
		INIT_LIST_HEAD(&flip->list);
		list_add(&flip->list, &pvr_queued);
	}

	/* Always do the flips, the hardware buffers the actual
	 * pointer set to the next vblank.  The "sync" argument
	 * controls whether or not we need to set up the wait queue
	 * for a later emgd_wait_flips() call */
	emgdhmi_flip_framebuffer(0);
	emgdhmi_flip_framebuffer(1);
	flip_result = 0;

	if(!sync) {
		check_swaps[0] = 1;
		check_swaps[1] = secondary_active;
		flip_finished = 0;
	}

	spin_unlock_irqrestore(&bufcfg_lock, flags);
}

/* Runs in interrupt context */
void emgdhmi_vsync(int screen)
{
	spin_lock(&bufcfg_lock);

	if(!check_swaps[screen]) {
		spin_unlock(&bufcfg_lock);
		return;
	}
	check_swaps[screen] = 0;
	if(!check_swaps[!screen]) {
		pvr_flip_t *flip, *tmp;
		list_for_each_entry_safe(flip, tmp, &pvr_queued, list) {
			flip->funcs->pfnPVRSRVCmdComplete(flip->cookie, IMG_TRUE);
			list_move(&flip->list, &pvr_finished);
		}

		flip_finished = 1;
		wake_up(&flip_waitq);
	}

	spin_unlock(&bufcfg_lock);
}

int emgdhmi_wait_flips(void)
{
	pvr_flip_t *flip, *tmp;
	unsigned long flags;
	int result = wait_event_interruptible(flip_waitq, flip_finished);
	if(!result)
		result = flip_result;

	spin_lock_irqsave(&bufcfg_lock, flags);
	list_for_each_entry_safe(flip, tmp, &pvr_finished, list) {
		list_del(&flip->list);
		OS_FREE(flip);
	}
	spin_unlock_irqrestore(&bufcfg_lock, flags);

	return result;
}

/* vblank handler for the 2nd display in vertext mode */
int emgdhmi_secondary_vblank(void* pdevinfo)
{
	emgdhmi_vsync(1);
	return 1;
}

void emgdhmi_set_active()
{
	emgdhmi_active = 1;
}

int emgdhmi_mode()
{
	return emgdhmi_active;
}

void emgdhmi_enable_secondary()
{
	secondary_active = 1;
}
