/* security/kyocera/kclsm_sysfs.c  (kclsm LSM module sysfs interface)
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2017 KYOCERA Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <linux/kobject.h>
#include <linux/string.h>
#include <linux/sysfs.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/stddef.h>

#include <linux/kclsm_sysfs.h>
#include "kclsm.h"

#define	KC_DNA_STRT_DEAMON	"userapp_initd"
#define	KC_DNA_PID_MAX_NUM	256UL
#define KC_DNA_STORAGE_PATH		"/vendor/data/a"
#define KC_DNA_STORAGE_DIRECTRY		"/vendor/data/a/"
#define KC_DNA_PROGRAM_PATH		"/vendor/app/etc/a"
#define KC_DNA_PROGRAM_DIRECTRY		"/vendor/app/etc/a/"

uint32_t dna_daemon_pid;
struct dna_apps dna_apps_pids;
static struct kobject *kclsm_kobj;

static bool is_dna_daemon_pid_initialized(void)
{
	if (dna_daemon_pid)
		return true;
	else
		return false;
}

static int save_dna_daemon_pid(pid_t pid)
{
	dna_daemon_pid = pid;
	return 0;
}

static bool validate_parent(struct task_struct *task_p)
{
	pr_debug("called %s real_parent_pid:%d\n", __func__, task_p->real_parent->pid);
	if (task_p->real_parent->pid == 1)
		return true;
	else
		return false;
}

static bool validate_cmdline(struct task_struct *task_p)
{
	pr_debug("called %s cmdline:%s\n", __func__, task_p->comm);
	if (strcmp(task_p->comm, KC_DNA_STRT_DEAMON) == 0)
		return true;
	else
		return false;
}

static ssize_t dna_daemon_show(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf)
{
	pr_debug("called %s\n", __func__);

	return sprintf(buf, "%d\n", dna_daemon_pid);
}

static ssize_t dna_daemon_store(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count)
{
	uint32_t pid;

	pr_debug("called %s set_pid:%s\n", __func__, buf);

	if (is_dna_daemon_pid_initialized())
		return -EINVAL;

	if (validate_parent(current) == false)
		return -EINVAL;

	if (validate_cmdline(current) == false)
		return -EINVAL;

	if (sscanf(buf, "%du", &pid) != 1)
		return -EINVAL;

	save_dna_daemon_pid((pid_t)pid);

	return count;
}

static struct kobj_attribute kclsm_dna_daemon_attribute = __ATTR(dna_daemon,
					S_IRUSR | S_IWUSR, dna_daemon_show, dna_daemon_store);

static bool is_dna_apps_full(void)
{
	struct dna_apps *list_p;
	uint32_t count = 0;

	list_for_each_entry(list_p, &dna_apps_pids.list, list) {
		++count;
	}

	if (count < KC_DNA_PID_MAX_NUM)
		return false;
	else
		return true;
}

struct dna_apps *find_stored_dna_app(pid_t pid)
{
	struct dna_apps *list_p;

	list_for_each_entry(list_p, &dna_apps_pids.list, list) {
		if (list_p->pid == pid)
			return list_p;
	}

	return NULL;
}

static int add_dna_app(pid_t pid)
{
	struct dna_apps* new_data;

	new_data = kmalloc(sizeof(struct dna_apps), GFP_KERNEL);
	if (!new_data)
		return -ENOMEM;

	new_data->pid = pid;
	list_add(&new_data->list, &dna_apps_pids.list);

	return 0;
}

static int del_dna_app(struct dna_apps* list_p)
{
	list_del(&list_p->list);
	kfree(list_p);

	return 0;
}

bool is_target_path(const char* path)
{
	pr_debug("%s input_path=%s\n",__func__,path);
	if (strcmp(path, KC_DNA_STORAGE_PATH) == 0 ||
		strcmp(path, KC_DNA_PROGRAM_PATH) == 0)
		return true;

	if (strncmp(path, KC_DNA_STORAGE_DIRECTRY,
				strlen(KC_DNA_STORAGE_DIRECTRY)) == 0 ||
		strncmp(path, KC_DNA_PROGRAM_DIRECTRY,
				strlen(KC_DNA_PROGRAM_DIRECTRY)) == 0)
		return true;

	return false;
}

static ssize_t dna_apps_show(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf)
{
	char* setbuf = buf;
	struct dna_apps *list_p;

	pr_debug("called %s\n", __func__);

	list_for_each_entry(list_p, &dna_apps_pids.list, list) {
		setbuf += sprintf(setbuf, "%d\n", list_p->pid);
	}

	return (setbuf - buf);
}

static ssize_t dna_apps_store(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf, size_t count)
{
	uint32_t pid;
	struct dna_apps* list_p;
	int add_result;

	pr_debug("called %s set_pid:%s current_pid:%d\n", __func__, buf, current->pid);

	if (! is_dna_daemon_pid_initialized())
		return -EINVAL;

	if (current->pid != dna_daemon_pid)
		return -EINVAL;

	if (is_dna_apps_full())
		return -ENOMEM;

	if (sscanf(buf, "%du", &pid) != 1)
		return -EINVAL;

	list_p = find_stored_dna_app((pid_t)pid);
	if (list_p != NULL) {
		del_dna_app(list_p);
		return count;
	}

	add_result = add_dna_app((pid_t)pid);
	if (add_result == 0)
		return count;

	return add_result;
}

static ssize_t enable_show(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf)
{
	return sprintf(buf, "%d\n", get_kclsm_enable());
}

static ssize_t deny_uart2access_show(struct kobject *kobj,
		struct kobj_attribute *attr, const char *buf)
{
	return sprintf(buf, "%d\n", get_deny_uart2access());
}

static struct kobj_attribute kclsm_dna_apps_attribute =  __ATTR(dna_apps,
					S_IRUSR | S_IWUSR, dna_apps_show, dna_apps_store);
static struct kobj_attribute kclsm_enable_attribute =  __ATTR(kclsm_enable,
					S_IRUSR | S_IRGRP | S_IROTH, enable_show, NULL);
static struct kobj_attribute kclsm_deny_uart2access_attribute =  __ATTR(deny_uart2access,
					S_IRUSR | S_IRGRP | S_IROTH, deny_uart2access_show, NULL);

static struct attribute *kclsm_attrs[] = {
	&kclsm_dna_daemon_attribute.attr,
	&kclsm_dna_apps_attribute.attr,
	&kclsm_enable_attribute.attr,
	&kclsm_deny_uart2access_attribute.attr,
	NULL,
};

static struct attribute_group kclsm_attr_group = {
	.attrs = kclsm_attrs,
};

static int __init kclsm_sysfs_init(void)
{
	int retval;

	kclsm_kobj = kobject_create_and_add("kclsm", kernel_kobj);
	if (!kclsm_kobj) {
		pr_err("create kclsm failed\n");
		return -ENOMEM;
	}

	retval = sysfs_create_group(kclsm_kobj, &kclsm_attr_group);
	if (retval == 0) {
		pr_info("kclsm_sysfs initialized\n");
		INIT_LIST_HEAD(&dna_apps_pids.list);
	} else {
		pr_err("kclsm_sysfs init failed\n");
		kobject_put(kclsm_kobj);
	}

	return retval;
}

static void __exit kclsm_sysfs_exit(void)
{
	sysfs_remove_group(kclsm_kobj, &kclsm_attr_group);
	kobject_put(kclsm_kobj);
}

bool kclsm_sysfs_is_dna_ready(void)
{
	return is_dna_daemon_pid_initialized();
}

module_init(kclsm_sysfs_init);
module_exit(kclsm_sysfs_exit);
MODULE_LICENSE("GPL");
